///
/// Send messages of various kinds to clients.
/// @file       response.h - pianod project
/// @author     Perette Barella
/// @date       Initial: 2012-03-16.  C++ Rework: 2014-10-27.
/// @copyright  Copyright 2012–2016 Devious Fish. All rights reserved.
///

#include <config.h>

#include <string>
#include <vector>

#include <football.h>
#include "fundamentals.h"
#include "musictypes.h"
#include "connection.h"

#ifndef _RESPONSE_H
#define _RESPONSE_H

/* Log levels 0x01, 0x02, 0x04, 0x08, 0x10, and 0x20 correspond
 to statuses 0xx, 1xx,  2xx,  3xx,  4xx, and 5xx. */
inline LOG_TYPE loglevel_of (RESPONSE_CODE response) {
    int level = ((int) response) / 100;
    return (LOG_TYPE) (1 << ((level < 0 || level > 5) ? 5 : level));
}

extern void sendcflog (LOG_TYPE loglevel, Football::Thingie *there, const char *format, ...);

/// Simple type used to attach data to a response code.
class Response {
    friend Football::Thingie &operator<<(Football::Thingie &there, const Response &response);
private:
    bool empty = false; ///< True if there is no value in this response.
    RESPONSE_CODE message = RESPONSE_CODE (0); ///< Message code, initialized to "empty" value.
    std::string value;
public:
    Response (RESPONSE_CODE msg, std::string det);
    Response (RESPONSE_CODE msg, const char * det);
    Response (RESPONSE_CODE msg, long det);
    Response (RESPONSE_CODE msg, double det, int precision);
    Response (RESPONSE_CODE msg);
};


extern const char *ResponseText (RESPONSE_CODE code);

template<typename Output>
inline Output &operator<<(Output &there, const RESPONSE_CODE status) {
    sendcflog (loglevel_of (status), &there, "%03d %s\n", status, ResponseText (status));
    return there;
}
template<typename Output>
inline Output *operator<<(Output *there, const RESPONSE_CODE status) {
    sendcflog (loglevel_of (status), there, "%03d %s\n", status, ResponseText (status));
    return there;
}
extern Football::Thingie &operator<<(Football::Thingie &there, const Response &response);
inline Football::Thingie &operator<<(Football::Thingie *there, const Response &response) {
    return (*there << response);
}




/// A response aggregator.
class PartialResponse {
    friend PianodConnection &operator<<(PianodConnection &there, const PartialResponse &response);
public:
    enum aggregate_type { pessimistic, optimistic };
private:
    std::vector<Response> diagnostics;
    const RESPONSE_CODE mixed_reason;
    const RESPONSE_CODE action_message;
    std::string success_item;

    unsigned short failures = 0;
    unsigned short successes = 0;
    RESPONSE_CODE reason = S_NOOP;
public:
    inline bool allSuccess () const { return !failures && successes ; };
    inline bool allfailure () const { return failures && !successes; }
    inline bool anySuccess () const { return successes; };
    inline bool anyfailure () const { return failures; }
    inline bool partial () const { return failures && successes; }
    inline bool noop () const { return !failures && !successes; }
    inline bool multiple () const { return (failures + successes) > 0; }
    inline bool single () const { return (failures + successes) == 1; }
    RESPONSE_CODE operator()() const { return reason; }

    void fail (RESPONSE_CODE r);
    void fail (RESPONSE_CODE r, const MusicThingie *regarding);
    void succeed (RESPONSE_CODE = S_OK, const MusicThingie *regarding = nullptr);
    inline void succeed (const MusicThingie *regarding) {
        succeed (S_OK, regarding);
    };
    void operator()(const RESPONSE_CODE r,
                    const MusicThingie *regarding);

    PartialResponse (aggregate_type kind = optimistic,
                     RESPONSE_CODE action = E_NOT_IMPLEMENTED,
                     int expected_responses = 0)
    : mixed_reason (kind == optimistic ? S_PARTIAL : E_PARTIAL),
    action_message (action) {
        if (expected_responses) {
            diagnostics.reserve (expected_responses);
        }
    }
};

extern PianodConnection &operator<<(PianodConnection &there, const PartialResponse &response);
inline PianodConnection &operator<<(PianodConnection *there, const PartialResponse &response) {
    return (*there << response);
}




inline Football::Thingie &operator<<(Football::Thingie &there, const CommandError &error) {
    if (*(error.what())) {
        sendcflog (loglevel_of (error.reason()), &there,
                   "%03d %s: %s\n", error.reason(), ResponseText (error.reason()),  error.what());
    } else {
        sendcflog (loglevel_of (error.reason()), &there,
                   "%03d %s\n", error.reason(), ResponseText (error.reason()));
    }
    return there;
}
inline Football::Thingie &operator<<(Football::Thingie *there, const CommandError &error) {
    return (*there << error);
}


// Send any music thing to a connection
PianodConnection &operator<<(PianodConnection &there, const MusicThingie *thing);
inline PianodConnection &operator<<(PianodConnection &there, const MusicThingie &thing) {
    return (there << &thing);
};
inline PianodConnection &operator<<(PianodConnection *there, const MusicThingie &thing) {
    return (*there << &thing);
};

// Send song support
void sendRatings (PianodConnection *conn,
                  const PianodSong &song);
void sendSong (Football::Thingie &there, const PianodSong &song);

void sendUpdatedRatings (PianodConnection &conn, const PianodSong *song);

// Allow sending songs to the service
PianodService &operator<<(PianodService &there, const PianodSong &song);
inline PianodService &operator<<(PianodService &there, const PianodSong *song) {
    return there << *song;
}
inline PianodService &operator<<(PianodService *there, const PianodSong &song) {
    return *there << song;
}

// Send playlists
void sendRatings (PianodConnection &conn,
                  const PianodPlaylist &playlist);

// Send song lists
PianodConnection &operator<<(PianodConnection &there, const SongList &songs);
inline PianodConnection &operator<<(PianodConnection *there, const SongList &songs) {
    return (*there << songs);
}

// Send playlist lists
PianodConnection &operator<<(PianodConnection &there, const PlaylistList &playlists);
inline PianodConnection &operator<<(PianodConnection *there, const PlaylistList &playlists) {
    return (*there << playlists);
}


// Send thingie lists
PianodConnection &operator<<(PianodConnection &there, const ThingieList &things);
inline PianodConnection &operator<<(PianodConnection *there, const ThingieList &things) {
    return (*there << things);
}
inline PianodConnection &operator<<(PianodConnection &there, const ThingieList *things) {
    return (there << *things);
}
PianodConnection &sendSeedlist (PianodConnection &there,
                                const ThingieList &songs,
                                PianodPlaylist *playlist);

#endif // _RESPONSE_H_ 
