///
/// Send messages of various kinds to clients.
/// @file       response.cpp - pianod project
/// @author     Perette Barella
/// @date       Initial: 2012-03-16.  C++ Rework: 2014-10-27.
/// @copyright  Copyright 2012–2017 Devious Fish. All rights reserved.
///

#include <config.h>

#include <cstdarg>
#include <cassert>

#include <sstream>
#include <iomanip>

#include <fb_public.h>

#include "logging.h"
#include "response.h"
#include "fundamentals.h"
#include "user.h"
#include "mediaunit.h"

using namespace std;


void sendcflog (LOG_TYPE loglevel, Football::Thingie *there, const char *format, ...) {
	va_list parameters;
	va_start(parameters, format);
#warning Cflogging was removed... uh...
	// vcflog (loglevel, format, parameters);
	va_end(parameters);
	// Need to reinitialize the var args data as it's modified by use 
	va_start(parameters, format);
	there->vprintf (format, parameters);
	va_end(parameters);
}



const char *ResponseText (RESPONSE_CODE response) {
    if (response >= D_DIAGNOSTICS && response <= D_DIAGNOSTICS_END)
        response = RESPONSE_CODE (response + 100);
	switch (response) {
		case V_PLAYING:			return "Playing";
        case V_STALLED:			return "Stalled";
		case V_PAUSED:			return "Paused";
        case V_BETWEEN_TRACKS:  return "Intertrack";
		case V_IDLE:            return "Idle";
        case V_TRACK_COMPLETE:	return "Track playback complete";

        case V_QUEUE_STOPPED:	return "Stopped";
        case V_QUEUE_REQUEST:   return "Request";
        case V_QUEUE_RANDOM:    return "Random Play";

        case V_SELECTEDSOURCE:  return "SelectedSource";
		case V_SELECTEDPLAYLIST:return "SelectedPlaylist";

        case V_MIX_CHANGED:		return "Mix has been changed";
        case V_PLAYLISTS_CHANGED:return "Playlist list has changed";
        case V_PLAYLISTRATING_CHANGED:
            return "PlaylistRatingChanged";
        case V_SOURCES_CHANGED:
            return "Available sources have changed";
        case V_SONGRATING_CHANGED:
            return "SongRatingChanged";
        case V_QUEUE_CHANGED:
            return "QueueChanged";
        case V_YELL:			return "says";
        case V_SOURCE_STATUS:   return "Status";
        case V_SERVER_STATUS:   return "Status";
        case V_USERACTION:      assert (0);
                                return "UserAction";
        case V_SELECTIONMETHOD: return "QueueRandomize";

        case I_WELCOME:			return "pianod2 " VERSION ". Welcome!";
        case I_ID:				return "ID";
		case I_ALBUM:			return "Album";
		case I_ARTIST:			return "Artist";
		case I_SONG:			return "Title";
		case I_PLAYLIST:		return "Playlist";
		case I_RATING:			return "Rating";
		case I_COVERART:		return "CoverArt";
		case I_GENRE:			return "Genre";
		case I_PLAYLISTRATING:	return "PlaylistRating";
		case I_CHOICEEXPLANATION:
								return "Explanation";
        case I_OWNER:           return "Owner";
        case I_SOURCE:          return "Source";
        case I_NAME:            return "Name";
        case I_YEAR:            return "Year";
        case I_DURATION:        return "Duration";
        case I_ACTIONS:         return "Actions";

        case I_ROOM:            return "Room";

		case I_VOLUME:			return "Volume";
		case I_AUDIOQUALITY:	return "Quality";
		case I_HISTORYSIZE:		return "HistoryLength";
		case I_AUTOTUNE_MODE:	return "AutotuneMode";
		case I_PAUSE_TIMEOUT:	return "PauseTimeout";
		case I_PLAYLIST_TIMEOUT:return "PlaylistTimeout";
		case I_PROXY:			return "Proxy";
		case I_CONTROLPROXY:	return "ControlProxy";
		case I_PANDORA_USER:	return "PandoraUser";
		case I_PANDORA_PASSWORD:return "PandoraPassword";
        case I_CACHE_MINIMUM:   return "CacheMinimum";
        case I_CACHE_MAXIMUM:   return "CacheMaximum";
		case I_OUTPUT_DRIVER:	return "OutputDriver";
		case I_OUTPUT_DEVICE:	return "OutputDevice";
		case I_OUTPUT_ID:		return "OutputID";
		case I_OUTPUT_SERVER:	return "OutputServer";
		case I_INFO_URL:		return "SeeAlso";
		case I_USER_PRIVILEGES:	return "Privileges";
        case I_INFO:            return "Information";
		case S_OK:				return "Success";
		case S_ANSWER_YES:		return "True, yes, 1, on";
		case S_ANSWER_NO:		return "False, no, 0, off";
		case S_DATA:			return "Data";
		case S_DATA_END:		return "No data or end of data";
		case S_SIGNOFF:			return "Good-bye";
        case S_NOOP:            return "Doing nothing was successful";
        case S_PARTIAL:         return "Partial success";
        case S_MATCH:           return "Matches";
        case S_ROUNDING:        return "Success, but value was approximated";
        case S_PENDING:         return "Success pending";

		case E_BAD_COMMAND:		return "Bad command";
		case E_UNAUTHORIZED:	return "Not authorized for requested action";
		case E_NAK:				return "Action failed";
		case E_DUPLICATE:		return "Already exists";
		case E_NOTFOUND:		return "Requested item not found";
		case E_WRONG_STATE:		return "Action is not applicable to current state";
		case E_CREDENTIALS:		return "Invalid login or password";
		case E_REQUESTPENDING:	return "Temporary failure, future completion unknown";
		case E_INVALID:			return "Invalid parameter";
		case E_TRANSFORM_FAILED:return "Playlist personalization failed";
		case E_QUOTA:			return "Quota exceeded";
		case E_LOGINREQUIRED:	return "Must be logged in";
        case E_UNSUPPORTED:     return "Not supported";
		case E_CONFLICT:		return "Conflict encountered";
        case E_RESOURCE:        return "Insufficient resources";
        case E_RANGE:           return "Limit has been reached";
        case E_WRONGTYPE:       return "Wrong type for action";
        case E_PERSISTENT:      return "Persistent expression or value required";
        case E_AMBIGUOUS:       return "Ambiguous expression";
        case E_PARTIAL:         return "Partial failure";
        case E_VARIOUS:         return "Assorted failures";
        case E_NO_ASSOCIATION:  return "No association";
        case E_TYPE_DISALLOWED: return "Cannot specify type with expression form";
        case E_EXPRESSION:      return "Syntax error in expression";
        case E_PLAYLIST_REQUIRED:
                                return "Song is not associated with a playlist";

        case E_TIMEOUT:         return "Timeout";
        case E_METAPLAYLIST:    return "Operation only valid on primary playlists";
        case E_MEDIA_ACTION:    return "Action not supported by source";
        case E_MEDIA_VALUE:     return "Value is not supported by source";
        case E_MEDIA_MANAGER:   return "Action not possible on media manager";
        case E_MEDIA_FAILURE:   return "Source failed to execute a request";
        case E_MEDIA_TRANSIENT: return "Action not supported on transient playlist";
        case E_BUG:             return "There is a bug in " PACKAGE;
		case E_NOT_IMPLEMENTED:	return "Not implemented";

        case F_FAILURE:			return "Internal server error";
        case F_PLAYER_EMPTY:    return "Nothing to play";
		case F_NETWORK_FAILURE:	return "Network failure ";
		case F_SHUTDOWN:		return "Service shutting down";
		case F_AUTHENTICATION:	return "Authentication failure";
		case F_RESOURCE:		return "Insufficent resources";
		case F_PANDORA:			return "Error communicating with Pandora";
		case F_INCOMPLETE:		return "Command execution incomplete";
        case F_PERMISSION:      return "Permission denied";
        case F_EXCEPTION:       return "Exception";
        case F_NETWORK_TIMEOUT: return "Network timeout";
        case F_CANNOT_OUTPUT:   return "Cannot open audio output";
        case F_AUDIO_FAILURE:   return "Audio failure";


		case A_SIGNED_IN:		return "signed in";
		case A_SIGNED_OUT:		return "has disconnected";
		case A_KICKED:			return "kicked";
        case A_IMBECILE:        return "executed an imbecilic command";
		case A_SKIPPED:			return "skipped the song";
		case A_STOPPED:			return "requested stop";
        case A_REQUESTS:        return "set requests-only mode";
        case A_RANDOMPLAY:      return "enabled random play";
		case A_PAUSED:			return "paused playback";
		case A_RESUMED:			return "resumed playback";
		case A_CHANGED_MIX:		return "changed the mix";
		case A_MIX_ADDED:		return "added to the mix";
		case A_MIX_REMOVED:		return "removed from the mix";
		case A_SELECTED_PLAYLIST:return "selected the playlist";
		case A_CREATED_PLAYLIST:return "created the playlist";
		case A_RENAMED_PLAYLIST:return "renamed the playlist";
		case A_DELETED_PLAYLIST:return "deleted the playlist";
        case A_SOURCE_ADD:      return "added a source";
        case A_SOURCE_BORROW:   return "borrowed a source";
        case A_SOURCE_REMOVE:   return "removed a source";
        case A_REQUEST_ADD:     return "requested music";
        case A_REQUEST_CANCEL:  return "cancelled requests";
        case A_REQUEST_CLEAR:   return "cleared requests";
        case A_SHUTDOWN:        return "initiated player shutdown";

        case D_DIAGNOSTICS:
        case D_NOTFOUND:
        case D_DIAGNOSTICS_END:
            assert (!"Switching on diagnostics-range status");
	}
	assert (!"Unfound status in ResponseText() switch");
	flog (LOG_WHERE (LOG_ERROR), "Unknown status ", (int) response);
	return "Unknown status";
}




Response::Response (RESPONSE_CODE msg, std::string det) {
    message = msg;
    value = det;
}
Response::Response (RESPONSE_CODE msg, const char * det) {
    message = msg;
    value = det;
}
Response::Response (RESPONSE_CODE msg, long det) {
    message = msg;
    std::ostringstream str;
    str << det;
    value = str.str();
}
Response::Response (RESPONSE_CODE msg, double det, int precision) {
    message = msg;
    std::ostringstream str;
    str << std::setprecision (precision) << std::fixed << det;
    value = str.str();
}
Response::Response (RESPONSE_CODE msg) {
    empty = true;
    message = msg;
}

Football::Thingie &operator<<(Football::Thingie &there, const Response &response) {
    assert (response.message != RESPONSE_CODE (0));
    if (response.empty) {
        return there << response.message;
    }
    sendcflog (loglevel_of (response.message), &there,
               "%03d %s: %s\n", response.message, ResponseText (response.message), response.value.c_str());
    return there;
};


/*
 *                  Response aggregation
 */

void PartialResponse::fail (RESPONSE_CODE r) {
    diagnostics.push_back (Response {RESPONSE_CODE (r - 100) } );
    if (noop())
        reason = r;
    else if (successes)
        reason = mixed_reason;
    else if (r != reason)
        reason = E_VARIOUS;
    failures++;
};


void PartialResponse::fail (RESPONSE_CODE r, const MusicThingie *regarding) {
    diagnostics.push_back (Response { RESPONSE_CODE (r - 100), regarding->id() } );
    if (noop())
        reason = r;
    else if (successes)
        reason = mixed_reason;
    else if (r != reason)
        reason = E_VARIOUS;
    failures++;
};


void PartialResponse::succeed (RESPONSE_CODE r, const MusicThingie *regarding) {
    if (noop()) {
        reason = r;
    } else if (failures) {
        reason = mixed_reason;
    } else {
        assert (reason == r);
    }
    successes++;
    if (successes == 1 && regarding) {
        success_item = regarding->name();
    }
};


void PartialResponse::operator()(const RESPONSE_CODE r,
                                 const MusicThingie *regarding) {
    if (isSuccess(r))
        succeed (r, regarding);
    else
        fail (r, regarding);
};


PianodConnection &operator<<(PianodConnection &there, const PartialResponse &response) {
    if (response.noop()) {
        there << S_NOOP;
    } else {
        if (response.reason == S_PARTIAL || response.reason == E_PARTIAL || response.reason == E_VARIOUS) {
            for (const auto &diag : response.diagnostics) {
                there << diag;
            }
        }
        if (response.reason == S_OK) {
            there << Response (S_MATCH, response.successes);
        } else {
            there << response();
        }
        if (response.anySuccess() && response.action_message != E_NOT_IMPLEMENTED) {
            if (response.successes == 1 && !response.success_item.empty()) {
                there.announce(response.action_message, response.success_item);
            } else {
                there.announce(response.action_message);
            }
        }
    }
    return there;
};




/*
 *                  Send anything
 */

/** Send an abstract thingie (determine the concrete type and send that.) */
PianodConnection &operator<<(PianodConnection &there, const MusicThingie *thing) {
    return thing->transmit (there);
}





/*
 *                  Send songs
 */


/** Send ratings, seed information and actions to a connection.
    @param conn The connection to send to.
    @param song The song whose ratings to send. */
void sendRatings (PianodConnection *conn,
                  const PianodSong &song) {
    const PianodPlaylist *playlist = song.playlist();

    // Suggestions never have ratings.
    if (song.isSuggestion()) return;

    // Primaries, seeds and ratings have some kind of rating.
    // (Except primaries don't have a rating for unauthenticated users.)
    // Announce seeds as unrated 0.0, so seed flags stay in place.
    Rating songRating = Rating::UNRATED;
    if (song.ratingScheme() == RatingScheme::Owner ||
        (song.ratingScheme() == RatingScheme::Individual && conn->user)) {
        songRating = song.rating (conn->user);
    }

    ostringstream ratings;
    ratings << RATINGS [songRating] << setprecision (1) << fixed  << " " << ratingAsFloat (songRating);

    // Include seed flags if they apply.
    if (playlist) {
        if (playlist->canSeed (MusicThingie::Type::Song)
            && playlist->seed (MusicThingie::Type::Song, &song))
            ratings << " seed";
        if (playlist->canSeed (MusicThingie::Type::Album)
            && playlist->seed (MusicThingie::Type::Album, &song))
            ratings << " albumseed";
        if (playlist->canSeed (MusicThingie::Type::Artist)
            && playlist->seed (MusicThingie::Type::Artist, &song))
            ratings << " artistseed";
    }
    conn << Response (I_RATING, ratings.str());

    // If this song can have a playlist rating, announce that.
    if (playlist && song.isPrimary()) {
        Rating playlist_rating = playlist->rating (conn->user);

        ostringstream playlist_ratings;
        playlist_ratings << RATINGS [playlist_rating] << setprecision (1) << fixed  << " " << ratingAsFloat (playlist_rating);
        conn << Response (I_PLAYLISTRATING, playlist_ratings.str());
    }

    // Provide a list of actions current user can take on this song
    const char *rate = nullptr, *queue = nullptr;
    const char *seed = "", *albumseed = "", *artistseed = "";
    if ((song.ratingScheme() == RatingScheme::Individual && conn->user) ||
        (song.ratingScheme() == RatingScheme::Owner && song.isEditableBy(conn->user))) {
        rate = " rate";


        if (playlist && playlist->isEditableBy(conn->user)) {
            // Include seed flags if they apply.
            if (playlist->canSeed (MusicThingie::Type::Song))
                seed = " seed";
            if (playlist->canSeed (MusicThingie::Type::Album))
                albumseed = " albumseed";
            if (playlist->canSeed (MusicThingie::Type::Artist))
                artistseed = " artistseed";
        }
    } else {
        // Check that coding assumption has not changed.
        // I.e., not ratable implies never seedable.
        assert (!(playlist && playlist->isEditableBy(conn->user)));
    }
    if (song.isUsableBy (conn->user) && song.canQueue() && song.isPrimary())
        queue = " request";
    if (rate || queue) {
        conn->printf ("%03d %s:%s%s%s%s%s\n",
                      I_ACTIONS, ResponseText (I_ACTIONS),
                      rate ? rate : "", queue ? queue : "",
                      seed, albumseed, artistseed);
    }
}

/** Send updated ratings.
    @param conn The connection initiating the ratings change.
    @param song The ratings to send. */
void sendUpdatedRatings (PianodConnection &conn, const PianodSong *song) {
    bool all_users = (song->ratingScheme () == RatingScheme::Owner);
    for (const auto &connection : conn.service()) {
        if (all_users || connection->user == conn.user) {
            sendRatings (connection, *song);
        }
    }
}


void sendSong (Football::Thingie &there, const PianodSong &song) {
    there << Response (I_ID, song.id());
    if (!song.albumTitle().empty())
        there << Response (I_ALBUM, song.albumTitle());
    if (!song.artist().empty())
        there << Response (I_ARTIST, song.artist());
    there << Response (I_SONG, song.title());
    if (!song.coverArtUrl().empty())
        there << Response (I_COVERART, song.coverArtUrl());
    PianodPlaylist *playlist = song.playlist();
    if (playlist)
        there << Response (I_PLAYLIST, playlist->playlistName());
    if (!song.genre().empty())
        there << Response (I_GENRE, song.genre());
    if (!song.infoUrl().empty())
        there << Response (I_INFO_URL, song.infoUrl());
    if (song.year())
        there << Response (I_YEAR, song.year());
    if (song.duration())
        there.printf ("%03d %s: %d:%02d\n",
                      I_DURATION, ResponseText (I_DURATION),
                      song.duration() / 60,
                      song.duration() % 60);
    there
    << Response (I_SOURCE, song.source()->kind())
    << Response (I_NAME, song.source()->name());
}

/** Send song details to a connection or service.
    When sending to a service, ratings details may vary if appropriate. */
PianodService &operator<<(PianodService &there, const PianodSong &song) {
    sendSong (there, song);
    for (auto connect : there) {
        sendRatings (connect, song);
    }
    return there;
}

/** Send a list of songs to a connection */
PianodConnection &operator<<(PianodConnection &there, const SongList &songs) {
    for (auto song : songs) {
        there << S_DATA << *song;
    }
    return there;
};



/*
 *                      Send Playlists
 */

/** Send playlist ratings and actions to a connection.
    @param conn The connection to send to.
    @param playlist The playlist whose info to send. */
void sendRatings (PianodConnection &conn,
                  const PianodPlaylist &playlist) {
    conn.printf ("%03d %s: rate",
                 I_ACTIONS, ResponseText (I_ACTIONS));
    if (playlist.isUsableBy (conn.user) && playlist.canQueue() && playlist.isPrimary())
        conn << " request";
    if (playlist.isEditableBy(conn.user)) {
        // Include seed flags if they apply.
        if (playlist.canSeed (MusicThingie::Type::Song))
            conn << " seed";
        if (playlist.canSeed (MusicThingie::Type::Album))
            conn << " albumseed";
        if (playlist.canSeed (MusicThingie::Type::Artist))
            conn << " artistseed";
    }
    conn << "\n";
}

/** Send a list of playlists to a connection */
PianodConnection &operator<<(PianodConnection &there, const PlaylistList &playlists) {
    for (auto playlist : playlists) {
        there << S_DATA;
        there << playlist;
    }
    return there;
};


/*
 *                  Send abstract lists
 */

PianodConnection &operator<<(PianodConnection &there, const ThingieList &things) {
    for (auto thing : things) {
        there << S_DATA << thing;
    }
    return there;
};



/*
 *              Send seed lists
 */

PianodConnection &sendSeedlist (PianodConnection &there,
                                const ThingieList &things,
                                PianodPlaylist *playlist) {
    for (auto thing : things) {
        there << S_DATA;
        auto song = thing->asSong();;
        const char *kind = nullptr;
        if (song) {
            // Could be a songseed or a rating.
            sendSong (there, *song);
            if (playlist->canSeed(MusicThingie::Type::Song) && playlist->seed(MusicThingie::Type::Song, song)) {
                kind = "seed";
            } else {
                assert (song->ratingScheme() == RatingScheme::Owner);
                Rating songRating = song->rating (nullptr);
                there.printf ("%03d %s: %s %1.1f\n",
                              I_RATING, ResponseText (I_RATING),
                              RATINGS [songRating],
                              (double) ratingAsFloat (songRating));
            }
        } else {
            auto artist = thing->asArtist();
            if (artist) {
                auto album = thing->asAlbum();
                if (album) {
                    kind = "albumseed";
                    there << album;
                } else {
                    kind = "artistseed";
                    there << artist;
                }
            } else {
                auto play = thing->asPlaylist();
                assert (play);
                kind = "playlistseed";
                there << play;
            }
        }
        if (kind) {
            there << Response (I_RATING, kind);
        }
    }
    return there;
}
