///
/// Evaluators for Parsnip command-line parsing.
/// @file       parsnip_evaluate.h - Parsnip serialization & parsing
/// @author     Perette Barella
/// @date       2020-05-06
/// @copyright  Copyright 2012-2020 Devious Fish. All rights reserved.
/// history     This parser is based on the Football parser from 2012.
///

#include "parsnip_command.h"

namespace Parsnip {
    inline void parser_assert (bool requirement, const char *explanation) {
        if (!requirement) {
            throw std::runtime_error (explanation);
        }
    }

    inline void parser_assert (bool requirement, const char *explanation, const std::string &detail) {
        if (!requirement) {
            throw std::runtime_error (std::string (explanation) + ": " + detail);
        }
    }

    /** If an evaluator isn't constructed, create it.
        If it exists and is the right type, leave it alone.
        If it is a TerminatorEvaluator, upgrade it to the type we want.
        @tparam DesiredEvaluator The type of evaluator we want.
        @param parser A pointer container for the existing evaluator.
        @param existed If not null, set to indicate whether parser was already the specified type.
        @return A pointer to the parser. */
    template <class DesiredEvaluator>
    static inline DesiredEvaluator *uptype_construct (EvaluatorRef &parser, bool *existed = nullptr) {
        DesiredEvaluator *possess = dynamic_cast<DesiredEvaluator *> (parser.get());
        if (existed) {
            // If not null, set existed to indicate if object was already of required type.
            *existed = (possess != nullptr);
        }
        if (!possess) {
            possess = new DesiredEvaluator (parser.get());
            parser.reset (possess);
        }
        return possess;
    }

    /// An evaluator representing abrupt end of the command pattern.
    class TerminatorEvaluator : public Evaluator {
        virtual Evaluator *getNextEvaluator (const StringType &value) override;
        virtual Parsnip::Data evaluateToken (ArgvCursor *cursor) override;
        virtual void convertToOptionEvaluator() override;
    };

    /// An evaluator representing abrupt end of the command pattern.
    class EndOfOptionEvaluator : public Evaluator {
        virtual Evaluator *getNextEvaluator (const StringType &value) override;
        virtual Parsnip::Data evaluateToken (ArgvCursor *cursor) override;
        virtual void convertToOptionEvaluator() override;

    public:
        using Evaluator::Evaluator;
    };

    /// Evaluate a value/fill-in field.
    class ValueEvaluator : public Evaluator {
    protected:
        EvaluatorRef next_evaluator;

        virtual Evaluator *getNextEvaluator (const StringType &value) override;

    public:
        using Evaluator::Evaluator;
        virtual bool operator== (const Evaluator &other) const override;
        virtual void convertToOptionEvaluator() override;

        /// Construct a value/fill-in token handler
        static void construct (EvaluatorRef &parser,
                               const ArgvCursor &cursor,
                               CommandId id,
                               AggregateParser *parent_parser,
                               std::string name,
                               std::string keywords);
    };

    /// Evaluate a string value field.
    class StringEvaluator : public ValueEvaluator {
    public:
        using ValueEvaluator::ValueEvaluator;
        virtual Parsnip::Data evaluateToken (ArgvCursor *cursor) override;
    };

    /// Evaluate and validate an integral value field.
    class IntegerEvaluator : public ValueEvaluator {
        long minimum;   ///< Minimum acceptable value.
        long maximum;   ///< Maximum acceptable value.
        int radix = 1;  ///< 0 = automatic, otherwise the base.

    public:
        using ValueEvaluator::ValueEvaluator;
        virtual Parsnip::Data evaluateToken (ArgvCursor *cursor) override;
        virtual bool operator== (const Evaluator &other) const override;

        static IntegerEvaluator *construct (EvaluatorRef &evaluator,
                                            const std::string &name,
                                            const std::string &min,
                                            const std::string &max);
    };

    /// Evaluate and validate a numeric value field.
    class RealEvaluator : public ValueEvaluator {
        double minimum;  ///< Minimum acceptable value.
        double maximum;  ///< Maximum acceptable value.

    public:
        using ValueEvaluator::ValueEvaluator;
        virtual Parsnip::Data evaluateToken (ArgvCursor *cursor) override;
        virtual bool operator== (const Evaluator &other) const override;

        static RealEvaluator *construct (EvaluatorRef &parser,
                                         const std::string &name,
                                         const std::string &min,
                                         const std::string &max);
    };

    /// Evaluate a keyword in a command line.
    class KeywordEvaluator : public Evaluator {
        friend class ValueEvaluator;

        EvaluatorRef numeric_evaluator;
        std::unordered_map<StringType, EvaluatorRef> tokens;
        bool numbers_present{false};  /// True if some of the tokens are numeric.

    public:
        using Evaluator::Evaluator;

        inline static bool is_numeric (const std::string &value) {
            return ((!value.empty() && isdigit (value[0]))
                    || (value.size() >= 2 && value[0] == '-' && isdigit (value[1])));
        }

        inline static std::string tolower (std::string value) {
            for (auto &ch : value) {
                ch = ::tolower (ch);
            }
            return value;
        }

        virtual Parsnip::Data evaluateToken (ArgvCursor *cursor) override;
        virtual Evaluator *getNextEvaluator (const StringType &token) override;
        virtual bool operator== (const Evaluator &other) const override;
        virtual void convertToOptionEvaluator() override;

        /// Construct a keyword token handler
        static void construct (EvaluatorRef &parser,
                               const ArgvCursor &cursor,
                               CommandId id,
                               AggregateParser *parent_parser,
                               const std::string &name,
                               std::string keywords);
    };

    /// An evaluator that represents the rest of the command line.
    class RemainderEvaluator : public Evaluator {
        friend class Evaluator;
        friend class RemainingValuesEvaluator;
        friend class ValueEvaluator;

    protected:
        EvaluatorRef terminating_evaluator;

    public:
        using Evaluator::Evaluator;

        virtual Evaluator *getNextEvaluator (const StringType &value) override;
        virtual bool operator== (const Evaluator &other) const override;
        virtual void convertToOptionEvaluator() override;
    };

    /// Gather the rest of the command line as an untokenized string.
    class RawRemainderEvaluator : public RemainderEvaluator {
    public:
        using RemainderEvaluator::RemainderEvaluator;

        virtual Parsnip::Data evaluateToken (ArgvCursor *cursor) override;
    };

    /// Gather the rest of the command line tokens in a list.
    class RemainingValuesEvaluator : public RemainderEvaluator {
        friend ValueEvaluator;

    public:
        using RemainderEvaluator::RemainderEvaluator;

        virtual Parsnip::Data evaluateToken (ArgvCursor *cursor) override;
    };

    /// Evaluate remainder with a different parser.
    class OptionEvaluator : public RemainderEvaluator {
        friend class OptionParser;
        OptionParserRef option_parser;
        bool regurgitate {false};
        bool iterate {false};

    protected:
        virtual Parsnip::Data evaluateToken (class ArgvCursor *cursor) override;

    public:
        using RemainderEvaluator::RemainderEvaluator;
        virtual bool operator== (const Evaluator &) const override;

        static OptionEvaluator *construct (EvaluatorRef &evaluator,
                                           std::string option_parser_type,
                                           bool iterative,
                                           const AggregateParser *parent_parser);
    };

}  // namespace Parsnip
