///
/// Evaluators for parsnip command-line parsing.
/// @file       parsnip_evaluate.cpp - Parsnip serialization & parsing
/// @author     Perette Barella
/// @date       2020-05-06
/// @copyright  Copyright 2012-2020 Devious Fish. All rights reserved.
/// history     This parser is based on the Football parser from 2012.
///

#include <string>

#include <cctype>

#include "parsnip.h"
#include "parsnip_argv.h"
#include "parsnip_command.h"
#include "parsnip_evaluate.h"

namespace Parsnip {

    /** Convert a terminator evaluator to an end-of-options evaluator.
        @param evaluator A pointer container for the existing evaluator. */
    static inline void options_evaluator_convert (EvaluatorRef &evaluator) {
        auto eval = evaluator.get();
        assert (eval);
        if (typeid (*eval) == typeid (TerminatorEvaluator)) {
            evaluator.reset (new EndOfOptionEvaluator (evaluator.get()));
        }
    }

    /*
     *              Evaluator class implementation
     */

    /** Upgrade an evaluator.  The original *must* be a TerminatorEvaluator;
        other type should never be upgraded.
        Herein we initialize the new evaluator with the relevant bits. */
    Evaluator::Evaluator (Evaluator *original) {
        if (original) {
            parser_assert (dynamic_cast<TerminatorEvaluator *> (original), "Conflicting patterns");
            name = original->name;
            command_id = original->command_id;
        }
    }

    /** Assign a name by which to store values for value evaluators.
        There must be no prior name, or the same as the prior name.
        @param nam The name.  If empty the assignment is ignored. */
    void Evaluator::set_name (EvaluatorRef &target, const std::string &nam) {
        if (!target) {
            target.reset (new TerminatorEvaluator);
        }
        if (target->name.empty()) {
            target->name = nam;
        } else if (!nam.empty()) {
            parser_assert (target->name == nam, "Conflicting value names", nam + "/" + target->name);
        }
    }

    /** Set the command ID in an evaluator.  The ID must not be redefined/changed.
        @param id The ID being set. */
    void Evaluator::setCommandId (CommandId id) {
        parser_assert (command_id == NoCommand, "Pattern redefined", std::to_string (int (id)));
        command_id = id;
    }

    /** Check if two evaluators are the same.  Subclasses should override this to check their details.
        @param other Another evaluator to which to compare. */
    bool Evaluator::operator== (const Evaluator &other) const {
        return (typeid (*this) == typeid (other) && this->name == other.name && this->command_id == other.command_id);
    }

    /*
     *              Terminal Evalutors
     */

    /// An evaluator representing abrupt end of the command pattern.
    Evaluator *TerminatorEvaluator::getNextEvaluator (const StringType &value) {
        throw RunOnCommand (value);
    }

    Parsnip::Data TerminatorEvaluator::evaluateToken (ArgvCursor *cursor) {
        assert (!"Unreachable");
        throw std::runtime_error ("Evaluation of statement termination");
    }

    void TerminatorEvaluator::convertToOptionEvaluator() {
        assert (!"Unreachable");
    }

    /// An evaluator representing of one option phrase.
    Evaluator *EndOfOptionEvaluator::getNextEvaluator (const StringType &value) {
        return nullptr;
    }

    Parsnip::Data EndOfOptionEvaluator::evaluateToken (ArgvCursor *cursor) {
        assert (!"Unreachable");
        throw std::runtime_error ("Evaluation after end of option");
    }

    void EndOfOptionEvaluator::convertToOptionEvaluator() {
        // No-op.
    }

    /*
     *              Value Evaluators
     */

    Evaluator *ValueEvaluator::getNextEvaluator (const StringType &value) {
        return next_evaluator.get();
    }

    bool ValueEvaluator::operator== (const Evaluator &other) const {
        if (this->Evaluator::operator== (other)) {
            auto far = static_cast<const ValueEvaluator *> (&other);
            return *next_evaluator == *(far->next_evaluator);
        }
        return false;
    }

    void ValueEvaluator::convertToOptionEvaluator() {
        if (next_evaluator) {
            options_evaluator_convert (next_evaluator);
            next_evaluator->convertToOptionEvaluator();
        }
    }

    /// @return the current token as string data.
    Parsnip::Data StringEvaluator::evaluateToken (ArgvCursor *cursor) {
        return Parsnip::Data ((*cursor)++.value());
    }

    /// @return the current token as a long integer
    /// @throw If value is not in range.
    Parsnip::Data IntegerEvaluator::evaluateToken (ArgvCursor *cursor) {
        const std::string &value = (*cursor)++.value();
        if (value.empty()) {
            throw NotNumeric ("(empty value)");
        }
        char *err;
        errno = 0;
        long int result = strtol (value.c_str(), &err, radix);
        if (*err != '\0' || errno != 0) {
            throw NotNumeric ('\'' + value + '\'');
        }
        if (result < minimum || result > maximum) {
            throw NumberOutOfRange (value + " (range: " + std::to_string (minimum) + "-" + std::to_string (maximum)
                                    + ")");
        }
        return Parsnip::Data (result);
    }

    bool IntegerEvaluator::operator== (const Evaluator &other) const {
        if (this->ValueEvaluator::operator== (other)) {
            auto far = static_cast<const IntegerEvaluator *> (&other);
            return (minimum == far->minimum && maximum == far->maximum && radix == far->radix);
        }
        return false;
    }

    /** Construct an evaluator (or validate the existing validator) for
        accepting an integer with a limited range. */
    IntegerEvaluator *IntegerEvaluator::construct (EvaluatorRef &parser,
                                                   const std::string &name,
                                                   const std::string &min,
                                                   const std::string &max) {
        bool existed;
        IntegerEvaluator *ip = uptype_construct<IntegerEvaluator> (parser, &existed);
        Evaluator::set_name (ip->next_evaluator, name);

        parser_assert (!min.empty() && !max.empty(), "Min or max empty");
        // Check first digit, skipping a leading +/-
        bool min_adaptive = (min[0] == '0' || min.substr (0, 2) == "-0");
        bool max_adaptive = (max[0] == '0' || max.substr (0, 2) == "-0");
        int rad = ((min_adaptive || max_adaptive) ? rad == 0 : rad = 10);
        parser_assert (ip->radix == 1 || rad == ip->radix, "Inconsistent radix");
        ip->radix = rad;

        long prior = ip->minimum;
        ip->minimum = std::stol (min, nullptr, ip->radix);
        parser_assert (!existed || prior == ip->minimum, "Inconsistent minimum");

        prior = ip->maximum;
        ip->maximum = std::stol (max, nullptr, ip->radix);
        parser_assert (!existed || prior == ip->maximum, "Inconsistent maximum");
        parser_assert (ip->minimum < ip->maximum, "Minimum must be less than maximum");
        return ip;
    }

    bool RealEvaluator::operator== (const Evaluator &other) const {
        if (this->ValueEvaluator::operator== (other)) {
            auto far = static_cast<const RealEvaluator *> (&other);
            return (minimum == far->minimum && maximum == far->maximum);
        }
        return false;
    }

    /// Interpret the token as double, range check it, and store it.
    Parsnip::Data RealEvaluator::evaluateToken (ArgvCursor *cursor) {
        const std::string &value = (*cursor)++.value();
        if (value.empty()) {
            throw NotNumeric ("(empty value)");
        }
        errno = 0;
        char *err;
        double result = strtod (value.c_str(), &err);
        if (*err != '\0' || errno != 0) {
            throw NotNumeric ('\'' + value + '\'');
        }
        if (result < minimum || result > maximum) {
            throw NumberOutOfRange (value + " (range: " + std::to_string (minimum) + "-" + std::to_string (maximum)
                                    + ")");
        }
        return Parsnip::Data (result);
    }

    /** Construct an evaluator (or validate the existing validator) for
        accepting real number with a limited range. */
    RealEvaluator *RealEvaluator::construct (EvaluatorRef &parser,
                                             const std::string &name,
                                             const std::string &min,
                                             const std::string &max) {
        bool existed;
        RealEvaluator *rp = uptype_construct<RealEvaluator> (parser, &existed);
        Evaluator::set_name (rp->next_evaluator, name);

        double prior = rp->minimum;
        rp->minimum = std::stod (min);
        parser_assert (!existed || prior == rp->minimum, "Inconsistent minimum");

        prior = rp->maximum;
        rp->maximum = std::stod (max);
        parser_assert (!existed || prior == rp->maximum, "Inconsistent minimum");
        parser_assert (rp->minimum < rp->maximum, "Minimum must be less than maximum");
        return rp;
    }

    /*
     *              Keyword Evaluator
     */

    /// Store the current keyword as a string, if it's marked for recording.
    Parsnip::Data KeywordEvaluator::evaluateToken (ArgvCursor *cursor) {
        if (numeric_evaluator && is_numeric (cursor->value())) {
            return numeric_evaluator->evaluateToken (cursor);
        }
        return (*cursor)++.value();
    }

    /// Select an evaluator for the next element based on current token.
    Evaluator *KeywordEvaluator::getNextEvaluator (const StringType &token) {
        if (numeric_evaluator && is_numeric (token)) {
            return numeric_evaluator->getNextEvaluator (token);
        }
        auto it = tokens.find (tolower (token));
        if (it == tokens.end()) {
            throw InvalidKeyword (token);  // Invalid command
        }
        return it->second.get();
    }

    bool KeywordEvaluator::operator== (const Evaluator &other) const {
        if (!this->Evaluator::operator== (other)) {
            return false;
        }
        auto far = static_cast<const KeywordEvaluator *> (&other);
        if (tokens.size() == far->tokens.size() && numbers_present == far->numbers_present
            && (numeric_evaluator == far->numeric_evaluator || *numeric_evaluator == *(far->numeric_evaluator))) {
            for (auto &token : tokens) {
                auto it = far->tokens.find (token.first);
                if (it == far->tokens.end()) {
                    return false;
                }
                if (*(token.second) != *(it->second.get())) {
                    return false;
                }
            }
            return true;
        }
        return false;
    };

    void KeywordEvaluator::convertToOptionEvaluator() {
        if (numeric_evaluator) {
            numeric_evaluator->convertToOptionEvaluator();
        }
        for (auto &token : tokens) {
            options_evaluator_convert (token.second);
            token.second->convertToOptionEvaluator();
        }
    }

    /*
     *              Remainder Evaluators
     */

    /// An evaluator that represents the rest of the command line.
    Evaluator *RemainderEvaluator::getNextEvaluator (const StringType &value) {
        return terminating_evaluator.get();
    }

    bool RemainderEvaluator::operator== (const Evaluator &other) const {
        if (this->Evaluator::operator== (other)) {
            auto far = static_cast<const RemainderEvaluator *> (&other);
            return *terminating_evaluator == *(far->terminating_evaluator);
        }
        return false;
    }

    void RemainderEvaluator::convertToOptionEvaluator() {
        options_evaluator_convert (terminating_evaluator);
        terminating_evaluator->convertToOptionEvaluator();
    }

    /// @return the original, untokenized, unmodified string.
    Parsnip::Data RawRemainderEvaluator::evaluateToken (ArgvCursor *cursor) {
        Parsnip::Data result = cursor->remainingString();
        while (!cursor->isEnd()) {
            (*cursor)++;
        }
        return std::move (result);
    }

    /// @return any remaining tokens in a list
    Parsnip::Data RemainingValuesEvaluator::evaluateToken (ArgvCursor *cursor) {
        Parsnip::Data list{Parsnip::Data::List};
        while (!cursor->isEnd()) {
            list.push_back (terminating_evaluator->evaluateToken (cursor));
        }
        return list;
    }

    Parsnip::Data OptionEvaluator::evaluateToken (class ArgvCursor *cursor) {
        Parsnip::Data results{Parsnip::Data::Dictionary};
        if (regurgitate) {
            assert (!cursor->isStart());
            *cursor = *cursor - 1;
        }
        if (iterate) {
            while (!cursor->isEnd()) {
                option_parser->evaluator->evaluate (cursor, results);
            }
        } else {
            option_parser->evaluator->evaluate (cursor, results);
        }
        return results;
    }

    bool OptionEvaluator::operator== (const Evaluator &other) const {
        if (RemainderEvaluator::operator== (other)) {
            auto far = static_cast<const OptionEvaluator *> (&other);
            return option_parser == far->option_parser;
        }
        return false;
    }

    /** Construct an evaluator (or validate the existing validator)
        that parsers the remainder with an option parser.
        @param evaluator The evaluator being constructed.
        @param option_parser_type The name of the option parser to use.
        @param iterative If true, option parser should iterate.
        @param parent_parser The aggregate parser containing option parsers to utilize. */
    OptionEvaluator *OptionEvaluator::construct (EvaluatorRef &evaluator,
                                                 std::string option_parser_type,
                                                 bool iterative,
                                                 const AggregateParser *parent_parser) {
        auto oe = uptype_construct<OptionEvaluator> (evaluator);
        parser_assert (!oe->option_parser, "Redefined statement");
        if (option_parser_type.substr (0, 2) == "--") {
            oe->regurgitate = true;
            option_parser_type.erase (0, 2);
        }
        oe->option_parser = parent_parser->option_parsers.at (option_parser_type);
        oe->iterate = iterative;
        return oe;
    }

}  // namespace Parsnip
