///
/// Parsnip command-line parsing.
/// @file       parsnip_command.h - Parsnip serialization & parsing
/// @author     Perette Barella
/// @date       2020-05-06
/// @copyright  Copyright 2012-2020 Devious Fish. All rights reserved.
/// history     This parser is based on the Football parser from 2012.
///

#pragma once

#include <string>
#include <memory>
#include <vector>
#include <unordered_map>
#include <exception>

#include "parsnip.h"

namespace Parsnip {
    /// NumberOutOfRange: The number was outside the range defined in the command pattern.
    class NumberOutOfRange : public Exception {
    public:
        inline NumberOutOfRange() : Exception ("Value out of range"){};
        inline NumberOutOfRange (const std::string &value) : Exception ("Value out of range", value){};
    };

    /// Not numeric: A command pattern specified a number, but the token was not valid number.
    class NotNumeric : public Exception {
    public:
        inline NotNumeric() : Exception ("Not a number"){};
        inline NotNumeric (const std::string &value) : Exception ("Not a number", value){};
    };

    /// A token encountered was not one of the expected keywords.
    class InvalidKeyword : public Exception {
    public:
        inline InvalidKeyword() : Exception ("Invalid keyword"){};
        inline InvalidKeyword (const std::string &value) : Exception ("Invalid keyword", value){};
    };

    /// More tokens were expected.
    class IncompleteCommand : public Exception {
    public:
        inline IncompleteCommand() : Exception ("Incomplete command"){};
        inline IncompleteCommand (const std::string &value) : Exception ("Incomplete command", value){};
    };

    /// There were extra tokens at the end of the command pattern.
    class RunOnCommand : public Exception {
    public:
        inline RunOnCommand() : Exception ("Run-on command"){};
        inline RunOnCommand (const std::string &value) : Exception ("Run-on command", value){};
    };
    
    class DuplicateOption : public Exception {
    public:
        inline DuplicateOption() : Exception ("Option respecified: "){};
        inline DuplicateOption (const std::string &value) : Exception ("Option respecified", value){};
    };

    class Evaluator;
    class AggregateParser;
    using EvaluatorRef = std::unique_ptr<Evaluator>;

    extern const std::string EmptyString;

    /// Evaluator base class.
    class Evaluator {
        friend class ValueEvaluator;
        friend class KeywordEvaluator;
        friend class RemainingValuesEvaluator;
        friend class RemainderEvaluator;
        friend class OptionParser;

    public:
        using StringType = std::string;
        using CommandId = int;

    private:
        static const CommandId NoCommand = -1;

    protected:
        /// If this completes a command, the command ID.
        CommandId command_id{NoCommand};
        /// If not empty, the name by which this value shall be accessed.
        std::string name;

        Evaluator() = default;

    public:
        Evaluator (Evaluator *original);
        virtual ~Evaluator() = default;

    protected:
        static void set_name (EvaluatorRef &target, const std::string &name);
        void setCommandId (CommandId id);

        /** Determine if and how to evaluate additional command line tokens.
            @param token The current token being evaluated.
            @return The evaluator for the next token, or nullptr if evaluation is complete. */
        virtual Evaluator *getNextEvaluator (const StringType &token) = 0;

        /** Evaluate command line tokens.
            @param cursor Provides tokens from the command line.
            Must be bumped forward as they are processed.
            @return Parsnip data object containing evaluated token. */
        virtual Parsnip::Data evaluateToken (class ArgvCursor *cursor) = 0;

        /** Make any changes to the parse tree to convert it to an options evaluator. */
        virtual void convertToOptionEvaluator() = 0;

    public:
        static void construct (EvaluatorRef &evaluator,
                               const ArgvCursor &cursor,
                               CommandId id,
                               AggregateParser *parent_parser = nullptr);
        int evaluate (ArgvCursor *cursor, Parsnip::Data &result_dict);
        virtual bool operator== (const Evaluator &) const;
        inline bool operator!= (const Evaluator &other) const {
            return !(*this == other);
        }
    };

    /// A class for parsing command lines.
    class Parser {
    protected:
        EvaluatorRef evaluator;

    public:
        using CommandId = int;
        using StringType = Evaluator::StringType;

        // Structure for defining new parser patterns.
        struct Definition {
            CommandId command_id;
            const char *statement;
        };
        using Definitions = std::vector<Definition>;

        // Structure for evaluation results.
        struct Result {
            ///< ID of command pattern that was matched.
            CommandId command_id;
            /// Parameters collected from the command line.
            Parsnip::Data parameters{Parsnip::Data::Dictionary};
        };

        Parser() = default;
        Parser (const Definitions &defs);
        void addStatements (const Definitions &defs);
        Result evaluate (const StringType &command) const;
        bool operator== (const Parser &) const;
    };

    /// A class for parsing command line options (name-value pairs, for instance).
    class OptionParser {
        friend class OptionEvaluator;
        EvaluatorRef evaluator;

    public:
        // Structure for defining new option patterns.
        using Definitions = std::vector<const char *>;

        OptionParser() = default;
        OptionParser (const Definitions &defs);
        void addOptions (const Definitions &defs);
        bool operator== (const OptionParser &) const;
    };
    using OptionParserRef = std::shared_ptr<OptionParser>;

    /// An advanced class for parcing command lines that allows use of option parsers.
    class AggregateParser : public Parser {
        friend class OptionEvaluator;
        std::unordered_map<std::string, OptionParserRef> option_parsers;

    public:
        using Parser::Parser;
        void addOptionParser (const std::string &name, OptionParserRef &add);
        void addStatements (const Definitions &defs);
    };

}  // namespace Parsnip
