///
/// Parsnip serialization.
/// A flexible data type for serializing data and parsing
/// messages into structures.
/// @file       parsnip.h - Parsnip serialization & parsing
/// @author     Perette Barella
/// @date       2019-05-01
/// @copyright  Copyright 2019-2020 Devious Fish. All rights reserved.
///

#pragma once

#include <config.h>

#include <exception>
#include <iostream>
#include <limits>
#include <memory>
#include <ostream>
#include <sstream>
#include <string>
#include <type_traits>
#include <unordered_map>
#include <vector>
#include <functional>

#include <cassert>
#include <climits>
#include <cmath>
#include <cstdlib>
#include <cstring>

/// Serialization and parsing library.
namespace Parsnip {

    /// Class representing issues related to serialization & parsing.
    class Exception : public std::exception {
    public:
        virtual const char *what() const noexcept override {
            return reason.c_str();
        };

    protected:
        /// Storage for the exception explanation.
        std::string reason;

        /** Construct an exception. */
        Exception (const std::string &why) : reason (why){};

        /** Construct an exception.
            @param why The primary explanation for the error.
            @param detail Specifics about the error, such as the problem location or data. */
        Exception (const std::string &why, const std::string &detail) : reason (why) {
            reason += ": ";
            reason += detail;
        };
    };

    /// DataRangeError indicates a value received in a stream can't be represented.
    class DataRangeError : public Exception {
    public:
        inline DataRangeError() : Exception ("Value out of representable range"){};
        inline DataRangeError (const std::string &value) : Exception ("Value exceeds datatype range", value){};
    };

    /** Generic data type.
        A datatype that can hold strings, numbers, booleans, lists, or dictionaries. */
    class Data {
        friend class IncorrectDataType;

    public:
        static const struct dictionary_t {
        } Dictionary;  ///< Dictionary flag type/value.
        static const struct list_t {
        } List;                             ///< List flag type/value.
        static const bool Flexible{false};  ///< Flag to make strings flexible.
        static const int NoIndent{-32767};  ///< Prevent indentation

        using ListType = std::vector<Data>;

    protected:
        enum class Type {
            Null,
            Dictionary,
            List,
            String,          ///< The value is string, even if the string contains a number.
            FlexibleString,  ///< The value is a string, but may convert to a number if the value is numeric.
            Integer,
            Real,
            Boolean
        };

        using DictionaryType = std::unordered_map<std::string, Data>;
        using StringType = std::string;  // As long as everything is UTF-8 encoded anyway...

        Type datatype = Type::Null;
        union data_t {
            DictionaryType *dictionary;
            ListType *list;
            StringType *str;
            long integer;
            double real;
            bool boolean;
        } data;

        void release();

    private:
        Data (Type kind);
        void mandateType (Type type) const;

        inline void loadDictionary(){};

        /// Dictionary constructor helper function, inserting a moved Data.
        template <typename... More>
        void loadDictionary (const char *name, Data &&value, More &&... remainder) {
            (*(data.dictionary))[name] = std::move (value);
            loadDictionary (std::forward<More> (remainder)...);
        }

        inline void loadArray(){};

        /// List constructor helper function, inserting a moved Data.
        template <typename... More>
        void loadArray (Data &&value, More... remainder) {
            data.list->push_back (std::move (value));
            loadArray (std::forward<More> (remainder)...);
        }

    public:
        using size_type = ListType::size_type;

        /** An iterator used to walk items in Data's lists. */
        struct iterator : std::iterator<std::forward_iterator_tag, Data> {
            const ListType *list = nullptr;
            int position = 0;
            iterator (const Data *l);
            iterator (const Data *l, size_type pos);
            const Data &operator*() const noexcept;
            iterator &operator++() noexcept;
            iterator operator++ (int) noexcept;
            bool operator!= (const iterator &compare) const noexcept;
        };

        /// Default constructor.
        Data (std::nullptr_t null = nullptr) {
            // Do nothing
        }

        /** String constructor.
            @param value The string value to assign.
            @param type_certainty True if known to be string, false if it may be a number or boolean in string form. */
        Data (const char *value, bool type_certainty = true) {
            data.str = new StringType (value);
            datatype = type_certainty ? Type::String : Type::FlexibleString;
        }

        /// String constructor, accepting C++ string.
        inline Data (const std::string &value, bool type_certainty = true) {
            data.str = new StringType (value);
            datatype = type_certainty ? Type::String : Type::FlexibleString;
        }

        /// Integer constructor.
        template <typename T,
                  typename
                  = typename std::enable_if<!std::is_same<T, bool>::value && std::is_integral<T>::value, int>::type>
        inline Data (T value) noexcept {
            datatype = Type::Integer;
            data.integer = value;
        }

        /// Floating point constructor.
        inline Data (double value) noexcept {
            datatype = Type::Real;
            data.real = value;
        }

        /// Boolean constructor.
        inline Data (bool value) noexcept {
            datatype = Type::Boolean;
            data.boolean = value;
        }

        /** Dictionary constructor.
            Use Dictionary flag variable as first parameter.
            Subsequent parameters are name, value ... pairs (function must have
            an odd number of arguments). */
        template <typename... T>
        Data (dictionary_t, T... params) : Data (Type::Dictionary) {
            loadDictionary (std::forward<T> (params)...);
        }

        /** List constructor.
            Use List flag variable as first parameter.
            Subsequent parameters are inserted in order as list members. */
        template <typename... T>
        Data (list_t, T... params) : Data (Type::List) {
            loadArray (std::forward<T> (params)...);
        }

        /// Move construction.
        inline Data (Data &&from) noexcept {
            datatype = from.datatype;
            data = from.data;
            from.datatype = Type::Null;
        }

        /// Move assignment
        inline Data &operator= (Data &&from) noexcept {
            if (this != &from) {
                release();
                datatype = from.datatype;
                data = from.data;
                from.datatype = Type::Null;
            }
            return *this;
        }

        // Copy constructor & assignment
        Data (const Data &from);
        Data &operator= (const Data &from);

        inline ~Data() {
            release();
        }

        bool operator== (const Data &compare) const;
        inline bool operator!= (const Data &compare) const {
            return !operator== (compare);
        }

        /** Check if data contains a value.
        @return True if empty, false otherwise. */
        inline bool isNull (void) const noexcept {
            return (datatype == Type::Null);
        }

        // Retrieve values by asking for type directly.
        const StringType &asString() const;
        long asLong (int base = 0) const;
        int asInteger (int base = 0) const;
        double asDouble() const;
        bool asBoolean() const;
        ListType asList() const;

        /// Template function for retrieving values by providing a desired type.
        /// Extract data as a string.
        template <typename DataType,
                  typename = typename std::enable_if<std::is_same<DataType, std::string>::value>::type>
        inline const std::string &as() const {
            return asString();
        }

        // Extract data as an integer type, except boolean.
        template <typename DataType,
                  typename
                  = typename std::enable_if<!std::is_same<DataType, bool>::value && std::is_integral<DataType>::value,
                                            int>::type>
        inline DataType as (int base = 0) const {
            long result = asLong (base);
            if (result < std::numeric_limits<DataType>::min() || result > std::numeric_limits<DataType>::max()) {
                throw DataRangeError (std::to_string (result));
            }
            return result;
        }

        // Extract data as a floating point type.
        template <typename DataType, typename = typename std::enable_if<std::is_floating_point<DataType>::value>::type>
        inline DataType as() const {
            double result = asDouble();
            if (result < std::numeric_limits<DataType>::min() || result > std::numeric_limits<DataType>::max()) {
                throw DataRangeError (std::to_string (result));
            }
            return result;
        }

        // Extract data as a boolean.
        template <typename DataType, typename = typename std::enable_if<std::is_same<DataType, bool>::value>::type>
        inline bool as() const {
            return asBoolean();
        }

        // Retrieve data as the object itself.
        template <typename DataType, typename = typename std::enable_if<std::is_same<DataType, Data>::value>::type>
        inline const Data &as() const {
            return *this;
        }

        // Dictionary accessors
        const Data &operator[] (const std::string &word) const;
        Data &operator[] (const std::string &word);
        bool contains (const std::string &word) const;

        /// Iterate over all dictionary members, calling a function
        /// with each key and value.  This implementation covers simple types.
        template <typename DataType, typename = typename std::enable_if<std::is_trivial<DataType>::value>::type>
        void foreach (std::function<void (const std::string &, DataType)> func) const {
            mandateType (Type::Dictionary);
            for (const auto &it : *(data.dictionary)) {
                func (it.first, it.second.as<DataType>());
            }
        }

        /// Dictionary iterator for strings and Data.
        template <typename DataType>
        void foreach (std::function<void (const std::string &, const DataType &)> func) const {
            mandateType (Type::Dictionary);
            for (const auto &it : *(data.dictionary)) {
                func (it.first, it.second.as<DataType>());
            }
        }

        // List accessors
        const Data &operator[] (int index) const;
        Data &operator[] (int index);
        void push_back (const Data &value);
        void push_back (Data &&value);
        size_type size() const;

        /// List iterator for simple types.  Invokes a function with each list member
        /// as the function parameter.
        template <typename DataType, typename = typename std::enable_if<std::is_trivial<Type>::value>::type>
        void foreach (std::function<void (const DataType)> func) const {
            mandateType (Type::List);
            for (const auto &it : *(data.list)) {
                func (it.as<DataType>());
            }
        }

        /// List iterator for strings and Data.
        template <typename DataType>
        void foreach (std::function<void (const DataType &)> func) const {
            mandateType (Type::List);
            for (const auto &it : *(data.list)) {
                func (it.as<DataType>());
            }
        }

        /** Retrieve forward iterator.
            @return Forward iterator positioned at first element.
            @throws IncorrectDataType if not a list. */
        inline Data::iterator begin() const {
            return iterator{this};
        }

        /** Retrieve end iterator.
            @return Forward iterator positioned after last element.
            @throws IncorrectDataType if not a list. */
        inline Data::iterator end() const {
            iterator it{this, size()};
            return it;
        }

        static const char *type_name (Type ty);

        /** Dictionary factory.
            @initial_data Name-value pairs to be inserted into the dictionary.
            @warning Initializer lists cannot use rvalue references; hence,
            elements are copied.  This is fine for simple items, but should be
            avoided when inserting complicatated structures.  Prefer move
            assignment or Parsnip::Data (Parsnip::Data::Dictionary, ...).*/
        template <typename T = Data>
        static Data make_dictionary (std::initializer_list <std::pair <const char *, const T>> initial_data) {
            Data data (Type::Dictionary);
            for (auto &it : initial_data) {
                data [it.first] = it.second;
            }
            return data;
        }

        /** List factory.
            Parameters are inserted in order as list members. */
        template <typename... T>
        static Data make_list(T&&... params) {
            Data data (Type::List);
            data.loadArray (std::forward<T> (params)...);
            return data;
        }

    public:
        std::string toJson (int indent = NoIndent) const;
        std::ostream &toJson (std::ostream &target, int indent = NoIndent, bool suppress = false) const;
        std::ostream &dumpJson (const std::string &intro, std::ostream &target = std::clog) const;
    };

    Data parse_json (std::istream &from, bool check_termination = true);
    Data parse_json (const std::string &from);

    /// DataFormatError represents a syntax error in a datastream.
    class DataFormatError : public Exception {
    public:
        inline DataFormatError (const std::string &detail) : Exception ("Invalid data format", detail){};
    };

    /// An element requested from a constant dictionary does not exist.
    class NoSuchKey : public Exception {
    public:
        inline NoSuchKey (const std::string &key) : Exception ("No such key", key){};
    };

    /// The data's type is inconsistent with the use requested of it.
    class IncorrectDataType : public Exception {
    public:
        IncorrectDataType (Data::Type expected, Data::Type actual) : Exception ("Incorrect data type", "Expected: ") {
            reason += Data::type_name (expected);
            reason += ", Actual: ";
            reason += Data::type_name (actual);
        }
    };

    inline std::ostream &operator<< (std::ostream &out, const Parsnip::Data &data) {
        data.toJson (out);
        return out;
    }

}  // namespace Parsnip
