///
/// Pandora Station implementation.
///	@file		pandorastation.cpp - pianod2
///	@author		Perette Barella
///	@date		2015-03-17
///	@copyright	Copyright (c) 2015-2020 Devious Fish. All rights reserved.
///

#include <config.h>

#include <cassert>

#include <string>

#include "pandoratypes.h"
#include "pandorastation.h"
#include "pandoramessages.h"
#include "pandora.h"

namespace Pandora {

    const int StationListCacheTime = 7200;
    const int StationSeedCacheTime = 10000;
    const int StationFeedbackCacheTime = 3600;

    /** Construct a skip tracker.  The tracker is a simple linked list.

        When a skip occurs, the time of its expiration is added to the list.  When checks
        are done, the front time is checked; if that time is past, it is removed.  Skips
        are allowed when the size of the list is below the skip limit.

        @limit The number of skips to allow.
        @interv The duration over which the limit applies. */
    SkipTracker::SkipTracker (int limit, int interv) : skip_limit (limit), interval (interv) {
    }

    /** Check if (and when) a skip is available.
        @param when If non-null, is set on return to the time at which the next skip is allowed.
        @return True if a skip is allowed, false otherwise. */
    bool SkipTracker::canSkip (time_t *when) {
        purge();
        if (when) {
            if (size() >= skip_limit) {
                *when = front();
            } else {
                *when = time (nullptr);
            }
        }
        return (size() < skip_limit);
    }

    /** Check if a skip is available, and record one if it is.
        @param when If non-null, is set on return to the time at which the next skip is allowed.
        @return True if a skip happened, false if it was not allowed. */
    bool SkipTracker::skip (time_t *when) {
        purge();
        if (canSkip (when)) {
            push_back (time (nullptr) + interval);
            return true;
        }
        return false;
    }

    /*
     *              Pandora Stations (Playlists)
     */

    Station::Station (Source *owner, const Parsnip::Data &message)
    : EncapsulatedPlaylist (owner), skips (owner->userFeatures().station_skip_limit, 3600) {
        playlistId (message["stationId"].asString());
        playlistName (message["name"].asString());
        is_shared = message["isShared"].asBoolean();
        may_rename = message["allowRename"].asBoolean();
        may_transform = message["isTransformAllowed"].asBoolean();
        if (may_transform != is_shared) {
            flog (LOG_WHERE (LOG_PANDORA),
                  "Station ",
                  playlistId(),
                  " (",
                  playlistName(),
                  "): may_transform (",
                  may_transform ? "true" : "false",
                  ") != is_shared (",
                  is_shared ? "true" : "false",
                  ")");
        }
        std::string gen;
        for (const auto &g : message["genre"]) {
            const std::string &gp = g.asString();
            // Assemble genres, but don't go over 30 characters.
            if (gen.empty()) {
                gen = gp;
            } else if (gen.size() + gp.size() < 30) {
                gen += ", " + gp;
            }
        }
        genre (gen);
    };

    /** This is not a full-copy assignment, this is to update station
        details in the master when changes are received. */
    Station &Station::operator= (const Station &update) {
        assert (playlistId() == update.playlistId());
        playlistName (update.playlistName());
        is_shared = update.is_shared;
        may_rename = update.may_rename;
        may_transform = update.may_transform;
        genre (update.genre());
        return *this;
    }

    bool Station::includedInMix (void) const {
        return in_quick_mix;
    }

    void Station::includedInMix (bool include) {
        in_quick_mix = include;
    }

    /** Convert a shared Pandora station into a private one.
        Shared stations cannot be renamed or have their seeds adjusted by the listener. */
    void Station::takePossession() {
        if (is_shared) {
            if (!may_transform) {
                throw CommandError (E_TRANSFORM_FAILED, "Transform of station disallowed");
            }
            RequestTransformStation transform (pandora(), this);
            Status status = pandora()->executeRequest (transform);
            if (status != Status::Ok) {
                throw CommandError (E_NAK, status_strerror (status));
            }
            *this = *(transform.getUpdatedStation());
        }
    }

    bool Station::canSeed (MusicThingie::Type seed_type) const {
        return (seed_type == MusicThingie::Type::Artist || seed_type == MusicThingie::Type::Song);
    }

    bool Station::seed (MusicThingie::Type seed_type, const MusicThingie *music) const {
        assert (canSeed (seed_type));
        assert (music);

        std::string id = pandora()->getRelevantSeedId (seed_type, music);
        if (id.empty()) {
            return false;
        }
        // Checking for seeding should not change anything, but we need to
        // break constness to refresh seed caches.
        const_cast<Station *> (this)->refreshSeeds();
        return song_info.isSeed (id);
    }

    void Station::seed (MusicThingie::Type seed_type, MusicThingie *music, bool value) {
        assert (MusicThingie::isPrimary (seed_type));
        assert (music);
        assert (canSeed (seed_type));

        std::string id = pandora()->getRelevantSeedId (seed_type, music);
        if (id.empty()) {
            throw CommandError (E_WRONGTYPE);
        }
        refreshSeeds();
        takePossession();
        RequestAlterStationSeed seed_alteration (pandora(), this, id, value);
        Status status = pandora()->executeRequest (seed_alteration);
        if (status != Status::Ok) {
            throw CommandError (E_NAK, status_strerror (status));
        }
        song_info.setSeed (id, value);
    }

    /// Retrieve a station's seed information on file with the servers.
    void Station::refreshSeeds() {
        if (seed_expiration > time (nullptr)) {
            return;
        }

        MusicAutoReleasePool pool;
        RequestStationDetails seed_request (pandora(), this);
        Status status = pandora()->executeRequest (seed_request);
        if (status != Status::Ok) {
            throw CommandError (E_NAK, status_strerror (status));
        }
        const ThingieList &seeds = seed_request.getSeeds();
        pandora()->library.unabridge (seeds);
        for (auto &entry : song_info) {
            entry.second.is_seed = false;
        }
        for (const auto &seed : seeds) {
            const Song *song = static_cast<const Song *> (seed->asSong());
            if (song) {
                song_info.setSeed (song->songPandoraId(), true);
            } else {
                const Artist *artist = static_cast<const Artist *> (seed->asArtist());
                if (artist) {
                    song_info.setSeed (artist->artistPandoraId(), true);
                }
            }
        }
        if (seed_request.positiveFeedback() == 0) {
            setNoRatings (true, &positive_rating_expiration);
        }
        if (seed_request.negativeFeedback() == 0) {
            setNoRatings (false, &negative_rating_expiration);
        }
        song_info.purge();
        seed_expiration = time (nullptr) + StationSeedCacheTime;
    }

    /** Remove all ratings, if we know there are none, for a station.
        @param direction True for positive ratings, false for negative.
        @param cache_expiration Updated with a new cache expiration time. */
    void Station::setNoRatings (bool direction, time_t *cache_expiration) {
        Rating rating = direction ? ThumbsUp : ThumbsDown;
        // Clear out old ratings
        for (auto &entry : song_info) {
            if (entry.second.rating == rating) {
                entry.second.rating = Rating::UNRATED;
            }
        }
        song_info.purge();
        *cache_expiration = time (nullptr) + StationFeedbackCacheTime;
    }

    /** Retrieve the latest ratings on file with the servers.
        @param direction True for positive ratings, false for negative.
        @param cache_expiration The time at which cached ratings will expire. */
    void Station::refreshRatings (bool direction, time_t *cache_expiration) {
        if (*cache_expiration < time (nullptr)) {
            return;
        }

        MusicAutoReleasePool pool;
        RequestStationFeedback request_ratings (pandora(), this, direction);
        Status status = pandora()->executeRequest (request_ratings);
        if (status != Status::Ok) {
            throw CommandError (E_NAK, status_strerror (status));
        }
        const SongList &song_ratings = request_ratings.getResponse();
        pandora()->library.unabridge (song_ratings);

        Rating rating = (direction ? ThumbsUp : ThumbsDown);

        // Clear out old ratings
        for (auto &entry : song_info) {
            if (entry.second.rating == rating) {
                entry.second.rating = Rating::UNRATED;
            }
        }
        for (const auto &song : song_ratings) {
            SongRating *rated_song = static_cast<SongRating *> (song);
            song_info.storeRating (rated_song->songPandoraId(), rating, rated_song->feedback_id);
        }
        song_info.purge();
        *cache_expiration = time (nullptr) + StationFeedbackCacheTime;
    }

    /// Retrieve the latest ratings on file with the servers.
    void Station::refreshRatings() {
        refreshRatings (true, &positive_rating_expiration);
        refreshRatings (false, &negative_rating_expiration);
    }

    // Forcibly update some ratings by expiring the cache and updating.
    void Station::forceRefreshRatings (bool direction) {
        time_t *expiration = (direction ? &positive_rating_expiration : &negative_rating_expiration);
        *expiration = 0;
        refreshRatings (direction, expiration);
        refreshRatings (false, &negative_rating_expiration);
    }

    ThingieList Station::getSeeds (void) const {
        // The API request thinks we don't need to change; we're just returning an existing list.
        // However, we may need to refresh the cache.  So, we're going to break constness.
        const_cast<Station *> (this)->refreshRatings();
        const_cast<Station *> (this)->refreshSeeds();

        ThingieList results;
        // Things with an info record either have a rating or are a seed.  We return both.
        for (const auto &info : song_info) {
            MusicThingie *item = pandora()->library.fulfill (info.first);
            if (item) {
                Song *song = static_cast<Song *> (item->asSong());
                if (song) {
                    if (!song->playlist()) {
                        // Assign this playlist to the song's or rating's playlist.
                        song->playlist (const_cast<Station *> (this));
                    } else if (song->playlist() != this) {
                        // Make a copy of the song with this station as the playlist.
                        Song *temp = new Song (*song);
                        temp->playlist (const_cast<Station *> (this));
                        item = temp;
                    }
                }
                results.push_back (item);
            } else {
                flog (LOG_ERROR, "No details available, id: ", info.first);
            }
        }
        return results;
    }

    void Station::rename (const std::string &new_name) {
        takePossession();
        if (!may_rename) {
            throw CommandError (E_UNSUPPORTED, "Rename of station disallowed");
        }
        RequestRenameStation rename (pandora(), this, new_name);
        Status status = pandora()->executeRequest (rename);
        if (status != Status::Ok) {
            throw CommandError (E_NAK, status_strerror (status));
        }
        playlistName (new_name);
    }

    void Station::erase() {
        RequestRemoveStation remove (pandora(), this);
        Status status = pandora()->executeRequest (remove);
        if (status != Status::Ok) {
            throw CommandError (E_NAK, status_strerror (status));
        }
        pandora()->removeStationByStationId (this->playlistId());
    }

    /** Get songs known to belong to a playlist.
        @return A list of songs from the playlist. */
    SongList Station::songs() {
        return (pandora()->library.getPlayableSongs (this, Filter::All));
    };

    /** Get songs belonging to a playlist.
        @param filter Match only matsongs matching this.  If omitted, match all.
        @return A list of matching songs from the playlist. */
    SongList Station::songs (const Filter &filter = Filter::All) {
        return (pandora()->library.getPlayableSongs (this, filter));
    }

}  // namespace Pandora
