///
/// Pandora shuffle station & mix management.
///	@file		shuffle.cpp - pianod2
///	@author		Perette Barella
///	@date		2020-03-29
///	@copyright	Copyright (c) 2020 Devious Fish. All rights reserved.
///

#include <config.h>

#include <string>
#include <vector>

#include "pandoracomm.h"
#include "pandoratypes.h"
#include "pandora.h"

namespace Pandora {
    /// Class representing the Pandora shuffle station.
    class ShuffleStation : public Station {
    public:
        ShuffleStation (Source *src, const Parsnip::Data &message) : Station (src, message) {
            playlistType (PianodPlaylist::MIX);
        }
    };

    /// Class representing a mix of all of a user's Pandora stations.
    class MixEverythingStation : public Station {
    public:
        MixEverythingStation (Source *src, const Parsnip::Data &message) : Station (src, message) {
            playlistType (PianodPlaylist::EVERYTHING);
            playlistName ("Everything");
        }
    };

    /// Pandora Message: Request shuffle station list and shuffle station info.
    class RetrieveShuffleStation : public Request {
    protected:
        mutable std::vector<std::string> shuffle_list;
        mutable Retainer<ShuffleStation *> shuffle_station;
        mutable Retainer<MixEverythingStation *> everything_station;

    public:
        RetrieveShuffleStation (Source *src) : Request (src, "v1/station/shuffle"){};
        virtual Parsnip::Data retrieveRequestMessage() const override {
            return Parsnip::Data (Parsnip::Data::Dictionary);
        }
        virtual void extractResponse (const Parsnip::Data &message) const override {
            shuffle_station = new ShuffleStation (source, message);
            everything_station = new MixEverythingStation (source, message);
            for (const auto &item : message["shuffleStationIds"]) {
                shuffle_list.push_back (item.asString());
            }
        }
        inline Station *getShuffleStation() const {
            assert (shuffle_station);
            return shuffle_station.get();
        }
        inline Station *getEverythingStation() const {
            assert (everything_station);
            return everything_station.get();
        }
        inline const std::vector<std::string> &getShuffleStations() const {
            assert (shuffle_station);  // Use as flag that request completed.
            return shuffle_list;
        }
    };

    /// Pandora message: set quickmix/shuffle stations.
    class SetShuffleStations : public RetrieveShuffleStation {
        bool shuffle_all{false};

    public:
        /// Request shuffle station list and shuffle station info.
        SetShuffleStations (Source *src, const std::vector<std::string> &station_ids, bool everything = false)
        : RetrieveShuffleStation (src), shuffle_all (everything) {
            shuffle_list = station_ids;
        };

        virtual Parsnip::Data retrieveRequestMessage() const override {
            Parsnip::Data ids (Parsnip::Data::List);
            for (const auto &id : shuffle_list) {
                ids.push_back (Parsnip::Data (id));
            }
            shuffle_list.clear();
            return Parsnip::Data (Parsnip::Data::Dictionary,
                                        "shuffleType",
                                        shuffle_all ? "all" : "my",
                                        "stationIds",
                                        std::move (ids));
        }
    };

    /** Update the quickmix/station shuffle by sending
        a list of desired stations to Pandora. */
    void Source::pushMixToServers() {
        std::vector<std::string> station_ids;
        for (auto &station : stations) {
            if (station.second->includedInMix()) {
                station_ids.push_back (station.second->playlistId());
            }
        }

        SetShuffleStations request (this, station_ids);
        Status status = comm.execute (request);
        if (status != Status::Ok) {
            flog (LOG_WHERE (LOG_ERROR), "Could not push updated mix stations to server.");
        }
    }

    /// Update the quickmix/station shuffle to play all stations.
    void Source::setMixAllOnServers() {
        std::vector<std::string> station_ids;
        for (auto &station : stations) {
            station_ids.push_back (station.second->playlistId());
        }

        SetShuffleStations request (this, station_ids, true);
        Status status = comm.execute (request);
        if (status != Status::Ok) {
            flog (LOG_WHERE (LOG_ERROR), "Could not set everything mix on server.");
        }
    }

    /// Retrieve the mix playlist.
    PianodPlaylist *Source::getMixPlaylist (void) {
        return mix_playlist.get();
    };

    /// Retrieve the everything playlist.
    PianodPlaylist *Source::getEverythingPlaylist (void) {
        return everything_playlist.get();
    };

    /** Initialize the quickmix/station shuffle.
       Retrieve list of quickmix stations from server, then
       set mixing flag for all stations listed.
       @param source The source owning the stations.
       @param stations The hash of stations. */
    bool Station::initializeMix (Source *source, StationLookup &stations) {
        assert (!source->mix_playlist);

        RetrieveShuffleStation shuffle_retrieve (source);
        Status status = source->executeRequest (shuffle_retrieve);
        if (status != Status::Ok) {
            return false;
        }

        auto mixers = shuffle_retrieve.getShuffleStations();

        // Turn off quickmix on all stations
        for (auto &station : stations) {
            station.second->in_quick_mix = false;
        }

        // Turn quickmix on again for those in mixing list
        for (const auto &item : mixers) {
            stations[item]->in_quick_mix = true;
        }

        source->mix_playlist = shuffle_retrieve.getShuffleStation();
        source->everything_playlist = shuffle_retrieve.getEverythingStation();
        return true;
    }
}  // namespace Pandora
