///
/// Rating and seed storage.
/// Stores and retrieves station data (seeds & feedback) for stations.
///	@file		mediaunits/pandora/pandoraseeds.cpp - pianod
///	@author		Perette Barella
///	@date		2020-04-02
///	@copyright	Copyright 2014-2020 Devious Fish.  All rights reserved.
///

#include <config.h>

#include <cassert>

#include <string>

#include "pandoratypes.h"
#include "pandorastation.h"

using namespace std;

namespace Pandora {

    /** Store a rating.
        @param song_id The ID of the song.
        @param rating The rating.
        @param feedback_id The feedback ID necessary to remove the rating.  May be empty string if unknown. */
    void StationSongInfo::storeRating (const std::string &song_id,
                                       const Rating rating,
                                       const std::string &feedback_id) {
        if (rating == Rating::UNRATED) {
            auto it = find (song_id);
            if (it != end()) {
                if (it->second.is_seed) {
                    it->second.rating = Rating::UNRATED;
                    it->second.feedback_id.clear();
                } else {
                    erase (it);
                }
            }
        } else {
            assert (rating == ThumbsUp || rating == ThumbsDown);
            SongInfo &info = (*this)[song_id];
            info.rating = rating;
            if (!feedback_id.empty()) {
                info.feedback_id = feedback_id;
            }
        }
    }

    /** Retrieve a rating.
        @param song_id The ID of the song.
        @param feedback_id If non-null, set to the feedback ID of the rating, if known.
        @return The rating of the specified song. */
    Rating StationSongInfo::getRating (const std::string &song_id, std::string *feedback_id) const {
        auto it = find (song_id);
        if (it == end()) {
            return Rating::UNRATED;
        }
        if (feedback_id) {
            *feedback_id = it->second.feedback_id;
        }
        return it->second.rating;
    }

    /** Mark a song as a seed.
        @param song_id The ID of the song.
        @param is_seed True to make the song a seed, false for not a seed. */
    void StationSongInfo::setSeed (const std::string &song_id, const bool is_seed) {
        if (!is_seed) {
            auto it = find (song_id);
            if (it != end()) {
                if (it->second.rating == Rating::UNRATED) {
                    erase (it);
                } else {
                    it->second.is_seed = is_seed;
                }
            }
        } else {
            (*this)[song_id].is_seed = is_seed;
        }
    }

    /** Mark a song as a seed.
        @param song_id The ID of the song.
        @return True if the specified song is a seed. */
    bool StationSongInfo::isSeed (const std::string &song_id) const {
        auto it = find (song_id);
        return (it == end() ? false : it->second.is_seed);
    }

    /// Remove superfluous entries from the table.
    void StationSongInfo::purge() {
        for (auto it = begin(); it != end();) {
            if (!it->second.is_seed && it->second.rating == Rating::UNRATED) {
                it = erase (it);
            } else {
                it++;
            }
        }
    }
}  // namespace Pandora
