///
/// Media player for Pandora.
/// This was once derived from pianobar, but that code has evolved away.
/// @file       mediaunits/pandora/pandoraplayer.cpp - pianod project
/// @author     Perette Barella
/// @date       2014-10-24
/// @copyright  Copyright 2012-2017 Devious Fish. All rights reserved.
///

#include <config.h>

#include "fundamentals.h"
#include "musictypes.h"
#include "retainer.h"

#include "pandoratypes.h"
#include "pandoramessages.h"
#include "pandora.h"

#include "mediaplayer.h"
#include "playerwrapper.h"

namespace Pandora {
    class Wrapper : public Media::PlayerWrapper {
        Source *pandora;
        Retainer<PlayableSong *> song;
        bool sent_play_notification {false};
        bool sent_pause_notification {false};

    public:
        Wrapper (Source *src, PlayableSong *sng, std::unique_ptr<Player> &&player)
        : Media::PlayerWrapper (std::move (player)), song (sng), pandora (src) {
        }

        virtual void pause (void) override {
            PlayerWrapper::pause();
            // Notify of pause unless we're right near the end anyway.
            float remaining = playRemaining();
            if ((remaining < 0.0f || remaining > 15.0f) && !sent_pause_notification) {
                sent_pause_notification = true;
                PlaybackPauseNotification start_notification (pandora, song.get());
                pandora->executeRequest (start_notification);
            }
        }

        virtual void abort (void) override {
            if (!sent_pause_notification) {
                PlaybackPauseNotification start_notification (pandora, song.get());
                pandora->executeRequest (start_notification);
            }
            PlayerWrapper::abort();
        }
        virtual void play (void) override {
            PlayerWrapper::play();
            if (!sent_play_notification) {
                PlaybackStartNotification start_notification (pandora, song.get());
                pandora->executeRequest (start_notification);
                sent_play_notification = true;
            } else if (sent_pause_notification) {
                PlaybackResumedNotification start_notification (pandora, song.get());
                pandora->executeRequest (start_notification);
                sent_pause_notification = false;
            }
        }

        //        // Indicate when stream will timeout from inactivity when paused.
        //        virtual time_t getPauseTimeout (void) override {
        //        }
    };

    class AdvertWrapper : public Media::PlayerWrapper {
        Source *pandora;
        Retainer<Advert *> advert;
        bool sent_play_notification {false};
        bool sent_pause_notification {false};
        mutable int pending_quarter_notification {0};

    public:
        AdvertWrapper (Source *src, Advert *ad, std::unique_ptr<Player> &&player)
        : Media::PlayerWrapper (std::move (player)), advert (ad), pandora (src) {
        }
        
        void sendNotifications (Advert::NotificationTarget &targets) const {
            for (const auto &notify : targets) {
                pandora->simpleNotification (notify);
            }
        }

        virtual void pause (void) override {
            PlayerWrapper::pause();
            if (PlayerWrapper::currentState() != Done && !sent_pause_notification) {
                sent_pause_notification = true;
                sendNotifications (advert->on_pause);
            }
        }

        virtual void play (void) override {
            PlayerWrapper::play();
            if (!sent_play_notification) {
                sent_play_notification = true;
                sendNotifications (advert->on_quarters [0]);
                pending_quarter_notification = 1;
            } else if (sent_pause_notification) {
                sendNotifications (advert->on_resume);
                sent_pause_notification = false;
            }
        }
        
        virtual float playPoint (void) const override {
            float point = PlayerWrapper::playPoint();
            float duration = trackDuration();
            if (point > 0.0 && duration > 0.0) {
                int completed_quarters = (point * 4) / duration;
                if (completed_quarters >= pending_quarter_notification &&
                    pending_quarter_notification < 5) {
                    sendNotifications (advert->on_quarters [pending_quarter_notification++]);
                    return PlayerWrapper::playPoint();
                }
            }
            return point;
        }

        virtual State currentState (void) const override {
            State current = PlayerWrapper::currentState();
            if (current == Done && pending_quarter_notification < 5) {
                sendNotifications (advert->on_quarters [4]);
                pending_quarter_notification = 5;
            }
            return current;
        }
    };

    Media::Player *Source::getPlayer (const AudioSettings &audio, PianodSong *s) {
        PlayableSong *song = dynamic_cast<PlayableSong *> (s);
        if (song) {
            if (!song->is_fresh || song->expired()) {
                RequestTrackReplay replay (this, song, last_played.get());
                Status status = comm.execute (replay);
                if (status != Status::Ok) {
                    flog (LOG_WHERE (LOG_WARNING), "Can not refresh track token for playback: ", status_strerror (status));
                    return (nullptr);
                }
                song = replay.getResponse();
                library.update (song);
            }
            std::unique_ptr<Media::Player> player (Media::Player::getPlayer (audio, song->audio_url, song->audio_gain));
            last_played = song;
            song->is_fresh = false;
            return new Wrapper (this, song, std::move (player));
        }
        Advert *ad = dynamic_cast <Advert *> (s);
        assert (ad);
        if (ad) {
            std::unique_ptr<Media::Player> player (Media::Player::getPlayer (audio, ad->audio_url, 0));
            last_played = song;
            return new AdvertWrapper (this, ad, std::move (player));
        }
        return nullptr;
    };
}  // namespace Pandora
