///
/// Pandora messages.
/// @file       mediaunits/pandora/pandoramessages.h - pianod project
/// @author     Perette Barella
/// @date       2020-03-24
/// @copyright  Copyright 2020 Devious Fish. All rights reserved.
///

#include <config.h>

#include <string>

#include "logging.h"
#include "fundamentals.h"
#include "musictypes.h"
#include "encapmusic.h"

#include "pandoratypes.h"
#include "pandoramessages.h"
#include "pandora.h"

namespace Pandora {
    const std::string Type_Track = "TR";
    const std::string Type_Album = "AL";
    const std::string Type_Artist = "AR";
    const std::string Type_Station = "ST";
    const std::string Type_StationFactory = "SF";

    Request::Request (Source *src, const char *ep) : source (src), endpoint (ep){};

    bool Request::debug() const {
        return false;
    }

    /** Extract assorted artists, albums and songs from a Pandora annotation response.
        @param message Parsnip Data from which to extract the things.
        @return The assembled list of things, converted to native constructs. */
    ThingieList Request::extractAssortedThingsFromDictionary (const Parsnip::Data &annotations) const {
        MusicAutoReleasePool pool;
        ThingieList results;
        std::function<void (const std::string &, const Parsnip::Data &)> handler{
                [&results, this] (const std::string &key, const Parsnip::Data &thing) -> void {
                    try {
                        const std::string &type = thing["type"].asString();
                        if (type == Type_Track) {
                            results.push_back (new Song (source, thing));
                        } else if (type == Type_Album) {
                            results.push_back (new Album (source, thing));
                        } else if (type == Type_Artist) {
                            results.push_back (new Artist (source, thing));
                        } else {
                            flog (LOG_WHERE (LOG_WARNING), "Unknown thing in Pandora response: ", type);
                            if (logging_enabled (LOG_PANDORA)) {
                                thing.dumpJson (key);
                            }
                        }
                    } catch (const Parsnip::NoSuchKey &ex) {
                        flog (LOG_WHERE (LOG_ERROR), "Unable to decode musicthingie: ", ex.what());
                        thing.dumpJson ("Problem item: " + key);
                    }
                }};
        annotations.foreach (handler);
        return results;
    }

    /*
     *              Notifications
     */

    Notification::Notification (Source *src, const char *ep) : Request (src, ep){};

    void Notification::extractResponse (const Parsnip::Data &message) const {
        // Response is empty or unimportant.
    }

    /*
     *              Retrieve Version Number
     */

    RetrieveVersionRequest::RetrieveVersionRequest (Source *src) : Request (src, "v5/collections/getVersion"){};

    Parsnip::Data RetrieveVersionRequest::retrieveRequestMessage() const {
        return Parsnip::Data (Parsnip::Data::Dictionary);
    }

    void RetrieveVersionRequest::extractResponse (const Parsnip::Data &message) const {
        result = message.asInteger();
    }

    /*
     *              Retrieve Station List
     */

    RequestStationList::RequestStationList (Source *src) : Request (src, "v1/station/getStations"){};

    Parsnip::Data RequestStationList::retrieveRequestMessage() const {
        return Parsnip::Data (Parsnip::Data::Dictionary, "pageSize", 250, "startIndex", 0);
    }

    void RequestStationList::extractResponse (const Parsnip::Data &message) const {
        MusicAutoReleasePool pool;
        results.reset (new PlaylistList());
        for (const auto &station_info : message["stations"]) {
            try {
                results->push_back (new Station (source, station_info));
                if (station_info["stationType"].asString() != "SEEDED_STATION") {
                    flog (LOG_WARNING, "Station type other than SEEDED_STATION");
                    if (logging_enabled (LOG_WARNING)) {
                        station_info.dumpJson ("Station " + station_info["stationId"].asString());
                    }
                }
            } catch (const Parsnip::NoSuchKey &ex) {
                flog (LOG_WHERE (LOG_ERROR), "Unable to decode musicthingie: ", ex.what());
                station_info.dumpJson ("Problem station");
            }
        }
    }

    /*
     *              Create a station
     */

    RequestCreateStation::RequestCreateStation (Source *src,
                                                const std::string &nam,
                                                const std::string &creator_pandora_id)
    : Request (src, "v1/station/createStation"), name (nam), create_from_id (creator_pandora_id){};

    Parsnip::Data RequestCreateStation::retrieveRequestMessage() const {
        Parsnip::Data new_name;  // Defaults to null.
        if (!name.empty()) {
            new_name = name;
        }
        return Parsnip::Data (Parsnip::Data::Dictionary,
                                    "stationName",
                                    new_name,
                                    "creationSource",
                                    nullptr,
                                    "stationCode",
                                    nullptr,
                                    "pandoraId",
                                    create_from_id,
                                    "searchQuery",
                                    nullptr,
                                    "creativeId",
                                    nullptr,
                                    "lineId",
                                    nullptr);
    }

    void RequestCreateStation::extractResponse (const Parsnip::Data &message) const {
        result = new Station (source, message);
    }

    /*
     *              Remove a station
     */

    RequestRemoveStation::RequestRemoveStation (Source *src, Station *station)
    : Request (src, "v1/station/removeStation"), station_id (station->playlistId()) {
        assert (!station_id.empty());
        assert (station->playlistType() == PianodPlaylist::PlaylistType::SINGLE);
    };

    Parsnip::Data RequestRemoveStation::retrieveRequestMessage() const {
        return Parsnip::Data (Parsnip::Data::Dictionary, "stationId", station_id);
    }

    void RequestRemoveStation::extractResponse (const Parsnip::Data &) const {
        // Response is empty.
    }

    /*
     *              Transform a shared station to a personal station
     */

    RequestTransformStation::RequestTransformStation (Source *src, Station *station)
    : Request (src, "v1/station/transformShared"), station_id (station->playlistId()) {
        assert (!station_id.empty());
        assert (station->playlistType() == PianodPlaylist::PlaylistType::SINGLE);
    };

    Parsnip::Data RequestTransformStation::retrieveRequestMessage() const {
        return Parsnip::Data (Parsnip::Data::Dictionary, "stationId", station_id);
    }

    void RequestTransformStation::extractResponse (const Parsnip::Data &message) const {
        updated_info = new Station (source, message);
    }

    /*
     *              Rename a station
     */

    RequestRenameStation::RequestRenameStation (Source *src, Station *station, const std::string &new_name)
    : Request (src, "v1/station/updateStation"), station_id (station->playlistId()), name (new_name) {
        assert (!station_id.empty());
        assert (!name.empty());
        assert (station->playlistType() == PianodPlaylist::PlaylistType::SINGLE);
    };

    Parsnip::Data RequestRenameStation::retrieveRequestMessage() const {
        return Parsnip::Data (Parsnip::Data::Dictionary, "stationId", station_id, "name", name);
    }

    void RequestRenameStation::extractResponse (const Parsnip::Data &message) const {
        // Disregard response
    }

    /*
     *              Retrieve Station Seeds
     */

    RequestStationSeeds::RequestStationSeeds (Source *src, const char *endpoint, Station *sta)
    : Request (src, endpoint), station_id (sta->playlistId()){};

    RequestStationSeeds::RequestStationSeeds (Source *src, Station *sta)
    : Request (src, "v1/station/getSeeds"), station_id (sta->playlistId()){};

    Parsnip::Data RequestStationSeeds::retrieveRequestMessage() const {
        return Parsnip::Data (Parsnip::Data::Dictionary, "stationId", station_id);
    }

    /** Determine the type of seed, then construct a native seed from the serialized form.
        @return Pointer to the native object, or nullptr on error. */
    MusicThingie *RequestStationSeeds::extractAssortedSeed (const Parsnip::Data &seed_info) const {
        try {
            Station *station = source->getStationByStationId (station_id);
            assert (station);
            if (seed_info.contains ("song")) {
                return new SongSeed (source, seed_info, station);
            } else if (seed_info.contains ("artist")) {
                return new ArtistSeed (source, seed_info);
            } else if (seed_info.contains ("genre")) {
                return new GenreSeed (source, seed_info);
            } else {
                flog (LOG_ERROR, "Unknown seed type");
                seed_info.dumpJson ("Seed details");
            }
        } catch (const Parsnip::NoSuchKey &ex) {
            flog (LOG_WHERE (LOG_ERROR), "Unable to decode seed: ", ex.what());
            seed_info.dumpJson ("Problem seed");
        }
        return nullptr;
    }

    void RequestStationSeeds::extractResponse (const Parsnip::Data &message) const {
        ThingieList seed_list;
        for (const auto &seed_info : message["seeds"]) {
            MusicThingie *seed = extractAssortedSeed (seed_info);
            if (seed) {
                seed_list.push_back (seed);
            }
        }
        seeds.reset (new ThingieList (std::move (seed_list)));
    }

    /*
     *              Retrieve Station Details (seeds and a few other tidbits)
     */

    RequestStationDetails::RequestStationDetails (Source *src, Station *sta)
    : RequestStationSeeds (src, "v1/station/getStationDetails", sta){};

    Parsnip::Data RequestStationDetails::retrieveRequestMessage() const {
        return Parsnip::Data (Parsnip::Data::Dictionary,
                                    "stationId",
                                    station_id,
                                    "isCurrentStation",
                                    false);
    }

    void RequestStationDetails::extractResponse (const Parsnip::Data &message) const {
        // Ignore "initialSeed", it is irrelevant.
        positive_feedback = message["positiveFeedbackCount"].asInteger();
        negative_feedback = message["negativeFeedbackCount"].asInteger();
        RequestStationSeeds::extractResponse (message);
    }

    /*
     *              Alter (add or remove) a station's seeds
     */

    RequestAlterStationSeed::RequestAlterStationSeed (Source *src,
                                                      Station *station,
                                                      const std::string &id,
                                                      bool positive)
    : Request (src, positive ? "v1/station/addSeed" : "v1/station/deleteSeed"),
      station_id (station->playlistId()),
      seed_id (id){};

    Parsnip::Data RequestAlterStationSeed::retrieveRequestMessage() const {
        return Parsnip::Data (Parsnip::Data::Dictionary,
                                    "stationId",
                                    station_id,
                                    "musicId",
                                    pandora_to_music_id (seed_id));
    }

    void RequestAlterStationSeed::extractResponse (const Parsnip::Data &message) const {
        // Nothing to do
    }

    /*
     *              Retrieve a list of feedback (ratings)
     */

    RequestStationFeedback::RequestStationFeedback (Source *src, Station *station, bool pos)
    : Request (src, "v1/station/getStationFeedback"), station_id (station->playlistId()), positive (pos){};

    Parsnip::Data RequestStationFeedback::retrieveRequestMessage() const {
        return Parsnip::Data (Parsnip::Data::Dictionary,
                                    "pageSize",
                                    100,
                                    "startIndex",
                                    1,
                                    "stationId",
                                    station_id,
                                    "positive",
                                    positive);
    }

    void RequestStationFeedback::extractResponse (const Parsnip::Data &message) const {
        SongList rated_list;
        for (const auto &song_info : message["feedback"]) {
            rated_list.push_back (new SongRating (source, song_info));
        }
        results.reset (new SongList (std::move (rated_list)));
    }

    /*
     *              Mark a song as overplayed/do not play for awhile.
     */

    RequestAddTiredSong::RequestAddTiredSong (Source *src, const std::string &token)
    : Request (src, "v1/station/addFeedback"), track_token (token){};

    Parsnip::Data RequestAddTiredSong::retrieveRequestMessage() const {
        return Parsnip::Data (Parsnip::Data::Dictionary, "trackToken", track_token);
    }

    void RequestAddTiredSong::extractResponse (const Parsnip::Data &message) const {
        // Response is empty.
    }

    /*
     *              Add feedback to a song that has played.
     */

    RequestAddFeedback::RequestAddFeedback (Source *src, const std::string &token, bool like)
    : Request (src, "v1/station/addFeedback"), track_token (token), positive (like){};

    Parsnip::Data RequestAddFeedback::retrieveRequestMessage() const {
        return Parsnip::Data (Parsnip::Data::Dictionary, "trackToken", track_token, "isPositive", positive);
    }

    void RequestAddFeedback::extractResponse (const Parsnip::Data &message) const {
        feedback_id.reset (new std::string (message["feedbackId"].asString()));
    }

    /*
     *              Delete feedback for a song.
     */

    RequestDeleteFeedback::RequestDeleteFeedback (Source *src, const std::string &token, bool like)
    : Request (src, "v1/station/deleteFeedback"), feedback_id (token), positive (like){};

    Parsnip::Data RequestDeleteFeedback::retrieveRequestMessage() const {
        return Parsnip::Data (Parsnip::Data::Dictionary, "feedbackId", feedback_id, "isPositive", positive);
    }

    void RequestDeleteFeedback::extractResponse (const Parsnip::Data &message) const {
        // Do nothing.
    }

    /*
     *              Retrieve a list of items for the playback queue
     */

    RequestQueueTracks::RequestQueueTracks (Source *src, const Station *station, const PlayableSong *last)
    : Request (src, "v1/playlist/getFragment"), station_id (station->playlistId()), last_play (last) {};

    Parsnip::Data RequestQueueTracks::retrieveRequestMessage() const {
        Parsnip::Data req (Parsnip::Data::Dictionary,
                                    "stationId",
                                    station_id,
                                    "isStationStart",
                                    true,
                                    "fragmentRequestReason",
                                    "Normal",
                                    "audioFormat",
                                    source->userFeatures().hifi_audio_encoding ? "mp3-hifi" : "mp3",
                                    "startingAtTrackId",
                                    nullptr,
                                    "onDemandArtistMessageArtistUidHex",
                                    nullptr,
                                    "onDemandArtistMessageIdHex",
                                    nullptr);
        if (last_play) {
            req ["lastPlayedTrackToken"] = last_play->trackToken();
        }
        return req;
    }

    void RequestQueueTracks::extractResponse (const Parsnip::Data &message) const {
        MusicAutoReleasePool pool;
        results.reset (new SongList());
        for (const auto &track_info : message["tracks"]) {
            try {
                results->push_back (new PlayableSong (source, track_info));
            } catch (const Parsnip::Exception &ex) {
                flog (LOG_WHERE (LOG_ERROR), "Unable to decode track: ", ex.what());
                track_info.dumpJson ("Problem queue track");
            }
        }
    }

    /*
     *              Request to replay a previously played track
     */

    RequestTrackReplay::RequestTrackReplay (Source *src, const PlayableSong *song, const PlayableSong *current)
    : Request (src, "v1/ondemand/getReplayTrack"), request (song), last_played (current){};

    Parsnip::Data RequestTrackReplay::retrieveRequestMessage() const {
        assert (request->playlist());
        Parsnip::Data req{Parsnip::Data::Dictionary,
                                "stationId",
                                request->playlist()->playlistId(),
                                "artistUid",
                                nullptr,
                                "lastPlayedTrackToken",
                                nullptr,
                                "trackToken",
                                request->trackToken()};
        if (last_played) {
            req ["lastPlayedTrackToken"] = last_played->trackToken();
        }
        return req;
    }

    void RequestTrackReplay::extractResponse (const Parsnip::Data &message) const {
        result = new PlayableSong (source, message ["replayTrack"]);
    }

    /*
     *              Playback start notification
     */

    /// Pandora message: Notify servers of playback start
    PlaybackStartNotification::PlaybackStartNotification (Source *src, const PlayableSong *song)
    : Notification (src, "v1/station/trackStarted"), track_token (song->trackToken()){};

    Parsnip::Data PlaybackStartNotification::retrieveRequestMessage() const {
        return Parsnip::Data (Parsnip::Data::Dictionary, "trackToken", track_token);
    }

    /*
     *              Playback paused notification
     */

    /// Pandora message: Notify servers of playback start
    PlaybackPauseNotification::PlaybackPauseNotification (Source *src, const PlayableSong *)
    : Notification (src, "v1/station/playbackPaused"){};

    Parsnip::Data PlaybackPauseNotification::retrieveRequestMessage() const {
        return Parsnip::Data (Parsnip::Data::Dictionary, "sync", false);
    }

    /*
     *              Playback resumed notification
     */

    PlaybackResumedNotification::PlaybackResumedNotification (Source *src, const PlayableSong *)
    : Notification (src, "v1/station/playbackResumed"){};

    Parsnip::Data PlaybackResumedNotification::retrieveRequestMessage() const {
        return Parsnip::Data (Parsnip::Data::Dictionary, "forceActive", false);
    }

    /*
     *              Search Request
     */

    const std::string SearchRequest::Type_ANY = "";
    SearchRequest::SearchRequest (Source *src, const std::string &q, const std::string &type)
    : Request (src, "v3/sod/search"), query (q), desired_type (type){};

    Parsnip::Data SearchRequest::retrieveRequestMessage() const {
        Parsnip::Data type_list;
        if (desired_type.empty()) {
            type_list = Parsnip::Data{Parsnip::Data::List,
                                            Type_Album,
                                            Type_Artist,
                                            Type_Track,
                                            Type_StationFactory};
        } else {
            type_list = Parsnip::Data{Parsnip::Data::List, desired_type};
        }

        return Parsnip::Data{Parsnip::Data::Dictionary,
                                   "query",
                                   query,
                                   "types",
                                   std::move (type_list),
                                   "listener",
                                   Parsnip::Data (nullptr),
                                   "start",
                                   0,
                                   "count",
                                   20,
                                   "annotate",
                                   false};
    }

    void SearchRequest::extractResponse (const Parsnip::Data &message) const {
        std::vector<std::string> items;
        for (const auto &item : message["results"]) {
            items.push_back (item.asString());
        }
        results.reset (new std::vector<std::string> (items));
    }

    /*
     *              Retrieve annoations (item details) from server
     */

    RetrieveAnnotations::RetrieveAnnotations (Source *src, const std::vector<std::string> &items)
    : Request (src, "v4/catalog/annotateObjectsSimple"), retrieval_items (items){};

    Parsnip::Data RetrieveAnnotations::retrieveRequestMessage() const {
        Parsnip::Data list{Parsnip::Data::List};
        for (const auto item : retrieval_items) {
            list.push_back (Parsnip::Data (item));
        }
        return Parsnip::Data{Parsnip::Data::Dictionary, "pandoraIds", std::move (list)};
    }

    void RetrieveAnnotations::extractResponse (const Parsnip::Data &message) const {
        results.reset (new ThingieList (extractAssortedThingsFromDictionary (message)));
    }

    /*
     *              Retrieve annoation for a single item from server.
     */
    RetrieveSingleAnnotation::RetrieveSingleAnnotation (Source *src, const std::string &item)
    : RetrieveAnnotations (src, std::vector<std::string>{item}) {
    }

    /*
     *              Retrieve advertisements from server
     */

    RetrieveAdverts::RetrieveAdverts (Source *src, PlayableSong *last_play, Station *sta)
    : Request (src, "v1/ad/getAdList"), last_song (last_play), station (sta){};

    Parsnip::Data RetrieveAdverts::retrieveRequestMessage() const {
        Parsnip::Data req{Parsnip::Data::Dictionary,
                                "currentTrackDeliveryType",
                                nullptr,
                                "currentTrackToken",
                                nullptr,
                                "currentTrackSecondsPlayed",
                                0,
                                "currentTrackStationId",
                                nullptr,
                                "secondsUntilAd",
                                5,
                                "audioAdIndex",
                                1,
                                "videoAdIndex",
                                1,
                                "adBlockerEnabled",
                                false,
                                "nextTrackStationId",
                                nullptr,
                                "skipForReplay",
                                false,
                                "currentTrackType",
                                "Track",
                                "userInitiatedTrackEndActionType",
                                "AUDIO_UPDATE",
                                "httpReferrer",
                                "",
                                "nextTrackDeliveryType",
                                nullptr,
                                "userInitiatedTrackEnd",
                                false,
                                "secondsSinceDisplayAd",
                                1,
                                "userTimedOut",
                                false};
        if (last_song) {
            req["currentTrackToken"] = last_song->trackToken();
            int last_duration = last_song->duration();
            req["currentTrackSecondsPlayed"] = (last_duration < 5 ? last_duration : (last_duration - 5));
            req["secondsSinceDisplayAd"] = last_duration / 2;
            if (last_song->playlist()) {
                req["nextTrackStationId"] = last_song->playlist()->playlistId();
            }
        }
        if (station) {
            req["nextTrackStationId"] = station->playlistId();
        }
        return req;
    }

    void RetrieveAdverts::extractResponse (const Parsnip::Data &message) const {
        SongList ads;
        for (const auto &advert : message["ads"]) {
            try {
                ads.push_back (new Advert (source, advert));
            } catch (const Parsnip::NoSuchKey &ex) {
                flog (LOG_WHERE (LOG_ERROR), "Unable to decode advert", ex.what());
                advert.dumpJson ("Advert: ");
            }
        }
        adverts.reset (new SongList (std::move (ads)));
    }
}  // namespace Pandora
