///
/// Command handlers for Pandora-specific stuff.
///	@file		mediaunits/pandora/pandoracommand.cpp - pianod
///	@author		Perette Barella
///	@date		2014-11-30
///	@copyright	Copyright 2014-2020 Devious Fish.  All rights reserved.
///

#include <config.h>

#include <cassert>

#include <football.h>

#include "pandora.h"

#include "fundamentals.h"
#include "sources.h"
#include "connection.h"
#include "response.h"
#include "user.h"
#include "users.h"
#include "mediamanager.h"

using namespace std;


typedef enum pandora_options_t {
    PROXY = 1,
    CONTROLPROXY,
    PAUSETIMEOUT,
    PLAYLISTTIMEOUT,
    CACHESIZE,
} PANDORA_OPTION;

static const FB_PARSE_DEFINITION statementOptions [] = {
    { PROXY,			"proxy {url} ..." },							// Set the proxy for all accesses
    { CONTROLPROXY,     "control proxy {url} ..." },					// Set the proxy for control connection only
    { PAUSETIMEOUT,     "pause timeout {#duration:15-86400} ..." },		// Timeout if paused too long
    { PLAYLISTTIMEOUT,  "playlist timeout {#duration:1800-86400} ..." },// Expire items queued too long
    { CACHESIZE,        "cache minimum {#minimum:800-989999} maximum {#maximum:1000-999999} ..." },
    { CMD_INVALID, NULL }
};


class PandoraOptions : public Media::SourceParameterParser<Pandora::ConnectionParameters, PANDORA_OPTION> {
    virtual const FB_PARSE_DEFINITION *statements (void) {
        return statementOptions;
    };
    virtual int handleOption (PANDORA_OPTION option, Pandora::ConnectionParameters &dest) override {
        switch (option) {
            case CONTROLPROXY:
                dest.control_proxy = argv ("url");
                if (strncasecmp (dest.control_proxy.c_str(), "http:", 5) != 0)
                    return FB_PARSE_RANGE;
                break;
            case PAUSETIMEOUT:
                dest.pause_timeout = atoi (argv ("duration"));
                break;
            case PLAYLISTTIMEOUT:
                dest.playlist_expiration = atoi (argv ("duration"));
                break;
            case PROXY:
                dest.proxy = argv ("url");
                if (strncasecmp (dest.proxy.c_str(), "http:", 5) != 0)
                    return FB_PARSE_RANGE;
                break;
            case CACHESIZE:
                dest.cache_minimum = atoi (argv ("minimum"));
                dest.cache_maximum = atoi (argv ("maximum"));
                // Make sure maximum is somewhat larger than minimum.
                if (dest.cache_minimum + 100 > dest.cache_maximum)
                    return FB_PARSE_RANGE;
                break;
            default:
                return common_parameters::handleOption (option, dest);
        }
        return FB_PARSE_SUCCESS;
    }
public:
    PandoraOptions () {
        addStatements (statementOptions);
    }
};


static PandoraOptions optionsParser;

typedef enum pandora_commands_t {
    PANDORAUSER = CMD_RANGE_PANDORA,
    PANDORAEXISTING,
    PANDORASETTINGS,
} COMMAND;

static const FB_PARSE_DEFINITION statementList[] = {
    { PANDORAUSER,		"pandora [plus:one|plus] user {user} {password} [{connection_options}] ..." },
                                                                        // Set Pandora password username/password
    { PANDORAEXISTING,  "pandora use {name} [{connection_options}] ..." },
                                                                        // Use existing/revised user's Pandora credentials
    { PANDORASETTINGS,  "pandora settings" },                           // Retrieve connection parameters
    { CMD_INVALID, NULL }
};


class PandoraCommands : public Football::Interpreter<PianodConnection, COMMAND> {
private:
    virtual bool hasPermission (PianodConnection &conn, COMMAND command) override;
    virtual void handleCommand (PianodConnection &conn, COMMAND command) override;
    virtual const FB_PARSE_DEFINITION *statements (void) override {
        return statementList;
    };
};

bool PandoraCommands::hasPermission (PianodConnection &conn, COMMAND) {
    return conn.havePrivilege (Privilege::Service);
}

void PandoraCommands::handleCommand (PianodConnection &conn, COMMAND command) {
    switch (command) {
        case PANDORAUSER:
            if (!conn.authenticated()) {
                conn << E_LOGINREQUIRED;
            } else {
                Pandora::ConnectionParameters params;
                params.username = conn.argv ("user");
                params.password = conn.argv ("password");
                params.owner = conn.user;
                params.name = conn.username();
                if (optionsParser.interpret(conn.argvFrom ("connection_options"), params, &conn) == FB_PARSE_SUCCESS) {
                    media_manager->add (new Pandora::Source (params), conn);
                }
            }
            return;
        case PANDORAEXISTING:
        {
            UserData::StringDictionary *persisted;
            User *owner;
            RESPONSE_CODE status = user_manager->findStoredSource (SOURCE_NAME_PANDORA,
                                                                   conn.argv ("name"),
                                                                   conn.user,
                                                                   &persisted,
                                                                   &owner);
            if (status != S_OK) {
                conn << status;
                return;
            }
            try {
                Pandora::ConnectionParameters params (*persisted);
                params.owner = owner;
                if (optionsParser.interpret(conn.argvFrom ("connection_options"), params, &conn) == FB_PARSE_SUCCESS) {
                    media_manager->add (new Pandora::Source (params), conn, owner != conn.user);
                }
            } catch (const invalid_argument &e) {
                conn << Response (E_CREDENTIALS, e.what());
            }
            return;
        }
        case PANDORASETTINGS:
        {
            if (conn.source()->isDecendableBy(conn.user)) {
                Pandora::Source *source = dynamic_cast<Pandora::Source *>(conn.source());
                if (source) {
                    conn << S_DATA
                    << Response (I_PROXY, source->connectionParams()->proxy)
                    << Response (I_CONTROLPROXY, source->connectionParams()->control_proxy)
                    << Response (I_PAUSE_TIMEOUT, source->connectionParams()->pause_timeout)
                    << Response (I_PLAYLIST_TIMEOUT, source->connectionParams()->playlist_expiration)
                    << Response (I_CACHE_MINIMUM, source->connectionParams()->cache_minimum)
                    << Response (I_CACHE_MAXIMUM, source->connectionParams()->cache_maximum)
                    << S_DATA_END;
                } else {
                    conn << Response (E_WRONG_STATE, "Pandora source required");
                }
            } else {
                conn << E_UNAUTHORIZED;
            }
            return;
        }
        default:
            conn << E_NOT_IMPLEMENTED;
            flog (LOG_WHERE (LOG_WARNING), "Unimplemented command ", command);
            break;
    }
}

static PandoraCommands interpreter;

void register_pandora_commands (PianodService *service) {
    service->addCommands(&interpreter);
}

void restore_pandora_source (UserData::StringDictionary *persisted, User *user) {
    Media::Source *m = nullptr;
    try {
        Pandora::ConnectionParameters params (*persisted);
        params.owner = user;
        m = new Pandora::Source (params);
        if (!media_manager->add (m))
            delete m;
    } catch (...) {
        delete m;
        throw;
    }
    return;
}
