///
/// Pandora data types for pianod.
/// @file       mediaunits/pandora/pandora.h - pianod project
/// @author     Perette Barella
/// @date       2014-10-23
/// @copyright  Copyright 2012-2020 Devious Fish. All rights reserved.
///

#pragma once

#include <config.h>

#include <ctime>

#include <string>
#include <unordered_map>
#include <functional>

#include "fundamentals.h"
#include "musictypes.h"
#include "musiccache.h"
#include "mediaplayer.h"
#include "mediaunit.h"
#include "mediaparameters.h"

#include "pandoratypes.h"
#include "pandorastation.h"
#include "pandoracomm.h"

/// Pandora source, player and related datatype specializations.
namespace Pandora {
    class Source;

    /** A library for stuff retrieved from Pandora.
        The `unabridge` member functions check the library for
        an item.  If not there, they add a new entry.  If it is
        there, they fill in missing fields in both the library's
        copy and the caller's copy.
        
        The `abridged` function accepts a constant song.  Since it
        cannot be updated. the library copy is updated and returned
        to the caller.
        
        The `update` member functions always replace the library's
        copy with the caller's.  If there are missing fields, they
        are filled with details from the library's copy prior to
        replacing it.
        
        The `fulfill` functions accept a list of Pandora item IDs.
        If in the library, they are returned as-is.  Otherwise,
        details are gathered from Pandora prior to returning. */

    class Library : public ThingiePool {
        Source *source;

    public:
        Library (Source *src, const ThingiePoolParameters &params);

        void add (MusicThingie *thing) = delete;
        void add (const SongList &songs) = delete;
        void add (const ThingieList &things) = delete;

        Song *unabridge (Song *song);
        const Song *unabridged (const Song *song);

        MusicThingie *unabridge (MusicThingie *thingie);
        void unabridge (const SongList &songs);
        void unabridge (const ThingieList &things);

        Song *update (Song *song);
        void update (const SongList &songs);

        MusicThingie *fulfill (const std::string &pandora_id);
        ThingieList fulfill (const std::vector<std::string> &list);

        SongList getPlayableSongs (std::function<bool (PlayableSong *)> matcher, const Filter &filter) const;
        SongList getPlayableSongs (Station *station, const Filter &filter) const;
        SongList getPlayableSongs (Artist *artist) const;
        SongList getPlayableSongs (Album *album) const;

        Parsnip::Data persist () const;
        void restore (const Parsnip::Data &data);
    };

    class Source : public Media::Source {
        friend bool Station::initializeMix (Source *, StationLookup &stations);

    private:
        // Member variables
        StationLookup stations;                   ///< A collection of stations, indexed by station ID.
        time_t station_list_expiration = 0;       ///< When cached station list expires
        Retainer<Station *> mix_playlist;         ///< The mix/shuffle meta-playlist.
        Retainer<Station *> everything_playlist;  ///< The everything meta-playlist.
        Communication comm;                       ///< Pandora server communicator.
        Retainer <PlayableSong *> last_played;    ///< Last played song
        time_t write_time {0};                    ///< Time at which data should be persisted
        std::unique_ptr<Parsnip::Data> recovery; ///< Data from persistence file

    public:
        Library library;               ///< Lookup table of things we gave out.
        SkipTracker skips{60, 86400};  ///< Skip counts for limiting

        inline const ConnectionParameters *connectionParams() const {
            return static_cast<const ConnectionParameters *> (parameters);
        }
        Source (const ConnectionParameters &params);
        ~Source();
        using Media::Source::alert;

        // Identity
        virtual const char *kind (void) const override;

        // Capabilities
        virtual bool canExpandToAllSongs () const override;
        virtual bool requireNameForCreatePlaylist (void) const override;

        // General source stuff
        virtual bool flush (void) override;
        virtual float periodic (void) override;

        // Playlist methods
        virtual PlaylistList getPlaylists (const Filter &filter = Filter::All) override;
        virtual PianodPlaylist *getMixPlaylist (void) override;
        virtual PianodPlaylist *getEverythingPlaylist (void) override;
        virtual Media::Player *getPlayer (const AudioSettings &audio, PianodSong *song) override;

        virtual MusicThingie *getAnythingById (const Media::SplitId &id) override;
        virtual ThingieList getSuggestions (const Filter &filter, SearchRange where) override;

        // Creating stations
        virtual PianodPlaylist *createPlaylist (const char *name, MusicThingie::Type type, MusicThingie *from) override;

        // Miscellaneous
        virtual SongList getRandomSongs (PianodPlaylist *playlist, const UserList &, Media::SelectionMethod) override;

        // Typecasts
        virtual MusicThingie *getSuggestion (MusicThingie *thing, MusicThingie::Type type, SearchRange where) override;

        // Functions for use within Pandora library.
        const std::string getRelevantSeedId (MusicThingie::Type seedType, const MusicThingie *music);
        Station *getStationByStationId (const std::string &station_id);
        void removeStationByStationId (const std::string &station_id);
        inline Status executeRequest (Request &request) {
            return comm.execute (request);
        }
        inline Status simpleNotification (const std::string &url) {
            return comm.sendSimpleNotification (url);
        }
        inline const UserFeatures &userFeatures() const {
            return comm.getFeatures();
        }
        
        /** Mark source data for persistence.  If called frequently,
            flushing will happen sooner. */
        inline void markDirty () {
            time_t now = time (nullptr);
            if (write_time == 0) {
                write_time = now + 3600 * 6;
            } else if (write_time > now) {
                write_time -= 5;
            }
        }


    private:
        // Internal functions
        bool updateStationList (PianodPlaylist::PlaylistType mixSetting = PianodPlaylist::SINGLE);
        bool initializeMix();
        void pushMixToServers();
        void setMixAllOnServers();
        bool restore ();
    };
}  // namespace Pandora
