///
/// Class for source parameters and their parsing.
///	@file		mediaparameters.h - pianod2
///	@author		Perette Barella
///	@date		2015-01-28
///	@copyright	Copyright (c) 2015-2020 Devious Fish. All rights reserved.
///

#pragma once 

#include <config.h>

#include <string>

#include <footparser.h>

#include "fundamentals.h"
#include "ownership.h"
#include "datastore.h"


namespace Media {
    /// How a source participates in track-source substitutions;
    enum ProxyMode {
        None, ///< Source does not participate.
        Donor, ///< Source provides substitute tracks.
        Recipient ///< Source wants its tracks replaced when possible.
    };

    enum PersistenceMode {
        Temporary, ///< Provided from command line, store neither settings nor indexes.
        Loaded, ///< Values loaded from stored data, do not update settings but do update indexes.
        Remember, ///< Store it, store indexes, but manually restore only.
        Restore, ///< Store it, store indexes, and automatically restore on startup.
    };

    /** Source parameter manager.  Implements common parameters, and can
        be extended for each source type to support custom parameters. */
    struct SourceParameters {
        friend int interpretCommonSourceParameters (Football::GenericOptionParser &parser,
                                                    int cmd, SourceParameters &result);
        bool userSpecified = false; // For private use by parameter parser.
    public:
        ProxyMode song_proxy = ProxyMode::None;
        User *owner = nullptr;
        Ownership::Type permissions = Ownership::Type::PRIVATE;
        PersistenceMode persistence = PersistenceMode::Temporary; ///< Persistence state.
        std::string name;
        bool waitForReady = false; ///< Wait on initialization.  Flag never persisted.

        SourceParameters (void) { };
        SourceParameters (Ownership::Type perm, User *user = nullptr);
        SourceParameters (Ownership::Type perm,
                          const std::string &sourcename);
        SourceParameters (const UserData::StringDictionary &src);
        virtual ~SourceParameters ();
        virtual bool persist (UserData::StringDictionary &dest) const;
    };


    class SourceParameters;

    extern FB_PARSE_DEFINITION standard_options [];
    extern int interpretCommonSourceParameters (Football::GenericOptionParser &parser,
                                                int cmd, SourceParameters &result);

    /// Parser for the basic set of source parameters.
    template <typename DataObjectType, typename OptionEnum> class SourceParameterParser
    : public Football::OptionParser<DataObjectType, OptionEnum> {
        typedef Football::OptionParser<DataObjectType, OptionEnum> base_type;
    public:
        typedef SourceParameterParser<DataObjectType, OptionEnum> common_parameters;
        using base_type::argv;
        using base_type::addStatements;
        SourceParameterParser () {
            addStatements (standard_options);
        };
        virtual int handleOption (OptionEnum cmd, DataObjectType &result) override {
            return interpretCommonSourceParameters (*this, cmd, result);
        };
    };
}

