///
/// ffmpeg audio player and metadata reader.
///	@file		ffmpegplayer.cpp - pianod2
///	@author		Perette Barella
///	@date		2015-03-02
///	@copyright	Copyright (c) 2015-2016 Devious Fish. All rights reserved.
///

#include <config.h>

#include <stdio.h>

#include <string>

#pragma GCC diagnostic push
#pragma GCC diagnostic ignored "-Wdocumentation"
#pragma GCC diagnostic ignored "-Wdeprecated"
extern "C" {
#include <libavformat/avformat.h>
#include <libavcodec/avcodec.h>
#include <libavutil/replaygain.h>
#include <libavfilter/avfilter.h>
#ifdef HAVE_LIBAVFILTER_AVFILTERGRAPH_H
#include <libavfilter/avfiltergraph.h>
#endif
#include <libavutil/dict.h>
}
#pragma GCC diagnostic pop


#include "fundamentals.h"
#include "logging.h"
#include "mediaplayer.h"
#include "audiooutput.h"
#include "ffmpegplayer.h"

using namespace std;
namespace Audio {

    /** Base abstract class for reading a media file or URL using ffmpeg.
        @param media_url The filename or URL of the media.
        @param timeout The timeout for reading the media stream, in seconds. */
    LibavMediaReader::LibavMediaReader (const string &media_url,
                                        int timeout) :
    url (media_url) {
        is_network = (strncasecmp (media_url.c_str(), "http", 4) == 0);

        string timeout_str = to_string (timeout * 1000000); // microseconds
        AVDictionary *options = nullptr;
        av_dict_set (&options, "timeout", timeout_str.c_str(), 0);
        // Request 2MB TCP/IP buffering
        string buffer_str = to_string (1024*1024*2);
        if (is_network) {
            av_dict_set (&options, "recv_buffer_size", buffer_str.c_str(), 0);
        }

        int status = avformat_open_input (&transport, url.c_str(), nullptr, &options);
        av_dict_free (&options);
        if (status < 0) {
            throw LavAudioException ("avformat_open_input", status);
        }
    }


    LibavMediaReader::~LibavMediaReader (void) {
        avformat_close_input (&transport);
    }


    /** Process a replaygain packet by extracting the track gain,
        or if not found then then the album gain. */
    void LibavMediaReader::processReplayGain (AVReplayGain *gain_info, int size) {
        assert (gain_info);
        if (size != sizeof (AVReplayGain)) {
            flog (LOG_WHERE (LOG_WARNING),
                  "Replay gain packet is ", size, "bytes, expected", sizeof (AVReplayGain));
            return;
        }

        // Docs say gain is in "microbels", millionths (1E-6) but also
        // instruct division by 100,000 (so 1E-5).
        float gain;
        if (gain_info->track_gain != INT32_MIN) {
            gain = gain_info->track_gain / 100000.0;
        } else if (gain_info->album_gain != 0) { // Yay consistency!
            gain = gain_info->album_gain / 100000.0;
        } else {
            flog (LOG_WHERE (LOG_WARNING), "Received replay gain with null values");
            return;
        }
        setGain (gain);
    }



    /** Prepare a ffmpeg source for playing its audio stream.
        Chooses codec, extracts replay gain if available, and sets
        streams other than the audio are set to ignore.
        @param codec_context On return, is set to an initialized
        codec_context with appropriate codec selected for the media.
        @return The audio stream index.
        @throw LavAudioException if something goes wrong. */
    int LibavMediaReader::initializeStream (std::unique_ptr <AVCodecContext, CodecDeleter> &codec_context) {
        int status = avformat_find_stream_info (transport, nullptr);
        if (status < 0){
            throw LavAudioException ("avformat_find_stream_info", status);
        }

#ifndef NDEBUG
        av_dump_format (transport, 0, url.c_str(), false);
#endif
        // Find the audio stream within multiplexed data
        int audio_stream = av_find_best_stream (transport, AVMEDIA_TYPE_AUDIO,
                                                -1, -1, // Not stream/related stream
                                                NULL, 0);
        if (audio_stream < 0)
            throw LavAudioException ("av_find_best_stream", status);

        // Don't listen on other streams
        for (int i = 0; i < transport->nb_streams; i++) {
            AVStream *stream = transport->streams [i];
            // Look for replay gain here too
            if (i == audio_stream) {
                for (int j = 0; j < stream->nb_side_data; j++) {
                    if (stream->side_data [j].type == AV_PKT_DATA_REPLAYGAIN) {
                        processReplayGain ((AVReplayGain *) stream->side_data[j].data,
                                           stream->side_data[j].size);
                    }
                }
            } else {
                stream->discard = AVDISCARD_ALL;
            }
        }
        // Find an appropriate codec for the stream
#ifdef HAVE_AVCODECPARAMETERS
        AVCodecID codec_id =  transport->streams [audio_stream]->codecpar->codec_id;
#else
        AVCodecID codec_id =  transport->streams [audio_stream]->codec->codec_id;
#endif
        AVCodec *codec = avcodec_find_decoder (codec_id);
        if (codec == NULL)
            throw AudioException ("Codec not found");

        codec_context.reset (avcodec_alloc_context3 (codec));
        if (!codec_context) {
            throw AudioException ("avcodec_alloc_context3 returned null");
        }
#ifdef HAVE_AVCODECPARAMETERS
        status = avcodec_parameters_to_context(codec_context.get(), transport->streams [audio_stream]->codecpar);
        if (status < 0) {
            throw LavAudioException ("avcodec_parameters_to_context", status);
        }
#else
        status = avcodec_copy_context (codec_context.get(), transport->streams [audio_stream]->codec);
        if (status < 0) {
            throw LavAudioException ("avcodec_copy_context", status);
        }
#endif

        // Initialize the codec
        status = avcodec_open2 (codec_context.get(), codec, nullptr);
        if (status < 0)
            throw LavAudioException ("avcodec_open2", status);

        return audio_stream;
    }

    


    /** Play a media file or URL using ffmpeg.
        @param AudioSettings Describe the output device.
        @param media_url The filename or URL of the media.
        @param audio_gain Gain to apply when playing file, in decibels.
        If ReplayGain is encountered during playback, that is preferred
        over this value. */
    LavPlayer::LavPlayer (const AudioSettings &AudioSettings,
                              const string &media_url,
                              float audio_gain) :
    LibavMediaReader (media_url), audio (AudioSettings), gain (audio_gain) {
    };

    void LavPlayer::setVolume (float volume) {
        audio.volume = volume;
        if (output) {
            output->setVolume (audio.volume + gain);
        }
    }

    float LavPlayer::trackDuration (void) const {
        return duration;
    };

    float LavPlayer::playPoint (void) const {
        return playpoint;
    }

    Media::Player::State LavPlayer::currentState (void) const  {
        return state;
    };

    /** Called when the playback thread is pausing. */
    void LavPlayer::pausing (void) {
        if (is_network) {
            av_read_pause (transport);
       }
    }
    /** Called when the playback thread is resuming playback. */
    void LavPlayer::resuming (void) {
        if (is_network) {
            av_read_play (transport);
        }
    }

    void LavPlayer::setGain (float new_gain) {
        // If the output is open, set the volume there.
        // Otherwise, it will be set when the output is opened.
        gain = new_gain;
        if (output) {
            output->setVolume (gain + audio.volume);
        }
        flog (LOG_WHERE (LOG_GENERAL), "Audio stream replaygain: ", gain, "dB");
    }



    /** @internal
        Decode and play audio a frame from ffmpeg.
        @param frame One frame from the media stream.
        @param codec The codec state for decoding the stream. */
    bool LavPlayer::playPacket (AVFrame *frame,
                                  AVCodecContext *codec,
                                  AVPacket packet /* intentionally by value */) {
        // Check for side-band replay gain data
        int size;
        auto gain_info = (AVReplayGain *) av_packet_get_side_data (&packet,
                                                                   AV_PKT_DATA_REPLAYGAIN,
                                                                   &size);
        if (gain_info) {
            processReplayGain (gain_info, size);
        }



#ifdef HAVE_AVCODEC_RECEIVE_FRAME
        // ffmpeg 3.1 @deprecated avcodec_decode_audio4
        // Use avcodec_send_packet() and avcodec_receive_frame().
        int status = avcodec_send_packet (codec, &packet);
        if (status != 0) {
            flogav("avcodec_send_packet", status);
            return false;
        }

        while (true) {
            status = avcodec_receive_frame (codec, frame);
            if (status == AVERROR (EAGAIN))
                return true;
            if (status != 0) {
                flogav("avcodec_receive_frame", status);
                return false;
            }
            output->play (frame);
        };
#else
        while (packet.size > 0) {
            int got_frame = 0;
            const int decoded = avcodec_decode_audio4 (codec, frame, &got_frame,
                                                       &packet);
            if (decoded < 0) {
                flogav("avcodec_decode_audio4", decoded);
                return false;
            }

            if (got_frame != 0)
                output->play (frame);
            packet.data += decoded;
            packet.size -= decoded;
        }

        return true;
#endif
    }

    /** @internal
        Play a media stream from ffmpeg.
        @param format The multiplexed media thing.
        @param codec The codec context for the audio stream.
        @param audio_stream The index of the audio stream.
        @return Pianod S_OK or a F_* error code. */
    RESPONSE_CODE LavPlayer::playStream (AVFormatContext *format,
                                           AVCodecContext *codec,
                                           const int audio_stream) {
        assert (format);
        assert (codec);
        assert (output);

        AVFrame *frame = av_frame_alloc ();
        if (!frame)
            return F_RESOURCE;

        AVPacket packet;
        av_init_packet (&packet);

        int status;
        while ((status = av_read_frame (format, &packet)) == 0) {
            if (packet.stream_index == audio_stream) {
                playPacket (frame, codec, packet);
            }
            playpoint = (av_q2d (format->streams [audio_stream]->time_base) * packet.pts);
            av_packet_unref (&packet);
            if (checkForPauseOrQuit()) {
                break;
            }
        }
        av_frame_free (&frame);
        if (status != 0 && status != AVERROR_EOF) {
            flogav ("av_read_frame", status);
            return F_FAILURE;
        }
        return S_OK;
    }

    RESPONSE_CODE LavPlayer::playerThread (void) {
        RESPONSE_CODE response = F_FAILURE;

        try {
            std::unique_ptr <AVCodecContext, CodecDeleter> codec;
            int audio_stream = initializeStream (codec);
            if (transport->streams [audio_stream]->duration != AV_NOPTS_VALUE) {
                duration = (av_q2d (transport->streams [audio_stream]->time_base) *
                            (double) transport->streams [audio_stream]->duration);
            }

            // Open the output.
            output.reset (LavAdapter::getOutput (audio, codec.get(), transport->streams [audio_stream]));
            if (output) {
                output->setVolume (audio.volume);
                state = Cueing;
                if (!checkForPauseOrQuit()) {
                    state = Playing;
                    response = playStream (transport, codec.get(), audio_stream);
                }
            }
            avcodec_close (codec.get());
        } catch (...) {
            state = Done;
            throw;
        }
        state = Done;
        return response;
    }


}
