///
/// Football option parser template.
/// C++ classes implementing an options parser.
/// @file       footparser.h - Football socket abstraction layer
/// @author     Perette Barella
/// @date       2016-04-18
/// @copyright  Copyright 2014–2017 Devious Fish. All rights reserved.
///

#ifndef pianod2_footparser_h
#define pianod2_footparser_h

#include <config.h>

#include <stdarg.h>
#include <stdbool.h>
#include <sys/types.h>
#include "assert.h"

#include <string>
#include <vector>

#include "fb_public.h"
#include "fb_service.h"

namespace Football {
    class Connection;

    /** Non-template framework for option parser. */
    class GenericOptionParser {
    protected:
        FB_PARSER *parser = fb_create_parser();
        int offset = 0; ///< Current option's starting offset into argv.
        int capacity = 0; ///< Capacity of argv/argn arrays.
        char **_argv = nullptr; ///< Array of options, for passing to parser.
        char **_argn = nullptr; ///< Named argument info returned by parser.
    public:
        GenericOptionParser ();
        ~GenericOptionParser ();

        // Disallow copy and move operations on parsers.
        GenericOptionParser (const GenericOptionParser &) = delete;
        GenericOptionParser (GenericOptionParser &&) = delete;
        GenericOptionParser &operator=(const GenericOptionParser &) = delete;
        GenericOptionParser &operator=(GenericOptionParser &&) = delete;

        bool addStatements (const FB_PARSE_DEFINITION *statements);

        /** Store an option into the result datatype.
         @param cmd The option that is being parsed.
                @param result Storage for the option value.
                @return FB_PARSE_SUCCESS, one of the FB_PARSE_ERROR values, or
                positive value of your designing.  Value is passed
                to Connection::commandError().
                @see FB_PARSE_ERROR */
        virtual int invokeHandleOption (int cmd, void *result) = 0;
    private:
        /// @internal Find the index position of a named argument.
        int argnPosition (const char *name);
        bool ensureCapacity (int size);

    protected:
        /** Interpret options.
            @param options A tokenized set of command line pieces, in sequential order.
            @param result Storage for option values.
            @param conn A connection associated with the parsing.
            If provided, encountered errors are reported to the connection before return.
            @return FB_PARSE_SUCCESS or a FB_PARSE_ERROR value.
            @see FB_PARSE_ERROR */
        int interpret (const std::vector<std::string> &options, void *result, Connection *conn);
    public:
        const char *argv (const char *name);
        const char *argv (int index);
        bool argvEquals (const char *name, const char *value);
    };



    /** Class to parse command line key-value options.
        To extend this class, provide a data type into which option values will be stored,
        and an enumeration type identifying the various options.
        Then, extend OptionParser implementing handleOption().

        Define the statement like this:
        - { statementId, "statement action {required1} {required2} [{options}] ..." }

        Define the options like this:
        - { optionId1,   "option {one} ..." }
        - { optionId2,   "another opt {two} and {three} ..." }
        - { optionId3,   "with flag ... }

        Valid command lines would thus include:
        - statement action "foo" "bar"
        - statement action "foo" "bar" option "firstoption" with flag
        - statement action "foo" "bar" with flag another opt "value for two" and "value for three"

        The option parser does not prevent options being respecified, so this is also valid:
        - statement action "foo bar" with flag option "first" with flag

        A future enhancement may allow specification per-option whether this is allowed.
        @tparam DataObjectType a datatype into which option values are placed.
        @tparam OptionEnum a type used to identify option IDs.
        */
    template <class DataObjectType, class OptionEnum> class OptionParser
    : public GenericOptionParser {
        static_assert (sizeof (OptionEnum) == sizeof (int),
                       "Enumerator size must be same as int");
    public:
        using GenericOptionParser::addStatements;
        struct ParseDefinition {
            const OptionEnum response; /**< The value to return when statement pattern is matched. */
            const char *statement; /**< The command line */
        };

        /** Add statements to the parser.
            @param statements An array of statement definitions to add.
            @return True on success, false on failure. */
        inline bool addStatements (const ParseDefinition *statements) {
            return GenericOptionParser::addStatements (reinterpret_cast <const FB_PARSE_DEFINITION *> (statements));
        }

        /** Store an option into the result datatype.
            @param cmd The option that is being parsed.
            @param result Storage for the option value.
            @return FB_PARSE_SUCCESS, one of the FB_PARSE_ERROR values, or
            positive value of your designing.  Value is passed
            to Connection::commandError().
            @see FB_PARSE_ERROR */
        virtual int handleOption (OptionEnum cmd, DataObjectType &result) = 0;

        /** Interpret options.
            @param options A tokenized set of command line pieces, in sequential order.
            @param result Storage for option values.
            @param conn A connection associated with the parsing.
            If provided, encountered errors are reported to the connection before return.
            @return FB_PARSE_SUCCESS or a FB_PARSE_ERROR value.
            @see FB_PARSE_ERROR */
        inline int interpret (const std::vector<std::string> &options, DataObjectType &result, Connection *conn = nullptr) {
            return GenericOptionParser::interpret (options, &result, conn);
        }
    private:
        virtual int invokeHandleOption (int cmd, void *result) override final {
            return handleOption (static_cast<OptionEnum> (cmd),
                                 *reinterpret_cast <DataObjectType *> (result));
        }
    };
    

}

#endif
