///
/// Football public C++ declarations.
/// Contains C++ classes which wrap standard C Football structures.
/// Methods exposed by the objects present a new API that is more
/// suitable for the C++ object-oriented approach.
/// @file       football.h - Football socket abstraction layer
/// @author     Perette Barella
/// @date       2014-10-27
/// @copyright  Copyright 2012–2016 Devious Fish. All rights reserved.
///

#ifndef __football__football__
#define __football__football__

#include <config.h>

#include <stdarg.h>
#include <stdbool.h>
#include <sys/types.h>
#include "assert.h"

#include <string>
#include <vector>
#include <map>
#include <iostream>
#include <functional>

#include "fb_public.h"
#include "fb_service.h"

/// Football C++ connections, services and parsers.
namespace Football {
    /// Lists of commands
    using HelpList = std::vector<const char *>;

    /** Base class for services, events, and connections.
        Provides common mechanism for outputting to associated connection(s). */
    class Thingie : public std::basic_streambuf<char> {
    protected:
        virtual void *myThingie (void) const = 0;
    public:
        virtual ~Thingie();
        virtual ssize_t printf (const char *format, ...);
        virtual ssize_t vprintf (const char *format, va_list parameters);
        virtual ssize_t bprintf (const char *format, ...);
        virtual ssize_t bvprintf (const char *format, va_list parameters);
        // Overrides to implement streambuf
        std::streamsize xsputn(const char_type* data, std::streamsize count);
        int_type overflow(int_type c);
    };

    /// Helper functions for sending output to a connection or service.
    inline Thingie &operator << (Thingie &there, const char *message) {
        there.printf ("%s", message);
        return there;
    };

    // Declare all these inbred bastards
    class Connection;
    class ServiceBase;

    // C++ Parser/command handler
    typedef ssize_t Command;
    class InterpreterBase {
        friend Connection;
    protected:
        virtual void handleCommand (Connection &conn, Command command) = 0;
        virtual bool hasPermission (Connection &conn, Command command);
    public:
        virtual const FB_PARSE_DEFINITION *statements (void) = 0;
        friend class ServiceBase;
    };


    /// C++ Connection wrapper
    class Connection: public Thingie {
        friend class Iterator;
        friend class ServiceBase;
        friend class Arena;
    private:
        FB_CONNECTION *connection = nullptr; ///< Pointer to underlying Football connection
        FB_EVENT *currentEvent = nullptr; ///< Pointer to underlying event being processed.
        mutable int offset = 0; ///< Offset into argv when reinterpreting.
        int argname_capacity = 0;
        char **argnames = nullptr;
        static Connection *getFromOld (FB_CONNECTION *connection);
        virtual inline void *myThingie (void) const { return connection; }
        int argvIndex (const char *itemname) const;
    public:
        using ReinterpretHandler = std::function<void (Connection &conn, Command command, const char *error)>;
        using ConstReinterpretHandler = std::function<void (const Connection &conn, Command command, const char *error)>;

        ServiceBase *service (void);
        ~Connection();

        // Actions
        bool transfer (ServiceBase *service, bool invokeNewConnectionHandler = false);
        void close (void);
        const HelpList getHelp (const char *search = NULL);
        void acceptInput (bool mode);

        // Getters & Actions - event related
        const char *command (void) const;
        char *const *argv() const;
        const char *argv (const char *itemname) const;
        inline const char *operator[] (const char *itemname) const {
            return argv (itemname);
        }
        std::vector<std::string> argvFrom (const char *itemname) const;
        std::vector<std::string> argvSlice (const char *itemname) const;
        const char *argvFromUntokenized (const char *itemname) const;
        bool argvEquals (const char *itemname, const char *value) const;
        bool argvEquals (const char *itemname, const std::string &value) const;
        int argc() const;
        void reinterpret (const char *itemname);
        void reinterpret (const char *start_token, const FB_PARSER *parser, ConstReinterpretHandler &handler) const;
        inline void reinterpret (const char *start_token, const FB_PARSER *parser, ReinterpretHandler &handler) {
            static_cast <const Connection *> (this)->reinterpret (start_token, parser,
                                                                  *reinterpret_cast <ConstReinterpretHandler *> (&handler));
        };

        // Stubs with default behavior, but probably will be overriden
        virtual void newConnection ();
        virtual void connectionClose ();
        virtual void commandError (ssize_t reason, const char *where);
        virtual void commandException (std::exception_ptr except);
        virtual void permissionDenied ();

    private:
        void dispatchCommand ();
        void setConnection (FB_CONNECTION *conn) { connection = conn; };

#ifdef FOOTBALL_TRANSITIONAL
        // Getters to get easy access to underlying C data structures.
        FB_CONNECTION *operator()(void) { return connection; }
#endif
    };


    /** C++ Service wrapper
        Manages the service's connections and interpreters.
        */
    class ServiceBase: public Thingie {
        friend class Iterator;
        friend class Connection;
        friend class Arena;
    private:
        using CommandMap = std::map <Command, InterpreterBase *>;
        using CommandPair = std::pair <Command, InterpreterBase *>;
        using ServiceClients = std::vector <InterpreterBase *>;

    protected:
        FB_SERVICE *service; ///< Pointer to underlying Football service structure
    private:
        FB_PARSER *parser = nullptr; ///< Single parser shared by all interpreters
        CommandMap commands; ///< Lookup command IDs to find appropriate interpreter
        ServiceClients clients; ///< List of service's interpreters
    protected:
        virtual inline void *myThingie (void) const { return service; }
        static ServiceBase *getFromOld (FB_SERVICE *service);
        // Constructor
        ServiceBase (FB_SERVICE_OPTIONS options, ServiceBase *parent = nullptr);
        ~ServiceBase();
        bool routeEvent (const FB_EVENT &event);
        // Factories
        virtual Connection *allocNewConnection () = 0;
        // Internal methods
        InterpreterBase *getInterpreter (Command command);
        const HelpList getHelp (Connection &destination, const char *);
    public:
        void close (void);
        Connection *newConnectionFromFile (const std::string &filename);
        Connection *newLoopbackConnection (int *loopback);
        bool addCommands (InterpreterBase *cmds);

        // Stubs, expected to be overridden
        virtual void serviceShutdown (void);

#ifdef FOOTBALL_TRANSITIONAL
        // Getters to get easy access to underlying C data structures.
        FB_SERVICE *operator()(void) { return service; }
#endif
    };


    // Now make templatized classes that make things easy/safe to use

    /** A command interpreter.
        To use this, first create a base type that will be shared by all interpreters.
        This will clarify your connection and command ID types.
        Then, extend that class for each interpreter.  Each must implement:

        - statements()
        - hasPermission()
        - handleCommand()

        @tparam UserConnection The connection type (extended from Football::Connection).
        This is passed to hasPermission and handleCommand methods, providing those
        methods with access to the trigger event/command, connection and connection
        context (maintained in your extended class, not the old Football void*).
        @tparam CommandEnum The type used by command IDs.
        */
    template <class UserConnection, class CommandEnum> class Interpreter : public InterpreterBase {
        static_assert (sizeof (CommandEnum) == sizeof (int),
                       "Size of command enumerator must match size of int.");
    private:
        virtual bool hasPermission (Connection &conn, Command command) override {
            return hasPermission ((UserConnection &) conn, (CommandEnum) command);
        };
        virtual void handleCommand (Connection &conn, Command command) override {
            handleCommand ((UserConnection &) conn, (CommandEnum) command);
        }
    public:
        virtual bool hasPermission (UserConnection &, CommandEnum) { return true; };
        virtual void handleCommand (UserConnection &conn, CommandEnum command) = 0;
    };


    /** Football Service.
        The service is customized for use with a particular connection type.
        @tparam Connection The connection type (extended from Football::Connection)
        used for the service's connections.
        */
    template <class Connection> class Service : public ServiceBase {
    public:
        struct iterator : public std::iterator<std::forward_iterator_tag, Connection> {
            int position;
            const FB_SERVICE &service;
            /// Advance the iterator past any opening/closing connections.
            inline void advancePastBadness (void) {
                while (position < service.connection_count && !service.connections [position]->relatedObject) {
                    assert (service.connections [position]->state != FB_SOCKET_STATE_OPEN);
                    position++;
                }
            }
            iterator (const FB_SERVICE &svc) : service (svc) {
                position = 0;
                advancePastBadness();
            };
            iterator (const FB_SERVICE &svc, int pos) : service (svc) {
                assert (pos == svc.connection_count);
                position = pos;
                advancePastBadness();
            };
            Connection *operator *() {
                return (Connection *)(service.connections [position]->relatedObject);
            }
            iterator &operator++() {
                position++;
                advancePastBadness();
                return *this;
            }
            iterator operator++(int) {
                iterator item = *this;
                position++;
                advancePastBadness();
                return item;
            }
            bool operator!=(iterator &compare) {
                return position != compare.position;
            }
        };
        struct const_iterator : public iterator {
            using iterator::iterator;
            inline const Connection *operator *() { return iterator::operator *(); };
        };
        Service (const FB_SERVICE_OPTIONS options,
                 Service *parent = nullptr) : ServiceBase (options, parent) {
        };
        Connection *newConnectionFromFile (const std::string &filename) {
            return (Connection *) ServiceBase::newConnectionFromFile (filename);
        };
        iterator begin() { iterator it (*service); return it ; };
        iterator end() { iterator it (*service, service->connection_count); return it ; };
        const_iterator begin() const { const_iterator it (*service); return it ; };
        const_iterator end() const { const_iterator it (*service, service->connection_count); return it ; };
    protected:
        /// Factories for related types instead of generics.
        virtual Connection *allocNewConnection () {
            return new (std::nothrow) Connection ();
        };
    };
    
    
    /** Football Arena: Manager for services.
        The arena runs all the services, providing event management and dispatching.
        Services are automatically added to the arena when created; use the Arena's
        poll routines to service them and their connections.  Events are dispatched
        to the virtual methods provided in Football::Connection, with support from
        Football::Interpreter; the poll return value indicates if service was done
        (or false if a timeout happened).
        */
    class Arena {
        friend ServiceBase;
    public:
        static bool ready (void);
        
        static bool poll (void);
        static bool pollWait (void);
        static bool pollUntil (time_t untilwhen);
        static bool pollWithTimeout (double timeout);
    private:
        static bool handleEvent (FB_EVENT *);
    };
    
    
} // </namespace>


#endif // ifndef __football__football__ 
