///
/// Football C++ service wrapper.
/// Implementations for Football::ServiceBase methods.
/// @file       fb_servicepp.cpp - Football socket abstraction layer
/// @author     Perette Barella
/// @date       2014-11-20
/// @copyright  Copyright 2014-2016 Devious Fish. All rights reserved.
///

#include <config.h>

#include <stdio.h>
#include <assert.h>

#include <iostream>
#include <exception>
#include <stdexcept>
#include <iterator>
#include <map>

#include "football.h"
#include "fb_service.h"

namespace Football {
    using namespace std;

    // Services
    /** Create a new service with requested options.  Creates the underlying
        standard C stuff, and a C++ object that wraps it.
        @param options The service options such as ports, behavior settings, etc.
        @see fb_create_service () */
    ServiceBase::ServiceBase (FB_SERVICE_OPTIONS options,
                              ServiceBase *parent) {
        assert (options.context_size == 0);
        if (parent) options.parent = parent->service;
        service = fb_create_service (&options);
        if (service) {
            service->relatedObject = this;
            return;
        }
        throw invalid_argument("check logs for further information");
    }

    ServiceBase::~ServiceBase () {
        fb_parser_destroy(parser);
        
    }
    

    /// Initiate shutdown of a service.
    void ServiceBase::close (void) {
        fb_close_service (service);
    }

    /** Create a new "connection" that reads from a file.
        Replies to the connection are transparently disposed.
        An appropriate event is dispatched to the connection when created,
        but nothing is read from the file yet.  The caller may adjust the
        context or privileges; reading is done by the Arena (service
        manager) poll routines.
        @param filename The name of the file to read from.
        @return An initialized Football::Connection, or NULL on error.
        */
    Connection *ServiceBase::newConnectionFromFile (const string &filename) {
        FB_EVENT *event = fb_accept_file (service, (char *) filename.c_str());
        if (!event) return nullptr;
        Arena::handleEvent (event);
        return Connection::getFromOld (event->connection);
    }


    /** Create a new connection that loops back to the service.
        Bidirectional communication is supported.
        Replies to the connection are transparently disposed.
        An appropriate event is dispatched to the connection when created,
        but nothing is read from the file yet.  The caller may adjust the
        context or privileges; reading is done by the Arena (service
        manager) poll routines.
        @param loopback On return, set to the socket used for loopback.
        @return A socket number, or -1 on error.
        */
    Connection *ServiceBase::newLoopbackConnection (int *loopback) {
        FB_EVENT *event = fb_loopback_socket (service);
        if (!event) return nullptr;
        Arena::handleEvent (event);
        *loopback = event->socket;
        return Connection::getFromOld (event->connection);
    }


    /** @internal
        Retrieve a C++ service object from the old C-style service structure.
        @param service The old standard C service structure.
        @return The C++ service object. */
    ServiceBase *ServiceBase::getFromOld(FB_SERVICE *service) {
        assert (service);
        assert (service->relatedObject);
        return (ServiceBase *) service->relatedObject;
    }


    /** Attach a ParserClient/Interpreter to a service, and add its parse
        definitions to the service's parser.
        @param client The ParserClient, whose methods will be called
        when commands are matched.
        @return true on success, false on error.
        @throw Underlying datastructures may throw bad_alloc, leaving
        parser in an indeterminate state. */
    bool ServiceBase::addCommands (InterpreterBase *client) {
        // First, make sure the commands don't exist already.
        const FB_PARSE_DEFINITION *cmds = client->statements();
        clients.push_back (client);

        for (const FB_PARSE_DEFINITION *cmd = cmds; cmd->statement != NULL; cmd++) {
            CommandMap::iterator existing = commands.find (cmd->response);
            if (existing != commands.end()) {
                assert (!"Command ID shared by multiple interpreters");
                fb_log (FB_WHERE (FB_LOG_ERROR), "Command ID reused: %d, statement: %s\n",
                        cmd->statement, cmd->response);
                return false;
            }
        }

        // The commands are all safe to add.
        if (parser == NULL) {
            parser = fb_create_parser ();
            if (!parser) {
                assert (!"Could not create parser");
                fb_log (FB_WHERE (FB_LOG_ERROR), "Failure creating new parser");
                return false;
            }
        }
        if (!fb_parser_add_statements (parser, cmds)) {
            assert (!"Could not add statements to parser");
            fb_log (FB_WHERE (FB_LOG_ERROR), "Failure adding statements");
            return false;
        };
        for (size_t x = 0; cmds [x].statement != NULL; x++) {
            CommandMap::iterator existing = commands.find (cmds [x].response);
            if (existing == commands.end()) {
                commands.insert (CommandPair (cmds [x].response, client));
            }
        }
        return true;
    }



    /** Find the correct ParserClient/Interpreter to handle a command.
        @param command The command ID.  Must be unique across all parser clients.
        @return The ParserClient which handles the specified command. */
    InterpreterBase *ServiceBase::getInterpreter (Command command) {
        CommandMap::iterator client = commands.find (command);
        if (client == commands.end()) return NULL;
        return client->second;
    }



    /** @internal
        Retrieve a list of allowed/matching commands to a client.
        Iterates through registered ParserClients/Interpreters for this service.
        For each command, checks that the connection hasPermission() for the command.
        @destination The connection which is requesting help
        @search Optional.  Text to search for.  If omitted/NULL, all commands are included.
        When specified, only commands containing the search text as a substring
        are included in results (but in all cases, only if they hasPermission() too).
        @return true if any help was sent, false otherwise. */
    const HelpList ServiceBase::getHelp(Connection &destination, const char *search) {
        vector<const char *> results;
        for (ServiceClients::iterator client = clients.begin(); client != clients.end(); client++) {
            const FB_PARSE_DEFINITION *cmds = (*client)->statements();
            for (size_t x = 0; cmds [x].statement != NULL; x++) {
                if ((*client)->hasPermission (destination, cmds[x].response)) {
                    if (search == NULL ||
                        strcasestr(cmds [x].statement, search) != NULL) {
                        results.push_back (cmds [x].statement);
                    }
                }
            }
        }
        return results;
    }
    
    
    /** Invoked when a service is shutting down.  Corresponds to
        FB_EVENT_STOPPED; by the time this is received, all
        connections have been closed. */
    void ServiceBase::serviceShutdown (void) {
        // Nothing to do by default
    }
    
} // </namespace>

