///
/// Football C++ connection and event wrappers and extensions.
/// Implementations for Football::Thingie and Football::Connection methods.
/// @file       fb_connection.cpp - Football socket abstraction layer
/// @author     Perette Barella
/// @date       2014-10-26
/// @copyright  Copyright 2014-2016 Devious Fish. All rights reserved.
///

#include <config.h>

#include <cstdio>
#include <cstdarg>
#include <string>
#include <cassert>
#include <fcntl.h>

#include <typeinfo>
#include <iostream>
#include <exception>
#include <stdexcept>
#include <iterator>
#include <vector>
#include <map>
#include <future> // for exception types
#include <regex> // for exception types

#include "football.h"
#include "fb_service.h"

namespace Football {
    using namespace std;

    /** Determine if a connection is allowed to use a command.  Used by the
        command dispatcher and the sendHelp methods.
        Default hasPermission: Allow everything.
        @param event The event/connection to check permissions for.
        @param command The command the connection is requesting to use.
        @return true if the command is allowed, false otherwise. */
    bool InterpreterBase::hasPermission (Connection &event, Command command) {
        (void) event; (void) command; // silence unused parameter warnings
        return true;
    }




    Thingie::~Thingie() {
        // Virtually needed.
    }

    /** Add messages to output queues.
        - 'b' variants broadcast.
        - 'v' accepts a pointer to format arguments.
        @param format a printf-style format string
        @return -1 on error, or number of bytes written/queued on success or partial success.
        */
    ssize_t Thingie::printf (const char *format, ...) {
        va_list parameters;
        va_start(parameters, format);
        ssize_t result = vprintf (format, parameters);
        va_end (parameters);
        return result;
    }

    /// @see Football::Thingie::printf
    ssize_t Thingie::vprintf (const char *format, va_list parameters) {
        return fb_vfprintf (myThingie(), format, parameters);
    }

    /// @see Football::Thingie::printf
    ssize_t Thingie::bprintf (const char *format, ...) {
        va_list parameters;
        va_start(parameters, format);
        ssize_t result = bvprintf (format, parameters);
        va_end (parameters);
        return result;
    }

    /// @see Football::Thingie::printf
    ssize_t Thingie::bvprintf (const char *format, va_list parameters) {
        return fb_bvfprintf (myThingie(), format, parameters);

    }

    streamsize Thingie::xsputn(const Thingie::char_type* data, streamsize count) {
        return fb_fprintf (myThingie(), "%*s", data, (int) count);
    };
    Thingie::int_type Thingie::overflow(Thingie::int_type c) {
        fb_fprintf (myThingie(), "%c", c);
        return c;
    }




    // Connections

    Connection::~Connection() {
        delete[] argnames;
    };


    /** @internal
        Find the C++ connection object from the old-style C connection.
        If there isn't one yet, create it one.
        @param connection The Football standard C connection structure.
        @return The C++ connection object. */
    Connection *Connection::getFromOld (FB_CONNECTION *connection) {
        assert (connection);
        assert (connection->service->relatedObject);
        if (!connection->relatedObject) {
            ServiceBase *svc = ServiceBase::getFromOld(connection->service);
            Connection *conn = svc->allocNewConnection();
            if (conn) {
                connection->relatedObject = conn;
                conn->connection = connection;
            } else {
                fb_close_connection (connection);
            }
        }
        return (Connection *) connection->relatedObject;
    }





    /*
     *              Actions
     */

    /// Initiate connection closure.
    void Connection::close (void) {
        fb_close_connection(connection);
    }

    /// Transfer a connection to another service.
    bool Connection::transfer (ServiceBase *newservice, bool invokeNewConnectionHandler) {
        if (fb_transfer(connection, newservice->service)) {
            if (invokeNewConnectionHandler) {
                newConnection ();
            }
            return true;
        }
        return false;
    }

    /** Get a connection's parent service.
        @return The C++ service object for the connection.  */
    ServiceBase *Connection::service () {
        assert (this);
        assert (this->connection);
        return ServiceBase::getFromOld (connection->service);
    }

    /** Send help to the connection.
        @param search Limit help to commands matching pattern, if given.
        @return Parse definitions of eligible commands. */
    const HelpList Connection::getHelp (const char *search) {
        return service()->getHelp (*this, search);
    }

    void Connection::acceptInput (bool mode) {
        fb_accept_input(connection, mode);
    }

    /*
     *              Connection event handlers
     */


    /** Invoked when a new connection has arrived/greeted.  Corresponds
        to FB_EVENT_CONNECT. */
    void Connection::newConnection () {
        *this << "[Football::Connection] Connected\n";
    }


    /** Invoked when a connection is closing.  Corresponds to
        FB_EVENT_CLOSE event.  This is the last notification/event
        for a connection before it closes. */
    void Connection::connectionClose () {
        *this << "[Football::Connection] Connection closing\n";
    }

    /** Invoked for:
        - parse errors originating with parsers attached to the service
        - exceptions thrown by command handlers attached to the service
        - parse errors from an options parser in use by the connection
        - values other than FB_PARSE_SUCCESS returned by option handler
        @param reason The nature of the parsing error.
        @param where The keyword at which the statement fails parsing.  May be null.
        @see FB_PARSE_ERROR in fb_public.h for related constants for reason.
        @see fb_parser.c for `where` explanation. */
    void Connection::commandError (ssize_t reason, const char *where) {
        (void) reason; // silence unused parameter warning
        assert (this);
        if (reason <= FB_ERROR_EXCEPTION) {
            printf ("%s%s%s",
                    reason == FB_ERROR_BADALLOC ? "Allocation error" : "Error",
                    where ? ": " : "", where ? where : "");
        } else {
            printf ("Invalid command%s%s\n",
                    where ? " at: " : "", where ? where : "");
        }
    }

    /** Default implementation of commandException builds the best message
        possible from the exception type,and passes it to commandError.
        @param except An exception that happened while dispatching a command. */
    static struct { const std::type_info *type; const char *name;} exception_names[] = {
        { &typeid (std::exception), "exception" },

        { &typeid (std::logic_error), "logic error" },
        { &typeid (std::invalid_argument), "invalid argument" },
        { &typeid (std::domain_error), "domain error" },
        { &typeid (std::length_error), "length error" },
        { &typeid (std::out_of_range), "out-of-range value" },
        { &typeid (std::future_error), "future error" },

        { &typeid (std::range_error), "range error" },
        { &typeid (std::overflow_error), "numeric overflow" },
        { &typeid (std::underflow_error), "numeric underflow" },
        { &typeid (std::regex_error), "regular expression error" },

        { &typeid (std::system_error), "system error" },
        { &typeid (std::ios_base::failure), "System or I/O error" },

        { &typeid (std::runtime_error), "runtime error" },
        { &typeid (std::bad_typeid), "bad typeid" },
        { &typeid (std::bad_cast), "invalid typecast" },
        { &typeid (std::bad_weak_ptr), "invalid weak_ptr" },
        { &typeid (std::bad_function_call), "bad function call" },
        { &typeid (std::bad_exception), "bad exception" }
    };
    void Connection::commandException (std::exception_ptr except) {
        const char *type = "Exception name missing";
        const char *detail = nullptr;
        try {
            std::rethrow_exception (except);
        } catch (const bad_alloc &e) {
            commandError (FB_ERROR_BADALLOC, e.what());
        } catch (const exception &e) {
            detail = e.what();
            for (auto const &ex : exception_names) {
                if (*(ex.type) == typeid(e)) {
                    type = ex.name;
                    break;
                }
            }
            fb_log (FB_WHERE(FB_LOG_WARNING),
                    "handleCommand threw a %s exception%s%s\n\tCommand: %s",
                    typeid (e).name(), detail ? ": " : "", detail ? detail : "",
                    currentEvent->argr [offset]);
            string message {type};
            if (detail && *detail) {
                message += ": ";
                message += detail;
            }
            commandError (FB_ERROR_EXCEPTION, message.c_str());
        } catch (...) {
            type = "Custom exception";
            fb_log (FB_WHERE(FB_LOG_ERROR),
                    "handleCommand threw something unknown\n\tCommand: %s",
                    currentEvent->argr [offset]);
            commandError (FB_ERROR_EXCEPTION, "Unknown thing thrown/caught");
        }
    }

    /** Invoked when a connection is ineligible to use a requested command.
        Permission is determined by the interpreter's hasPermission() method. */
    void Connection::permissionDenied () {
        assert (this);
        *this << "Permission denied\n";
    }


    /** @internal
        Dispatch a command from a connection.
        - If the command is invalid, invoke invalidCommand() handler.
        - If the connection doesn't have privilege to the requested command,
        invoke permissionDenied() handler.
        - Otherwise, call the appropriate Interpreter's
        handleCommand() to do the work. */
    void Connection::dispatchCommand () {
        assert (argc() >= 0);
        char *where = nullptr;
        Command command;
        if (argname_capacity < argc()) {
            delete[] argnames;
            argname_capacity = argc() + 100;
            argnames = new (nothrow) char *[argname_capacity];
            if (!argnames) {
                fb_log (FB_WHERE(FB_LOG_WARNING), "argnames allocation failed");
                commandError (FB_ERROR_BADALLOC, nullptr);

            }
        }
        memset (argnames + offset, 0, (argname_capacity - offset) * sizeof (*argnames));
        command = fb_interpret_recurse (service()->parser, argv(), argnames + offset, &where);
        if (command <= 0) {
            commandError (command, where);
        } else {
            InterpreterBase *interpreter = service()->getInterpreter (command);
            if (!interpreter->hasPermission(*this, command)) {
                permissionDenied ();
            } else {
                try {
                    interpreter->handleCommand(*this, command);
                } catch (...) {
                    fb_log (FB_WHERE(FB_LOG_WARNING),
                            "handleCommand threw a %s exception\n\tCommand: %s",
                            typeid (current_exception()).name(), currentEvent->argr [offset]);
                    commandException (std::current_exception());
                }
            }
        }
    }






    /*
     *              Getters for event stuff
     */

    /** Get the raw command line for an input event.
        @return The command line. */
    const char *Connection::command (void) const {
        assert (this);
        assert (currentEvent->type == FB_EVENT_INPUT || currentEvent->type == FB_EVENT_CONNECT);
        return currentEvent->command;
    }

    /** Return the entire argv array for an input event.
        @return pointer to the start of argv. */
    char * const*Connection::argv (void) const {
        assert (this);
        assert (currentEvent->type == FB_EVENT_INPUT || currentEvent->type == FB_EVENT_CONNECT);
        return currentEvent->argv + offset;
    }

    /** Return the number of keywords in the event's argv array.
        @return Number of items in argv. */
    inline int Connection::argc (void) const {
        assert (this);
        assert (currentEvent->type == FB_EVENT_INPUT || currentEvent->type == FB_EVENT_CONNECT);
        return currentEvent->argc - offset;
    }

    /** Get the index of an named argv element.
        @return The index, or -1 if not found. */
    int Connection::argvIndex (const char *itemname) const {
        // Nota bene: Using <= comparison because we want to check
        // the terminator null element too.
        for (int i = offset; i <= currentEvent->argc; i++) {
            if (argnames [i] && strcasecmp (argnames [i], itemname) == 0) {
                return i;
            }
        }
        return -1;
    }

    /** Return a single keyword from the argv array.
        @param itemname The name of the requested argv array element.
        @return The requested string, or NULL if `itemname` doesn't exist. */
    const char *Connection::argv (const char *itemname) const {
        int i = argvIndex (itemname);
        return i >= 0 ? currentEvent->argv [i] : nullptr;
    }

    /** Return remaining arguments starting at a named argument.
        @param itemname The name of the starting point.
        @return A vector containing the remaining arguments,
        or an empty list if the named item is not found. */
    vector<string> Connection::argvFrom (const char *itemname) const {
        vector<string> result;
        int i = argvIndex (itemname);
        if (i >= 0) {
            while (currentEvent->argv [i]) {
                result.push_back (string (currentEvent->argv [i++]));
            }
        }
        return result;
    };

    /** Return a slice corresponding to a named argument.
        @param itemname The name of the argument.
        @return A vector containing the remaining arguments,
        or an empty list if the named item is not found. */
    vector<string> Connection::argvSlice (const char *itemname) const {
        vector<string> result;
        int i = argvIndex (itemname);
        if (i >= 0) {
            int end = argvIndex ("...");
            if (end == i+1) {
                while (currentEvent->argv [i]) {
                    result.push_back (string (currentEvent->argv [i++]));
                }
            } else {
                result.push_back (string (currentEvent->argv [i]));
            }
        }
        return result;
    };

    /** Return all elements subsequent to a named argument.
        @param itemname The name of the starting point.
        @return The unsplit command string, starting from the named item,
        or a null pointer if the named argument is not found. */
    const char *Connection::argvFromUntokenized (const char *itemname) const {
        int i = argvIndex (itemname);
        return i >= 0 ? currentEvent->argr [i] : nullptr;
    };


    /** Check if a named argv element has a particular value.
        @param itemname The name of the argv element, which is allowed to not exist.
        @param value The comparison value.  If nullptr, checks the name item is not present.
        @return true if the specified value exists and matches,
        or (if value == nullptr) that the specified value doesn't exist. */
    bool Connection::argvEquals (const char *itemname, const char *value) const {
        const char *actual_value = argv (itemname);
        if (actual_value == nullptr && value == nullptr) return true;
        if (actual_value == nullptr || value == nullptr) return false;
        return (strcasecmp (actual_value, value) == 0);
    }
    
    /** See `argvEquals(const char *itemname, const char *value)` except that value
        must not be nullptr. */
    bool Connection::argvEquals (const char *itemname, const string &value) const {
        return argvEquals (itemname, value.c_str());
    }
    
    
    /** Reparse an event's command, using an alternate point in argv.
        This recursively invokes the command dispatcher with the new parsing point.
        Values returned by this class's argv, argv(n), and argc reflect the
        offset for the duration of the recursive invocation.
        @param start_token The name of the token to start interpreting at.
        May be invoked recursively; offsets become cumulative. */
    void Connection::reinterpret (const char *start_token) {
        assert (this);
        int old_offset = offset;
        offset = argvIndex (start_token);
        assert (offset >= 0);
        if (offset >= 0)
            dispatchCommand ();
        offset = old_offset;
    }

    /** Parse more of an event's command by applying a parser, then calling a supplied function.
        @param start_token The token to start applying the parser at.
        @param parser The parser to apply.
        @param handler The handler to call. */
    void Connection::reinterpret (const char *start_token, const FB_PARSER *parser,
                                  ConstReinterpretHandler &handler) const {
        assert (this);
        assert (parser);
        int old_offset = offset;
        offset = argvIndex (start_token);
        assert (offset >= 0);
        if (offset >= 0) {
            assert (argc() >= 0);
            char *where = nullptr;
            memset (argnames + offset, 0, (argname_capacity - offset) * sizeof (*argnames));
            Command command = fb_interpret_recurse (parser, argv(), argnames + offset, &where);
            handler (*this, command, where);
        }
        offset = old_offset;
    }

} // </namespace>
