///
/// Command handlers for audio engine.
/// Control playback, select music, rate songs and playlists,
/// create/remove playlists and add/remove seeds.
///	@file		enginecommand.cpp - pianod
///	@author		Perette Barella
///	@date		2014-12-08
///	@copyright	Copyright 2014-2020 Devious Fish.  All rights reserved.
///

#include <config.h>

#include <cstdlib>

#include <exception>
#include <memory>
#include <initializer_list>

#include <football.h>

#include "fundamentals.h"
#include "lookup.h"
#include "engine.h"
#include "predicate.h"
#include "tuner.h"
#include "connection.h"
#include "response.h"
#include "filter.h"
#include "querylist.h"
#include "servicemanager.h"
#include "mediaunit.h"
#include "mediaplayer.h"
#include "mediamanager.h"
#include "audiooptionsparser.h"


using namespace std;

#define SEEDVERB " <verb:add|delete|toggle>"
#define SEEDPREP " <to|from|for>"

static const FB_PARSE_DEFINITION statementList[] = {
    // Status, queue and history
    { TIMESTATUS,		"" },								// On null input, display time.
    { QUERYSTATUS,		"status" },							// Current song, etc.
    { QUERYHISTORY,		"history list [{#index}]" },		// Previously played songs
    { QUERYQUEUE,		"queue list [{#index}]" },			// Show upcoming songs
    { GETHISTORYSIZE,	"get history length" },                 // Read the length of the history
    { SETHISTORYSIZE,	"set history length {#length:1-50}" },	// Set the length of the history
    //	{ GETOUTPUTSERVER,	"get audio output settings" },	// libao setting


    { GETVOLUME,		"volume" },							// Query volume level
    { SETVOLUME,		"volume level {#level:-100-100}" },	// Directly set the volume level
    { ADJUSTVOLUME,     "volume <adjustment:up|down>" },    // Adjust the volume level
    { GETCROSSFADETIME, "crossfade duration" },             // Query song overlap
    { SETCROSSFADETIME, "crossfade duration {#seconds:0.0-15}" }, // How long to overlap songs
    { GETCROSSFADELEVEL,"crossfade level" },                // Query crossfade volume adjustment
    { SETCROSSFADELEVEL,"crossfade level {#level:0.0-50}" },// How much volume adjustment when crossfading
    { WAITFORENDOFSONG,	"wait for end of [current] song [{options}] ..." }, // Delay further input until after song ends
    { WAITFORNEXTSONG,	"wait for next song [{options}] ..." },	// Delay further input until next song begins

    { NEXTSONG,			"skip" },							// Skip the rest of the current song
    { PAUSEPLAYBACK,	"pause" },							// Pause playback
    { STOPPLAYBACK,		"stop [now]" },						// Stop playback when song finished
    { RESUMEPLAYBACK,   "resume" },
    { PLAYQUEUEMODE,    "play [mode:request|random]" },     // Play player mode
    { TOGGLEPLAYBACK,	"pause toggle" },					// Toggle music playback
    { PLAYMIX,          "play <mode:mix|auto|everything>" },// Select the quickmix playlist, start playback
    { PLAYPLAYLIST,		"play playlist" LIST_PLAYLIST },	// Select a playlist, start playback
    { PLAYDIRECT,       "play from" SIMPLE_LIST_PREDICATE },    // Select a transient playlist, start playback
    { SELECTQUEUEMODE,  "select <mode:request|random>" },   // Play player mode
    { SELECTMIX,        "select <mode:mix|auto|everything>"},// Select the quickmix playlist
    { SELECTPLAYLIST,	"select playlist" LIST_PLAYLIST },  // Select a playlist
    { SELECTDIRECT,     "select from" SIMPLE_LIST_PREDICATE },  // Select a transient playlist

    // Playlist management commands
    { PLAYLISTRENAME,   "playlist rename" SINGLE_PLAYLIST " to {newname}" },
    { PLAYLISTDELETE,   "playlist delete" LIST_PLAYLIST },
    { PLAYLISTCREATE,   "playlist create [smart] from" LIST_PREDICATE },
    { PLAYLISTCREATE,   "playlist create [smart] name {name} from" LIST_PREDICATE },
    { PLAYLISTFROMFILTER,
        "playlist create name {name} where {expression} ..." },
    // Content control -- ratings and seeds
    { GETSUGGESTIONS,   "find [range:all|suggestion|request]" LIST_PREDICATE },
    { RATE,             "rate song {rating}" },                     // Rate song
    { RATE,             "rate song {rating}" LIST_PREDICATE },
    { RATEPLAYLIST,     "rate playlist {rating}" },                 // Rate playlists
    { RATEPLAYLIST,     "rate playlist {rating}" LIST_PLAYLIST },
    { SEEDLIST,         "seed list" },
    { SEEDLIST,         "seed list playlist" LIST_PLAYLIST },      // List the seeds and ratings
    { SEEDALTER,        "seed" SEEDVERB OPTIONALTYPE }, // Make currently playing item into seed for its playlist
    { SEEDALTER,		"seed" SEEDVERB OPTIONALTYPE SINGLE_PREDICATE },    // Make something into a seed for its playlist.
    { SEEDALTER,        "seed" SEEDVERB OPTIONALTYPE SINGLE_PREDICATE SEEDPREP " playlist" LIST_PLAYLIST }, // Make something into a seed for another playlist.
    { SEEDALTER,        "seed" SEEDVERB OPTIONALTYPE SEEDPREP " playlist" LIST_PLAYLIST }, // Make currently playing item into seed for another playlist
    { SEEDALTER,        "playlist modify" SINGLE_PLAYLIST SEEDVERB " seed" LIST_PREDICATE },

    // Song info and requests
    { LISTSONGSBYFILTER,"song list" LIST_PREDICATE },           // List songs matching the expression
    { LISTSONGSBYPLAYLIST, "playlist song list" LIST_PLAYLIST },// List songs from playlist matching the expression
    { REQUESTMUSIC,     "song request" LIST_PREDICATE },        // DEPRECATED FORM
    { REQUESTMUSIC,     "request" LIST_PREDICATE },             // Request some music
    { REQUESTCLEAR,     "request clear" },                      // Clear request queue
    { REQUESTCANCEL,    "request cancel" SIMPLE_LIST_PREDICATE },      // Remove items from request queue
    { ALTERAUDIOCONFIG, "room reconfigure [{audio_options}] ..." }, // Change audio settings

#ifndef NDEBUG
    { FILTERECHO,       "test filter echo" SIMPLE_LIST_PREDICATE }, // Test predicate filter generation
#endif

    { CMD_INVALID,      NULL }
};

static const LookupTable <SearchRange> SearchRanges {
    { "all",            SearchRange::EXHAUSTIVE },
    { "suggestion",     SearchRange::SHALLOW },
    { "request",        SearchRange::REQUESTABLE },
    { "known",          SearchRange::KNOWN }
};


const FB_PARSE_DEFINITION *AudioEngine::statements (void) {
    return statementList;
};

/** Check a list of somethings for authorized use.
    @tparam ListType The type of the list.
    @param items_to_check The list of items.
    @param user The user requesting authorization.
    @param diagnostics A diagnostics collector for failed authorizations.
    @param usage The manner in which the items are to be used.
    @return A list of somethings for which use is authorized. */
template<typename ListType>
ListType AuthorizedUse (ListType items_to_check,
                        User *user,
                        PartialResponse *diagnostics,
                        Ownership::Action usage) {
    ListType valid;
    for (auto item : items_to_check) {
        if (item->hasPermission(user, usage)) {
            valid.push_back (item);
        } else {
            diagnostics->fail (E_UNAUTHORIZED, item);
        }
    }
    return valid;
}


/* Send the queue or history.  Allow negative indexes from either to refer to the other;
 0 refers to current track.  This makes implementing paging on clients easier because
 it won't have to use separate commands. */
void AudioEngine::sendSongLists (PianodConnection &conn, ENGINECOMMAND command) {
    if (conn.argv ("index")) {
        // Index given, send one specific song
        long index = atoi (conn.argv ("index"));
        if (index == 0) {
            if (current_song) {
                conn << S_DATA << *current_song << S_DATA_END;
            } else {
                conn << E_WRONG_STATE;
            }
        } else {
            // Deal with negative indexes by negating the request type.
            if (index < 0) {
                index = -index;
                command = (command == QUERYHISTORY ? QUERYQUEUE : QUERYHISTORY);
            }
            SongList *src = &song_history;
            if (command == QUERYQUEUE && index == 1 && cueing_song) {
                conn << S_DATA << *cueing_song << S_DATA_END;
            } else {
                if (command == QUERYQUEUE) {
                    if (cueing_song)
                        index--;
                    src = &requests;
                    if (index > src->size()) {
                        index -= src->size();
                        src = &random_queue;
                    }
                }
                if (index > src->size()) {
                    conn << E_NOTFOUND;
                } else {
                    conn << S_DATA << (*src) [index - 1] << S_DATA_END;
                }
            }
        }
    } else {
        // No index specified, send the whole list.
        if (command == QUERYHISTORY) {
            conn << song_history << S_DATA_END;
        } else {
            if (cueing_song)
                conn << S_DATA << *cueing_song;
            conn << requests << random_queue << S_DATA_END;
        }
    }
}




bool AudioEngine::hasPermission (PianodConnection &conn, ENGINECOMMAND command) {
    switch (command) {
        case TIMESTATUS:
        case QUERYSTATUS:
        case QUERYHISTORY:
        case QUERYQUEUE:
        case GETHISTORYSIZE:
        case GETVOLUME:
        case GETCROSSFADETIME:
        case GETCROSSFADELEVEL:
        case WAITFORENDOFSONG:
        case WAITFORNEXTSONG:
        case RATEPLAYLIST:
        // These are based on ownership.  A listener with service may be an owner
        case PLAYLISTRENAME:
        case PLAYLISTDELETE:
        case PLAYLISTCREATE:
        case LISTSONGSBYPLAYLIST:
        case SEEDLIST:
        case SEEDALTER:
        // Needed by potential owners and for a future "Request" privilege.
        case GETSUGGESTIONS:
        case LISTSONGSBYFILTER:
        // Owner-based, but also sources like filesystem allow all real users:
        case RATE:
            return conn.haveRank (Rank::Listener);

        case REQUESTMUSIC:
        case REQUESTCLEAR:
        case REQUESTCANCEL:
            return conn.haveRank (Rank::Standard) || conn.havePrivilege (Privilege::Queue);

        case SETVOLUME:
        case ADJUSTVOLUME:
        case NEXTSONG:
        case STOPPLAYBACK:
        case PAUSEPLAYBACK:
        case RESUMEPLAYBACK:
        case TOGGLEPLAYBACK:
        case PLAYQUEUEMODE:
        case PLAYPLAYLIST:
        case PLAYMIX:
        case PLAYDIRECT:
        case SELECTQUEUEMODE:
        case SELECTMIX:
        case SELECTPLAYLIST:
        case SELECTDIRECT:
        // Crossfades should be a user setting, like volume.
        // But if the audio output doesn't support it (libsdl), it can cause trouble,
        // so it should be admin-only.  Errr... taking a reasonable capability away
        // for this reason seems egregious, so it remains available.
        case SETCROSSFADETIME:
        case SETCROSSFADELEVEL:
            return conn.haveRank (Rank::Standard);
        default:
            return conn.haveRank (Rank::Administrator);
    }
};

/** Check that required state/privileges are available.
    @throw CommandError if requirements are not met. */
void AudioEngine::require (PianodConnection &conn, unsigned long requirements) const {
    assert (conn.source());
    assert (current_playlist);
    // Source-related requirements
    if (requirements & REQUIRE_SOURCE) {
        if (!conn.source()->isReady()) {
            throw CommandError (E_WRONG_STATE, "Source is not ready.");
        }
        if ((requirements & REQUIRE_EXPAND) && !conn.source()->canExpandToAllSongs()) {
            throw CommandError (E_MEDIA_ACTION, "Song lists from current source");
        }
    }
    // Playlist-related requirements
    if (requirements & REQUIRE_PLAYLIST) {
        if ((requirements & REQUIRE_EXPAND) && !current_playlist->source()->canExpandToAllSongs()) {
            throw CommandError (E_MEDIA_ACTION, "Song lists from current playlist");
        }
    }
    // Other requirements
    if ((requirements & REQUIRE_PLAYER) && !player) {
        throw CommandError (E_WRONG_STATE, "Not playing");
    }
}


/** Get a thing from a predicate and ensure it is usable.
    @param conn The connection requesting the items.
    @param usage The manner in which the object is to be accessed.
    @return The thing
    @throw CommandError on invalid request, not found, or insufficient privileges. */
MusicThingie *AudioEngine::getThingOrCurrent (const PianodConnection &conn,
                                              Ownership::Action usage) const {
    // Get a thing by ID or current song
    MusicThingie *thing = nullptr;
    if (Predicate::havePredicate(conn)) {
        thing = Predicate::getSpecifiedThing (conn);
    } else if (current_song) {
        thing = current_song.get();
    } else {
        throw CommandError (E_WRONG_STATE, "No current song");
    }
    assert (thing);
    if (!thing->hasPermission (conn.user, usage))
        throw CommandError (E_UNAUTHORIZED);
    return thing;
};


/** Validate and/or convert a thingie to a requested type.
    @param thing The thing to convert
    @param want The type to convert it to.
    @return A pointer to the object, or nullptr if it does not convert.
    @throw CommandError When converting a song to a playlist, but the song has no playlist. */
static MusicThingie *thingie_cast (MusicThingie *thing, MusicThingie::Type want) {
    // Adjust the type, such as getting playlist from song, and validate type.
    switch (want) {
        case MusicThingie::Type::Artist:
            return thing->asArtist();
        case MusicThingie::Type::Album:
            return thing->asAlbum();
        case MusicThingie::Type::Song:
            return thing->asSong();
        case MusicThingie::Type::Playlist:
        {
            PianodPlaylist *pl;
            PianodSong *song;
            if ((pl = thing->asPlaylist())) {
                return pl;
            } else if ((song = thing->asSong())) {
                thing = song->playlist();
                if (!thing) {
                    throw CommandError (E_INVALID, "Song does not have a playlist.");
                }
                return thing;
            };
            break;
        }
        default:
            assert (!"Switch/case unmatched");
            break;
    }
    return nullptr;
}


/** Get a thing from a predicate, ensure usability and it conforms to a type.
    @param conn The connection requesting the items.
    @param want Indicates the type of thing wanted.
    @param usage The manner in which the object is to be accessed.
    @return The thing
    @throw CommandError on invalid request, not found, or insufficient privileges. */
MusicThingie *AudioEngine::getThingOrCurrent (const PianodConnection &conn,
                                              MusicThingie::Type want,
                                              Ownership::Action usage) const {
    // Get a thing by ID or current song
    MusicThingie *orig_thing = getThingOrCurrent(conn, usage);
    assert (orig_thing);
    MusicThingie *thing = thingie_cast (orig_thing, want);
    if (!thing)
        throw CommandError (E_WRONGTYPE, (*orig_thing)());
    return thing;
};



/** Get some things from a predicate and ensure they is usable.
    @param conn The connection requesting the things.
    @param diagnostics Status aggregator for problems encountered.
    @param usage The manner in which the object is to be accessed.
    @return The things; may be the empty set.
    @throw CommandError on invalid request, not found, or insufficient privileges
    for a single-item predicate. */
ThingieList AudioEngine::getThingsOrCurrent (const PianodConnection &conn,
                                             PartialResponse *diagnostics,
                                             Ownership::Action usage) const {
    assert (diagnostics);
    ThingieList candidates;
    if (!Predicate::havePredicate(conn)) {
        candidates.push_back (getThingOrCurrent (conn, usage));
        return candidates;
    }
    return AuthorizedUse (Predicate::getSpecifiedThings (conn),
                          conn.user, diagnostics, usage);
};



/** Get things from a predicate, ensure usability and it conforms to a type.
    @param conn The connection requesting the things.
    @param want Indicates the type of thing wanted.
    @param diagnostics Status aggregator for problems encountered.
    @param usage The manner in which the object is to be accessed.
    @return The things; may be the empty set.
    @throw CommandError on invalid request, not found, or insufficient privileges
    for a single-item predicate. */
ThingieList AudioEngine::getThingsOrCurrent (const PianodConnection &conn,
                                             MusicThingie::Type want,
                                             PartialResponse *diagnostics,
                                             Ownership::Action usage) const {
    assert (diagnostics);
    ThingieList candidates;
    if (!Predicate::havePredicate(conn)) {
        candidates.push_back (getThingOrCurrent (conn, usage));
        return candidates;
    }
    candidates = Predicate::getSpecifiedThings (conn);
    ThingieList results;
    for (auto item : candidates) {
        try {
            MusicThingie *thing = thingie_cast (item, want);
            if (!thing) {
                diagnostics->fail (E_WRONGTYPE, item);
            } else if (thing->hasPermission (conn.user, usage)) {
                results.push_back (thing);
            } else {
                diagnostics->fail (E_UNAUTHORIZED, thing);
            }
        } catch (const CommandError &e) {
            diagnostics->fail (e.reason(), item);
        }
    }
    return results;
}


/** Get a songlist predicate, or use current song.
    @param conn The connection requesting the songs.
    @param diagnostics Status aggregator for problems encountered.
    @param usage The manner in which the object is to be accessed.
    @return The songs; may be the empty set.
    @throw CommandError on invalid request, not found, or insufficient privileges
    for a single-item predicate. */

SongList AudioEngine::getSongsOrCurrent (const PianodConnection &conn,
                                         PartialResponse *diagnostics,
                                         Ownership::Action usage) const {
    assert (diagnostics);
    // Privilege checks done by called function.
    ThingieList things = getThingsOrCurrent (conn, MusicThingie::Type::Song,
                                             diagnostics, usage);
    SongList songs;
    songs.reserve (things.size());
    for (auto song : things) {
        assert (song->asSong());
        songs.push_back (song->asSong());
    }
    return songs;
}



/** Get a playlist predicate, or use current playlist.
 @param conn The connection requesting the things.
 @param usage The manner in which the object is to be accessed.
 @return The things; may be the empty set.
 @throw CommandError on invalid request, not found, or insufficient privileges
 for a single-item predicate. */
PianodPlaylist *AudioEngine::getPlaylistOrCurrent (const PianodConnection &conn,
                                                    Ownership::Action usage) const {
    PianodPlaylist *candidate;
    if (Predicate::havePlaylistPredicate (conn)) {
        candidate = Predicate::getSpecifiedPlaylist (conn);
    } else {
        if (!current_song && !current_playlist)
            throw CommandError (E_AMBIGUOUS, "No current playlist");
        if (current_song && current_song->playlist() != current_playlist.get())
            throw CommandError (E_AMBIGUOUS);
        candidate = current_song ? current_song->playlist() : current_playlist.get();
        if (candidate->playlistType() != PianodPlaylist::SINGLE)
            throw CommandError (E_METAPLAYLIST);
    }
    if (!candidate->hasPermission (conn.user, usage))
        throw CommandError (E_UNAUTHORIZED, candidate->id());
    return candidate;
}





/** Get a playlist predicate, or use current playlist.
    @param conn The connection requesting the things.
    @param diagnostics Status aggregator for problems encountered.
    @param usage The manner in which the object is to be accessed.
    @return The things; may be the empty set.
    @throw CommandError on invalid request, not found, or insufficient privileges
    for a single-item predicate. */
PlaylistList AudioEngine::getPlaylistsOrCurrent (const PianodConnection &conn,
                                                 PartialResponse *diagnostics,
                                                 Ownership::Action usage) const {
    PlaylistList candidates;
    if (Predicate::havePlaylistPredicate (conn)) {
        candidates = Predicate::getSpecifiedPlaylists (conn);
    } else {
        candidates.push_back (getPlaylistOrCurrent (conn,usage));
    }
    return AuthorizedUse (candidates, conn.user, diagnostics, usage);
}


/** Get a playlist predicate, or use one implied by a thing.
    @param conn The connection requesting the things.
    @param diagnostics Status aggregator for problems encountered.
    @param usage The manner in which the object is to be accessed.
    @return The things; may be the empty set.
    @throw CommandError on invalid request, not found, or insufficient privileges
    for a single-item predicate. */
PlaylistList AudioEngine::getPlaylistsOrCurrent (const PianodConnection &conn,
                                                 PartialResponse *diagnostics,
                                                 Ownership::Action usage,
                                                 const ThingieList &default_playlist) const {
    PlaylistList candidates;
    if (Predicate::havePlaylistPredicate (conn)) {
        candidates = Predicate::getSpecifiedPlaylists (conn);
    } else {
        // Use implied playlist from a list of things; all must be songs and all have the same playlist.
        PianodPlaylist *play = nullptr;
        if (default_playlist.size() > 0) {
            auto first = default_playlist.front()->asSong();
            if (first)
                play = first->playlist();
            if (play) {
                for (auto thing : default_playlist ) {
                    auto song = thing->asSong();
                    if (!song)
                        throw CommandError (E_NO_ASSOCIATION, thing->id());
                    if (song->playlist() != play) {
                        throw CommandError (E_CONFLICT, "Songs have inconsistent playlists");

                    }
                }
            }
        }
        if (!play)
            throw CommandError (E_NO_ASSOCIATION);
        candidates.push_back (play);
    }

    return AuthorizedUse (candidates, conn.user, diagnostics, usage);
}


/** Add, remove, or toggle some seeds to/from some playlists.
    @param action Indicates add, remove or toggle.
    @param seeds The list of seeds to add/remove.
    @param targets the playlists to add/remove from.
    @param fixed_type If true, typecast seeds to `seed_type` when adding/removing.
    @param seed_type Type of seed to typecast to.
    @param diagnostics Status aggregator for problems encountered. */
static void seedAction (const CartsVerb action,
                 const ThingieList &seeds,
                 const PlaylistList &targets,
                 const bool fixed_type,
                 MusicThingie::Type seed_type,
                 PartialResponse *diagnostics) {
    for (auto seed : seeds) {
        for (auto target : targets) {
            assert (target);
            assert (target->playlistType() == PianodPlaylist::SINGLE);
            MusicThingie *source_seed = seed;
            try {
                if (!fixed_type)
                    seed_type = seed->primaryType();
                if (seed->source() != target->source()) {
                    source_seed = target->source()->getSuggestion (seed, seed_type,
                                                                   action == CartsVerb::Add ? SearchRange::SHALLOW : SearchRange::KNOWN);
                }
                target->seed(seed_type, source_seed,
                             action == CartsVerb::Add ? true :
                             action == CartsVerb::Remove ? false :
                             /* Toggle */ !target->seed(seed_type, seed));
                (*diagnostics) (S_OK, seed);
            } catch (const CommandError &e) {
                (*diagnostics) (e.reason(), seed);
            }
        }
    }
}





void AudioEngine::handleCommand (PianodConnection &conn, ENGINECOMMAND command) {
    assert (conn.source());
    assert (current_playlist);

    // This first batch of commands may be used at anytime.
    switch (command) {
        case TIMESTATUS:
            sendPlaybackStatus (conn);
            return;
        case QUERYSTATUS:
            if (current_song) {
                conn << S_DATA << *current_song;
            }
            conn << S_DATA_END;
            sendQueueMode (conn);
            sendPlaybackStatus (conn);
            sendSelectedPlaylist (conn);
            conn.sendSelectedSource();
            return;
        case QUERYHISTORY:
        case QUERYQUEUE:
            sendSongLists (conn, command);
            return;
        case SETHISTORYSIZE:
            history_size = atoi (conn ["length"]);
            conn.service() << Response (I_HISTORYSIZE, history_size);
            conn << S_OK;
            return;
        case GETHISTORYSIZE:
            conn << S_DATA << Response (I_HISTORYSIZE, history_size) << S_DATA_END;
            return;
        case WAITFORENDOFSONG:
            if (conn ["current"] && !current_song)
                conn << E_WRONG_STATE;
            else
                conn.waitForEventWithOptions (WaitEvent::Type::TrackEnded, this);
            return;
        case WAITFORNEXTSONG:
            conn.waitForEventWithOptions (WaitEvent::Type::TrackStarted, this);
            return;
        case GETVOLUME:
            conn << S_DATA << Response (I_VOLUME, audio.volume) << S_DATA_END;
            return;
        case SETVOLUME:
            audio.volume = atoi (conn ["level"]);
            if (player) {
                player->setVolume (audio.volume);
            }
            conn << S_OK;
            service << Response (I_VOLUME, audio.volume);
            return;
        case ADJUSTVOLUME:
        {
            // Use a ±100 scale. 0 is "standard" level; >0 causes distortion. 
            if (conn.argvEquals("adjustment", "up")) {
                if (audio.volume >= 100) {
                    conn << Response (E_RANGE, "Already at maximum volume");
                    return;
                }
                audio.volume ++;
            } else {
                if (audio.volume <= -100) {
                    conn << Response (E_RANGE, "Already at minimum volume");
                    return;
                }
                audio.volume --;
            }
            if (player) {
                player->setVolume (audio.volume);
            }
            conn << S_OK;
            service << Response (I_VOLUME, audio.volume);
            return;
        }
        case GETCROSSFADETIME:
            conn << S_DATA << Response (I_INFO, audio.crossfade_time, 1) << S_DATA_END;
            return;
        case SETCROSSFADETIME:
        {
            AudioSettings result = audio;
            result.crossfade_time = strtod (conn ["seconds"], nullptr);
            AudioOptions::validate (result);
            audio.crossfade_time = result.crossfade_time;
            conn << S_OK;
            return;
        }
        case GETCROSSFADELEVEL:
            conn << S_DATA << Response (I_INFO, audio.crossfade_level, 1) << S_DATA_END;
            return;
        case SETCROSSFADELEVEL:
            audio.crossfade_level = strtod (conn ["level"], nullptr);
            conn << S_OK;
            return;
        case RESUMEPLAYBACK:
        case PAUSEPLAYBACK:
        case TOGGLEPLAYBACK:
        {
            playbackState (command == RESUMEPLAYBACK ? PlaybackState::Playing :
                           command == PAUSEPLAYBACK ? PlaybackState::Paused :
                           playback_state == PlaybackState::Playing ? PlaybackState::Paused : PlaybackState::Playing,
                           &conn);
            conn << S_OK;
            return;
        }
        case STOPPLAYBACK:
            if (conn ["now"] && player) {
                aborting_playback = true;
                player->abort ();
            }
            queueMode (QueueMode::Stopped);
            conn << S_OK;
            return;
        case PLAYQUEUEMODE:
        case SELECTQUEUEMODE:
        {
            if (quit_requested)
                throw CommandError (E_WRONG_STATE, "Shutdown pending");
            queueMode (conn.argvEquals ("mode", "request") ? QueueMode::Requests : QueueMode::RandomPlay, &conn);
            if (command == PLAYQUEUEMODE)
                playbackState (PlaybackState::Playing, &conn);
            conn << S_OK;
            return;
        }
        case PLAYPLAYLIST:
        case PLAYMIX:
        case PLAYDIRECT:
        case SELECTPLAYLIST:
        case SELECTMIX:
        case SELECTDIRECT:
        {
            if (quit_requested)
                throw CommandError (E_WRONG_STATE, "Shutdown pending");
            PianodPlaylist *new_playlist;
            if (command == PLAYDIRECT || command == SELECTDIRECT) {
                require (conn, REQUIRE_SOURCE | REQUIRE_EXPAND);
                auto filter = Predicate::getPredicate (conn);
                if (!filter->canPersist())
                    throw CommandError (E_PERSISTENT);
                new_playlist = conn.source()->getTransientPlaylist (*filter.get());
                if (!new_playlist || (new_playlist->source() != media_manager && new_playlist->songs().size() == 0)) {
                    throw CommandError (E_NOTFOUND);
                }
            } else if (command == PLAYMIX || command == SELECTMIX) {
                new_playlist = (conn.argvEquals ("mode", "everything")
                               ? conn.source()->getEverythingPlaylist()
                               : conn.source()->getMixPlaylist());
                if (!new_playlist) {
                    // In case a source doesn't have a mix or everything playlist.
                    conn << E_MEDIA_ACTION;
                    return;
                }
                mix.automatic (conn.argvEquals ("mode", "auto"));
            } else {
                new_playlist = Predicate::getSpecifiedPlaylist (conn);
                mix.automatic (false);
            }
            assert (new_playlist);
            current_playlist = new_playlist;
            conn.announce (A_SELECTED_PLAYLIST, current_playlist->playlistName());
            sendSelectedPlaylist ();
            mix.recalculatePlaylists ();

            if (command == PLAYPLAYLIST || command == PLAYMIX || command == PLAYDIRECT) {
                queueMode (QueueMode::RandomPlay, &conn);
                playbackState (PlaybackState::Playing, &conn);
            }
            conn << S_OK;
            return;
        }
        case PLAYLISTRENAME:
        {
            auto play = Predicate::getSpecifiedPlaylist (conn);
            assert (play);

            if (!play->isEditableBy (conn.user)) {
                conn << E_UNAUTHORIZED;
            } else {
                const char *name = conn.argv ("newname");
                play->rename (name);
                conn << S_OK;
                conn.announce (A_RENAMED_PLAYLIST, name);
                service_manager->broadcast (V_PLAYLISTS_CHANGED);
            }
            return;
        }
        case PLAYLISTDELETE:
        {
            PartialResponse response (PartialResponse::aggregate_type::optimistic,
                                      A_DELETED_PLAYLIST);


            auto playlists = AuthorizedUse (Predicate::getSpecifiedPlaylists (conn),
                                            conn.user, &response, Ownership::Action::ALTER);
            for (auto play : playlists) {
                try {
                    play->erase();
                    response (S_OK, play);
                } catch (const CommandError &e) {
                    response (e.reason(), play);
                }
            }
            conn << response;
            service_manager->broadcast (V_PLAYLISTS_CHANGED);
            mix.updatePlaylists();
            return;
        }
        case RATE:
        {
            // Determine the assigned rating
            const char *ratingText = conn.argv ("rating");
            Rating rating = Rating::UNRATED;
            bool overplayed = (strcasecmp (ratingText, "overplayed") == 0);

            // Validating rating value before doing real work.
            if (!overplayed)
                rating = RATINGS [ratingText];

            PartialResponse result;
            SongList songs = getSongsOrCurrent (conn, &result);
            for (auto song : songs) {
                RESPONSE_CODE status = E_BUG;
                try {
                    if (song->ratingScheme() == RatingScheme::Nobody) {
                        if (song->isSeed() || song->isSuggestion()) {
                            status = E_WRONGTYPE;
                        } else {
                            status = E_MEDIA_ACTION;
                        }
                    } else if (song->ratingScheme() == RatingScheme::Individual && !conn.user) {
                        status = E_LOGINREQUIRED;
                    } else if (song->ratingScheme() != RatingScheme::Individual
                        && !song->isEditableBy (conn.user)) {
                        status = E_UNAUTHORIZED;
                    } else if (overplayed) {
                        status = song->rateOverplayed (conn.user);
                    } else {
                        status = song->rate(rating, conn.user);
                        if (isSuccess (status)) {
                            if (song == current_song.get()) {
                                sendUpdatedRatings (conn, song);
                            } else if (song->ratingScheme() == RatingScheme::Owner) {
                                conn.service() << Response (V_SONGRATING_CHANGED,
                                                            song->id());
                            } else {
                                service_manager->broadcast (Response (V_SONGRATING_CHANGED,
                                                                     song->id()),
                                                           conn.user);
                            }
                        }
                    }
                } catch (const CommandError &e) {
                    status = e.reason();
                }
                result (status, song);
            }
            conn << result;
            return;
        }
        case RATEPLAYLIST:
        {
            Rating rating = RATINGS [conn ["rating"]];
            PartialResponse status;
            PlaylistList playlists = getPlaylistsOrCurrent(conn, &status);

            for (auto play : playlists) {
                RESPONSE_CODE result = play->rate (rating, conn.user);
                if (isSuccess (result)) {
                    service_manager->broadcast (Response (V_PLAYLISTRATING_CHANGED,
                                                         play->id()),
                                               conn.user);
                    mix.recalculatePlaylists();
                    track_acquisition_time = 0;
                }
                status (result, play);
            }
            conn << status;
            return;
        }
        case SEEDLIST:
        {
            PianodPlaylist *play = getPlaylistOrCurrent(conn, Ownership::Action::READ);
            assert (play);
            sendSeedlist (conn, play->getSeeds(), play) << S_DATA_END;
            return;
        }
        case SEEDALTER:
        {
            // Determine kind of seed
            CartsVerb action = CartsWord [conn ["verb"]];
            auto type = conn.argv ("type");
            MusicThingie::Type seed_kind = MusicThingie::Type::Song;;
            ThingieList seeds;
            PartialResponse response;
            if (type) {
                // Acquire thing of necessary type, either current or specified
                seed_kind = THINGIETYPES [type];
                seeds = getThingsOrCurrent (conn, seed_kind, &response);
            } else {
                // Acquire thing and use implied type
                seeds = getThingsOrCurrent (conn, &response);
            }

            // Acquire the target: song's playlist, current playlist, or specified
            PlaylistList targets = getPlaylistsOrCurrent (conn, &response, Ownership::Action::ALTER, seeds);

            seedAction (action, seeds, targets, type, seed_kind, &response);
            conn << response;
            return;
        }
        case PLAYLISTCREATE:
        {
            if (!conn.source()->isEditableBy (conn.user)) {
                conn << E_UNAUTHORIZED;
                return;
            }

            PartialResponse response (PartialResponse::aggregate_type::optimistic,
                                      A_CREATED_PLAYLIST);

            // Determine kind of starter seeds, then the seeds.
            // Although not usec to create a smart playlist, the first one
            // may be used to assign a name.
            auto type = conn.argv ("type");
            MusicThingie::Type seed_kind = MusicThingie::Type::Song;
            ThingieList seeds;
            if (type) {
                // Acquire thing of necessary type, either current or specified
                seed_kind = THINGIETYPES [type];
                seeds = getThingsOrCurrent (conn, seed_kind, &response);
            } else {
                // Acquire thing and use implied type
                seeds = getThingsOrCurrent (conn, &response);
            }

            // Get and validate the playlist name
            bool smart = (conn ["smart"]);
            string name = conn ["name"];
            if (name.empty() && (smart || conn.source()->requireNameForCreatePlaylist())) {
                name = seeds.front()->name() + " Playlist";
            }
            if (!name.empty() && conn.source()->getPlaylistByName (name.c_str())) {
                conn << E_DUPLICATE;
                return;
            }

            PianodPlaylist *playlist;
            if (smart) {
                // Create a smart playlist.
                auto p = Predicate::getPredicate (conn);
                if (!p->canPersist()) {
                    conn << E_PERSISTENT;
                    return;
                };
                playlist = conn.source()->createPlaylist (name.c_str(), *p);
                response.succeed();
            } else {
                // Create a standard playlist.

                // Create the playlist with first item.
                MusicThingie *seed = seeds.front();
                seeds.pop_front();
                if (!type)
                    seed_kind = seed->primaryType();
                if (seed->source() != conn.source())
                    seed = conn.source()->getSuggestion (seed, seed_kind);
                playlist = conn.source()->createPlaylist (name.empty() ? nullptr : name.c_str(), seed_kind, seed);
                response (name.empty() || playlist->name() == name ? S_OK : S_ROUNDING, seed);

                // Subsequently seeds are handled by seed_add.
                PlaylistList playlist_as_list;
                playlist_as_list.push_back (playlist);
                seedAction (CartsVerb::Add, seeds, playlist_as_list, type, seed_kind, &response);
            }
            conn << response;
            if (response.anySuccess()) {
                conn.announce (A_CREATED_PLAYLIST, playlist->name());
                service_manager->broadcast (V_PLAYLISTS_CHANGED);
                mix.updatePlaylists();
            }
            return;
        }
        case PLAYLISTFROMFILTER:
        {
            if (!conn.source()->isEditableBy (conn.user)) {
                conn << E_UNAUTHORIZED;
                return;
            }
            assert (conn ["expression"]);
            Filter f (conn.argvFromUntokenized ("expression"));
            if (!f.canPersist()) {
                conn << E_PERSISTENT;
                return;
            };
            if (conn.source()->getPlaylistByName (conn ["name"])) {
                conn << E_DUPLICATE;
                return;
            }
            conn.source()->createPlaylist (conn ["name"], f);
            conn << S_OK;
            conn.announce (A_CREATED_PLAYLIST, conn ["name"]);
            service_manager->broadcast (V_PLAYLISTS_CHANGED);
            mix.updatePlaylists();
            return;
        }
        case LISTSONGSBYPLAYLIST:
        {
            require (conn, REQUIRE_SOURCE | REQUIRE_EXPAND);
            auto play = getPlaylistOrCurrent (conn, Ownership::Action::READ);
            // A predicate specifying a source could return a non-request playlist.
            if (!play->source()->canExpandToAllSongs()) {
                throw CommandError (E_MEDIA_ACTION, "Requests from specified source");
            }
            conn << play->songs() << S_DATA_END;
            return;
        }
        case LISTSONGSBYFILTER:
        {
            auto matches = Predicate::getSpecifiedSongs (conn, SearchRange::REQUESTS);
            conn << matches << S_DATA_END;
            return;
        }
        case REQUESTMUSIC:
        {
            auto requested = Predicate::getSpecifiedSongs (conn, SearchRange::REQUESTS);
            SongList::size_type size = requested.size();
            requests.join (requested);
            conn << Response (S_MATCH, size);
            if (size == 1) {
                conn.announce (A_REQUEST_ADD, requested.front()->name());
            } else if (size > 1) {
                conn.announce (A_REQUEST_ADD);
            }
            service << V_QUEUE_CHANGED;
            return;
        }
        case REQUESTCLEAR:
            if (!requests.empty()) {
                requests.clear();
                conn.announce(A_REQUEST_CLEAR);
            }
            conn << S_OK;
            service << V_QUEUE_CHANGED;
            return;
        case REQUESTCANCEL:
        {
            auto filter = Predicate::getPredicate (conn);
            PartialResponse response (PartialResponse::optimistic, A_REQUEST_CANCEL);
            SongList retained_requests;
            // Check cueing songs
            if (cueing_song && filter->matches (cueing_song.get())) {
                time_t when;
                if (cueing_song->canSkip (&when)) {
                    response.succeed (cueing_song.get());
                    cueing_player->abort();
                } else {
                    response.fail (E_QUOTA, cueing_song.get());
                }
            }
            // Check the request queue
            retained_requests.reserve (requests.size());
            for (auto song : requests) {
                if (filter->matches (song)) {
                    response.succeed (song);
                } else {
                    retained_requests.push_back (song);
                }
            }
            // Check the random queue
            SongList retained_random;
            for (auto song : random_queue) {
                if (filter->matches (song)) {
                    time_t when;
                    if (song->canSkip (&when)) {
                        response.succeed (song);
                    } else {
                        response.fail (E_QUOTA, song);
                    }
                } else {
                    retained_random.push_back (song);
                }
            }
            if (response.noop()) {
                response.fail (E_NOTFOUND);
            } else if (response.anySuccess()) {
                requests = retained_requests;
                random_queue = retained_random;
                service << V_QUEUE_CHANGED;
            }
            conn << response;
            return;
        }
        case GETSUGGESTIONS:
        {
            SearchRange range = (conn ["range"] ? SearchRanges [conn ["range"]] :
                                 SearchRange::SHALLOW);
            ThingieList results { Predicate::getSpecifiedThings (conn, range) };
            conn << results << S_DATA_END;
            return;
        }
        case ALTERAUDIOCONFIG:
        {
            AudioSettings altered = audio;
            if (AudioOptions::parser.interpret(conn.argvFrom ("audio_options"), altered, conn) == FB_PARSE_SUCCESS) {
                audio = altered;
                service << Response (I_VOLUME, audio.volume);
                conn << S_OK;
            }
            return;
        }
#ifndef NDEBUG
        case FILTERECHO:
        {
            auto f = Predicate::getPredicate (conn);
            string expr = f->toString();
            conn << S_DATA << Response (I_INFO, expr) << S_DATA_END;
            return;
        }
#endif
        default:
            /* Continue to next switch statement, after requirements check. */;
    }



    // From here on, commands require a player.
    require (conn, REQUIRE_PLAYER);
    assert (current_song);
    switch (command) {
        case NEXTSONG:
        {
            time_t nextSkip = 0;
            if (current_song->canSkip (&nextSkip)) {
                aborting_playback = true;
                player->abort ();
                conn << Response (A_SKIPPED, current_song->title()) << S_OK;
            } else {
                conn << E_QUOTA;
                if (nextSkip) {
                    nextSkip -= time(nullptr);
                    conn.printf ("%03d %s: Next skip in %d:%02d.\n",
                                 V_SOURCE_STATUS, ResponseText (V_SOURCE_STATUS), (int) nextSkip / 60, (int) nextSkip % 60);
                }
            }
            return;
        }
        default:
            flog (LOG_WHERE (LOG_WARNING),
                   "Unimplemented command ", command);
            break;
    }
}

/* Send the current status, including:
    - Playing, paused, stopped, between tracks
    - Track duration, Current playhead position, track remaining if playing or paused
    @param there The destination of the data.
    @param only_if_accurate Only send data if playpoint and duration are accurate.
    If not, defer output and return false.
    @return true if the data reported was complete, false if playpoint or duration
    were not available and reported as 0. */
bool AudioEngine::sendPlaybackStatus (Football::Thingie &there, bool only_if_accurate) {
    bool data_valid = true;
    if (player && player->ready()) {
        RESPONSE_CODE state = (playback_state == PlaybackState::Paused ? V_PAUSED :
                               stall.onset ? V_STALLED : V_PLAYING);

        long duration = player->trackDuration();
        long playpoint = player->playPoint();
        data_valid = (duration >= 0 && playpoint >= 0);
        if (only_if_accurate && !data_valid)
            return false;
        if (duration < 0)
            duration = 0;
        if (playpoint < 0)
            playpoint = 0;
        long song_remaining = data_valid ? player->playRemaining() : 0;
        bool remaining_is_negative = song_remaining < 0;
        if (remaining_is_negative) {
            // song is longer than expected
            song_remaining = -song_remaining;
        }

        there.printf ("%03d %s: %02i:%02i/%02i:%02i/%c%02li:%02li\n",
                      state, ResponseText (state),
                      playpoint / 60, playpoint % 60,
                      duration / 60, duration % 60,
                      (remaining_is_negative ? '+' : '-'),
                      song_remaining / 60, song_remaining % 60);
    } else {
        there << ((playback_state == PlaybackState::Paused ||
                   (queue_mode == QueueMode::Requests && requests.empty()) ||
                   queue_mode == QueueMode::Stopped) ? V_IDLE : V_BETWEEN_TRACKS);
    }
    return data_valid;
}

/** Pause or resume playback.
    @param state Indicates whether to play or pause.
    @param conn Connection requesting change, or nullptr. */
void AudioEngine::playbackState (PlaybackState state, PianodConnection *conn) {
    if (playback_state != state) {
        playback_state = state;

        // Apply that state to the player.
        if (player) {
            if (playback_state == PlaybackState::Playing) {
                player->play();
                if (cueing_player && transition_state >= TransitionProgress::Crossfading)
                    cueing_player->play();
            } else {
                player->pause();
                if (cueing_player)
                    cueing_player->pause();
            }
            if (playback_state == PlaybackState::Paused && !pause_expiration) {
                pause_expiration = (pause_timeout ?
                                    time (nullptr) + pause_timeout :
                                    FAR_FUTURE);
            } else if (playback_state == PlaybackState::Playing) {
                pause_expiration = 0;
                stall.playback_effective_start = time (nullptr) - (long) (player->playPoint());
                stall.onset = 0;
                stall.onset_playpoint = 0;
            }
            if (conn) {
                conn->announce (playback_state == PlaybackState::Playing ? A_RESUMED : A_PAUSED);
            }
        } else if (playback_state == PlaybackState::Playing && queue_mode != QueueMode::Stopped) {
            conn->announce (A_RESUMED);
        }
        sendPlaybackStatus();
    }
}

/** Set the queueing mode.
 @param mode Indicates stopped, requests only, or random play.
 @param conn Connection requesting change, or nullptr. */
void AudioEngine::queueMode (QueueMode mode, PianodConnection *conn) {
    if (mode != queue_mode) {
        queue_mode = mode;
        sendQueueMode();
        if (conn) {
            conn->announce (mode == QueueMode::Stopped ? A_STOPPED :
                            mode == QueueMode::Requests ? A_REQUESTS : A_RANDOMPLAY);
        }
    }
}


void AudioEngine::sendQueueMode (Football::Thingie &there) {
    there << (queue_mode == QueueMode::Stopped ? V_QUEUE_STOPPED :
              queue_mode == QueueMode::Requests ? V_QUEUE_REQUEST : V_QUEUE_RANDOM);
}

void AudioEngine::sendSelectedPlaylist (Football::Thingie &there) {
    assert (current_playlist);
    const char *mode = "bug";
    switch (current_playlist->playlistType()) {
        case PianodPlaylist::EVERYTHING:
            mode = "everything";
            break;
        case PianodPlaylist::MIX:
            mode = mix.automatic() ? "auto" : "mix";
            break;
        case PianodPlaylist::SINGLE:
        case PianodPlaylist::TRANSIENT:
            mode = "playlist";
            break;
        default:
            assert (!"Switch/case unmatched");
            break;
    }

    there.printf ("%03d %s: %s %s\n",
                  V_SELECTEDPLAYLIST, ResponseText (V_SELECTEDPLAYLIST),
                  mode, current_playlist->playlistName().c_str());
}


/** Send audio engine status to a client.  Used on login. */
void AudioEngine::sendStatus (PianodConnection &there) {
    there << Response (I_VOLUME, audio.volume);
    sendSelectedPlaylist (there);
    there.sendSelectedSource();
    sendQueueMode(there);
    sendPlaybackStatus (there);
    there << Response (I_ROOM, there.service().roomName());
    if (current_song) {
        there << *current_song;
    }
    mix.sendStatus (there);
}

/** Update audio engine status to a client.  Used on authentication. */
void AudioEngine::updateStatus (PianodConnection &there) {
    if (current_song) {
        sendRatings (&there, *current_song);
    }
}


UserList AudioEngine::getAutotuneUsers () {
    return mix.getAutotuneUsers ();
};
