///
/// Audio engine -- manage current audio player and sources.
///	@file		engine.h - pianod
///	@author		Perette Barella
///	@date		2014-11-30
///	@copyright	Copyright (c) 2014-2020 Devious Fish. All rights reserved.
///

#pragma once

#include <config.h>

#include <ctime>

#include <football.h>

#include "musictypes.h"
#include "retainer.h"
#include "connection.h"
#include "response.h"
#include "mediaunit.h"
#include "tuner.h"

/// Audio engine commands such as start, stop, pause, change playlists or seeds.
typedef enum engine_commands_t {
    TIMESTATUS = CMD_RANGE_ENGINE,
    GETVOLUME,
    SETVOLUME,
    GETCROSSFADETIME,
    SETCROSSFADETIME,
    GETCROSSFADELEVEL,
    SETCROSSFADELEVEL,
    ADJUSTVOLUME,
    GETHISTORYSIZE,
    SETHISTORYSIZE,
    WAITFORENDOFSONG,
    WAITFORNEXTSONG,
    QUERYSTATUS,
    QUERYHISTORY,
    QUERYQUEUE,
    NEXTSONG,
    STOPPLAYBACK,
    PAUSEPLAYBACK,
    RESUMEPLAYBACK,
    TOGGLEPLAYBACK,
    PLAYQUEUEMODE,
    PLAYPLAYLIST,
    PLAYMIX,
    PLAYDIRECT,
    SELECTQUEUEMODE,
    SELECTMIX,
    SELECTPLAYLIST,
    SELECTDIRECT,
    PLAYLISTRENAME,
    PLAYLISTDELETE,
    PLAYLISTCREATE,
    PLAYLISTFROMFILTER,
    GETSUGGESTIONS,
    LISTSONGSBYFILTER,
    LISTSONGSBYPLAYLIST,
    REQUESTMUSIC,
    REQUESTCLEAR,
    REQUESTCANCEL,
    RATE,
    RATEPLAYLIST,
    SEEDLIST,
    SEEDALTER,
    ALTERAUDIOCONFIG,
    // EXPLAINSONGCHOICE, Not implemented/future
    // CREATEBOOKMARK, Not implemented/future
#ifndef NDEBUG
    FILTERECHO
#endif
} ENGINECOMMAND;

/** Audio engine, responsible for one "room" worth of audio.
    Each audio engine has its own queue, selected playlist(s),
    and playback status.
 */
class AudioEngine : public Football::Interpreter<PianodConnection, ENGINECOMMAND> {
private:
    /// Prerequisites that may be necessary to execute a particular command.
    typedef enum requirement_t {
        REQUIRE_SOURCE = 0x01, ///< Marks other flags as source-related
        REQUIRE_PLAYLIST = 0x02, ///< Marks other flags as playlist-related.
        REQUIRE_PLAYER = 0x04, ///< Require an audio player exist
        REQUIRE_EXPAND = 0x10, ///< Require expand to song list capability
    } REQUIREMENT;
    
    enum class PlaybackState {
        Paused,
        Playing
    };

    /// States player progresses through when segueing between tracks.
    enum class TransitionProgress {
        Playing,    ///< the audio player is not transitioning
        Purged,     ///< the queue has een purged in preparation for end-of-song
        Cueing,     ///< a second audio player has been initialized and is initializing/prefetching.
        Crossfading,///< the audio players are both playing as audio is cross-faded
        Done        ///< transition is done, but old song keeps right on going.
    };

    /// Behaviors possible when the player is idle.
    enum class QueueMode {
        Stopped,        ///< Don't start playing anything.
        Requests,       ///< Play only requested songs
        RandomPlay,     ///< Random selections from the current source if no requests.
    };

    /// Whether and how completely track information was announced at start of playback.
    enum class Announced {
        Never,          // Track was never announced
        Partially,      // Track was announced but playpoint/duration incomplete.
        Completely      // Track was announced with full and accurate data.
    };

    /// Structure used to track player progress/status.
    struct {
        time_t playback_effective_start = 0; ///< When not stalled, current time minus playback point.
        time_t onset = 0; ///< Clock time at which a playback stall was detected.
        time_t onset_playpoint = 0; ///< Playback point at which a playback stall was detected.
    } stall;

    static const int pause_timeout = 1800; ///< Pause timeout applied to all sources/songs
    static const int aquire_tracks_retry = 600; ///< Retry period if unable to get random tracks.

    const float prefetch_time = 5; ///< Number of seconds before cuing at which queue is refreshed.

    int history_size = 10; ///< Maximum number of tracks retained in history.

    volatile bool quit_requested = false; ///< Flag for signal handlers to request shutdown.
    bool quit_initiated = false; ///< Flag set once a shutdown request has been initiated.

    time_t track_acquisition_time = 0; ///< The next time at which random track aquisition may be attempted.

    PianodService *service = nullptr;

    Retainer <PianodSong *> current_song = nullptr;
    Retainer <PianodSong *> cueing_song = nullptr;
    SongList requests;
    SongList random_queue;
    SongList song_history;

    Retainer <PianodPlaylist *> current_playlist = nullptr;
    Media::Player *player = nullptr;
    Media::Player *cueing_player = nullptr;
    TransitionProgress transition_state = TransitionProgress::Playing;

    Announced track_announced = Announced::Never;
    PlaybackState playback_state = PlaybackState::Paused;
    QueueMode queue_mode = QueueMode::Stopped;
    time_t pause_expiration = 0;
    bool aborting_playback = false;
    bool empty_warning_given = false;

    AudioSettings audio;
    Tuner::Tuner mix;

    // Standard parser things
    virtual bool hasPermission (PianodConnection &conn, ENGINECOMMAND command) override;
    virtual void handleCommand (PianodConnection &conn, ENGINECOMMAND command) override;
    virtual const FB_PARSE_DEFINITION *statements (void) override;

    // Interpreter helpers
    MusicThingie *getThingOrCurrent (const PianodConnection &conn,
                                     Ownership::Action usage = Ownership::Action::USE) const;
    MusicThingie *getThingOrCurrent (const PianodConnection &conn,
                                     MusicThingie::Type want,
                                     Ownership::Action usage = Ownership::Action::USE) const;
    ThingieList getThingsOrCurrent (const PianodConnection &conn,
                                    PartialResponse *diagnostics,
                                    Ownership::Action usage = Ownership::Action::USE) const;
    ThingieList getThingsOrCurrent (const PianodConnection &conn,
                                    MusicThingie::Type want,
                                    PartialResponse *diagnostics,
                                    Ownership::Action usage = Ownership::Action::USE) const;
    SongList getSongsOrCurrent (const PianodConnection &conn,
                                PartialResponse *diagnostics,
                                Ownership::Action usage = Ownership::Action::USE) const;
    PianodPlaylist *getPlaylistOrCurrent (const PianodConnection &conn,
                                           Ownership::Action usage) const;
    PlaylistList getPlaylistsOrCurrent (const PianodConnection &conn,
                                        PartialResponse *diagnostics,
                                        Ownership::Action usage = Ownership::Action::USE) const;
    PlaylistList getPlaylistsOrCurrent (const PianodConnection &conn,
                                        PartialResponse *diagnostics,
                                        Ownership::Action usage,
                                        const ThingieList &default_playlist) const;
    void require (PianodConnection &conn, unsigned long requirements) const;


    // Internal routines
    inline bool queueEmpty (void) const {
        return requests.empty() && random_queue.empty();
    }
    bool startPlayer (void);
    void promotePlayer (void);
    void cleanupPlayer (void);
    float monitorPlayer (void);

    // Status and other senders
    void playbackState (PlaybackState state, PianodConnection *conn = nullptr);
    void queueMode (QueueMode mode, PianodConnection *conn = nullptr);
    void sendSongLists (PianodConnection &conn, ENGINECOMMAND command);
    bool sendPlaybackStatus (Football::Thingie &there, bool only_if_accurate = false);
    void sendQueueMode (Football::Thingie &there);
    void sendSelectedPlaylist (Football::Thingie &there);

    // Alternate senders 
    inline bool sendPlaybackStatus (bool only_if_accurate = false) {
        return sendPlaybackStatus (*service, only_if_accurate);
    };
    inline void sendQueueMode (void) { sendQueueMode (*service); };
    inline void sendSelectedPlaylist (void) { sendSelectedPlaylist (*service); };

    // Helper methods
    void PurgeUnselectedSongs (void);
    bool considerCreatingPlayer (void);
    bool acquireRandomTracks (void);

    // Callback methods and handlers
    void sourceReady (const Media::Source *);
    bool sourceRemovalCheck (const Media::Source *source);
    void sourceRemoved (const Media::Source *source);
    void sourceOffline (const Media::Source *);
    void sourceStatus (RESPONSE_CODE status, const char *detail);

    void playlistsChanged ();
    void mixChanged (bool automatic, const char *why);

public:
    AudioEngine (PianodService *svc, const AudioSettings &audio_options);
    ~AudioEngine (void);
    void shutdown (bool immediate);
    float periodic(void); /// Should be called intermittently by the main run-loop
    void sendStatus (PianodConnection &there);
    void updateStatus (PianodConnection &there);
    void registerWithInterpreter (PianodService *service);
    void usersChangedNotification();
    UserList getAutotuneUsers ();
    inline const AudioSettings &audioSettings () { return audio; };
};
