///
/// Pianod specializations of Football connections & services.
///	@file		connection.cpp - pianod
///	@author		Perette Barella
///	@date		2014-11-28
///	@copyright	Copyright 2014-2020 Devious Fish.  All rights reserved.
///

#include <config.h>

#include <cassert>

#include "football.h"
#include "footparser.h"
#include "fundamentals.h"
#include "response.h"
#include "connection.h"
#include "enum.h"
#include "musictypes.h"
#include "mediamanager.h"
#include "servicemanager.h"
#include "engine.h"
#include "filter.h"
#include "querylist.h"


bool PianodConnection::broadcast_user_actions = true;
time_t WaitEvent::nextTimeout = FAR_FUTURE;

PianodConnection::~PianodConnection () {
}


/// Parser for options for events/WAIT commands.
namespace WaitOptions {

    /** Event/Wait for options */
    enum class Option {
        Timeout = 1
    };

    /** WaitOptionsParser is the parser that fills in the WaitEvent structure. */
    class Parser : public Football::OptionParser<WaitEvent, Option> {
        virtual int handleOption (Option option, WaitEvent &dest) override;
    public:
        Parser () {
            static const Parser::ParseDefinition waitOptionStatements [] = {
                { Option::Timeout,         "timeout {#duration:0-999999999} ..." }, // how long to wait before failing
                { (Option) CMD_INVALID,    NULL }
            };
            addStatements (waitOptionStatements);
        };
    };
    

    /** AudioOptions is the parser that fills in the AudioSettings structure. */
    int Parser::handleOption (Option option, WaitEvent &dest) {
        switch (option) {
            case Option::Timeout:
                dest.timeout = time (nullptr) + atoi (argv ("duration"));
                return FB_PARSE_SUCCESS;
        }
        assert (0);
        return FB_PARSE_FAILURE;
    }

    /// An instance of the wait options parser.
    Parser parser;
}




/*
 *              Event handlers
 */

/** Handle football-layer errors that happen on pianod connections.
    This overrides the built-in method, formatting errors in accordance
    with pianod protocol. */
void PianodConnection::commandError (ssize_t reason, const char *where) {
    switch (reason) {
            // Parser Errors 
        case FB_PARSE_FAILURE:
            this << E_BUG;
            return;
        case FB_PARSE_INCOMPLETE:
            printf ("%03d Command incomplete after %s\n", E_BAD_COMMAND, where);
            return;
        case FB_PARSE_INVALID_KEYWORD:
            printf ("%03d Bad command %s\n", E_BAD_COMMAND, where);
            return;
        case FB_PARSE_NUMERIC:
            printf ("%03d Numeric value expected: %s\n", E_BAD_COMMAND, where);
            return;
        case FB_PARSE_RANGE:
            printf ("%03d Numeric value out of range: %s\n", E_BAD_COMMAND, where);
            return;
        case FB_PARSE_EXTRA_TERMS:
            printf ("%03d Run-on command at %s\n", E_BAD_COMMAND, where);
            return;
        case FB_ERROR_BADALLOC:
            if (where)
                this << Response (E_RESOURCE, where);
            else
                this << E_RESOURCE;
            return;
        case FB_ERROR_EXCEPTION:
            if (where)
                this << Response (E_NAK, where);
            else
                this << E_NAK;
            return;
        default:
            assert (reason > 0);
            this << (RESPONSE_CODE) (reason < 0 ? E_NAK : reason);
    }
}

/** Handle custom exceptions that may occur.  Pass other exceptions to Football.
     @param except An exception that happened while dispatching a command. */
void PianodConnection::commandException (std::exception_ptr except) {
    try {
        std::rethrow_exception (except);
    } catch (const Query::impossible &e) {
        this << Response (E_MEDIA_ACTION,
                          "Query complexity exceeds source ability");
    } catch (const CommandError &error) {
        this << error;
    } catch (...) {
        Football::Connection::commandException(except);
    }
}

void PianodConnection::permissionDenied() {
    this << E_UNAUTHORIZED;
}


void PianodConnection::newConnection () {
    // Greet the connection, allocate any resources
    this << S_OK << I_WELCOME;
    sendEffectivePrivileges();
    source (media_manager);
    service().audioEngine()->sendStatus (*this);
};

void PianodConnection::updateConnection () {
    service().audioEngine()->updateStatus (*this);
}

void PianodConnection::connectionClose () {
    // Free user context resources 
    if (user) {
        announce (A_SIGNED_OUT);
        user = nullptr;
        service().usersChangedNotification();
    }
}

/*
 *              Getters
 */


/** Get connected user's rank.
    @return User's rank, or visitor rank if not authenticated. */
Rank PianodConnection::effectiveRank (void) {
    return authenticated() ? user->getRank () : User::getVisitorRank ();
}

// Determine if a user or visitor has a rank or better. 
bool PianodConnection::haveRank (Rank minimum) {
    return (effectiveRank () >= minimum);
};

/** Determine if the user has a privilege.
    Visitors cannot be assigned privileges but they may be implied by rank.
    @param priv The privilege to check.
    @return True if the user either has the privilege or it is implied by rank. */
bool PianodConnection::havePrivilege (Privilege priv) {
    assert (Enum<Privilege>().isValid (priv));
    // Administrators inherently get certain privileges 
    if (haveRank (Rank::Administrator) &&
        (priv == Privilege::Service || priv == Privilege::Tuner)) {
        return true;
    }
    if (haveRank (Rank::Standard) && priv == Privilege::Queue)
        return true;
    return authenticated() ? user->havePrivilege (priv) : false;
};

/*
 *              Events
 */

/** Begin waiting for an event on a connection.
    @param type The type of event to wait for.
    @param detail A pointer representing a specific event instance to wait for. */
void PianodConnection::waitForEvent (WaitEvent::Type type, const void *detail) {
    assert (pending.event == WaitEvent::Type::None);
    assert (type != WaitEvent::Type::None);
    pending.event = type;
    pending.parameter = detail;
    acceptInput (false);
}

/** Interpret options and begin waiting for an event on a connection.
    @param type The type of event to wait for.
    @param detail A pointer representing a specific event instance to wait for.
    @return True on success, false if event options are invalid. */
bool PianodConnection::waitForEventWithOptions (WaitEvent::Type type, const void *detail) {
    assert (pending.event == WaitEvent::Type::None);
    assert (type != WaitEvent::Type::None);
    if (WaitOptions::parser.interpret (argvFrom("options"), pending, this) != FB_PARSE_SUCCESS)
        return false;
    pending.event = type;
    pending.parameter = detail;
    acceptInput (false);
    if (pending.timeout < WaitEvent::nextTimeout)
        WaitEvent::nextTimeout = pending.timeout;
    return true;
}

/** Process an event for a connection.
    If waiting for the event, the status is announced and input is resumed.
    If not waiting, or waiting for a different event, nothing happens.
    @param type The type of event occurring.
    @param detail A pointer representing a specific event instance.
    @param reply The status to report if the event applies to the connection. */
void PianodConnection::event (WaitEvent::Type type, const void *detail, RESPONSE_CODE reply) {
    assert (type != WaitEvent::Type::None);
    if (pending.event == type && pending.parameter == detail) {
        acceptInput (true);
        this << reply;
        if (pending.close_after_event) {
            Football::Connection::close();
        }
        pending = WaitEvent {};
    }
}

/** Check if a pending event has timed out.  If so, fire it with failure.
    Otherwise, check then pending event for a closer next timeout time. */
void PianodConnection::checkTimeouts () {
    if (pending.timeout) {
        time_t now = time (nullptr);
        if (pending.timeout <= now) {
            event (pending.event, pending.parameter, E_TIMEOUT);
        } else {
            if (pending.timeout < WaitEvent::nextTimeout) {
                WaitEvent::nextTimeout = pending.timeout;
            }
        }
    }
}


/** Close after events are handled, or now if not waiting on one. */
void PianodConnection::close_after_events () {
    if (pending.event == WaitEvent::Type::None) {
        close();
    } else {
        pending.close_after_event = true;
    }
}

/*
 *              Transmitters
 */


/** Announce to all connections that a user performed some action.
    If the broadcast feature is disabled, logs user action instead.
    @param code Enumeration identifying user action taken.
    @param parameter Details of action, usually target object's name. */
void PianodConnection::announce (RESPONSE_CODE code, const char *parameter) {
    assert (this);
    assert (code == V_YELL || code >= 1000);

    if (broadcast_user_actions || code == V_YELL) {
        sendcflog (LOG_USERACTION, &service(), "%03d %s %s%s%s\n",
                  code == V_YELL ? V_YELL : V_USERACTION, username().c_str(),
                  ResponseText (code), parameter ? ": " : "",
                  parameter ? parameter : "");
    } else {
        // Log it in case action logging is turned on 
        flog (LOG_WHERE (LOG_USERACTION), username(), " ",
              ResponseText (code), parameter ? ": " : "", parameter ? parameter : "");
        // If status indicates something happened,  send the generic version of the message. 
    }
}

/** Report the selected source to the connection. */
void PianodConnection::sendSelectedSource() {
    assert (_source);
    printf("%03d %s: %d %s %s\n",
           V_SELECTEDSOURCE, ResponseText (V_SELECTEDSOURCE),
           (int) _source->serialNumber(),
           _source->kind(), _source->name().c_str());
}


/** Transmit the user's effective privileges. */
void PianodConnection::sendEffectivePrivileges () {
    std::string privs = User::getRankName (effectiveRank());
    privs.reserve (int (Privilege::Count) * 20); // Preallocate space for efficiency.
    for (Privilege p : Enum <Privilege>()) {
        if (havePrivilege (p)) {
            privs += " ";
            privs += User::getPrivilegeName (p);
        }
    }
    this << Response (I_USER_PRIVILEGES, privs);
}


/*
 *              PianodService
 */
PianodService::~PianodService() {
    if (engine) delete engine;
}

/** When a service has been completely shut down, remove it from the service manager. */
void PianodService::serviceShutdown (void) {
    service_manager->removeRoom (this);
}

void PianodService::usersChangedNotification (void) {
    engine->usersChangedNotification();
}
