///
/// Debug/event logging interface.
/// @file       logging.h - pianod project
/// @author     Perette Barella
/// @date       2012-03-16
/// @copyright  Copyright 2012–2020 Devious Fish. All rights reserved.
///

#pragma once

#include <config.h>

#include <cstring>
#include <cstdarg>
#include <iostream>
#include <utility>

typedef enum log_types_t {
	LOG_ERROR = 0, // Nonmaskable 
	LOG_000 = 0x01, // Corresponding to response message groups 
	LOG_100 = 0x02,
	LOG_200 = 0x04,
	LOG_300 = 0x08,
	LOG_400 = 0x10,
	LOG_500 = 0x20,
	// 0x40 not used yet
	LOG_GENERAL = 0x80,
	LOG_EVENT = 0x100,
	LOG_WARNING = 0x200,
	// 0x800 not used yet
	LOG_USERACTION = 0x1000,
    LOG_PROTOCOL = 0x2000,
    LOG_ALLOCS = 0x4000,
    LOG_CACHES = 0x8000,
    LOG_TUNING = 0x10000,
    LOG_BIASING = 0x20000,
    LOG_PANDORA = 0x100000
} LOG_TYPE;

// Logging
extern LOG_TYPE flogging_level;
extern void set_logging (unsigned logtype);

static inline const char *trim_filename (const char *path) {
    const char *name = strrchr (path, '/');
    return name ? name + 1 : path;
}

// LOG_WHERE macro inserts file, line and function for C++ flog calls only.
// Unlike Football, where the FB_WHERE macro is required, these are optional.
#ifdef NDEBUG
#define LOG_WHERE(level) (level)
#define LOG_FUNCTION(level) (level), __func__, ": "
#else
#define LOG_WHERE(level) (level), trim_filename(__FILE__), ":", __LINE__, " ", __func__, " (", #level, "): "
#define LOG_FUNCTION(level) (level), trim_filename(__FILE__), ":", __LINE__, " ", __func__, " (", #level, "): "
#endif

/// Implement empty argument list for flogging.
inline void flog_details (void) {
    std::clog << '\n';
};

inline bool logging_enabled (LOG_TYPE level) {
    return (level == LOG_ERROR || (flogging_level & level));
}

/// Recurse while there are arguments for flogging.
template<typename OutputItem, typename... MoreItems>
void flog_details (OutputItem message, MoreItems&&... more) {
    std::clog << message;
    flog_details (std::forward <MoreItems> (more)...);
}

/// Log messages when their logging category is enabled.
/// Errors are always logged.  Message to to standard out.
template<typename... MoreItems>
void flog (LOG_TYPE level, MoreItems&&... more) {
    if (logging_enabled (level)) {
        char date [24];
        time_t now = time (NULL);
        strftime(date, sizeof (date),"%Y-%m-%d %H:%M:%S: ", localtime(&now));
        std::clog << date;
        flog_details (std::forward<MoreItems> (more)...);
    }
}

