///
/// Parser for audio options.
/// Parse audio options for creating or reconfiguring a room.
///	@file		audiooptionsparser.cpp - pianod2
///	@author		Perette Barella
///	@date		2015-11-23
///	@copyright	Copyright (c) 2015-2017 Devious Fish. All rights reserved.
///

#include <config.h>

#include "audiooptionsparser.h"
#include "audiooutput.h"

namespace AudioOptions {
    /** Validate audio settings.
        @param settings The audio settings to validate.
        @throw An appropriate command error if the settings are invalid. */
    void validate (const AudioSettings settings) {
        if (settings.preroll_time == 0.0f && settings.crossfade_time > 0.0f) {
            throw CommandError (E_INVALID, "Cannot crossfade without preroll.");
        }
        try {
            if (!Audio::Output::isValidOutput (settings)) {
                throw CommandError (E_NOTFOUND, settings.output_library);
            }
        } catch (const Audio::AudioException &e) {
            throw CommandError (E_INVALID, e.what());
        }
        if (settings.preroll_time > 0.0f) {
            try {
                if (!Audio::Output::outputCanCrossfade (settings)) {
                    throw CommandError (E_INVALID, "Cannot crossfade.");
                }
            } catch (const Audio::AudioException &e) {
                throw CommandError (E_INVALID, e.what());
            }
        }
    }


    static const Parser::ParseDefinition audioOptionStatements [] = {
        { Option::Library,         "library {library} ..." },      // Choose output library
        { Option::Driver,          "driver {driver} ..." },		// output setting
        { Option::Device,          "device {device} ..." },		// output setting
        { Option::Id,              "id {#id} ..." },				// output setting
        { Option::Options,         "options {options} ..." },				// output setting
        { Option::Server,          "server {server} ..." },		// output setting
        { Option::Volume,          "volume {#level:-100-100 ..."}, // room volume
        { Option::CrossfadeTime,   "crossfade duration {#seconds:0.0-15} ..." },  // How much volume adjustment when crossfading
        { Option::CrossfadeLevel,  "crossfade level {#level:0.0-50} ..." }, // How long to overlap songs
        { Option::PrerollTime,     "preroll duration {#seconds:0.0-30} ..." }, // Advance cueing time to allow buffering
        { (Option) CMD_INVALID,    NULL }
    };

    /** AudioOptions is the parser that fills in the AudioSettings structure. */
    int Parser::handleOption (Option option, AudioSettings &dest) {
        switch (option) {
            case Option::Driver:
                dest.output_driver = argv ("driver");
                return FB_PARSE_SUCCESS;
            case Option::Device:
                dest.output_device = argv ("device");
                return FB_PARSE_SUCCESS;
            case Option::Id:
                dest.output_id = argv ("id");
                return FB_PARSE_SUCCESS;
            case Option::Options:
                dest.output_options = argv ("options");
                return FB_PARSE_SUCCESS;
            case Option::Server:
                dest.output_server = argv ("server");
                return FB_PARSE_SUCCESS;
            case Option::Library:
                dest.output_library = argv ("library");
                return FB_PARSE_SUCCESS;
            case Option::Volume:
                dest.volume = strtod (argv ("level"), nullptr);
                return FB_PARSE_SUCCESS;
            case Option::CrossfadeTime:
                dest.crossfade_time = strtod (argv ("seconds"), nullptr);
                return FB_PARSE_SUCCESS;
            case Option::CrossfadeLevel:
                dest.crossfade_level = strtod (argv ("level"), nullptr);
                return FB_PARSE_SUCCESS;
            case Option::PrerollTime:
                dest.preroll_time = strtod (argv ("seconds"), nullptr);
                return FB_PARSE_SUCCESS;
        }
        assert (0);
        return FB_PARSE_FAILURE;
    }

    Parser::Parser () {
        addStatements (audioOptionStatements);
    };

    int Parser::interpret(const std::vector<std::string> &options, AudioSettings &result, PianodConnection &conn) {
        if (options.size()) {
            int status = Football::OptionParser<AudioSettings, Option>::interpret (options, result, &conn);
            if (status != FB_PARSE_SUCCESS)
                return status;
            validate (result);
        }
        return FB_PARSE_SUCCESS;
    };

    Parser parser;

}
