<?php

/**
* @package   s9e\TextFormatter
* @copyright Copyright (c) 2010-2021 The s9e authors
* @license   http://www.opensource.org/licenses/mit-license.php The MIT License
*/
namespace s9e\TextFormatter\Plugins\MediaEmbed\Configurator\Collections;

class CachedDefinitionCollection extends SiteDefinitionCollection
{
	/** {@inheritdoc} */
	protected $items = [
		'abcnews'=>['attributes'=>[],'example'=>'https://abcnews.go.com/WNN/video/dog-goes-wild-when-owner-leaves-22936610','extract'=>['!abcnews\\.go\\.com/(?:video/embed\\?id=|[^/]+/video/[^/]+-)(?\'id\'\\d+)!'],'homepage'=>'https://abcnews.go.com/','host'=>['abcnews.go.com'],'iframe'=>['src'=>'//abcnews.go.com/video/embed?id={@id}'],'name'=>'ABC News','scrape'=>[],'tags'=>['news']],
		'acast'=>['attributes'=>[],'example'=>['https://play.acast.com/s/the-taskmaster-podcast/ep31.leemack-s11ep.8','https://shows.acast.com/chaotic-normal/episodes/the-rising-tide'],'extract'=>['@play\\.acast\\.com/s/(?\'show_id\'[-.\\w]+)/(?\'episode_id\'[-.\\w]+)(?:\\?seek=(?\'t\'\\d+))?@','@shows\\.acast\\.com/(?\'show_id\'[-.\\w]+)/(?:episodes/)?(?\'episode_id\'[-.\\w]+)(?:\\?seek=(?\'t\'\\d+))?@'],'homepage'=>'https://play.acast.com/','host'=>['acast.com'],'iframe'=>['height'=>188,'max-width'=>900,'src'=>'https://embed.acast.com/{@show_id}/{@episode_id}?seek={@t}','width'=>'100%'],'name'=>'Acast','oembed'=>['endpoint'=>'https://oembed.acast.com/v1/embed-player','scheme'=>'https://play.acast.com/s/{@show_id}/{@episode_id}'],'scrape'=>[['extract'=>['@"showId":"(?\'show_id\'[-0-9a-f]+)@','@"id":"(?\'episode_id\'[-0-9a-f]+)@'],'match'=>['@play\\.acast\\.com/s/[-.\\w]+/.@','@shows\\.acast\\.com/[-.\\w]+/.@'],'url'=>'https://feeder.acast.com/api/v1/shows/{@show_id}/episodes/{@episode_id}']],'tags'=>['podcasts']],
		'amazon'=>['attributes'=>['id'=>['required'=>true]],'example'=>['https://www.amazon.ca/gp/product/B00GQT1LNO/','https://www.amazon.co.jp/gp/product/B003AKZ6I8/','https://www.amazon.co.uk/dp/B00EO4NN5C/','https://www.amazon.com/dp/B002MUC0ZY','https://www.amazon.com/The-BeerBelly-200-001-80-Ounce-Belly/dp/B001RB2CXY/','https://www.amazon.com/gp/product/B00ST0KGCU/','https://www.amazon.de/Netgear-WN3100RP-100PES-Repeater-integrierte-Steckdose/dp/B00ET2LTE6/','https://www.amazon.es/Microsoft-Sculpt-Ergonomic-Desktop-L5V-00011/dp/B00FO10ZK0/','https://www.amazon.fr/Vans-Authentic-Baskets-mixte-adulte/dp/B005NIKPAY/','https://www.amazon.in/Vans-Unisex-Classic-Loafers-Moccasins/dp/B01I3LWMVC/','https://www.amazon.it/Super-Maxi-Pot-de-Nutella/dp/B0090GJ8VM/','https://www.amazon.com/dp/B0018CDWLS/'],'extract'=>['#/(?:dp|gp/product)/(?\'id\'[A-Z0-9]+)#','#amazon\\.(?:co\\.)?(?\'tld\'ca|de|es|fr|in|it|jp|uk)#'],'homepage'=>'https://affiliate-program.amazon.com/','host'=>['amazon.ca','amazon.co.uk','amazon.co.jp','amazon.com','amazon.de','amazon.es','amazon.fr','amazon.in','amazon.it'],'iframe'=>['height'=>240,'src'=>'//ws-<xsl:choose><xsl:when test="@tld=\'jp\'">fe</xsl:when><xsl:when test="@tld and contains(\'desfrinituk\',@tld)">eu</xsl:when><xsl:otherwise>na</xsl:otherwise></xsl:choose>.assoc-amazon.com/widgets/cm?l=as1&amp;f=ifr&amp;o=<xsl:choose><xsl:when test="@tld=\'ca\'">15&amp;t=<xsl:value-of select="$AMAZON_ASSOCIATE_TAG_CA"/></xsl:when><xsl:when test="@tld=\'de\'">3&amp;t=<xsl:value-of select="$AMAZON_ASSOCIATE_TAG_DE"/></xsl:when><xsl:when test="@tld=\'es\'">30&amp;t=<xsl:value-of select="$AMAZON_ASSOCIATE_TAG_ES"/></xsl:when><xsl:when test="@tld=\'fr\'">8&amp;t=<xsl:value-of select="$AMAZON_ASSOCIATE_TAG_FR"/></xsl:when><xsl:when test="@tld=\'in\'">31&amp;t=<xsl:value-of select="$AMAZON_ASSOCIATE_TAG_IN"/></xsl:when><xsl:when test="@tld=\'it\'">29&amp;t=<xsl:value-of select="$AMAZON_ASSOCIATE_TAG_IT"/></xsl:when><xsl:when test="@tld=\'jp\'">9&amp;t=<xsl:value-of select="$AMAZON_ASSOCIATE_TAG_JP"/></xsl:when><xsl:when test="@tld=\'uk\'">2&amp;t=<xsl:value-of select="$AMAZON_ASSOCIATE_TAG_UK"/></xsl:when><xsl:otherwise>1&amp;t=<xsl:value-of select="$AMAZON_ASSOCIATE_TAG"/></xsl:otherwise></xsl:choose>&amp;asins=<xsl:value-of select="@id"/>','style'=>['background'=>'url(https://www.amazon.com/favicon.ico) center no-repeat'],'width'=>120],'name'=>'Amazon Product','parameters'=>['AMAZON_ASSOCIATE_TAG'=>['title'=>'Amazon Associate tag (.com)'],'AMAZON_ASSOCIATE_TAG_CA'=>['title'=>'Amazon Associate tag (.ca)'],'AMAZON_ASSOCIATE_TAG_DE'=>['title'=>'Amazon Associate tag (.de)'],'AMAZON_ASSOCIATE_TAG_ES'=>['title'=>'Amazon Associate tag (.es)'],'AMAZON_ASSOCIATE_TAG_FR'=>['title'=>'Amazon Associate tag (.fr)'],'AMAZON_ASSOCIATE_TAG_IN'=>['title'=>'Amazon Associate tag (.in)'],'AMAZON_ASSOCIATE_TAG_IT'=>['title'=>'Amazon Associate tag (.it)'],'AMAZON_ASSOCIATE_TAG_JP'=>['title'=>'Amazon Associate tag (.co.jp)'],'AMAZON_ASSOCIATE_TAG_UK'=>['title'=>'Amazon Associate tag (.co.uk)']],'scrape'=>[]],
		'anchor'=>['attributes'=>[],'example'=>'https://anchor.fm/aconversationwith/episodes/Answering-Voicemails-A-Conversation-With-You--2-e6e7lg','extract'=>['@anchor.fm/(?:[-\\w]+/)*?episodes/(?:[-\\w]+-)(?\'id\'\\w+)(?![-\\w])@'],'homepage'=>'https://anchor.fm/','host'=>['anchor.fm'],'iframe'=>['height'=>102,'max-width'=>768,'src'=>'//anchor.fm/x/embed/episodes/x-{@id}','width'=>'100%'],'name'=>'Anchor','scrape'=>[],'tags'=>['podcasts']],
		'applepodcasts'=>['attributes'=>[],'choose'=>['otherwise'=>['iframe'=>['allow'=>'autoplay *;encrypted-media *','height'=>450,'max-width'=>900,'src'=>'https://embed.podcasts.apple.com/{@country}/podcast/episode/id{@podcast_id}?theme={$MEDIAEMBED_THEME}','width'=>'100%']],'when'=>['iframe'=>['allow'=>'autoplay *;encrypted-media *','height'=>175,'max-width'=>900,'src'=>'https://embed.podcasts.apple.com/{@country}/podcast/episode/id{@podcast_id}?theme={$MEDIAEMBED_THEME}&i={@episode_id}','width'=>'100%'],'test'=>'@episode_id']],'example'=>['https://podcasts.apple.com/us/podcast/the-office-deep-dive-with-brian-baumgartner/id1550331348','https://podcasts.apple.com/us/podcast/the-office-deep-dive-with-brian-baumgartner/id1550331348?i=1000514199106'],'extract'=>['@podcasts\\.apple\\.com/(?\'country\'\\w+)/podcast/[-\\w]*/id(?\'podcast_id\'\\d+)(?:\\?i=(?\'episode_id\'\\d+))?@'],'homepage'=>'https://podcasts.apple.com/','host'=>['podcasts.apple.com'],'name'=>'Apple Podcasts','scrape'=>[],'tags'=>['podcasts']],
		'audioboom'=>['attributes'=>[],'example'=>'https://audioboom.com/posts/2493448-robert-patrick','extract'=>['!audioboo(?:\\.f|m\\.co)m/(?:boo|post)s/(?\'id\'\\d+)!'],'homepage'=>'https://audioboom.com/','host'=>['audioboo.fm','audioboom.com'],'iframe'=>['height'=>150,'max-width'=>700,'src'=>'//audioboom.com/posts/{@id}/embed/v3','width'=>'100%'],'name'=>'Audioboom','scrape'=>[],'tags'=>['podcasts']],
		'audiomack'=>['attributes'=>[],'choose'=>['otherwise'=>['iframe'=>['height'=>252,'max-width'=>900,'src'=>'https://www.audiomack.com/embed/song/<xsl:choose><xsl:when test="@id"><xsl:value-of select="@id"/></xsl:when><xsl:otherwise><xsl:value-of select="@artist"/>/<xsl:value-of select="@title"/></xsl:otherwise></xsl:choose>','width'=>'100%']],'when'=>['iframe'=>['height'=>400,'max-width'=>900,'src'=>'https://www.audiomack.com/embed/album/<xsl:choose><xsl:when test="@id"><xsl:value-of select="@id"/></xsl:when><xsl:otherwise><xsl:value-of select="@artist"/>/<xsl:value-of select="@title"/></xsl:otherwise></xsl:choose>','width'=>'100%'],'test'=>'@mode=\'album\'']],'example'=>['https://www.audiomack.com/your-music-fix/song/jammin-kungs-remix-1','https://www.audiomack.com/chance-the-rapper/album/acid-rap'],'extract'=>['!audiomack\\.com/(?\'mode\'album|song)/(?\'artist\'[-\\w]+)/(?\'title\'[-\\w]+)!','!audiomack\\.com/(?\'artist\'[-\\w]+)/(?\'mode\'album|song)/(?\'title\'[-\\w]+)!'],'homepage'=>'https://www.audiomack.com/','host'=>['audiomack.com'],'name'=>'Audiomack','scrape'=>[],'tags'=>['music']],
		'audius'=>['attributes'=>[],'example'=>['https://audius.co/feedbachmusic/album/lights-ii-62147','https://audius.co/audiusplaylists/playlist/culture-over-clout-56948','https://audius.co/manilakilla/chet-porter-the-longest-day-ever-manila-killa-edit'],'extract'=>[],'homepage'=>'https://audius.co/','host'=>['audius.co'],'iframe'=>['height'=>'<xsl:choose><xsl:when test="@track_id">12</xsl:when><xsl:otherwise>48</xsl:otherwise></xsl:choose>0','max-width'=>'<xsl:choose><xsl:when test="@track_id">9</xsl:when><xsl:otherwise>4</xsl:otherwise></xsl:choose>00','src'=>'https://audius.co/embed/<xsl:choose><xsl:when test="@track_id">track/<xsl:value-of select="@track_id"/>?flavor=compact</xsl:when><xsl:when test="@album_id">album/<xsl:value-of select="@album_id"/>?flavor=card</xsl:when><xsl:otherwise>playlist/<xsl:value-of select="@playlist_id"/>?flavor=card</xsl:otherwise></xsl:choose>','width'=>'100%'],'name'=>'Audius','scrape'=>[['extract'=>['!"id"\\s*:\\s*"(?\'track_id\'\\w+)"!'],'match'=>['@audius\\.co/(?!v1/)(?\'user\'[-.\\w]+)/(?!album/|playlist/)(?\'slug\'[%\\-.\\w]+)@'],'url'=>'https://discoveryprovider.audius.co/v1/resolve?app_name=s9e-textformatter&url=/{@user}/{@slug}'],['extract'=>['!"id"\\s*:\\s*"(?\'album_id\'\\w+)"!'],'match'=>['@audius\\.co/(?!v1/)(?\'user\'[-.\\w]+)/album/(?\'slug\'[%\\-.\\w]+)@'],'url'=>'https://discoveryprovider.audius.co/v1/resolve?app_name=s9e-textformatter&url=/{@user}/album/{@slug}'],['extract'=>['!"id"\\s*:\\s*"(?\'playlist_id\'\\w+)"!'],'match'=>['@audius\\.co/(?!v1/)(?\'user\'[-.\\w]+)/playlist/(?\'slug\'[%\\-.\\w]+)@'],'url'=>'https://discoveryprovider.audius.co/v1/resolve?app_name=s9e-textformatter&url=/{@user}/playlist/{@slug}']],'tags'=>['music']],
		'bandcamp'=>['attributes'=>[],'example'=>['https://proleter.bandcamp.com/album/curses-from-past-times-ep','https://proleter.bandcamp.com/track/downtown-irony','https://therunons.bandcamp.com/track/still-feel'],'extract'=>[],'homepage'=>'https://bandcamp.com/','host'=>['bandcamp.com'],'iframe'=>['height'=>400,'src'=>'//bandcamp.com/EmbeddedPlayer/size=large/minimal=true/<xsl:choose><xsl:when test="@album_id">album=<xsl:value-of select="@album_id"/><xsl:if test="@track_num">/t=<xsl:value-of select="@track_num"/></xsl:if></xsl:when><xsl:otherwise>track=<xsl:value-of select="@track_id"/></xsl:otherwise></xsl:choose><xsl:if test="$MEDIAEMBED_THEME=\'dark\'">/bgcol=333333/linkcol=0f91ff</xsl:if>','width'=>400],'name'=>'Bandcamp','scrape'=>[['extract'=>['!/album=(?\'album_id\'\\d+)!'],'match'=>['!bandcamp\\.com/album/.!']],['extract'=>['!"album_id":(?\'album_id\'\\d+)!','!"track_num":(?\'track_num\'\\d+)!','!/track=(?\'track_id\'\\d+)!'],'match'=>['!bandcamp\\.com/track/.!']]],'tags'=>['music']],
		'bbcnews'=>['attributes'=>[],'example'=>['https://www.bbc.com/news/video_and_audio/must_see/42847060/calls-to-clean-off-banksy-mural-in-hull','https://www.bbc.co.uk/news/av/world-africa-48141248/fifteen-year-old-nigerian-builds-small-scale-construction-machines','https://www.bbc.co.uk/news/av/embed/p0783sg7/48125671'],'extract'=>['@bbc\\.co(?:m|\\.uk)/news/(?:av|video_and_audio)/(?:\\w+-)+(?\'id\'\\d+)@','@bbc\\.co(?:m|\\.uk)/news/(?:av|video_and_audio)/embed/(?\'id\'\\w+/\\d+)@','@bbc\\.co(?:m|\\.uk)/news/(?:av|video_and_audio)/\\w+/(?\'id\'\\d+)@','@bbc\\.co(?:m|\\.uk)/news/av-embeds/(?\'id\'\\d+)@'],'homepage'=>'https://www.bbc.com/news/video_and_audio/headlines/','host'=>['bbc.co.uk','bbc.com'],'iframe'=>['src'=>'//www.bbc.com/news/av-embeds/<xsl:choose><xsl:when test="starts-with(@playlist,\'/news/\')"><xsl:choose><xsl:when test="contains(@playlist,\'-\')"><xsl:value-of select="substring-after(substring-after(translate(@playlist,\'A\',\'#\'),\'news/\'),\'-\')"/></xsl:when><xsl:otherwise><xsl:value-of select="substring-after(translate(@playlist,\'A\',\'/\'),\'/news/\')"/></xsl:otherwise></xsl:choose></xsl:when><xsl:when test="contains(@id,\'/\')"><xsl:value-of select="substring-after(@id,\'/\')"/></xsl:when><xsl:otherwise><xsl:value-of select="@id"/></xsl:otherwise></xsl:choose>'],'name'=>'BBC News','scrape'=>[],'tags'=>['news']],
		'bitchute'=>['attributes'=>[],'example'=>'https://www.bitchute.com/video/y9AejeZ2vD0/','extract'=>['@bitchute\\.com/(?:embed|video)/(?\'id\'[-\\w]+)@'],'homepage'=>'https://www.bitchute.com/','host'=>['bitchute.com'],'iframe'=>['src'=>'https://www.bitchute.com/embed/{@id}/'],'name'=>'BitChute','oembed'=>['endpoint'=>'https://www.bitchute.com/oembed/','scheme'=>'https://www.bitchute.com/video/{@id}/'],'scrape'=>[],'tags'=>['videos']],
		'bleacherreport'=>['attributes'=>[],'example'=>'https://bleacherreport.com/articles/2687569-how-bell-brown-have-shattered-records-on-steelers-historic-playoff-run','extract'=>[],'homepage'=>'https://bleacherreport.com/videos','host'=>['bleacherreport.com'],'iframe'=>['height'=>320,'src'=>'//bleacherreport.com/video_embed?id={@id}&library=video-cms','width'=>320],'name'=>'Bleacher Report videos','scrape'=>[['extract'=>['!id="video-(?\'id\'[-\\w]+)!','!video_embed\\?id=(?\'id\'[-\\w]+)!'],'match'=>['!/articles/.!']]],'tags'=>['sports']],
		'brightcove'=>['attributes'=>['bckey'=>['required'=>false]],'example'=>['https://players.brightcove.net/219646971/default_default/index.html?videoId=4815779906001','https://link.brightcove.com/services/player/bcpid4773906090001?bckey=AQ~~,AAAAAA0Xi_s~,r1xMuE8k5Nyz5IyYc0Hzhl5eZ5cEEvlm&bctid=4815779906001'],'extract'=>['@link\\.brightcove\\.com/services/player/bcpid(?\'bcpid\'\\d+).*?bckey=(?\'bckey\'[-,~\\w]+).*?bctid=(?\'bctid\'\\d+)@','@players\\.brightcove\\.net/(?\'bcpid\'\\d+)/.*?videoId=(?\'bctid\'\\d+)@'],'homepage'=>'https://www.brightcove.com/','host'=>['link.brightcove.com','players.brightcove.net'],'iframe'=>['src'=>'https://<xsl:choose><xsl:when test="@bckey">link.brightcove.com/services/player/bcpid<xsl:value-of select="@bcpid"/>?bckey=<xsl:value-of select="@bckey"/>&amp;bctid=<xsl:value-of select="@bctid"/>&amp;secureConnections=true&amp;secureHTMLConnections=true&amp;autoStart=false&amp;height=360&amp;width=640</xsl:when><xsl:otherwise>players.brightcove.net/<xsl:value-of select="@bcpid"/>/default_default/index.html?videoId=<xsl:value-of select="@bctid"/></xsl:otherwise></xsl:choose>'],'name'=>'Brightcove','scrape'=>[],'tags'=>['videos']],
		'castos'=>['attributes'=>[],'example'=>'https://on-the-cuff.castos.com/podcasts/10156/episodes/frequently-asked-questions-on-mens-style','extract'=>['@(?\'host\'[-\\w]+)\\.castos\\.com/player/(?\'id\'\\d+)@'],'homepage'=>'https://castos.com/','host'=>['castos.com'],'iframe'=>['height'=>150,'max-width'=>900,'src'=>'https://player.castos.com/player/{@id}','width'=>'100%'],'name'=>'Castos','scrape'=>[['extract'=>['@(?\'host\'[-\\w]+)\\.castos\\.com/player/(?\'id\'\\d+)@'],'match'=>['@castos\\.com/(?:podcasts/[^/]*+/)?episodes/.@']]],'tags'=>['podcasts']],
		'cbsnews'=>['attributes'=>[],'choose'=>['otherwise'=>['flash'=>['flashvars'=>'si=254&contentValue={@id}','padding-height'=>40,'src'=>'//i.i.cbsi.com/cnwk.1d/av/video/cbsnews/atlantis2/cbsnews_player_embed.swf']],'when'=>[['iframe'=>['src'=>'https://www.cbsnews.com/embed/videos/{@id}/'],'test'=>'contains(@id, \'-\')'],['flash'=>['flashvars'=>'pType=embed&si=254&pid={@pid}','padding-height'=>38,'src'=>'//www.cbsnews.com/common/video/cbsnews_player.swf'],'test'=>'@pid']]],'example'=>'https://www.cbsnews.com/video/is-carbonated-water-a-healthy-option/','extract'=>['#cbsnews\\.com/videos?/(?!watch/)(?\'id\'[-\\w]+)#','#cbsnews\\.com/video/watch/\\?id=(?\'id\'\\d+)#'],'homepage'=>'https://www.cbsnews.com/video/','host'=>['cbsnews.com'],'name'=>'CBS News Video','scrape'=>[],'tags'=>['news']],
		'clyp'=>['attributes'=>[],'example'=>'https://clyp.it/0hdvyehn','extract'=>['@clyp\\.it/(?!user/)(?\'id\'\\w+)@'],'homepage'=>'https://clyp.it/','host'=>['clyp.it'],'iframe'=>['height'=>265,'max-width'=>900,'src'=>'https://clyp.it/{@id}/widget','width'=>'100%'],'name'=>'Clyp','scrape'=>[],'tags'=>['music']],
		'cnbc'=>['attributes'=>[],'example'=>['https://video.cnbc.com/gallery/?video=3000269279','https://www.cnbc.com/video/2019/06/05/carplay-android-auto-alexa-big-tech-gets-into-our-cars.html'],'extract'=>['!cnbc\\.com/gallery/\\?video=(?\'id\'\\d+)!'],'homepage'=>'https://www.cnbc.com/video/','host'=>['cnbc.com'],'iframe'=>['src'=>'https://player.cnbc.com/p/gZWlPC/cnbc_global?playertype=synd&byGuid={@id}&size=640_360'],'name'=>'CNBC','scrape'=>[['extract'=>['!byGuid=(?\'id\'\\d+)!'],'match'=>['!cnbc\\.com/video/20\\d\\d/\\d\\d/\\d\\d/\\w!']]],'tags'=>['news']],
		'cnn'=>['attributes'=>[],'example'=>['https://edition.cnn.com/videos/tv/2015/06/09/airplane-yoga-rachel-crane-ts-orig.cnn','https://us.cnn.com/video/data/2.0/video/bestoftv/2013/10/23/vo-nr-prince-george-christening-arrival.cnn.html'],'extract'=>['!cnn.com/videos/(?\'id\'.*\\.cnn)!','!cnn\\.com/video/data/2\\.0/video/(?\'id\'.*\\.cnn)!'],'homepage'=>'https://edition.cnn.com/video/','host'=>['cnn.com'],'iframe'=>['src'=>'//fave.api.cnn.io/v1/fav/?video={@id}&customer=cnn&edition=international&env=prod'],'name'=>'CNN','scrape'=>[],'tags'=>['news']],
		'cnnmoney'=>['attributes'=>[],'example'=>'https://money.cnn.com/video/technology/2014/05/20/t-twitch-vp-on-future.cnnmoney/','extract'=>['!money\\.cnn\\.com/video/(?\'id\'.*\\.cnnmoney)!'],'homepage'=>'https://money.cnn.com/video/','host'=>['money.cnn.com'],'iframe'=>['height'=>360,'src'=>'//money.cnn.com/.element/ssi/video/7.0/players/embed.player.html?videoid=video/{@id}&width=560&height=360','width'=>560],'name'=>'CNNMoney','scrape'=>[],'tags'=>['finance','news']],
		'codepen'=>['attributes'=>[],'example'=>'https://codepen.io/ainalem/pen/wvKOEMV','extract'=>['!codepen\\.io/(?\'user\'[-\\w]+)/(?:details|embed|full|live|pen)/(?\'id\'\\w+)!'],'homepage'=>'https://codepen.io/','host'=>['codepen.io'],'iframe'=>['height'=>400,'src'=>'https://codepen.io/{@user}/embed/{@id}?height=400&default-tab=html,result','width'=>'100%'],'name'=>'CodePen','scrape'=>[],'tags'=>['programming']],
		'comedycentral'=>['attributes'=>[],'example'=>['https://www.cc.com/video-clips/uu5qz4/key-and-peele-dueling-hats','https://www.comedycentral.com/video-clips/uu5qz4/key-and-peele-dueling-hats','https://tosh.cc.com/video-clips/aet4lh/rc-car-crash'],'extract'=>[],'homepage'=>'https://www.cc.com/','host'=>['cc.com','comedycentral.com'],'iframe'=>['src'=>'//media.mtvnservices.com/embed/{@id}'],'name'=>'Comedy Central','scrape'=>[['extract'=>['!(?\'id\'mgid:arc:(?:episode|video):[.\\w]+:[-\\w]+)!'],'match'=>['!c(?:c|omedycentral)\\.com/(?:full-episode|video-clip)s/!']]],'tags'=>['entertainment']],
		'coub'=>['attributes'=>[],'example'=>'https://coub.com/view/6veusoty','extract'=>['!coub\\.com/view/(?\'id\'\\w+)!'],'homepage'=>'https://coub.com/','host'=>['coub.com'],'iframe'=>['src'=>'//coub.com/embed/{@id}'],'name'=>'Coub','scrape'=>[],'tags'=>['videos']],
		'dailymotion'=>['amp'=>['custom-element'=>'amp-dailymotion','src'=>'https://cdn.ampproject.org/v0/amp-dailymotion-0.1.js','template'=>'<amp-dailymotion layout="responsive" width="640" height="360" data-start="{@t}" data-videoid="{@id}"/>'],'attributes'=>[],'example'=>['https://dai.ly/x5e9eog','https://www.dailymotion.com/video/x5e9eog'],'extract'=>['!dai\\.ly/(?\'id\'[a-z0-9]+)!i','!dailymotion\\.com/(?:live/|swf/|user/[^#]+#video=|(?:related/\\d+/)?video/)(?\'id\'[a-z0-9]+)!i','!start=(?\'t\'\\d+)!'],'homepage'=>'https://www.dailymotion.com/','host'=>['dai.ly','dailymotion.com'],'iframe'=>['src'=>'//www.dailymotion.com/embed/video/<xsl:value-of select="@id"/><xsl:if test="@t">?start=<xsl:value-of select="@t"/></xsl:if>'],'name'=>'Dailymotion','oembed'=>['endpoint'=>'https://www.dailymotion.com/services/oembed','scheme'=>'https://www.dailymotion.com/video/{@id}'],'scrape'=>[],'source'=>'http://www.dailymotion.com/doc/api/player.html','tags'=>['videos']],
		'democracynow'=>['attributes'=>[],'example'=>['https://www.democracynow.org/2014/7/2/dn_at_almedalen_week_at_swedens','https://www.democracynow.org/blog/2015/3/13/part_2_bruce_schneier_on_the','https://www.democracynow.org/shows/2006/2/20','https://www.democracynow.org/2015/5/21/headlines','https://m.democracynow.org/stories/15236'],'extract'=>['!democracynow.org/(?:embed/)?(?\'id\'(?:\\w+/)?\\d+/\\d+/\\d+(?:/\\w+)?)!'],'homepage'=>'https://www.democracynow.org/','host'=>['democracynow.org'],'iframe'=>['src'=>'//www.democracynow.org/embed/<xsl:choose><xsl:when test="contains(@id,\'/headlines\')">headlines/<xsl:value-of select="substring-before(@id,\'/headlines\')"/></xsl:when><xsl:when test="starts-with(@id,\'2\')">story/<xsl:value-of select="@id"/></xsl:when><xsl:when test="starts-with(@id,\'shows/\')">show/<xsl:value-of select="substring-after(@id,\'/\')"/></xsl:when><xsl:otherwise><xsl:value-of select="@id"/></xsl:otherwise></xsl:choose>'],'name'=>'Democracy Now!','scrape'=>[['extract'=>['!democracynow\\.org/(?\'id\'(?:\\w+/)?\\d+/\\d+/\\d+(?:/\\w+)?)\' rel=\'canonical!'],'match'=>['!m\\.democracynow\\.org/stories/\\d!']]]],
		'dumpert'=>['attributes'=>[],'example'=>'https://www.dumpert.nl/item/6622577_4652b140','extract'=>['!dumpert\\.nl/(?:item|mediabase)/(?\'id\'\\d+[/_]\\w+)!'],'homepage'=>'https://www.dumpert.nl/','host'=>['dumpert.nl'],'iframe'=>['src'=>'//www.dumpert.nl/embed/{translate(@id,\'/\',\'_\')}'],'name'=>'dumpert','scrape'=>[],'tags'=>['.nl','entertainment']],
		'eighttracks'=>['attributes'=>[],'example'=>['https://8tracks.com/lovinq/headphones-in-world-out','https://8tracks.com/lovinq/4982023'],'extract'=>['!8tracks\\.com/[-\\w]+/(?\'id\'\\d+)(?=#|$)!'],'homepage'=>'https://8tracks.com/','host'=>['8tracks.com'],'iframe'=>['height'=>400,'src'=>'//8tracks.com/mixes/{@id}/player_v3_universal','width'=>400],'name'=>'8tracks','scrape'=>[['extract'=>['!eighttracks://mix/(?\'id\'\\d+)!'],'match'=>['!8tracks\\.com/[-\\w]+/\\D!']]],'tags'=>['music']],
		'espn'=>['attributes'=>[],'example'=>['https://www.espn.com/video/clip?id=17474659','https://www.espn.com/espnw/video/13887284/kyrgios-angry-code-violation-almost-hitting-ref','https://broadband.espn.go.com/video/clip?id=17481969'],'extract'=>['#video/(?:clip(?:\\?id=|/_/id/))?(?\'id\'\\d+)#'],'homepage'=>'https://www.espn.com/','host'=>['espn.com','espn.go.com'],'iframe'=>['src'=>'//www.espn.com/core/video/iframe?id={@id}'],'name'=>'ESPN','scrape'=>[],'tags'=>['sports']],
		'facebook'=>['amp'=>['custom-element'=>'amp-facebook','src'=>'https://cdn.ampproject.org/v0/amp-facebook-0.1.js','template'=>'<amp-facebook layout="responsive" width="640" height="360"><xsl:if test="starts-with(@type,\'v\')"><xsl:attribute name="data-embed-as">video</xsl:attribute></xsl:if><xsl:attribute name="data-href">https://www.facebook.com/<xsl:choose><xsl:when test="@user"><xsl:value-of select="@user"/></xsl:when><xsl:otherwise>user</xsl:otherwise></xsl:choose>/<xsl:choose><xsl:when test="starts-with(@type,\'v\')">video</xsl:when><xsl:otherwise>post</xsl:otherwise></xsl:choose>s/<xsl:value-of select="@id"/></xsl:attribute></amp-facebook>'],'attributes'=>[],'example'=>['https://www.facebook.com/FacebookforDevelopers/posts/696408974555431','https://www.facebook.com/watch/?v=224353158889229','https://fb.watch/3zYsXdnxjf/'],'extract'=>['@/(?!(?:apps|developers|graph)\\.)[-\\w.]*facebook\\.com/(?:[/\\w]+/permalink|(?!marketplace/|pages/|groups/).*?)(?:/|fbid=|\\?v=)(?\'id\'\\d+)(?=$|[/?&#])@','@facebook\\.com/(?\'user\'[.\\w]+)/(?=(?:post|video)s?/)(?\'type\'[pv])@','@facebook\\.com/video/(?=post|video)(?\'type\'[pv])@','@facebook\\.com/watch/\\?(?\'type\'[pv])=@','@facebook.com/groups/[^/]*/(?\'type\'p)osts/(?\'id\'\\d+)@'],'homepage'=>'https://www.facebook.com/','host'=>['facebook.com','fb.watch'],'iframe'=>['data-s9e-livepreview-ignore-attrs'=>'style','onload'=>'var c=new MessageChannel;c.port1.onmessage=function(e){{style.height=e.data+\'px\'}};contentWindow.postMessage(\'s9e:init\',\'https://s9e.github.io\',[c.port2])','src'=>'https://s9e.github.io/iframe/2/facebook.min.html#{@type}{@id}'],'name'=>'Facebook','scrape'=>[['extract'=>['@facebook\\.com/watch/\\?(?\'type\'v)=(?\'id\'\\d+)@','@facebook\\.com/(?\'user\'[.\\w]+)/(?\'type\'v)ideos/(?\'id\'\\d+)@'],'header'=>'User-agent: PHP (not Mozilla)','match'=>['@fb\\.watch/.@']]],'tags'=>['social']],
		'falstad'=>['attributes'=>[],'example'=>'https://www.falstad.com/circuit/circuitjs.html?cct=$+1+0.000005+10.20027730826997+50+5+43+v+176+272+256+144+0+0+40+5+0+0+0.5+w+176+272+336+272+0+r+336+272+256+144+0+1000','extract'=>['!falstad\\.com/circuit/circuitjs\\.html\\?c(?:ct=(?\'cct\'[^&]+)|tz=(?\'ctz\'[-+=\\w]+))!'],'homepage'=>'https://www.falstad.com/circuit/circuitjs.html','host'=>['falstad.com'],'iframe'=>['height'=>500,'src'=>'https://www.falstad.com/circuit/circuitjs.html?running=false&amp;c<xsl:choose><xsl:when test="@cct">ct=<xsl:value-of select="@cct"/></xsl:when><xsl:otherwise>tz=<xsl:value-of select="@ctz"/></xsl:otherwise></xsl:choose>','style'=>['max-height'=>'90vh'],'width'=>'100%'],'name'=>'Falstad Circuit Simulator','scrape'=>[]],
		'flickr'=>['attributes'=>[],'example'=>['https://www.flickr.com/photos/8757881@N04/2971804544/lightbox/','https://flic.kr/8757881@N04/2971804544','https://flic.kr/p/5wBgXo'],'extract'=>['@flickr\\.com/photos/[^/]+/(?\'id\'\\d+)@','@flic\\.kr/(?!p/)[^/]+/(?\'id\'\\d+)@'],'homepage'=>'https://www.flickr.com/','host'=>['flic.kr','flickr.com'],'iframe'=>['height'=>500,'src'=>'https://www.flickr.com/photos/_/{@id}/player/','width'=>500],'name'=>'Flickr','scrape'=>[['extract'=>['@flickr\\.com/photos/[^/]+/(?\'id\'\\d+)@'],'match'=>['@flic\\.kr/p/(?\'short\'\\w+)@'],'url'=>'https://www.flickr.com/photo.gne?rb=1&short={@short}']],'tags'=>['images']],
		'foxnews'=>['attributes'=>[],'example'=>'https://video.foxnews.com/v/3592758613001/reddit-helps-fund-homemade-hot-sauce-venture/','extract'=>['!video\\.foxnews\\.com/v/(?\'id\'\\d+)!'],'homepage'=>'https://www.foxnews.com/','host'=>['video.foxnews.com'],'iframe'=>['src'=>'//video.foxnews.com/v/video-embed.html?video_id={@id}'],'name'=>'Fox News','scrape'=>[],'tags'=>['news']],
		'funnyordie'=>['attributes'=>[],'example'=>'https://www.funnyordie.com/videos/bf313bd8b4/murdock-with-keith-david','extract'=>['!funnyordie\\.com/videos/(?\'id\'[0-9a-f]+)!'],'homepage'=>'https://www.funnyordie.com/','host'=>['funnyordie.com'],'iframe'=>['src'=>'//www.funnyordie.com/embed/{@id}'],'name'=>'Funny or Die','scrape'=>[],'source'=>'http://support.funnyordie.com/discussions/problems/918-embed-user-videos-widget','tags'=>['entertainment']],
		'gamespot'=>['attributes'=>[],'example'=>['https://www.gamespot.com/destiny/videos/destiny-the-moon-trailer-6415176/','https://www.gamespot.com/events/game-crib-tsm-snapdragon/gamecrib-extras-cooking-with-dan-dinh-6412922/','https://www.gamespot.com/videos/beat-the-pros-pax-prime-2013/2300-6414307/'],'extract'=>['!gamespot\\.com.*?/(?:events|videos)/.*?-(?\'id\'\\d+)/(?:[#?].*)?$!'],'homepage'=>'https://www.gamespot.com/','host'=>['gamespot.com'],'iframe'=>['src'=>'//www.gamespot.com/videos/embed/{@id}/'],'name'=>'Gamespot','scrape'=>[],'tags'=>['gaming']],
		'getty'=>['attributes'=>['et'=>['required'=>true],'height'=>['defaultValue'=>360,'filterChain'=>['#uint']],'sig'=>['required'=>true],'width'=>['defaultValue'=>640,'filterChain'=>['#uint']]],'example'=>['https://gty.im/3232182','https://www.gettyimages.com/detail/3232182','https://www.gettyimages.com/detail/news-photo/the-beatles-travel-by-coach-to-the-west-country-for-some-news-photo/3232182','https://www.gettyimages.co.uk/detail/3232182'],'extract'=>[],'homepage'=>'https://www.gettyimages.com/','host'=>['gettyimages.be','gettyimages.cn','gettyimages.co.jp','gettyimages.co.uk','gettyimages.com','gettyimages.com.au','gettyimages.de','gettyimages.dk','gettyimages.es','gettyimages.fr','gettyimages.ie','gettyimages.it','gettyimages.nl','gettyimages.pt','gettyimages.se','gty.im'],'iframe'=>['height'=>'{@height}','src'=>'//embed.gettyimages.com/embed/{@id}?et={@et}&tld=com&sig={@sig}&caption=false&ver=1','width'=>'{@width}'],'name'=>'Getty Images','scrape'=>[['extract'=>['!/embed/(?\'id\'\\d+)!','!"height":[ "]*(?\'height\'\\d+)!','!"width":[ "]*(?\'width\'\\d+)!','!\\?et=(?\'et\'[-=\\w]+)!','!\\\\u0026sig=(?\'sig\'[-=\\w]+)!'],'match'=>['!(?:gty\\.im|gettyimages\\.[.\\w]+/detail(?=/).*?)/(?\'id\'\\d+)!'],'url'=>'https://embed.gettyimages.com/preview/{@id}']],'tags'=>['images']],
		'gfycat'=>['amp'=>['custom-element'=>'amp-gfycat','src'=>'https://cdn.ampproject.org/v0/amp-gfycat-0.1.js','template'=>'<amp-gfycat layout="responsive" width="{@width}" height="{@height}" data-gfyid="{@id}"/>'],'attributes'=>['height'=>['defaultValue'=>360,'filterChain'=>['#uint']],'width'=>['defaultValue'=>640,'filterChain'=>['#uint']]],'example'=>['https://gfycat.com/SereneIllfatedCapybara','https://giant.gfycat.com/SereneIllfatedCapybara.gif'],'extract'=>['#gfycat\\.com/(?!gaming|reactions|stickers|gifs/tag)(?:gifs/detail/|ifr(?:ame)?/)?(?\'id\'\\w+)#'],'homepage'=>'https://gfycat.com/','host'=>['gfycat.com'],'iframe'=>['height'=>'{@height}','padding-height'=>44,'src'=>'//gfycat.com/ifr/{@id}?hd=1','width'=>'{@width}'],'name'=>'Gfycat','oembed'=>['endpoint'=>'https://api.gfycat.com/v1/oembed','scheme'=>'https://gfycat.com/{@id}'],'scrape'=>[['extract'=>['!/ifr/(?\'id\'\\w+)!'],'match'=>['#gfycat\\.com/(?!gaming|reactions|stickers|gifs/tag)(?:gifs/detail/|ifr(?:ame)?/)?[a-z]#'],'url'=>'https://gfycat.com/ifr/{@id}'],['extract'=>['!"height":(?\'height\'\\d+)!','!"width":(?\'width\'\\d+)!'],'match'=>['//'],'url'=>'https://api.gfycat.com/v1/oembed?url=https://gfycat.com/{@id}']],'source'=>'https://developers.gfycat.com/iframe/','tags'=>['images'],'video'=>['height'=>'{@height}','poster'=>'https://thumbs.gfycat.com/{@id}-mobile.jpg','source'=>[['src'=>'https://giant.gfycat.com/{@id}.mp4','type'=>'video/mp4'],['src'=>'https://giant.gfycat.com/{@id}.webm','type'=>'video/webm'],['src'=>'https://thumbs.gfycat.com/{@id}-mobile.mp4','type'=>'video/mp4']],'width'=>'{@width}']],
		'gifs'=>['attributes'=>['height'=>['defaultValue'=>360,'filterChain'=>['#uint']],'width'=>['defaultValue'=>640,'filterChain'=>['#uint']]],'example'=>['https://gifs.com/gif/zm4DLy','https://j.gifs.com/Y6YZoO.gif'],'extract'=>['!gifs\\.com/(?:gif/)?(?\'id\'\\w+)!'],'homepage'=>'https://gifs.com/','host'=>['gifs.com'],'iframe'=>['height'=>'{@height}','src'=>'//gifs.com/embed/{@id}','width'=>'{@width}'],'name'=>'Gifs.com','scrape'=>[['extract'=>['!meta property="og:image:width" content="(?\'width\'\\d+)!','!meta property="og:image:height" content="(?\'height\'\\d+)!'],'match'=>['//'],'url'=>'https://gifs.com/gif/{@id}']],'tags'=>['images'],'video'=>['height'=>'{@height}','poster'=>'https://j.gifs.com/{@id}.jpg','source'=>[['src'=>'https://j.gifs.com/{@id}@large.mp4','type'=>'video/mp4'],['src'=>'https://j.gifs.com/{@id}.mp4','type'=>'video/mp4']],'width'=>'{@width}']],
		'giphy'=>['attributes'=>['height'=>['defaultValue'=>360,'filterChain'=>['#uint']],'width'=>['defaultValue'=>640,'filterChain'=>['#uint']]],'example'=>['https://media.giphy.com/media/UGUqsAtHh3T1e/source.gif','https://giphy.com/gifs/cheezburger-sloth-yawn-FsjDdnIRcroIM','https://i.giphy.com/l41m1wcjpWJZi5w7S.gif','https://giphy.com/videos/quinnxcii-coffee-VG8PNQDwsfOzU4mQAL'],'extract'=>['!giphy\\.com/(?\'type\'gif|video|webp)\\w+/(?:[-\\w]+-)*(?\'id\'\\w+)!','!giphy\\.com/media/(?\'id\'\\w+)/\\w+\\.(?\'type\'gif|webp)!','!i\\.giphy\\.com/(?\'id\'\\w+)\\.(?\'type\'gif|webp)!'],'homepage'=>'https://giphy.com/','host'=>['giphy.com'],'iframe'=>['height'=>'{@height}','src'=>'//giphy.com/embed/<xsl:value-of select="@id"/><xsl:if test="@type=\'video\'">/video</xsl:if>','width'=>'{@width}'],'name'=>'GIPHY','scrape'=>[['extract'=>['!"height"\\s*:\\s*(?\'height\'\\d+)!','!"width"\\s*:\\s*(?\'width\'\\d+)!'],'match'=>['//'],'url'=>'https://giphy.com/services/oembed?url=https://media.giphy.com/media/{@id}/giphy.gif']],'tags'=>['images'],'video'=>['height'=>'{@height}','poster'=>'https://media.giphy.com/media/{@id}/giphy_s.gif','source'=>['src'=>'https://media.giphy.com/media/{@id}/giphy.mp4','type'=>'video/mp4'],'width'=>'{@width}']],
		'gist'=>['amp'=>['custom-element'=>'amp-gist','src'=>'https://cdn.ampproject.org/v0/amp-gist-0.1.js','template'=>'<amp-gist layout="fixed-height" height="180"><xsl:attribute name="data-gistid"><xsl:choose><xsl:when test="contains(@id,\'/\')"><xsl:choose><xsl:when test="contains(substring-after(@id,\'/\'),\'/\')"><xsl:value-of select="substring-before(substring-after(@id,\'/\'),\'/\')"/></xsl:when><xsl:otherwise><xsl:value-of select="substring-after(@id,\'/\')"/></xsl:otherwise></xsl:choose></xsl:when><xsl:otherwise><xsl:value-of select="@id"/></xsl:otherwise></xsl:choose></xsl:attribute></amp-gist>'],'attributes'=>[],'example'=>['https://gist.github.com/JoshyPHP/0ee8433f5a9a779d08ef','https://gist.github.com/6806305','https://gist.github.com/JoshyPHP/6806305/ad88d904b082c8211afa040162402015aacb8599'],'extract'=>['@gist\\.github\\.com/(?\'id\'(?:[-\\w]+/)?[\\da-f]+(?:/[\\da-f]+)?\\b(?!/archive))@'],'homepage'=>'https://gist.github.com/','host'=>['github.com'],'iframe'=>['data-s9e-livepreview-ignore-attrs'=>'style','height'=>180,'onload'=>'var c=new MessageChannel;c.port1.onmessage=function(e){{style.height=e.data+\'px\'}};contentWindow.postMessage(\'s9e:init\',\'https://s9e.github.io\',[c.port2])','scrolling'=>'','src'=>'https://s9e.github.io/iframe/2/gist.min.html#{@id}','width'=>'100%'],'name'=>'GitHub Gist','scrape'=>[],'tags'=>['programming']],
		'globalnews'=>['attributes'=>[],'example'=>['https://globalnews.ca/video/1647385/mark-channels-his-70s-look/','https://globalnews.ca/video/rd/350539843530/?jwsource=cl'],'extract'=>['!globalnews\\.ca/video/(?\'id\'\\d+)!'],'homepage'=>'https://globalnews.ca/','host'=>['globalnews.ca'],'iframe'=>['src'=>'//globalnews.ca/video/embed/{@id}/'],'name'=>'Global News','scrape'=>[['extract'=>['!globalnews\\.ca/video/(?\'id\'\\d+)!'],'match'=>['!globalnews\\.ca/video/rd/!']]],'tags'=>['.ca','news']],
		'gofundme'=>['attributes'=>[],'example'=>'https://www.gofundme.com/2p37ao','extract'=>['@gofundme\\.com/(?\'id\'\\w+)(?![^#?])@'],'homepage'=>'https://www.gofundme.com/','host'=>['gofundme.com'],'iframe'=>['height'=>207,'padding-height'=>208,'src'=>'//www.gofundme.com/mvc.php?route=widgets/mediawidget&fund={@id}&image=1&coinfo=1','width'=>349],'name'=>'GoFundMe','scrape'=>[],'tags'=>['fundraising']],
		'googledrive'=>['attributes'=>[],'example'=>'https://drive.google.com/file/d/0B_4NRUjxLBejNjVmeG5MUzA3Q3M/view?usp=sharing','extract'=>['!drive\\.google\\.com/.*?(?:file/d/|id=)(?\'id\'[-\\w]+)!'],'homepage'=>'https://drive.google.com','host'=>['drive.google.com'],'iframe'=>['height'=>480,'src'=>'//drive.google.com/file/d/{@id}/preview','width'=>640],'name'=>'Google Drive','scrape'=>[],'tags'=>['documents','images','videos']],
		'googleplus'=>['attributes'=>['name'=>['filterChain'=>['urldecode']]],'example'=>['https://plus.google.com/+TonyHawk/posts/C5TMsDZJWBd','https://plus.google.com/106189723444098348646/posts/V8AojCoTzxV'],'extract'=>['!//plus\\.google\\.com/(?:u/\\d+/)?(?:\\+(?\'name\'[^/]+)|(?\'oid\'\\d+))/posts/(?\'pid\'\\w+)!'],'homepage'=>'https://plus.google.com/','host'=>['plus.google.com'],'iframe'=>['data-s9e-livepreview-ignore-attrs'=>'style','height'=>240,'onload'=>'var c=new MessageChannel;c.port1.onmessage=function(e){{style.height=e.data+\'px\'}};contentWindow.postMessage(\'s9e:init\',\'https://s9e.github.io\',[c.port2])','src'=>'https://s9e.github.io/iframe/2/googleplus.min.html#<xsl:choose><xsl:when test="@oid"><xsl:value-of select="@oid"/></xsl:when><xsl:otherwise>+<xsl:value-of select="@name"/></xsl:otherwise></xsl:choose>/posts/<xsl:value-of select="@pid"/>','width'=>450],'name'=>'Google+','scrape'=>[],'source'=>'https://developers.google.com/+/web/embedded-post/','tags'=>['social']],
		'googlesheets'=>['attributes'=>[],'choose'=>['otherwise'=>['iframe'=>['height'=>500,'src'=>'https://docs.google.com/spreadsheets/d/<xsl:value-of select="@id"/>/p<xsl:choose><xsl:when test="starts-with(@id,\'e/\')">ubhtml?widget=true&amp;headers=false</xsl:when><xsl:otherwise>review</xsl:otherwise></xsl:choose>#gid=<xsl:value-of select="@gid"/>','style'=>['resize'=>'vertical'],'width'=>'100%']],'when'=>['iframe'=>['height'=>372,'src'=>'https://docs.google.com/spreadsheets/d/{@id}/pubchart?oid={@oid}&format=interactive','width'=>600],'test'=>'@type=\'chart\'']],'example'=>['https://docs.google.com/spreadsheets/d/1f988o68HDvk335xXllJD16vxLBuRcmm3vg6U9lVaYpA','https://docs.google.com/spreadsheets/u/1/d/e/2PACX-1vQ7iYECa_ziPWygKBf9Og-PkQl5-gdnFRU-jP-z1FyjLfLXMAdRJN1J5Lx31NjnCq5lkRcL-T0Idh4_/pubhtml','https://docs.google.com/spreadsheets/d/e/2PACX-1vSbbciMU7t5dCB8auk-nyK-p-25TJjgUrzBuMCAiq_2ZDqoMSFAlZCctIC7y3TXS71ys_MRdoAU4Jqr/pubchart?oid=1729348495'],'extract'=>['@docs\\.google\\.com/spreadsheet(?:/ccc\\?key=|(?:[^e]/)+)(?\'id\'(?:e/)?[-\\w]+)@','@oid=(?\'oid\'\\d+)@','@#gid=(?\'gid\'\\d+)@','@/pub(?\'type\'chart)@'],'homepage'=>'https://www.google.com/sheets/about/','host'=>['docs.google.com'],'name'=>'Google Sheets','scrape'=>[],'tags'=>['documents']],
		'hudl'=>['attributes'=>['id'=>['required'=>false]],'example'=>['https://www.hudl.com/video/59dfca6090eeac23f0eef838','https://www.hudl.com/athlete/2067184/highlights/163744377','https://www.hudl.com/video/3/323679/57719969842eb243e47883f8'],'extract'=>['!hudl\\.com/athlete/(?\'athlete\'\\d+)/highlights/(?\'highlight\'[\\da-f]+)!','!hudl\\.com/video/\\d+/(?\'athlete\'\\d+)/(?\'highlight\'[\\da-f]+)!','@hudl\\.com/video/(?\'id\'\\w+)(?![\\w/])@'],'homepage'=>'https://www.hudl.com/','host'=>['hudl.com'],'iframe'=>['src'=>'//www.hudl.com/embed/<xsl:choose><xsl:when test="@id">video/<xsl:value-of select="@id"/></xsl:when><xsl:otherwise>athlete/<xsl:value-of select="@athlete"/>/highlights/<xsl:value-of select="@highlight"/></xsl:otherwise></xsl:choose>'],'name'=>'Hudl','scrape'=>[['extract'=>['!hudl\\.com/video/\\d+/(?\'athlete\'\\d+)/(?\'highlight\'[\\da-f]+)!','@hudl\\.com/video/(?\'id\'\\w+)(?![\\w/])@'],'match'=>['!hudl\\.com/v/!']]],'tags'=>['sports']],
		'hulu'=>['attributes'=>[],'example'=>'https://www.hulu.com/watch/484180','extract'=>[],'homepage'=>'https://www.hulu.com/','host'=>['hulu.com'],'iframe'=>['src'=>'https://secure.hulu.com/embed/{@id}'],'name'=>'Hulu','scrape'=>[['extract'=>['!eid=(?\'id\'[-\\w]+)!'],'match'=>['!hulu\\.com/watch/!']]]],
		'ign'=>['attributes'=>[],'example'=>'https://www.ign.com/videos/2013/07/12/pokemon-x-version-pokemon-y-version-battle-trailer','extract'=>['!(?\'id\'https?://.*?ign\\.com/videos/.+)!i'],'homepage'=>'https://www.ign.com/videos/','host'=>['ign.com'],'iframe'=>['height'=>263,'src'=>'//widgets.ign.com/video/embed/content.html?url={@id}','width'=>468],'name'=>'IGN','scrape'=>[],'tags'=>['gaming']],
		'imdb'=>['attributes'=>[],'example'=>['https://www.imdb.com/video/imdb/vi2482677785/','https://www.imdb.com/title/tt2294629/videoplayer/vi2482677785'],'extract'=>[],'homepage'=>'https://www.imdb.com/trailers/','host'=>['imdb.com'],'iframe'=>['src'=>'//www.imdb.com/video/imdb/vi{@id}/imdb/embed?autoplay=false&width=640'],'name'=>'IMDb trailers','scrape'=>[['extract'=>['!imdb\\.com/[/\\w]+?/vi(?\'id\'\\d+)/[/\\w]*?embed!'],'match'=>['!imdb\\.com/[/\\w]+?/vi(?\'id\'\\d+)!'],'url'=>'https://www.imdb.com/video/imdb/vi{@id}/imdb/embed']],'tags'=>['movies']],
		'imgur'=>['amp'=>['custom-element'=>'amp-imgur','src'=>'https://cdn.ampproject.org/v0/amp-imgur-0.1.js','template'=>'<amp-imgur layout="responsive" width="542" height="400" data-imgur-id="{@id}"/>'],'attributes'=>[],'example'=>['https://imgur.com/AsQ0K3P','https://imgur.com/a/9UGCL','https://imgur.com/gallery/9UGCL','https://i.imgur.com/u7Yo0Vy.gifv','https://i.imgur.com/UO1UrIx.mp4'],'extract'=>['@imgur\\.com/(?\'id\'a/\\w+)@','@i\\.imgur\\.com/(?\'id\'\\w{5,7})[lms]?\\.@','@imgur\\.com/(?\'id\'\\w+)(?![\\w./])@'],'homepage'=>'https://imgur.com/','host'=>['imgur.com'],'iframe'=>['data-s9e-livepreview-ignore-attrs'=>'style','height'=>400,'onload'=>'var c=new MessageChannel;c.port1.onmessage=function(e){{var d=e.data.split(\' \');style.height=d[0]+\'px\';style.width=d[1]+\'px\'}};contentWindow.postMessage(\'s9e:init\',\'https://s9e.github.io\',[c.port2])','src'=>'https://s9e.github.io/iframe/2/imgur.min.html#<xsl:if test="@type=\'album\'">a/</xsl:if><xsl:value-of select="@id"/>','width'=>542],'name'=>'Imgur','oembed'=>['endpoint'=>'https://api.imgur.com/oembed.json','scheme'=>'https://imgur.com/{@id}'],'scrape'=>[['extract'=>['@data-id="(?\'id\'[\\w/]+)"@'],'match'=>['@imgur\\.com/(?![art]/|user/)(?\'path\'(?:gallery/)?\\w+)(?![\\w.])@'],'url'=>'https://api.imgur.com/oembed.xml?url=/{@path}']],'tags'=>['images']],
		'indiegogo'=>['attributes'=>[],'example'=>'https://www.indiegogo.com/projects/gameheart-redesigned','extract'=>['!indiegogo\\.com/projects/(?\'id\'[-\\w]+)!'],'homepage'=>'https://www.indiegogo.com/','host'=>['indiegogo.com'],'iframe'=>['height'=>445,'src'=>'//www.indiegogo.com/project/{@id}/embedded','width'=>222],'name'=>'Indiegogo','scrape'=>[],'tags'=>['fundraising']],
		'instagram'=>['amp'=>['custom-element'=>'amp-instagram','src'=>'https://cdn.ampproject.org/v0/amp-instagram-0.1.js','template'=>'<amp-instagram layout="responsive" width="540" height="540" data-captioned="" data-shortcode="{@id}"/>'],'attributes'=>[],'example'=>['https://instagram.com/p/gbGaIXBQbn/','https://www.instagram.com/tv/BkQjCfsBIzi/'],'extract'=>['!instagram\\.com/(?:p|reel|tv)/(?\'id\'[-\\w]+)!'],'homepage'=>'https://instagram.com/','host'=>['instagram.com'],'iframe'=>['data-s9e-livepreview-ignore-attrs'=>'style','height'=>540,'onload'=>'var c=new MessageChannel;c.port1.onmessage=function(e){{style.height=e.data+\'px\'}};contentWindow.postMessage(\'s9e:init\',\'https://s9e.github.io\',[c.port2])','src'=>'https://s9e.github.io/iframe/2/instagram.min.html#<xsl:value-of select="@id"/><xsl:if test="$MEDIAEMBED_THEME=\'dark\'">#theme=dark</xsl:if>','width'=>540],'name'=>'Instagram','scrape'=>[],'source'=>'http://help.instagram.com/513918941996087','tags'=>['social']],
		'internetarchive'=>['attributes'=>['height'=>['defaultValue'=>360,'filterChain'=>['#uint']],'id'=>['filterChain'=>['htmlspecialchars_decode']],'width'=>['defaultValue'=>640,'filterChain'=>['#uint']]],'choose'=>['otherwise'=>['iframe'=>['height'=>'{@height}','src'=>'https://archive.org/embed/{@id}','width'=>'{@width}']],'when'=>['iframe'=>['height'=>170,'max-width'=>640,'src'=>'https://archive.org/embed/{@id}&list_height=150','width'=>'100%'],'test'=>'contains(@id,\'playlist=1\')']],'example'=>['https://archive.org/details/BillGate99','https://archive.org/details/DFTS2014-05-30'],'extract'=>[],'homepage'=>'https://archive.org/','host'=>['archive.org'],'name'=>'Internet Archive','scrape'=>[['extract'=>['!meta property="twitter:player" content="https://archive.org/embed/(?\'id\'[^/"]+)!','!meta property="og:video:width" content="(?\'width\'\\d+)!','!meta property="og:video:height" content="(?\'height\'\\d+)!'],'match'=>['!archive\\.org/(?:details|embed)/!']]]],
		'izlesene'=>['amp'=>['custom-element'=>'amp-izlesene','src'=>'https://cdn.ampproject.org/v0/amp-izlesene-0.1.js','template'=>'<amp-izlesene layout="responsive" width="640" height="360" data-videoid="{@id}"/>'],'attributes'=>[],'example'=>'https://www.izlesene.com/video/kardan-adamin-salincak-keyfi/10553555','extract'=>['!izlesene\\.com/video/[-\\w]+/(?\'id\'\\d+)!'],'homepage'=>'https://www.izlesene.com/','host'=>['izlesene.com'],'iframe'=>['src'=>'//www.izlesene.com/embedplayer/{@id}?autoplay=0'],'name'=>'İzlesene','scrape'=>[],'tags'=>['.tr']],
		'jsfiddle'=>['attributes'=>[],'example'=>'https://jsfiddle.net/1v03Lqnu/','extract'=>['@jsfiddle.net/(?:(?\'user\'\\w+)/)?(?!\\d+\\b|embedded\\b|show\\b)(?\'id\'\\w+)\\b(?:/(?\'revision\'\\d+)\\b)?@'],'homepage'=>'https://jsfiddle.net/','host'=>['jsfiddle.net'],'iframe'=>['height'=>400,'src'=>'//jsfiddle.net/{@id}/{@revision}/embedded/{$MEDIAEMBED_THEME}/','width'=>'100%'],'name'=>'JSFiddle','scrape'=>[],'tags'=>['programming']],
		'jwplatform'=>['attributes'=>[],'example'=>['https://content.jwplatform.com/players/X6tRZpKj-7Y21S9TB.html','https://content.jwplatform.com/previews/YfTSAegE-L0l35Tsd'],'extract'=>['!jwplatform\\.com/\\w+/(?\'id\'[-\\w]+)!'],'homepage'=>'https://www.jwplayer.com/products/jwplatform/','host'=>['jwplatform.com'],'iframe'=>['src'=>'//content.jwplatform.com/players/{@id}.html'],'name'=>'JW Platform','scrape'=>[],'tags'=>['videos']],
		'kaltura'=>['amp'=>['custom-element'=>'amp-kaltura-player','src'=>'https://cdn.ampproject.org/v0/amp-kaltura-player-0.1.js','template'=>'<amp-kaltura-player layout="responsive" width="640" height="360" data-service-url="cdnapisec.kaltura.com" data-uiconf="{@uiconf_id}" data-partner="{@partner_id}" data-entryid="{@entry_id}" data-param-streamertype="auto"/>'],'attributes'=>['entry_id'=>['required'=>true],'partner_id'=>['required'=>true],'uiconf_id'=>['required'=>true]],'example'=>['https://www.kaltura.com/tiny/b5tib','https://vcu.mediaspace.kaltura.com/media/t/1_hu0rc15a'],'extract'=>['@/p(?:artner_id)?/(?\'partner_id\'\\d+)/@','@/sp/(?\'sp\'\\d+)/@','@/uiconf_id/(?\'uiconf_id\'\\d+)/@','@\\bentry_id[=/](?\'entry_id\'\\w+)@'],'homepage'=>'https://corp.kaltura.com/','host'=>['kaltura.com'],'iframe'=>['padding-height'=>36,'src'=>'https://cdnapisec.kaltura.com/p/<xsl:value-of select="@partner_id"/>/sp/<xsl:choose><xsl:when test="@sp"><xsl:value-of select="@sp"/></xsl:when><xsl:otherwise><xsl:value-of select="@partner_id"/>00</xsl:otherwise></xsl:choose>/embedIframeJs/uiconf_id/<xsl:value-of select="@uiconf_id"/>/partner_id/<xsl:value-of select="@partner_id"/>?iframeembed=true&amp;entry_id=<xsl:value-of select="@entry_id"/>'],'name'=>'Kaltura','scrape'=>[['extract'=>['@kaltura\\.com/+p/(?\'partner_id\'\\d+)/sp/(?\'sp\'\\d+)/\\w*/uiconf_id/(?\'uiconf_id\'\\d+)/.*?\\bentry_id=(?\'entry_id\'\\w+)@'],'match'=>['@kaltura\\.com/(?:media/t|tiny)/.@']]],'tags'=>['videos']],
		'khl'=>['attributes'=>[],'example'=>['https://video.khl.ru/events/233677','https://video.khl.ru/quotes/251237'],'extract'=>[],'homepage'=>'https://www.khl.ru/','host'=>['video.khl.ru'],'iframe'=>['src'=>'//video.khl.ru/iframe/feed/start/{@id}?type_id=18&width=560&height=315'],'name'=>'Kontinental Hockey League (КХЛ)','scrape'=>[['extract'=>['!/feed/start/(?\'id\'[/\\w]+)!'],'match'=>['!video\\.khl\\.ru/(?:event|quote)s/\\d!']]],'tags'=>['.ru','sports']],
		'kickstarter'=>['attributes'=>[],'choose'=>['otherwise'=>['iframe'=>['height'=>420,'src'=>'//www.kickstarter.com/projects/{@id}/widget/card.html','width'=>220]],'when'=>['iframe'=>['height'=>360,'src'=>'//www.kickstarter.com/projects/{@id}/widget/video.html','width'=>640],'test'=>'@video']],'example'=>['https://www.kickstarter.com/projects/1869987317/wish-i-was-here-1','https://www.kickstarter.com/projects/1869987317/wish-i-was-here-1/widget/card.html','https://www.kickstarter.com/projects/1869987317/wish-i-was-here-1/widget/video.html'],'extract'=>['!kickstarter\\.com/projects/(?\'id\'[^/]+/[^/?]+)(?:/widget/(?:(?\'card\'card)|(?\'video\'video)))?!'],'homepage'=>'https://www.kickstarter.com/','host'=>['kickstarter.com'],'name'=>'Kickstarter','scrape'=>[],'tags'=>['fundraising']],
		'libsyn'=>['attributes'=>[],'example'=>'https://pexlives.libsyn.com/the-ark-in-space-ep4-commentary-with-el-and-jack','extract'=>[],'homepage'=>'https://www.libsyn.com/','host'=>['libsyn.com'],'iframe'=>['height'=>90,'max-width'=>900,'src'=>'//html5-player.libsyn.com/embed/episode/id/{@id}/thumbnail/no','width'=>'100%'],'name'=>'Libsyn','scrape'=>[['extract'=>['!embed/episode/id/(?\'id\'\\d+)!'],'match'=>['@(?!\\.mp3)....$@']]],'tags'=>['podcasts']],
		'liveleak'=>['attributes'=>[],'example'=>'https://www.liveleak.com/view?t=yIcw_1520190567','extract'=>['!liveleak\\.com/(?:e/|view\\?i=)(?\'id\'\\w+)!'],'homepage'=>'https://www.liveleak.com/','host'=>['liveleak.com'],'iframe'=>['src'=>'//www.liveleak.com/e/{@id}'],'name'=>'Liveleak','scrape'=>[['extract'=>['!liveleak\\.com/e/(?\'id\'\\w+)!'],'match'=>['!liveleak\\.com/view\\?t=!']]],'tags'=>['videos']],
		'livestream'=>['attributes'=>[],'example'=>['https://new.livestream.com/jbtvlive/musicmarathon','https://livestream.com/ccscsl/USChessChampionships/videos/83267610','https://livestre.am/58XNV'],'extract'=>['!livestream\\.com/accounts/(?\'account_id\'\\d+)/events/(?\'event_id\'\\d+)!','!/videos/(?\'video_id\'\\d+)!','!original\\.livestream\\.com/(?\'channel\'\\w+)/video\\?clipId=(?\'clip_id\'[-\\w]+)!'],'homepage'=>'https://new.livestream.com/','host'=>['livestre.am','livestream.com'],'iframe'=>['src'=>'//<xsl:choose><xsl:when test="@clip_id">cdn.livestream.com/embed/<xsl:value-of select="@channel"/>?layout=4&amp;autoplay=false&amp;clip=<xsl:value-of select="@clip_id"/></xsl:when><xsl:otherwise>livestream.com/accounts/<xsl:value-of select="@account_id"/>/events/<xsl:value-of select="@event_id"/><xsl:if test="@video_id">/videos/<xsl:value-of select="@video_id"/></xsl:if>/player?autoPlay=false</xsl:otherwise></xsl:choose>'],'name'=>'Livestream','scrape'=>[['extract'=>['!accounts/(?\'account_id\'\\d+)/events/(?\'event_id\'\\d+)!'],'match'=>['@livestream\\.com/(?!accounts/\\d+/events/\\d)@']],['extract'=>['!//original\\.livestream\\.com/(?\'channel\'\\w+)/video/(?\'clip_id\'[-\\w]+)!'],'match'=>['!livestre.am!']]],'tags'=>['livestreaming','videos']],
		'mailru'=>['attributes'=>[],'example'=>['https://my.mail.ru/corp/auto/video/testdrive/34.html','https://my.mail.ru/mail/alenka1957/video/1/7.html'],'extract'=>[],'homepage'=>'https://my.mail.ru/','host'=>['my.mail.ru'],'iframe'=>['src'=>'https://my.mail.ru/video/embed/{@id}'],'name'=>'Mail.Ru','scrape'=>[['extract'=>['!"itemId": ?"?(?\'id\'\\d+)!'],'match'=>['!my\\.mail\\.ru/\\w+/\\w+/video/\\w+/\\d!']]],'tags'=>['.ru']],
		'medium'=>['attributes'=>[],'example'=>'https://medium.com/@donnydonny/team-internet-is-about-to-win-net-neutrality-and-they-didnt-need-googles-help-e7e2cf9b8a95','extract'=>['#medium\\.com/(?:s/\\w+/|@?[-\\w]+/)?(?:\\w+-)*(?\'id\'[0-9a-f]+)(?!\\w)#'],'homepage'=>'https://medium.com/','host'=>['medium.com'],'iframe'=>['data-s9e-livepreview-ignore-attrs'=>'style','height'=>316,'max-width'=>900,'onload'=>'var c=new MessageChannel;c.port1.onmessage=function(e){{style.height=e.data+\'px\'}};contentWindow.postMessage(\'s9e:init\',\'https://s9e.github.io\',[c.port2])','src'=>'https://s9e.github.io/iframe/2/medium.min.html#{@id}','width'=>'100%'],'name'=>'Medium','scrape'=>[],'tags'=>['blogging']],
		'megaphone'=>['amp'=>['custom-element'=>'amp-megaphone','src'=>'https://cdn.ampproject.org/v0/amp-megaphone-0.1.js','template'=>'<amp-megaphone layout="fixed-height" height="200" data-episode="{@id}"><xsl:if test="$MEDIAEMBED_THEME=\'light\'"><xsl:attribute name="data-light"/></xsl:if></amp-megaphone>'],'attributes'=>[],'example'=>['https://cms.megaphone.fm/channel/lockedonheat?selected=LKN8165322853','https://player.megaphone.fm/LKN8165322853'],'extract'=>['@megaphone\\.fm/.*?\\?(?:e|selected)=(?\'id\'\\w+)@','@(?:dcs|player|traffic)\\.megaphone\\.fm/(?\'id\'\\w+)@','@megaphone\\.link/(?\'id\'\\w+)@'],'homepage'=>'https://megaphone.fm/','host'=>['megaphone.fm','megaphone.link'],'iframe'=>['height'=>200,'max-width'=>900,'src'=>'https://player.megaphone.fm/<xsl:value-of select="@id"/><xsl:if test="$MEDIAEMBED_THEME=\'light\'">?light=true</xsl:if>','width'=>'100%'],'name'=>'Megaphone','scrape'=>[],'tags'=>['podcasts']],
		'metacafe'=>['attributes'=>[],'example'=>'https://www.metacafe.com/watch/10785282/chocolate_treasure_chest_epic_meal_time/','extract'=>['!metacafe\\.com/watch/(?\'id\'\\d+)!'],'homepage'=>'https://www.metacafe.com/','host'=>['metacafe.com'],'iframe'=>['src'=>'//www.metacafe.com/embed/{@id}/'],'name'=>'Metacafe','scrape'=>[],'tags'=>['videos']],
		'mixcloud'=>['attributes'=>[],'example'=>'https://www.mixcloud.com/OneTakeTapes/timsch-one-take-tapes-2/','extract'=>['@mixcloud\\.com/(?!categories|tag)(?\'id\'[-\\w]+/[^/&]+)/@'],'homepage'=>'https://www.mixcloud.com/','host'=>['mixcloud.com'],'iframe'=>['height'=>400,'src'=>'//www.mixcloud.com/widget/iframe/?feed=http%3A%2F%2Fwww.mixcloud.com%2F{@id}%2F&embed_type=widget_standard','width'=>400],'name'=>'Mixcloud','oembed'=>['endpoint'=>'https://www.mixcloud.com/oembed/','scheme'=>'https://www.mixcloud.com/{@id}'],'scrape'=>[],'tags'=>['music']],
		'mlb'=>['attributes'=>[],'example'=>['https://www.mlb.com/video/statcast-stantons-two-homers/c-1898424783','https://m.mlb.com/video/v1205791883'],'extract'=>['#mlb\\.com/video/(?:[-\\w/]+/)?(?:c-|v|[-\\w]+-c)(?\'id\'\\d+)#'],'homepage'=>'https://mlb.com/video/','host'=>['mlb.com'],'iframe'=>['src'=>'https://www.mlb.com/video/share/c-{@id}'],'name'=>'MLB','scrape'=>[],'tags'=>['sports']],
		'mrctv'=>['attributes'=>[],'example'=>'https://mrctv.org/videos/cnn-frets-about-tobacco-companies-color-coding-tricks','extract'=>[],'homepage'=>'https://www.mrctv.org/','host'=>['mrctv.org'],'iframe'=>['src'=>'https://www.mrctv.org/embed/{@id}'],'name'=>'MRCTV','scrape'=>[['extract'=>['!mrctv\\.org/embed/(?\'id\'\\d+)!'],'match'=>['!mrctv\\.org/videos/.!']]]],
		'msnbc'=>['attributes'=>[],'choose'=>['otherwise'=>['iframe'=>['src'=>'https://www.msnbc.com/msnbc/embedded-video/{@id}']],'when'=>['iframe'=>['padding-height'=>80,'src'=>'//player.theplatform.com/p/7wvmTC/MSNBCEmbeddedOffSite?guid={@id}'],'test'=>'contains(@id, \'_\')']],'example'=>['https://www.msnbc.com/ronan-farrow-daily/watch/thats-no-moon--300512323725','https://on.msnbc.com/1qkH62o'],'extract'=>[],'homepage'=>'https://www.msnbc.com/watch','host'=>['msnbc.com'],'name'=>'MSNBC','scrape'=>[['extract'=>['@embedded-video/(?!undefined)(?\'id\'\\w+)@'],'match'=>['@msnbc\\.com/[-\\w]+/watch/@','@on\\.msnbc\\.com/.@']]],'tags'=>['news']],
		'natgeochannel'=>['attributes'=>[],'example'=>['https://channel.nationalgeographic.com/channel/brain-games/videos/jason-silva-on-intuition/','https://channel.nationalgeographic.com/wild/urban-jungle/videos/leopard-in-the-city/'],'extract'=>['@channel\\.nationalgeographic\\.com/(?\'id\'[-/\\w]+/videos/[-\\w]+)@'],'homepage'=>'https://channel.nationalgeographic.com/','host'=>['channel.nationalgeographic.com'],'iframe'=>['src'=>'//channel.nationalgeographic.com/{@id}/embed/'],'name'=>'National Geographic Channel','scrape'=>[]],
		'natgeovideo'=>['attributes'=>[],'example'=>['https://video.nationalgeographic.com/tv/changing-earth','https://video.nationalgeographic.com/video/weirdest-superb-lyrebird'],'extract'=>[],'homepage'=>'https://video.nationalgeographic.com/','host'=>['video.nationalgeographic.com'],'iframe'=>['src'=>'//player.d.nationalgeographic.com/players/ngsvideo/share/?guid={@id}'],'name'=>'National Geographic Video','scrape'=>[['extract'=>['@guid="(?\'id\'[-\\w]+)"@'],'match'=>['@video\\.nationalgeographic\\.com/(?:tv|video)/\\w@']]],'tags'=>['documentaries']],
		'nbcnews'=>['attributes'=>[],'example'=>'https://www.nbcnews.com/video/bob-dylan-awarded-nobel-prize-for-literature-785193027834','extract'=>['!nbcnews\\.com/(?:widget/video-embed/|video/[-\\w]+?-)(?\'id\'\\d+)!'],'homepage'=>'https://www.nbcnews.com/video/','host'=>['nbcnews.com'],'iframe'=>['src'=>'//www.nbcnews.com/widget/video-embed/{@id}'],'name'=>'NBC News','scrape'=>[],'tags'=>['news']],
		'nbcsports'=>['attributes'=>[],'example'=>'https://www.nbcsports.com/video/countdown-rio-olympics-what-makes-perfect-performance','extract'=>[],'homepage'=>'https://www.nbcsports.com/video','host'=>['nbcsports.com'],'iframe'=>['src'=>'//vplayer.nbcsports.com/p/BxmELC/nbcsports_embed/select/media/{@id}?parentUrl='],'name'=>'NBC Sports','scrape'=>[['extract'=>['!select/media/(?\'id\'\\w+)!'],'match'=>['!nbcsports\\.com/video/.!']]],'tags'=>['sports']],
		'nhl'=>['attributes'=>[],'example'=>'https://www.nhl.com/video/recap-min-2-ott-1-fot/t-277753022/c-46330703','extract'=>['#nhl\\.com/(?:\\w+/)?video(?:/(?![ct]-)[-\\w]+)?(?:/t-(?\'t\'\\d+))?(?:/c-(?\'c\'\\d+))?#'],'homepage'=>'https://www.nhl.com/video','host'=>['nhl.com'],'iframe'=>['src'=>'https://www.nhl.com/video/embed<xsl:if test="@t">/t-<xsl:value-of select="@t"/></xsl:if><xsl:if test="@c">/c-<xsl:value-of select="@c"/></xsl:if>?autostart=false'],'name'=>'NHL Videos and Highlights','scrape'=>[],'tags'=>['sports']],
		'npr'=>['attributes'=>[],'example'=>['https://www.npr.org/blogs/goatsandsoda/2015/02/11/385396431/the-50-most-effective-ways-to-transform-the-developing-world','https://n.pr/1Qky1m5'],'extract'=>[],'homepage'=>'https://www.npr.org/','host'=>['npr.org','n.pr'],'iframe'=>['height'=>228,'max-width'=>800,'src'=>'//www.npr.org/player/embed/{@i}/{@m}','width'=>'100%'],'name'=>'NPR','scrape'=>[['extract'=>['!player/embed/(?\'i\'\\d+)/(?\'m\'\\d+)!'],'header'=>'Cookie: trackingChoice=false; choiceVersion=1','match'=>['!npr\\.org/[/\\w]+/\\d+!','!n\\.pr/\\w!']]],'tags'=>['podcasts']],
		'nytimes'=>['attributes'=>[],'example'=>['https://www.nytimes.com/video/magazine/100000003166834/small-plates.html','https://www.nytimes.com/video/technology/personaltech/100000002907606/soylent-taste-test.html','https://www.nytimes.com/video/2012/12/17/business/100000001950744/how-wal-mart-conquered-teotihuacan.html','https://movies.nytimes.com/movie/131154/Crooklyn/trailers'],'extract'=>['!nytimes\\.com/video/[a-z]+/(?:[a-z]+/)?(?\'id\'\\d+)!','!nytimes\\.com/video/\\d+/\\d+/\\d+/[a-z]+/(?\'id\'\\d+)!'],'homepage'=>'https://www.nytimes.com/video/','host'=>['nytimes.com'],'iframe'=>['height'=>400,'src'=>'//graphics8.nytimes.com/video/players/offsite/index.html?videoId={@id}','width'=>585],'name'=>'The New York Times Video','scrape'=>[['extract'=>['!/video/movies/(?\'id\'\\d+)!'],'match'=>['!nytimes\\.com/movie(?:s/movie)?/(?\'playlist\'\\d+)/[-\\w]+/trailers!'],'url'=>'http://www.nytimes.com/svc/video/api/playlist/{@playlist}?externalId=true']],'tags'=>['movies','news']],
		'odysee'=>['attributes'=>['name'=>['filterChain'=>['#url'],'required'=>false]],'example'=>'https://odysee.com/@veritasium:f/the-secret-of-synchronization:2','extract'=>['#odysee\\.com/(?:\\$/\\w+/)?(?\'name\'[^:/]+)[:/](?\'id\'\\w{40})#'],'homepage'=>'https://odysee.com/','host'=>['odysee.com'],'iframe'=>['src'=>'https://odysee.com/$/embed/<xsl:choose><xsl:when test="@name"><xsl:value-of select="@name"/></xsl:when><xsl:otherwise>-</xsl:otherwise></xsl:choose>/<xsl:value-of select="@id"/>'],'name'=>'Odysee','scrape'=>[['extract'=>['#odysee\\.com/\\$/embed/(?\'name\'[^:/]+)/(?\'id\'\\w{40})#'],'match'=>['#odysee\\.com/@[^/:]+:\\w+/.#']]],'tags'=>['videos']],
		'orfium'=>['attributes'=>[],'example'=>['https://www.orfium.com/album/24371/everybody-loves-kanye-totom/','https://www.orfium.com/live-set/614763/foof-no-lights-5-foof/','https://www.orfium.com/playlist/511651/electronic-live-sessions-creamtronic/','https://www.orfium.com/track/625367/the-ambience-of-the-goss-vistas/'],'extract'=>['@album/(?\'album_id\'\\d+)@','@playlist/(?\'playlist_id\'\\d+)@','@live-set/(?\'set_id\'\\d+)@','@track/(?\'track_id\'\\d+)@'],'homepage'=>'https://www.orfium.com/','host'=>['orfium.com'],'iframe'=>['height'=>'<xsl:choose><xsl:when test="@album_id">550</xsl:when><xsl:otherwise>275</xsl:otherwise></xsl:choose>','max-width'=>900,'src'=>'https://www.orfium.com/embedded/<xsl:choose><xsl:when test="@album_id">album/<xsl:value-of select="@album_id"/></xsl:when><xsl:when test="@playlist_id">playlist/<xsl:value-of select="@playlist_id"/></xsl:when><xsl:when test="@set_id">live-set/<xsl:value-of select="@set_id"/></xsl:when><xsl:otherwise>track/<xsl:value-of select="@track_id"/></xsl:otherwise></xsl:choose>','width'=>'100%'],'name'=>'Orfium','scrape'=>[],'tags'=>['music']],
		'pastebin'=>['attributes'=>[],'example'=>'https://pastebin.com/9jEf44nc','extract'=>['@pastebin\\.com/(?!u/)(?:\\w+(?:\\.php\\?i=|/))?(?\'id\'\\w+)@'],'homepage'=>'https://pastebin.com/','host'=>['pastebin.com'],'iframe'=>['height'=>300,'scrolling'=>'','src'=>'//pastebin.com/embed_iframe.php?i={@id}','style'=>['resize'=>'vertical'],'width'=>'100%'],'name'=>'Pastebin','scrape'=>[]],
		'pinterest'=>['amp'=>['custom-element'=>'amp-pinterest','src'=>'https://cdn.ampproject.org/v0/amp-pinterest-0.1.js','template'=>'<amp-pinterest width="245" height="330" data-do="embedPin" data-url="https://www.pinterest.com/pin/{@id}/"/>'],'attributes'=>[],'example'=>['https://www.pinterest.com/pin/99360735500167749/','https://www.pinterest.com/pinterest/official-news/'],'extract'=>['@pinterest.com/pin/(?\'id\'\\d+)@','@pinterest.com/(?!_/|discover/|explore/|news_hub/|pin/|search/)(?\'id\'[-\\w]+/[-\\w]+)@'],'homepage'=>'https://www.pinterest.com/','host'=>['pinterest.com'],'iframe'=>['data-s9e-livepreview-ignore-attrs'=>'style','height'=>360,'onload'=>'var c=new MessageChannel;c.port1.onmessage=function(e){{style.height=e.data+\'px\'}};contentWindow.postMessage(\'s9e:init\',\'https://s9e.github.io\',[c.port2])','src'=>'https://s9e.github.io/iframe/2/pinterest.min.html#{@id}','width'=>'<xsl:choose><xsl:when test="contains(@id,\'/\')">730</xsl:when><xsl:otherwise>345</xsl:otherwise></xsl:choose>'],'name'=>'Pinterest','scrape'=>[],'source'=>'https://developers.pinterest.com/tools/widget-builder/','tags'=>['social']],
		'podbean'=>['attributes'=>[],'example'=>['https://dialhforheroclix.podbean.com/e/dial-h-for-heroclix-episode-46-all-ya-need-is-love/','https://www.podbean.com/ew/pb-gmyyz-d6e065'],'extract'=>['!podbean\\.com/(?:[-\\w]+/)*(?:player[-\\w]*/|\\w+/pb-)(?\'id\'[-\\w]+)!'],'homepage'=>'https://www.podbean.com/','host'=>['podbean.com'],'iframe'=>['height'=>150,'max-width'=>900,'src'=>'https://www.podbean.com/player-v2/?i={@id}','width'=>'100%'],'name'=>'Podbean','scrape'=>[['extract'=>['!podbean\\.com/player[^/]*/\\?i=(?\'id\'[-\\w]+)!'],'match'=>['@podbean\\.com/(?:media/shar)?e/(?!pb-)@']]],'tags'=>['podcasts']],
		'prezi'=>['attributes'=>[],'example'=>'https://prezi.com/5ye8po_hmikp/10-most-common-rookie-presentation-mistakes/','extract'=>['#//prezi\\.com/(?!(?:a(?:bout|mbassadors)|c(?:o(?:llaborate|mmunity|ntact)|reate)|exp(?:erts|lore)|ip(?:ad|hone)|jobs|l(?:ear|ogi)n|m(?:ac|obility)|pr(?:es(?:s|ent)|icing)|recommend|support|user|windows|your)/)(?\'id\'\\w+)/#'],'homepage'=>'https://prezi.com/','host'=>['prezi.com'],'iframe'=>['height'=>400,'src'=>'//prezi.com/embed/{@id}/','width'=>550],'name'=>'Prezi','scrape'=>[],'tags'=>['presentations']],
		'reddit'=>['amp'=>['custom-element'=>'amp-reddit','src'=>'https://cdn.ampproject.org/v0/amp-reddit-0.1.js','template'=>'<amp-reddit layout="responsive" width="450" height="300" data-embedtype="post" data-src="https://www.reddit.com/r/{@id}/"/>'],'attributes'=>[],'example'=>['https://www.reddit.com/r/pics/comments/9wp1qg/my_mom_recently_started_watercolor_painting_and/','https://www.reddit.com/r/pics/comments/9wp1qg/my_mom_recently_started_watercolor_painting_and/e9m6pn9/'],'extract'=>['!(?\'id\'\\w+/comments/\\w+(?:/\\w+/\\w+)?)!'],'homepage'=>'https://www.reddit.com/','host'=>['reddit.com'],'iframe'=>['data-s9e-livepreview-ignore-attrs'=>'style','height'=>165,'onload'=>'var c=new MessageChannel;c.port1.onmessage=function(e){{style.height=e.data+\'px\'}};contentWindow.postMessage(\'s9e:init\',\'https://s9e.github.io\',[c.port2])','src'=>'https://s9e.github.io/iframe/2/reddit.min.html#{@id}{@path}#theme={$MEDIAEMBED_THEME}','width'=>800],'name'=>'Reddit threads and comments','scrape'=>[],'source'=>'https://www.reddit.com/wiki/embeds','tags'=>['social']],
		'rumble'=>['attributes'=>[],'example'=>'https://rumble.com/vaxo02-i-cant-climb-.html','extract'=>['!rumble\\.com/embed/(?\'id\'\\w+)!'],'homepage'=>'https://rumble.com/','host'=>['rumble.com'],'iframe'=>['src'=>'https://rumble.com/embed/{@id}/'],'name'=>'Rumble','scrape'=>[['extract'=>['!video"?:"(?\'id\'\\w+)!'],'match'=>['#rumble\\.com/(?!embed/).#']]],'tags'=>['videos']],
		'rutube'=>['attributes'=>[],'example'=>['https://rutube.ru/play/embed/4782775','https://rutube.ru/tracks/4118278.html?v=8b490a46447720d4ad74616f5de2affd'],'extract'=>['!rutube\\.ru/(?:play/embed|tracks)/(?\'id\'\\d+)!'],'homepage'=>'https://rutube.ru/','host'=>['rutube.ru'],'iframe'=>['height'=>405,'src'=>'//rutube.ru/play/embed/{@id}','width'=>720],'name'=>'Rutube','scrape'=>[['extract'=>['!rutube\\.ru/play/embed/(?\'id\'\\d+)!'],'header'=>'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/89.0.0.0 Safari/537.36','match'=>['!rutube\\.ru/video/(?\'vid\'[0-9a-f]{32})!'],'url'=>'https://rutube.ru/api/oembed/?url=https://rutube.ru/video/{@vid}/']],'tags'=>['.ru']],
		'scribd'=>['attributes'=>[],'example'=>['https://www.scribd.com/doc/237147661/Calculus-2-Test-1-Review','https://www.scribd.com/presentation/324333881/600-1450-World-History-Jeopardy'],'extract'=>['!scribd\\.com/(?:mobile/)?(?:doc(?:ument)?|presentation)/(?\'id\'\\d+)!'],'homepage'=>'https://www.scribd.com/','host'=>['scribd.com'],'iframe'=>['height'=>500,'src'=>'https://www.scribd.com/embeds/{@id}/content?view_mode=scroll&show_recommendations=false','style'=>['resize'=>'vertical'],'width'=>'100%'],'name'=>'Scribd','scrape'=>[],'tags'=>['documents','presentations']],
		'sendvid'=>['attributes'=>[],'example'=>'https://sendvid.com/w99u3jjp','extract'=>['!sendvid\\.com/(?\'id\'\\w+)!'],'homepage'=>'https://www.sendvid.com/','host'=>['sendvid.com'],'iframe'=>['src'=>'//sendvid.com/embed/{@id}'],'name'=>'Sendvid','scrape'=>[],'tags'=>['videos']],
		'slideshare'=>['attributes'=>[],'example'=>'https://www.slideshare.net/Slideshare/how-23431564','extract'=>['!slideshare\\.net/[^/]+/[-\\w]+-(?\'id\'\\d{6,})$!'],'homepage'=>'https://www.slideshare.net/','host'=>['slideshare.net'],'iframe'=>['height'=>356,'src'=>'//www.slideshare.net/slideshow/embed_code/{@id}','width'=>427],'name'=>'SlideShare','scrape'=>[['extract'=>['!"presentationId":(?\'id\'\\d+)!'],'match'=>['@slideshare\\.net/[^/]+/\\w(?![-\\w]+-\\d{6,}$)@']]],'source'=>'http://help.slideshare.com/forums/67665-Embedding-Sharing','tags'=>['presentations']],
		'soundcloud'=>['amp'=>['custom-element'=>'amp-soundcloud','src'=>'https://cdn.ampproject.org/v0/amp-soundcloud-0.1.js','template'=>'<amp-soundcloud layout="fixed-height" height="150" data-playlistid="{@playlist_id}" data-secret-token="{@secret_token}" data-trackid="{@track_id}"/>'],'attributes'=>[],'example'=>['https://api.soundcloud.com/tracks/98282116','https://soundcloud.com/andrewbird/three-white-horses','https://soundcloud.com/tenaciousd/sets/rize-of-the-fenix/'],'extract'=>['@https?://(?:api\\.)?soundcloud\\.com/(?!pages/)(?\'id\'[-/\\w]+/[-/\\w]+|^[^/]+/[^/]+$)@i','@api\\.soundcloud\\.com/playlists/(?\'playlist_id\'\\d+)@','@api\\.soundcloud\\.com/tracks/(?\'track_id\'\\d+)(?:\\?secret_token=(?\'secret_token\'[-\\w]+))?@','@soundcloud\\.com/(?!playlists|tracks)[-\\w]+/[-\\w]+/(?=s-)(?\'secret_token\'[-\\w]+)@'],'homepage'=>'https://soundcloud.com/','host'=>['soundcloud.com'],'iframe'=>['height'=>'<xsl:choose><xsl:when test="@playlist_id or contains(@id,\'/sets/\')">450</xsl:when><xsl:otherwise>166</xsl:otherwise></xsl:choose>','max-width'=>900,'src'=>'https://w.soundcloud.com/player/?url=<xsl:choose><xsl:when test="@playlist_id">https%3A//api.soundcloud.com/playlists/<xsl:value-of select="@playlist_id"/></xsl:when><xsl:when test="@track_id">https%3A//api.soundcloud.com/tracks/<xsl:value-of select="@track_id"/>&amp;secret_token=<xsl:value-of select="@secret_token"/></xsl:when><xsl:otherwise><xsl:if test="not(contains(@id,\'://\'))">https%3A//soundcloud.com/</xsl:if><xsl:value-of select="@id"/></xsl:otherwise></xsl:choose>','width'=>'100%'],'name'=>'SoundCloud','scrape'=>[['extract'=>['@soundcloud:tracks:(?\'track_id\'\\d+)@'],'header'=>'User-agent: PHP (not Mozilla)','match'=>['@soundcloud\\.com/(?!playlists/\\d|tracks/\\d)[-\\w]+/[-\\w]@']],['extract'=>['@soundcloud://playlists:(?\'playlist_id\'\\d+)@'],'header'=>'User-agent: PHP (not Mozilla)','match'=>['@soundcloud\\.com/[-\\w]+/sets/@']]],'source'=>'https://soundcloud.com/pages/widgets','tags'=>['music']],
		'sporcle'=>['attributes'=>[],'example'=>'https://www.sporcle.com/games/bowsntoys/mothers-day-jokes','extract'=>['#sporcle.com/framed/.*?gid=(?\'id\'\\w+)#'],'homepage'=>'https://www.sporcle.com/','host'=>['sporcle.com'],'iframe'=>['data-s9e-livepreview-ignore-attrs'=>'style','height'=>535,'onload'=>'var c=new MessageChannel;c.port1.onmessage=function(e){{style.height=e.data+\'px\'}};contentWindow.postMessage(\'s9e:init\',\'*\',[c.port2])','src'=>'https://s9e.github.io/iframe/2/sporcle.min.html#{@id}','width'=>820],'name'=>'Sporcle','scrape'=>[['extract'=>['#encodedGameID\\W+(?\'id\'\\w+)#'],'match'=>['#sporcle\\.com/games/(?!\\w*category/)[-\\w]+/[-\\w]#']]],'tags'=>['entertainment']],
		'sportsnet'=>['attributes'=>[],'example'=>'https://www.sportsnet.ca/soccer/west-ham-2-hull-2/','extract'=>[],'homepage'=>'https://www.sportsnet.ca/','host'=>['sportsnet.ca'],'iframe'=>['src'=>'//players.brightcove.net/1704050871/rkedLxwfab_default/index.html?videoId={@id}'],'name'=>'Sportsnet','scrape'=>[['extract'=>['@bc_videos\\s*:\\s*(?\'id\'\\d+)@'],'match'=>['//']]],'tags'=>['.ca','sports']],
		'spotify'=>['attributes'=>[],'choose'=>['otherwise'=>['iframe'=>['allow'=>'encrypted-media','height'=>320,'padding-height'=>80,'src'=>'https://open.spotify.com/embed/{translate(@id,\':\',\'/\')}{@path}','width'=>320]],'when'=>['iframe'=>['allow'=>'encrypted-media','height'=>152,'max-width'=>900,'src'=>'https://open.spotify.com/embed/{@id}','width'=>'100%'],'test'=>'starts-with(@id, \'episode/\') or starts-with(@id, \'show/\')']],'example'=>['https://open.spotify.com/user/ryan.lloyd.french/playlist/4jHNeLTj7CnL5JwZzmnqOL','https://open.spotify.com/track/3CPjqkaMwrbht1lJpELoY0?si=K7cm7-dWTR6U7BjDn1oP7g','https://open.spotify.com/episode/4zhSJwkVdndcVvwCypfl9b'],'extract'=>['!(?:open|play)\\.spotify\\.com/(?:user/[-.\\w]+/)?(?\'id\'(?:album|artist|episode|playlist|show|track)(?:[:/][-.\\w]+)+)!'],'homepage'=>'https://www.spotify.com/','host'=>['link.tospotify.com','open.spotify.com','play.spotify.com'],'name'=>'Spotify','scrape'=>[['extract'=>['!(?:open|play)\\.spotify\\.com/(?:user/[-.\\w]+/)?(?\'id\'(?:album|artist|episode|playlist|show|track)(?:[:/][-.\\w]+)+)!'],'header'=>'User-agent: PHP (not Mozilla)','match'=>['!link\\.tospotify\\.com/.!']]],'source'=>'https://developer.spotify.com/technologies/widgets/spotify-play-button/','tags'=>['music','podcasts']],
		'spreaker'=>['attributes'=>[],'example'=>['https://www.spreaker.com/show/2-up-front_1','https://www.spreaker.com/user/publichousemedia/2-up-front-119-final-show-peter-wilt-rac'],'extract'=>['!spreaker\\.com/episode/(?\'episode_id\'\\d+)!'],'homepage'=>'https://www.spreaker.com','host'=>['spreaker.com'],'iframe'=>['height'=>'{400 - 200 * boolean(@episode_id)}','max-width'=>900,'src'=>'https://widget.spreaker.com/player?episode_id={@episode_id}&show_id={@show_id}&theme={$MEDIAEMBED_THEME}','width'=>'100%'],'name'=>'Spreaker','scrape'=>[['extract'=>['!episode_id=(?\'episode_id\'\\d+)!','!show_id=(?\'show_id\'\\d+)!'],'match'=>['!(?\'url\'.+/(?:show/|user/.+/).+)!'],'url'=>'https://api.spreaker.com/oembed?format=json&url={@url}']],'tags'=>['podcasts']],
		'steamstore'=>['attributes'=>[],'example'=>'https://store.steampowered.com/app/250520/','extract'=>['!store.steampowered.com/app/(?\'id\'\\d+)!'],'homepage'=>'https://store.steampowered.com/','host'=>['store.steampowered.com'],'iframe'=>['height'=>190,'max-width'=>900,'src'=>'//store.steampowered.com/widget/{@id}','width'=>'100%'],'name'=>'Steam store','scrape'=>[],'tags'=>['gaming']],
		'stitcher'=>['attributes'=>[],'example'=>'https://www.stitcher.com/show/national-geographics-overheard/episode/chasing-the-worlds-largest-tornado-79283587','extract'=>['!/(?:embed|splayer/f)/(?\'fid\'\\d+)/(?\'eid\'\\d+)!'],'homepage'=>'https://www.stitcher.com/','host'=>['stitcher.com'],'iframe'=>['height'=>150,'max-width'=>900,'src'=>'https://www.stitcher.com/embed/{@fid}/{@eid}','width'=>'100%'],'name'=>'Stitcher','scrape'=>[['extract'=>['!/embed/(?\'fid\'\\d+)/(?\'eid\'\\d+)!'],'match'=>['!/(?:podcast/|show/|s\\?)!']]],'tags'=>['podcasts']],
		'strawpoll'=>['attributes'=>[],'example'=>'https://strawpoll.me/738091','extract'=>['!strawpoll\\.me/(?\'id\'\\d+)!'],'homepage'=>'https://strawpoll.me/','host'=>['strawpoll.me'],'iframe'=>['scrolling'=>'','src'=>'//www.strawpoll.me/embed_1/{@id}'],'name'=>'Straw Poll','scrape'=>[]],
		'streamable'=>['attributes'=>[],'example'=>'https://streamable.com/e4d','extract'=>['!streamable\\.com/(?:e/)?(?\'id\'\\w+)!'],'homepage'=>'https://streamable.com/','host'=>['streamable.com'],'iframe'=>['src'=>'//streamable.com/e/{@id}'],'name'=>'Streamable','oembed'=>['endpoint'=>'https://api.streamable.com/oembed.json','scheme'=>'https://streamable.com/{@id}'],'scrape'=>[],'source'=>'https://support.streamable.com/article/46-streamable-api','tags'=>['videos']],
		'streamja'=>['attributes'=>[],'example'=>'https://streamja.com/2nr','extract'=>['@streamja\\.com/(?!login|signup|terms|videos)(?\'id\'\\w+)@'],'homepage'=>'https://streamja.com','host'=>['streamja.com'],'iframe'=>['src'=>'https://streamja.com/embed/{@id}'],'name'=>'Streamja','scrape'=>[],'tags'=>['sports']],
		'teamcoco'=>['attributes'=>[],'example'=>['https://teamcoco.com/video/serious-jibber-jabber-a-scott-berg-full-episode','https://teamcoco.com/video/73784/historian-a-scott-berg-serious-jibber-jabber-with-conan-obrien'],'extract'=>['!teamcoco\\.com/video/(?\'id\'\\d+)!'],'homepage'=>'https://teamcoco.com/','host'=>['teamcoco.com'],'iframe'=>['src'=>'//teamcoco.com/embed/v/{@id}'],'name'=>'Team Coco','scrape'=>[['extract'=>['!embed/v/(?\'id\'\\d+)!'],'match'=>['!teamcoco\\.com/video/\\D!']]],'tags'=>['entertainment']],
		'ted'=>['attributes'=>[],'example'=>['https://www.ted.com/talks/eli_pariser_beware_online_filter_bubbles.html','https://embed.ted.com/playlists/26/our_digital_lives.html'],'extract'=>['#ted\\.com/(?\'id\'(?:talk|playlist)s/[-\\w]+(?:\\.html)?)(?![-\\w]|/transcript)#i'],'homepage'=>'https://www.ted.com/','host'=>['ted.com'],'iframe'=>['src'=>'//embed.ted.com/<xsl:value-of select="@id"/><xsl:if test="not(contains(@id,\'.html\'))">.html</xsl:if>'],'name'=>'TED Talks','scrape'=>[],'source'=>'http://blog.ted.com/2011/04/01/now-you-can-embed-tedtalks-with-subtitles-enabled/','tags'=>['presentations']],
		'telegram'=>['attributes'=>[],'example'=>'https://t.me/telegram/83','extract'=>['@//t.me/(?!addstickers/|joinchat/)(?\'id\'\\w+/\\d+)@'],'homepage'=>'https://telegram.org/','host'=>['t.me'],'iframe'=>['data-s9e-livepreview-ignore-attrs'=>'style','height'=>96,'onload'=>'var c=new MessageChannel;c.port1.onmessage=function(e){{style.height=e.data+\'px\'}};contentWindow.postMessage(\'s9e:init\',\'https://s9e.github.io\',[c.port2])','src'=>'https://s9e.github.io/iframe/2/telegram.min.html#<xsl:value-of select="@id"/><xsl:if test="$MEDIAEMBED_THEME=\'dark\'">#theme=dark</xsl:if>','width'=>500],'name'=>'Telegram','scrape'=>[],'tags'=>['social']],
		'theatlantic'=>['attributes'=>[],'example'=>'https://www.theatlantic.com/video/index/358928/computer-vision-syndrome-and-you/','extract'=>['!theatlantic\\.com/video/index/(?\'id\'\\d+)!'],'homepage'=>'https://www.theatlantic.com/video/','host'=>['theatlantic.com'],'iframe'=>['src'=>'//www.theatlantic.com/video/iframe/{@id}/'],'name'=>'The Atlantic Video','scrape'=>[],'tags'=>['news']],
		'theguardian'=>['attributes'=>[],'example'=>'https://www.theguardian.com/world/video/2016/apr/07/tokyos-hedgehog-cafe-encourages-you-to-embrace-prickly-pets-video','extract'=>['!theguardian\\.com/(?\'id\'\\w+/video/20(?:0[0-9]|1[0-7])[-/\\w]+)!'],'homepage'=>'https://www.theguardian.com/video','host'=>['theguardian.com'],'iframe'=>['src'=>'//embed.theguardian.com/embed/video/{@id}'],'name'=>'The Guardian (obsolete)','scrape'=>[],'tags'=>['news']],
		'theonion'=>['attributes'=>[],'example'=>['https://www.theonion.com/video/nation-successfully-completes-mothers-day-by-918-a,35998/','https://www.theonion.com/video/the-onion-reviews-avengers-age-of-ultron-38524'],'extract'=>['!theonion\\.com/video/[-\\w]+[-,](?\'id\'\\d+)!'],'homepage'=>'https://www.theonion.com/video/','host'=>['theonion.com'],'iframe'=>['src'=>'//www.theonion.com/video_embed/?id={@id}'],'name'=>'The Onion','scrape'=>[],'tags'=>['entertainment']],
		'tiktok'=>['attributes'=>[],'example'=>['https://www.tiktok.com/@lauren.feagans/video/6789430799839104261','https://vm.tiktok.com/TTPdrc3YBJ/'],'extract'=>['#tiktok\\.com/(?:@[.\\w]+/video|v|i18n/share/video)/(?\'id\'\\d+)#'],'homepage'=>'https://www.tiktok.com','host'=>['tiktok.com'],'iframe'=>['height'=>700,'src'=>'//www.tiktok.com/embed/{@id}','width'=>340],'name'=>'TikTok','oembed'=>['endpoint'=>'https://www.tiktok.com/oembed?url=','scheme'=>'https://www.tiktok.com/video/{@id}'],'scrape'=>[['extract'=>['#tiktok\\.com/(?:@[.\\w]+/video|v|i18n/share/video)/(?\'id\'\\d+)#'],'match'=>['#//vm\\.tiktok\\.com/.#']]],'tags'=>['videos']],
		'tmz'=>['attributes'=>[],'example'=>'https://www.tmz.com/videos/0_2pr9x3rb/','extract'=>['@tmz\\.com/videos/(?\'id\'\\w+)@'],'homepage'=>'https://www.tmz.com/videos','host'=>['tmz.com'],'iframe'=>['src'=>'//www.kaltura.com/index.php/kwidget/cache_st/133592691/wid/_591531/partner_id/591531/uiconf_id/9071262/entry_id/{@id}'],'name'=>'TMZ','scrape'=>[],'tags'=>['gossip']],
		'tradingview'=>['attributes'=>[],'example'=>['https://www.tradingview.com/symbols/NASDAQ-AAPL/','https://www.tradingview.com/chart/US30/Yro4JNSD-us30/'],'extract'=>['!tradingview\\.com/(?:chart/[^/]+|i)/(?\'chart\'\\w+)!','!tradingview\\.com/symbols/(?\'symbol\'[-:\\w]+)!'],'homepage'=>'https://www.tradingview.com/','host'=>['tradingview.com'],'iframe'=>['height'=>400,'src'=>'https://s.tradingview.com/<xsl:choose><xsl:when test="@symbol">widgetembed/?interval=D&amp;symbol=<xsl:value-of select="translate(@symbol,\'-\',\':\')"/></xsl:when><xsl:otherwise>embed/<xsl:value-of select="@chart"/></xsl:otherwise></xsl:choose>','width'=>'100%'],'name'=>'TradingView','scrape'=>[],'tags'=>['finance']],
		'traileraddict'=>['attributes'=>[],'example'=>'https://www.traileraddict.com/the-amazing-spider-man-2/super-bowl-tv-spot','extract'=>[],'homepage'=>'https://www.traileraddict.com/','host'=>['traileraddict.com'],'iframe'=>['src'=>'//v.traileraddict.com/{@id}'],'name'=>'Trailer Addict','scrape'=>[['extract'=>['@v\\.traileraddict\\.com/(?\'id\'\\d+)@'],'match'=>['@traileraddict\\.com/(?!tags/)[^/]+/.@']]],'tags'=>['movies']],
		'trendingviews'=>['attributes'=>[],'example'=>['https://mydailyfreedom.com/tv/video/chick-fil-a-customers-sing-impromtu-lean-on-me-8213.html','https://trendingviews.co/video/chick-fil-a-customers-sing-impromtu-lean-on-me-8213.html'],'extract'=>['!(?:mydailyfreedom\\.com|trendingviews\\.co)/(?:tv/)?(?:embed|video)/(?:[^/]+-)?(?\'id\'\\d+)!'],'homepage'=>'https://mydailyfreedom.com/','host'=>['mydailyfreedom.com','trendingviews.co'],'iframe'=>['src'=>'https://mydailyfreedom.com/tv/embed/{@id}'],'name'=>'MyDailyFreedom','scrape'=>[],'tags'=>['videos']],
		'tumblr'=>['attributes'=>[],'example'=>'https://staff.tumblr.com/post/180624227620/','extract'=>['!(?\'name\'[-\\w]+)\\.tumblr\\.com/post/(?\'id\'\\d+)!'],'homepage'=>'https://www.tumblr.com/','host'=>['tumblr.com'],'iframe'=>['data-s9e-livepreview-ignore-attrs'=>'style','height'=>300,'onload'=>'var c=new MessageChannel;c.port1.onmessage=function(e){{style.height=e.data+\'px\'}};contentWindow.postMessage(\'s9e:init\',\'https://s9e.github.io\',[c.port2])','src'=>'https://s9e.github.io/iframe/2/tumblr.min.html#{@key}/{@id}','width'=>520],'name'=>'Tumblr','scrape'=>[['extract'=>['!did=\\\\u0022(?\'did\'[-\\w]+)!','!embed\\\\/post\\\\/(?\'key\'[-\\w]+)!'],'header'=>'User-agent: curl','match'=>['!\\w\\.tumblr\\.com/post/\\d!'],'url'=>'https://www.tumblr.com/oembed/1.0?url=https://{@name}.tumblr.com/post/{@id}']],'tags'=>['social']],
		'twentyfoursevensports'=>['attributes'=>[],'choose'=>['otherwise'=>['iframe'=>['height'=>200,'onload'=>'window.addEventListener(\'message\',function(e){{if(e.source===this.contentWindow&&e.data.height)this.style.height=e.data.height+\'px\'}})','src'=>'https://247sports.com/PlayerSport/{@player_id}/Embed/','width'=>600]],'when'=>['iframe'=>['src'=>'https://www.cbssports.com/video/player/embed?args=player_id%3D{@video_id}%26channel%3Dmobilevideo%26pcid%3D{@video_id}%26width%3D640%26height%3D360%26autoplay%3Dfalse%26comp_ads_enabled%3Dfalse%26uvpc%3Dhttp%3A%2F%2Fsports.cbsimg.net%2Fvideo%2Fuvp%2Fconfig%2Fv4%2Fuvp_247sports.xml%26uvpc_m%3Dhttp%3A%2F%2Fsports.cbsimg.net%2Fvideo%2Fuvp%2Fconfig%2Fv4%2Fuvp_247sports_m.xml%26partner%3D247%26partner_m%3D247_mobile%26utag%3D247sportssite%26resizable%3Dtrue'],'test'=>'@video_id']],'example'=>['https://247sports.com/Player/LeSean-McCoy-25234/','https://247sports.com/Video/FFT-Week-12-QBRB-Start-or-Sit-1122-8838631/'],'extract'=>['!247sports\\.com/PlayerSport/[-\\w]*?(?\'player_id\'\\d+)/Embed!'],'homepage'=>'https://247sports.com/','host'=>['247sports.com'],'name'=>'247Sports','scrape'=>[['extract'=>['!247sports\\.com/PlayerSport/[-\\w]*?(?\'player_id\'\\d+)/Embed!'],'header'=>'User-agent: Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/73.0.3683.86 Safari/537.36','match'=>['!247sports\\.com/Player/[-\\w]*?\\d!']],['extract'=>['!embedVideoContainer_(?\'video_id\'\\d+)!'],'header'=>'User-agent: Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/73.0.3683.86 Safari/537.36','match'=>['!247sports\\.com/Video/.!']]],'tags'=>['sports']],
		'twitch'=>['attributes'=>[],'example'=>['https://www.twitch.tv/twitch','https://www.twitch.tv/twitch/v/29415830?t=17m17s','https://www.twitch.tv/videos/29415830?t=17m17s','https://www.twitch.tv/chess/clip/CrepuscularOptimisticEagleOhMyDog-9jMi4oqxK5MN0F0C'],'extract'=>['#twitch\\.tv/(?:videos|\\w+/v)/(?\'video_id\'\\d+)?#','#www\\.twitch\\.tv/(?!videos/)(?\'channel\'\\w+)(?:/clip/(?\'clip_id\'[-\\w]+))?#','#t=(?\'t\'(?:(?:\\d+h)?\\d+m)?\\d+s)#','#clips\\.twitch\\.tv/(?:(?\'channel\'\\w+)/)?(?\'clip_id\'[-\\w]+)#'],'homepage'=>'https://www.twitch.tv/','host'=>['twitch.tv'],'iframe'=>['onload'=>'contentWindow.postMessage(\'\',\'https://s9e.github.io\')','src'=>'https://s9e.github.io/iframe/2/twitch.min.html#channel={@channel};clip_id={@clip_id};t={@t};video_id={@video_id}'],'name'=>'Twitch','scrape'=>[],'source'=>'https://dev.twitch.tv/docs/embed/video-and-clips','tags'=>['gaming','livestreaming']],
		'twitter'=>['amp'=>['custom-element'=>'amp-twitter','src'=>'https://cdn.ampproject.org/v0/amp-twitter-0.1.js','template'=>'<amp-twitter layout="responsive" width="550" height="273" data-tweetid="{@id}"><blockquote placeholder=""><a href="https://twitter.com/user/status/{@id}">Twitter</a></blockquote></amp-twitter>'],'attributes'=>[],'example'=>['https://twitter.com/IJasonAlexander/statuses/526635414338023424','https://mobile.twitter.com/DerekTVShow/status/463372588690202624','https://twitter.com/#!/IJasonAlexander/status/526635414338023424'],'extract'=>['@twitter\\.com/(?:#!/|i/)?\\w+/(?:status(?:es)?|tweet)/(?\'id\'\\d+)@'],'homepage'=>'https://twitter.com/','host'=>['twitter.com'],'iframe'=>['allow'=>'autoplay *','data-s9e-livepreview-ignore-attrs'=>'style','height'=>273,'onload'=>'var c=new MessageChannel;c.port1.onmessage=function(e){{style.height=e.data+\'px\'}};contentWindow.postMessage(\'s9e:init\',\'https://s9e.github.io\',[c.port2])','src'=>'https://s9e.github.io/iframe/2/twitter.min.html#<xsl:value-of select="@id"/><xsl:if test="$MEDIAEMBED_THEME=\'dark\'">#theme=dark</xsl:if>','width'=>550],'name'=>'Twitter','scrape'=>[],'tags'=>['social']],
		'ustream'=>['attributes'=>[],'choose'=>['otherwise'=>['iframe'=>['src'=>'//www.ustream.tv/embed/{@cid}?html5ui']],'when'=>['iframe'=>['src'=>'//www.ustream.tv/embed/recorded/{@vid}?html5ui'],'test'=>'@vid']],'example'=>['https://www.ustream.tv/channel/ps4-ustream-gameplay','https://www.ustream.tv/baja1000tv','https://www.ustream.tv/recorded/40688256'],'extract'=>['!ustream\\.tv/recorded/(?\'vid\'\\d+)!'],'homepage'=>'https://www.ustream.tv/','host'=>['ustream.tv'],'name'=>'Ustream','scrape'=>[['extract'=>['!embed/(?\'cid\'\\d+)!'],'match'=>['#ustream\\.tv/(?!explore/|platform/|recorded/|search\\?|upcoming$|user/)(?:channel/)?[-\\w]+#']]],'tags'=>['gaming']],
		'vbox7'=>['attributes'=>[],'example'=>'https://vbox7.com/play:3975300ec6','extract'=>['!vbox7\\.com/play:(?\'id\'[\\da-f]+)!'],'homepage'=>'https://vbox7.com/','host'=>['vbox7.com'],'iframe'=>['src'=>'//vbox7.com/emb/external.php?vid={@id}'],'name'=>'VBOX7','scrape'=>[],'tags'=>['.bg']],
		'veoh'=>['attributes'=>[],'example'=>'https://www.veoh.com/watch/v6335577TeB8kyNR','extract'=>['!veoh\\.com/(?:m/watch\\.php\\?v=|watch/)v(?\'id\'\\w+)!'],'flash'=>['padding-height'=>40,'src'=>'//www.veoh.com/swf/webplayer/WebPlayer.swf?version=AFrontend.5.7.0.1509&permalinkId=v{@id}&player=videodetailsembedded&videoAutoPlay=0&id=anonymous'],'homepage'=>'https://www.veoh.com/','host'=>['veoh.com'],'name'=>'Veoh','scrape'=>[],'tags'=>['videos']],
		'vevo'=>['attributes'=>[],'example'=>['https://www.vevo.com/watch/USUV71400682','https://www.vevo.com/watch/eminem/the-monster-explicit/USUV71302925'],'extract'=>['!vevo\\.com/watch/(.*?/)?(?\'id\'[A-Z]+\\d+)!'],'homepage'=>'https://vevo.com/','host'=>['vevo.com'],'iframe'=>['src'=>'https://embed.vevo.com/?isrc={@id}'],'name'=>'Vevo','scrape'=>[],'tags'=>['music']],
		'videodetective'=>['attributes'=>[],'example'=>'https://www.videodetective.com/movies/zootopia/658596','extract'=>['!videodetective\\.com/\\w+/[-\\w]+/(?:trailer/P0*)?(?\'id\'\\d+)!'],'homepage'=>'https://www.videodetective.com/','host'=>['videodetective.com'],'iframe'=>['src'=>'//www.videodetective.com/embed/video/?options=false&autostart=false&playlist=none&publishedid={@id}'],'name'=>'Video Detective','scrape'=>[]],
		'vimeo'=>['amp'=>['custom-element'=>'amp-vimeo','src'=>'https://cdn.ampproject.org/v0/amp-vimeo-0.1.js','template'=>'<amp-vimeo layout="responsive" width="640" height="360" data-videoid="{@id}"/>'],'attributes'=>['t'=>['filterChain'=>['#timestamp']]],'example'=>['https://vimeo.com/67207222','https://vimeo.com/channels/staffpicks/67207222'],'extract'=>['!vimeo\\.com/(?:channels/[^/]+/|video/)?(?\'id\'\\d+)(?:/(?\'h\'\\w+))?\\b!','!#t=(?\'t\'[\\dhms]+)!'],'homepage'=>'https://vimeo.com/','host'=>['vimeo.com'],'iframe'=>['src'=>'//player.vimeo.com/video/<xsl:value-of select="@id"/><xsl:if test="@h">?h=<xsl:value-of select="@h"/></xsl:if><xsl:if test="@t">#t=<xsl:value-of select="@t"/></xsl:if>'],'name'=>'Vimeo','oembed'=>['endpoint'=>'https://vimeo.com/api/oembed.json','scheme'=>'https://vimeo.com/{@id}'],'scrape'=>[],'source'=>'http://developer.vimeo.com/player/embedding','tags'=>['videos']],
		'vine'=>['amp'=>['custom-element'=>'amp-vine','src'=>'https://cdn.ampproject.org/v0/amp-vine-0.1.js','template'=>'<amp-vine layout="responsive" width="400" height="400" data-vineid="{@id}"/>'],'attributes'=>[],'example'=>'https://vine.co/v/bYwPIluIipH','extract'=>['!vine\\.co/v/(?\'id\'[^/]+)!'],'homepage'=>'https://vine.co/','host'=>['vine.co'],'iframe'=>['height'=>480,'src'=>'https://vine.co/v/{@id}/embed/simple?audio=1','width'=>480],'name'=>'Vine','scrape'=>[],'tags'=>['social','videos']],
		'vk'=>['attributes'=>[],'example'=>['https://vk.com/video121599878_165723901?hash=e06b0878046e1d32','https://vk.com/video_ext.php?oid=121599878&id=165723901&hash=e06b0878046e1d32'],'extract'=>['!vk(?:\\.com|ontakte\\.ru)/(?:[\\w.]+\\?z=)?video(?\'oid\'-?\\d+)_(?\'vid\'\\d+).*?hash=(?\'hash\'[0-9a-f]+)!','!vk(?:\\.com|ontakte\\.ru)/video_ext\\.php\\?oid=(?\'oid\'-?\\d+)&id=(?\'vid\'\\d+)&hash=(?\'hash\'[0-9a-f]+)!'],'homepage'=>'https://vk.com/','host'=>['vk.com','vkontakte.ru'],'iframe'=>['src'=>'//vk.com/video_ext.php?oid={@oid}&id={@vid}&hash={@hash}&hd=1'],'name'=>'VK','scrape'=>[['extract'=>['#meta property="og:video" content=".*?oid=(?\'oid\'-?\\d+).*?id=(?\'vid\'\\d+).*?hash=(?\'hash\'[0-9a-f]+)#'],'header'=>'User-agent: Mozilla/5.0 (X11; Linux x86_64; rv:62.0) Gecko/20100101 Firefox/62.0','match'=>['#^(?!.*?hash=)#']]],'tags'=>['.ru']],
		'vocaroo'=>['attributes'=>[],'example'=>'https://vocaroo.com/3OlvdB6eiiK','extract'=>['!voca(?:\\.ro|roo\\.com)/(?:i/)?(?\'id\'\\w+)!'],'homepage'=>'https://vocaroo.com/','host'=>['voca.ro','vocaroo.com'],'iframe'=>['height'=>80,'max-width'=>900,'src'=>'https://vocaroo.com/embed/{@id}','width'=>'100%'],'name'=>'Vocaroo','scrape'=>[]],
		'vox'=>['attributes'=>[],'example'=>'https://www.vox.com/2015/7/21/9005857/ant-man-marvel-apology-review#ooid=ltbzJkdTpKpE-O6hOfD3YJew3t3MppXb','extract'=>['!vox.com/.*#ooid=(?\'id\'[-\\w]+)!'],'homepage'=>'https://www.vox.com/','host'=>['vox.com'],'iframe'=>['src'=>'//player.ooyala.com/iframe.html#pbid=a637d53c5c0a43c7bf4e342886b9d8b0&ec={@id}'],'name'=>'Vox','scrape'=>[]],
		'washingtonpost'=>['attributes'=>[],'example'=>['https://www.washingtonpost.com/video/c/video/df229384-9216-11e6-bc00-1a9756d4111b','https://www.washingtonpost.com/video/world/aurora-display-lights-up-the-night-sky-over-finland/2016/10/14/df229384-9216-11e6-bc00-1a9756d4111b_video.html'],'extract'=>['#washingtonpost\\.com/video/c/\\w+/(?\'id\'[-0-9a-f]+)#','#washingtonpost\\.com/video/[-/\\w]+/(?\'id\'[-0-9a-f]+)_video\\.html#'],'homepage'=>'https://www.washingtonpost.com/video/','host'=>['washingtonpost.com'],'iframe'=>['src'=>'//www.washingtonpost.com/video/c/embed/{@id}'],'name'=>'Washington Post Video','scrape'=>[],'tags'=>['news']],
		'wistia'=>['amp'=>['custom-element'=>'amp-wistia-player','src'=>'https://cdn.ampproject.org/v0/amp-wistia-player-0.1.js','template'=>'<amp-wistia-player layout="responsive" width="640" height="360" data-media-hashed-id="{@id}"/>'],'attributes'=>[],'choose'=>['otherwise'=>['iframe'=>['src'=>'https://fast.wistia.net/embed/iframe/{@id}']],'when'=>['iframe'=>['height'=>218,'max-width'=>900,'src'=>'https://fast.wistia.net/embed/iframe/{@id}','width'=>'100%'],'test'=>'@type = \'audio\'']],'example'=>['https://thoughtworks.wistia.com/medias/b6al55s35k','https://fast.wistia.com/embed/iframe/od5qgyo2uy'],'extract'=>['!wistia.com/(?:(?:embed/iframe|medias)/|.*wmediaid=)(?\'id\'\\w+)!'],'homepage'=>'https://wistia.com/','host'=>['wistia.com'],'name'=>'Wistia','scrape'=>[['extract'=>['!"type":"(?:\\w+_)?(?\'type\'audio)!'],'match'=>['!wistia.com/(?:(?:embed/iframe|medias)/|.*wmediaid=)\\w!'],'url'=>'https://fast.wistia.net/embed/iframe/{@id}']],'tags'=>['podcasts','videos']],
		'wshh'=>['attributes'=>[],'example'=>['https://worldstar.com/video.php?v=wshhn7nmk3033k8m66ZG','https://www.worldstarhiphop.com/featured/71630'],'extract'=>['!worldstar(?:hiphop)?\\.com/(?:emb|featur)ed/(?\'id\'\\d+)!'],'homepage'=>'https://www.worldstarhiphop.com/','host'=>['worldstar.com','worldstarhiphop.com'],'iframe'=>['src'=>'//worldstarhiphop.com/embed/{@id}'],'name'=>'WorldStarHipHop','scrape'=>[['extract'=>['!(?:v: ?"?|worldstar(?:hiphop)?\\.com/embed/)(?\'id\'\\d+)!'],'match'=>['!worldstar(?:hiphop)?\\.com/(?:\\w+/)?video\\.php\\?v=\\w+!']]],'tags'=>['videos']],
		'wsj'=>['attributes'=>[],'example'=>['https://www.wsj.com/video/nba-players-primp-with-pedicures/9E476D54-6A60-4F3F-ABC1-411014552DE6.html','https://live.wsj.com/#!09FB2B3B-583E-4284-99D8-FEF6C23BE4E2','https://live.wsj.com/video/seahawks-qb-russell-wilson-on-super-bowl-win/9B3DF790-9D20-442C-B564-51524B06FD26.html'],'extract'=>['@wsj\\.com/[^#]*#!(?\'id\'[-0-9A-F]{36})@','@wsj\\.com/video/[^/]+/(?\'id\'[-0-9A-F]{36})@'],'homepage'=>'https://www.wsj.com/video/','host'=>['wsj.com'],'iframe'=>['height'=>288,'src'=>'//video-api.wsj.com/api-video/player/iframe.html?guid={@id}','width'=>512],'name'=>'The Wall Street Journal Online','scrape'=>[['extract'=>['@guid=(?\'id\'[-0-9A-F]{36})@'],'match'=>['@on\\.wsj\\.com/\\w@']]],'tags'=>['news']],
		'xboxclips'=>['attributes'=>[],'example'=>'https://gameclips.io/boulderBaby5568/035a50fa-2d54-4820-aa44-f0f43a873308','extract'=>['@(?:gameclips\\.io|xboxclips\\.com)/(?!game/)(?\'user\'[^/]+)/(?!screenshots/)(?\'id\'[-0-9a-f]+)@'],'homepage'=>'https://gameclips.io/','host'=>['gameclips.io','xboxclips.com'],'iframe'=>['height'=>315,'src'=>'//gameclips.io/{@user}/{@id}/embed','width'=>560],'name'=>'GameClips.io','scrape'=>[],'tags'=>['gaming']],
		'xboxdvr'=>['attributes'=>[],'example'=>'https://gamerdvr.com/gamer/LOXITANE/video/12463958','extract'=>['!(?:gamer|xbox)dvr\\.com/gamer/(?\'user\'[^/]+)/video/(?\'id\'\\d+)!'],'homepage'=>'https://gamerdvr.com/','host'=>['gamerdvr.com','xboxdvr.com'],'iframe'=>['src'=>'//gamerdvr.com/gamer/{@user}/video/{@id}/embed'],'name'=>'Gamer DVR','scrape'=>[],'tags'=>['gaming']],
		'youku'=>['attributes'=>[],'example'=>'https://v.youku.com/v_show/id_XMzY0NTMyMTgxMg==.html','extract'=>['!youku\\.com/v(?:_show|ideo)/id_(?\'id\'\\w+=*)!'],'homepage'=>'https://www.youku.com/','host'=>['youku.com'],'iframe'=>['src'=>'//player.youku.com/embed/{@id}'],'name'=>'Youku','scrape'=>[],'tags'=>['.cn']],
		'youmaker'=>['attributes'=>[],'example'=>['https://www.youmaker.com/v/EnDXOWo8OOvQ','https://www.youmaker.com/video/b3ce8457-2cbe-4661-84ec-829fa8fe0754'],'extract'=>['!youmaker\\.com/(?:embed|v(?:ideo)?)/(?\'id\'[-a-z0-9]+)!i'],'homepage'=>'https://www.youmaker.com/','host'=>['youmaker.com'],'iframe'=>['src'=>'https://www.youmaker.com/embed/{@id}'],'name'=>'YouMaker','scrape'=>[],'tags'=>['videos']],
		'youtube'=>['amp'=>['custom-element'=>'amp-youtube','src'=>'https://cdn.ampproject.org/v0/amp-youtube-0.1.js','template'=>'<amp-youtube layout="responsive" width="640" height="360" data-param-list="{@list}" data-param-start="{@t}" data-videoid="{@id}"/>'],'attributes'=>['id'=>['filterChain'=>['#identifier'],'required'=>false],'t'=>['filterChain'=>['#timestamp']]],'example'=>['https://www.youtube.com/watch?v=-cEzsCAzTak','https://youtu.be/-cEzsCAzTak','https://www.youtube.com/watch?feature=player_detailpage&v=jofNR_WkoCE#t=40','https://www.youtube.com/watch?v=pC35x6iIPmo&list=PLOU2XLYxmsIIxJrlMIY5vYXAFcO5g83gA'],'extract'=>['!youtube\\.com/(?:watch.*?v=|shorts/|v/|attribution_link.*?v%3D)(?\'id\'[-\\w]+)!','!youtu\\.be/(?\'id\'[-\\w]+)!','@[#&?]t=(?\'t\'\\d[\\dhms]*)@','![&?]list=(?\'list\'[-\\w]+)!'],'homepage'=>'https://www.youtube.com/','host'=>['youtube.com','youtu.be'],'iframe'=>['src'=>'https://www.youtube.com/embed/<xsl:value-of select="@id"/><xsl:if test="@list">?list=<xsl:value-of select="@list"/></xsl:if><xsl:if test="@t"><xsl:choose><xsl:when test="@list">&amp;</xsl:when><xsl:otherwise>?</xsl:otherwise></xsl:choose>start=<xsl:value-of select="@t"/></xsl:if>','style'=>['background'=>'url(https://i.ytimg.com/vi/{@id}/hqdefault.jpg) 50% 50% / cover']],'name'=>'YouTube','oembed'=>['endpoint'=>'https://www.youtube.com/oembed','scheme'=>'https://www.youtube.com/watch?v={@id}'],'scrape'=>[],'source'=>'https://support.google.com/youtube/bin/answer.py?hl=en&answer=171780','tags'=>['livestreaming','videos']]
	];
}