/*
 * Copyright 2014-2017 MongoDB, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <php.h>
#include <Zend/zend_interfaces.h>

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "phongo_compat.h"
#include "php_phongo.h"
#include "php_bson.h"

#if 0

zend_class_entry* php_phongo_serverdescription_ce;

/* {{{ proto string MongoDB\Driver\ServerDescription::getHost()
   Returns the hostname for this ServerDescription */
static PHP_METHOD(ServerDescription, getHost)
{
	php_phongo_server_t*         intern;
	mongoc_server_description_t* sd;

	PHONGO_PARSE_PARAMETERS_NONE();

	intern = Z_SERVERDESCRIPTION_OBJ_P(getThis());

	PHONGO_PARSE_PARAMETERS_NONE();

	if ((sd = mongoc_client_get_server_description(Z_MANAGER_OBJ_P(&intern->manager)->client, intern->server_id))) {
		RETVAL_STRING(mongoc_server_description_host(sd)->host);
		mongoc_server_description_destroy(sd);
		return;
	}

	phongo_throw_exception(PHONGO_ERROR_RUNTIME, "Failed to get server description");
} /* }}} */

/* {{{ proto array MongoDB\Driver\Server::getInfo()
   Returns the last isMaster result document for this Server */
static PHP_METHOD(Server, getInfo)
{
	zend_error_handling          error_handling;
	php_phongo_server_t*         intern;
	mongoc_server_description_t* sd;

	intern = Z_SERVER_OBJ_P(getThis());

	zend_replace_error_handling(EH_THROW, phongo_exception_from_phongo_domain(PHONGO_ERROR_INVALID_ARGUMENT), &error_handling);
	if (zend_parse_parameters_none() == FAILURE) {
		zend_restore_error_handling(&error_handling);
		return;
	}
	zend_restore_error_handling(&error_handling);

	if ((sd = mongoc_client_get_server_description(Z_MANAGER_OBJ_P(&intern->manager)->client, intern->server_id))) {
		const bson_t*         is_master = mongoc_server_description_ismaster(sd);
		php_phongo_bson_state state;

		PHONGO_BSON_INIT_DEBUG_STATE(state);

		if (!php_phongo_bson_to_zval_ex(bson_get_data(is_master), is_master->len, &state)) {
			/* Exception should already have been thrown */
			zval_ptr_dtor(&state.zchild);
			mongoc_server_description_destroy(sd);
			return;
		}

		mongoc_server_description_destroy(sd);

		RETURN_ZVAL(&state.zchild, 0, 1);
	}

	phongo_throw_exception(PHONGO_ERROR_RUNTIME, "Failed to get server description");
} /* }}} */

/* {{{ proto integer MongoDB\Driver\Server::getLatency()
   Returns the last measured latency for this Server */
static PHP_METHOD(Server, getLatency)
{
	zend_error_handling          error_handling;
	php_phongo_server_t*         intern;
	mongoc_server_description_t* sd;

	intern = Z_SERVER_OBJ_P(getThis());

	zend_replace_error_handling(EH_THROW, phongo_exception_from_phongo_domain(PHONGO_ERROR_INVALID_ARGUMENT), &error_handling);
	if (zend_parse_parameters_none() == FAILURE) {
		zend_restore_error_handling(&error_handling);
		return;
	}
	zend_restore_error_handling(&error_handling);

	if ((sd = mongoc_client_get_server_description(Z_MANAGER_OBJ_P(&intern->manager)->client, intern->server_id))) {
		RETVAL_LONG((zend_long) mongoc_server_description_round_trip_time(sd));
		mongoc_server_description_destroy(sd);
		return;
	}

	phongo_throw_exception(PHONGO_ERROR_RUNTIME, "Failed to get server description");
} /* }}} */

/* {{{ proto integer MongoDB\Driver\Server::getPort()
   Returns the port for this Server */
static PHP_METHOD(Server, getPort)
{
	zend_error_handling          error_handling;
	php_phongo_server_t*         intern;
	mongoc_server_description_t* sd;

	intern = Z_SERVER_OBJ_P(getThis());

	zend_replace_error_handling(EH_THROW, phongo_exception_from_phongo_domain(PHONGO_ERROR_INVALID_ARGUMENT), &error_handling);
	if (zend_parse_parameters_none() == FAILURE) {
		zend_restore_error_handling(&error_handling);
		return;
	}
	zend_restore_error_handling(&error_handling);

	if ((sd = mongoc_client_get_server_description(Z_MANAGER_OBJ_P(&intern->manager)->client, intern->server_id))) {
		RETVAL_LONG(mongoc_server_description_host(sd)->port);
		mongoc_server_description_destroy(sd);
		return;
	}

	phongo_throw_exception(PHONGO_ERROR_RUNTIME, "Failed to get server description");
} /* }}} */

/* {{{ proto integer MongoDB\Driver\Server::getType()
   Returns the node type of this Server */
static PHP_METHOD(Server, getType)
{
	zend_error_handling          error_handling;
	php_phongo_server_t*         intern;
	mongoc_server_description_t* sd;

	intern = Z_SERVER_OBJ_P(getThis());

	zend_replace_error_handling(EH_THROW, phongo_exception_from_phongo_domain(PHONGO_ERROR_INVALID_ARGUMENT), &error_handling);
	if (zend_parse_parameters_none() == FAILURE) {
		zend_restore_error_handling(&error_handling);
		return;
	}
	zend_restore_error_handling(&error_handling);

	if ((sd = mongoc_client_get_server_description(Z_MANAGER_OBJ_P(&intern->manager)->client, intern->server_id))) {
		RETVAL_LONG(php_phongo_server_description_type(sd));
		mongoc_server_description_destroy(sd);
		return;
	}

	phongo_throw_exception(PHONGO_ERROR_RUNTIME, "Failed to get server description");
} /* }}} */

ZEND_BEGIN_ARG_INFO_EX(ai_Server_void, 0, 0, 0)
ZEND_END_ARG_INFO()

static zend_function_entry php_phongo_server_me[] = {
	/* clang-format off */
	PHP_ME(Server, getHost, ai_Server_void, ZEND_ACC_PUBLIC | ZEND_ACC_FINAL)
	PHP_ME(Server, getInfo, ai_Server_void, ZEND_ACC_PUBLIC | ZEND_ACC_FINAL)
	PHP_ME(Server, getLatency, ai_Server_void, ZEND_ACC_PUBLIC | ZEND_ACC_FINAL)
	PHP_ME(Server, getPort, ai_Server_void, ZEND_ACC_PUBLIC | ZEND_ACC_FINAL)
	PHP_ME(Server, getType, ai_Server_void, ZEND_ACC_PUBLIC | ZEND_ACC_FINAL)
	ZEND_NAMED_ME(__construct, PHP_FN(MongoDB_disabled___construct), ai_Server_void, ZEND_ACC_PRIVATE | ZEND_ACC_FINAL)
	ZEND_NAMED_ME(__wakeup, PHP_FN(MongoDB_disabled___wakeup), ai_Server_void, ZEND_ACC_PUBLIC | ZEND_ACC_FINAL)
	PHP_FE_END
	/* clang-format on */
};
/* }}} */

/* {{{ MongoDB\Driver\ServerDescription object handlers */
static zend_object_handlers php_phongo_handler_server_description;

static int php_phongo_server_description_compare_objects(zval* o1, zval* o2) /* {{{ */
{
	php_phongo_server_t*         intern1;
	php_phongo_server_t*         intern2;
	mongoc_server_description_t *sd1, *sd2;
	int                          retval = 0;

	ZEND_COMPARE_OBJECTS_FALLBACK(o1, o2);

	intern1 = Z_SERVER_OBJ_P(o1);
	intern2 = Z_SERVER_OBJ_P(o2);

	sd1 = mongoc_client_get_server_description(Z_MANAGER_OBJ_P(&intern1->manager)->client, intern1->server_id);
	sd2 = mongoc_client_get_server_description(Z_MANAGER_OBJ_P(&intern2->manager)->client, intern2->server_id);

	if (sd1 && sd2) {
		retval = strcasecmp(mongoc_server_description_host(sd1)->host_and_port, mongoc_server_description_host(sd2)->host_and_port);
	} else {
		phongo_throw_exception(PHONGO_ERROR_RUNTIME, "Failed to get server description(s)");
	}

	if (sd1) {
		mongoc_server_description_destroy(sd1);
	}

	if (sd2) {
		mongoc_server_description_destroy(sd2);
	}

	return retval;
} /* }}} */

static void php_phongo_server_free_object(zend_object* object) /* {{{ */
{
	php_phongo_server_t* intern = Z_OBJ_SERVER(object);

	zend_object_std_dtor(&intern->std);

	if (!Z_ISUNDEF(intern->manager)) {
		zval_ptr_dtor(&intern->manager);
	}
} /* }}} */

static zend_object* php_phongo_server_create_object(zend_class_entry* class_type) /* {{{ */
{
	php_phongo_server_t* intern = NULL;

	intern = PHONGO_ALLOC_OBJECT_T(php_phongo_server_t, class_type);

	zend_object_std_init(&intern->std, class_type);
	object_properties_init(&intern->std, class_type);

	PHONGO_SET_CREATED_BY_PID(intern);

	intern->std.handlers = &php_phongo_handler_server;

	return &intern->std;
} /* }}} */

static HashTable* php_phongo_server_get_debug_info(phongo_compat_object_handler_type* object, int* is_temp) /* {{{ */
{
	php_phongo_server_t*         intern = NULL;
	zval                         retval = ZVAL_STATIC_INIT;
	mongoc_server_description_t* sd;

	*is_temp = 1;
	intern   = Z_OBJ_SERVER(PHONGO_COMPAT_GET_OBJ(object));

	if (!(sd = mongoc_client_get_server_description(Z_MANAGER_OBJ_P(&intern->manager)->client, intern->server_id))) {
		phongo_throw_exception(PHONGO_ERROR_RUNTIME, "Failed to get server description");
		return NULL;
	}

	php_phongo_server_to_zval(&retval, sd);
	mongoc_server_description_destroy(sd);

	return Z_ARRVAL(retval);
} /* }}} */
/* }}} */

void php_phongo_server_init_ce(INIT_FUNC_ARGS) /* {{{ */
{
	zend_class_entry ce;

	INIT_NS_CLASS_ENTRY(ce, "MongoDB\\Driver", "Server", php_phongo_server_me);
	php_phongo_server_ce                = zend_register_internal_class(&ce);
	php_phongo_server_ce->create_object = php_phongo_server_create_object;
	PHONGO_CE_FINAL(php_phongo_server_ce);
	PHONGO_CE_DISABLE_SERIALIZATION(php_phongo_server_ce);

	memcpy(&php_phongo_handler_server, phongo_get_std_object_handlers(), sizeof(zend_object_handlers));
	PHONGO_COMPAT_SET_COMPARE_OBJECTS_HANDLER(server);
	php_phongo_handler_server.get_debug_info = php_phongo_server_get_debug_info;
	php_phongo_handler_server.free_obj       = php_phongo_server_free_object;
	php_phongo_handler_server.offset         = XtOffsetOf(php_phongo_server_t, std);

	zend_declare_class_constant_long(php_phongo_server_ce, ZEND_STRL("TYPE_UNKNOWN"), PHONGO_SERVER_UNKNOWN);
	zend_declare_class_constant_long(php_phongo_server_ce, ZEND_STRL("TYPE_STANDALONE"), PHONGO_SERVER_STANDALONE);
	zend_declare_class_constant_long(php_phongo_server_ce, ZEND_STRL("TYPE_MONGOS"), PHONGO_SERVER_MONGOS);
	zend_declare_class_constant_long(php_phongo_server_ce, ZEND_STRL("TYPE_POSSIBLE_PRIMARY"), PHONGO_SERVER_POSSIBLE_PRIMARY);
	zend_declare_class_constant_long(php_phongo_server_ce, ZEND_STRL("TYPE_RS_PRIMARY"), PHONGO_SERVER_RS_PRIMARY);
	zend_declare_class_constant_long(php_phongo_server_ce, ZEND_STRL("TYPE_RS_SECONDARY"), PHONGO_SERVER_RS_SECONDARY);
	zend_declare_class_constant_long(php_phongo_server_ce, ZEND_STRL("TYPE_RS_ARBITER"), PHONGO_SERVER_RS_ARBITER);
	zend_declare_class_constant_long(php_phongo_server_ce, ZEND_STRL("TYPE_RS_OTHER"), PHONGO_SERVER_RS_OTHER);
	zend_declare_class_constant_long(php_phongo_server_ce, ZEND_STRL("TYPE_RS_GHOST"), PHONGO_SERVER_RS_GHOST);
} /* }}} */

#endif

/*
 * Local variables:
 * tab-width: 4
 * c-basic-offset: 4
 * End:
 * vim600: noet sw=4 ts=4 fdm=marker
 * vim<600: noet sw=4 ts=4
 */
