/*
  +----------------------------------------------------------------------+
  | PHP Version 5                                                        |
  +----------------------------------------------------------------------+
  | Copyright (c) 1997-2005 The PHP Group                                |
  +----------------------------------------------------------------------+
  | This source file is subject to version 3.0 of the PHP license,       |
  | that is bundled with this package in the file LICENSE, and is        |
  | available through the world-wide-web at the following url:           |
  | http://www.php.net/license/3_0.txt.                                  |
  | If you did not receive a copy of the PHP license and are unable to   |
  | obtain it through the world-wide-web, please send a note to          |
  | license@php.net so we can mail you a copy immediately.               |
  +----------------------------------------------------------------------+
  | Author: Rasmus Lerdorf <rasmus@php.net>                              |
  |         Ilia Alshanetsky <ilia@php.net>                              |
  +----------------------------------------------------------------------+
  $Id: gdchart.c,v 1.9 2005/12/29 22:56:53 iliaa Exp $ 
*/

/* 
 * Note, this extension was not written to be threadsafe because neither the
 * GD nor the GDCHART underlying code is threadsafe.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "php.h"
#include "php_ini.h"
#include "ext/standard/info.h"
#include "php_gdchart.h"

#include "gdchart-src/gdc.h"
#include "gdchart-src/gdcpie.h"
#include "gdchart-src/gdchart.h"

static zend_object_handlers gdc_object_handlers;
zend_class_entry *gdc_class_entry = NULL;

static unsigned long gdc_default_colors[11] = {
	0xFF4040L,0x80FF80L,0x8080FFL,0xFF80FFL,0xFFFF80L,0x80FFFFL,0x0080FFL,0x40FF40L,0x4040FFL,0x404040L,0x808080L
};

typedef struct _php_gdchart_map {
	char *name;
	int name_len;
	void *value;
} php_gdchart_map;

#define PHP_GDC_LONG_OPT 50
static php_gdchart_map php_gdc_long_map[PHP_GDC_LONG_OPT] = {
	{ "grid", sizeof("grid")-1, &GDC_grid },
	{ "ticks", sizeof("ticks")-1, &GDC_ticks},
	{ "angle", sizeof("angle")-1, &GDC_3d_angle},
	{ "xaxis", sizeof("xaxis")-1, &GDC_xaxis},
	{ "yaxis", sizeof("yaxis")-1, &GDC_yaxis},
	{ "border", sizeof("border")-1, &GDC_border},
	{ "yaxis2", sizeof("yaxis2")-1, &GDC_yaxis2},
	{ "bgColor", sizeof("bgColor")-1, &GDC_BGColor},
	{ "barWidth", sizeof("barWidth")-1, &GDC_bar_width},
	{ "capWidth", sizeof("capWidth")-1, &GDC_HLC_cap_width},
	{ "volColor", sizeof("volColor")-1, &GDC_VolColor},
	{ "pieAngle", sizeof("pieAngle")-1, &GDCPIE_3d_angle},
	{ "pieDepth", sizeof("pieDepth")-1, &GDCPIE_3d_depth},
	{ "thumbnail", sizeof("thumbnail")-1, &GDC_thumbnail},
	{ "hlcStyle", sizeof("hlcStyle")-1, &GDC_HLC_style},
	{ "hardSize", sizeof("hardSize")-1, &GDC_hard_size},
	{ "titleSize", sizeof("titleSize")-1, &GDC_title_size},
	{ "lineColor", sizeof("lineColor")-1, &GDC_LineColor},
	{ "plotColor", sizeof("plotColor")-1, &GDC_PlotColor},
	{ "gridColor", sizeof("gridColor")-1, &GDC_GridColor},
	{ "labelDist", sizeof("labelDist")-1, &GDCPIE_label_dist},
	{ "labelSize", sizeof("labelSize")-1, &GDCPIE_label_size},
	{ "edgeColor", sizeof("edgeColor")-1, &GDCPIE_EdgeColor},
	{ "zeroShelf", sizeof("zeroShelf")-1, &GDC_0Shelf},
	{ "stackType", sizeof("stackType")-1, &GDC_stack_type},
	{ "yvalStyle", sizeof("yvalStyle")-1, &GDC_yval_style},
	{ "hardXorig", sizeof("hardXorig")-1, &GDC_hard_xorig},
	{ "hardYorig", sizeof("hardYorig")-1, &GDC_hard_yorig},
	{ "titleColor", sizeof("titleColor")-1, &GDC_TitleColor},
	{ "xtitleSize", sizeof("xtitleSize")-1, &GDC_xtitle_size},
	{ "ytitleSize", sizeof("ytitleSize")-1, &GDC_ytitle_size},
	{ "perspective", sizeof("perspective")-1, &GDCPIE_perspective},
	{ "xtitleColor", sizeof("xtitleColor")-1, &GDC_XTitleColor},
	{ "ytitleColor", sizeof("ytitleColor")-1, &GDC_YTitleColor},
	{ "xlabelColor", sizeof("xlabelColor")-1, &GDC_XLabelColor},
	{ "ylabelColor", sizeof("ylabelColor")-1, &GDC_YLabelColor},
	{ "jpegQuality", sizeof("jpegQuality")-1, &GDC_jpeg_quality},
	{ "ytitle2Color", sizeof("ytitle2Color")-1, &GDC_YTitle2Color},
	{ "ylabel2Color", sizeof("ylabel2Color")-1, &GDC_YLabel2Color},
	{ "percentLabels", sizeof("percentLabels")-1, &GDCPIE_percent_labels},
	{ "xaxisfontSize", sizeof("xaxisfontSize")-1, &GDC_xaxisfont_size},
	{ "yaxisfontSize", sizeof("yaxisfontSize")-1, &GDC_yaxisfont_size},
	{ "xlabelSpacing", sizeof("xlabelSpacing")-1, &GDC_xlabel_spacing},
	{ "ylabelDensity", sizeof("ylabelDensity")-1, &GDC_ylabel_density},
	{ "interpolations", sizeof("interpolations")-1, &GDC_interpolations},
	{ "transparentBg", sizeof("transparentBg")-1, &GDC_transparent_bg},
	{ "hardGrapheigt", sizeof("hardGrapheigt")-1, &GDC_hard_grapheight},
	{ "otherThreshold", sizeof("otherThreshold")-1, &GDCPIE_other_threshold},
	{ "hardGraphwidth", sizeof("hardGraphwidth")-1, &GDC_hard_graphwidth},
	{ "annotationFontSize", sizeof("annotationFontSize")-1, &GDC_annotation_font_size},
};

#define PHP_GDC_STR_OPT 16
static php_gdchart_map php_gdc_string_map[PHP_GDC_STR_OPT] = {
	{ "title", sizeof("title")-1, &GDC_title},
	{ "xtitle", sizeof("xtitle")-1, &GDC_xtitle},
	{ "ytitle", sizeof("ytitle")-1, &GDC_ytitle},
	{ "ytitle2", sizeof("ytitle2")-1, &GDC_ytitle2},
	{ "bgImage", sizeof("bgImage")-1, &GDC_BGImage},
	{ "titleFont", sizeof("titleFont")-1, &GDC_title_font},
	{ "xaxisFont", sizeof("xaxisFont")-1, &GDC_xaxis_font},
	{ "yaxisFont", sizeof("yaxisFont")-1, &GDC_yaxis_font},
	{ "labelFont", sizeof("labelFont")-1, &GDCPIE_label_font},
	{ "ylabelFmt", sizeof("ylabelFmt")-1, &GDC_ylabel_fmt},
	{ "xlabelCtl", sizeof("xlabelCtl")-1, &GDC_xlabel_ctl},
	{ "xtitleFont", sizeof("xtitleFont")-1, &GDC_xtitle_font},
	{ "ytitleFont", sizeof("ytitleFont")-1, &GDC_ytitle_font},
	{ "percentFmt", sizeof("percentFmt")-1, &GDCPIE_percent_fmt},
	{ "ylabel2Fmt", sizeof("ylabel2Fmt")-1, &GDC_ylabel2_fmt},
	{ "annotationFont", sizeof("annotationFont")-1, &GDC_annotation_font},
};

#define PHP_GDC_DOUBLE_OPT 12
static php_gdchart_map php_gdc_double_map[PHP_GDC_DOUBLE_OPT] = {
	{ "depth", sizeof("depth")-1, &GDC_3d_depth},
	{ "xaxisAngle", sizeof("xaxisAngle")-1, &GDC_xaxis_angle},
	{ "titlePtsize", sizeof("titlePtsize")-1, &GDC_title_ptsize},
	{ "labelPtsize", sizeof("labelPtsize")-1, &GDCPIE_label_ptsize},
	{ "xaxisPtsize", sizeof("xaxisPtsize")-1, &GDC_xaxis_ptsize},
	{ "yaxisPtsize", sizeof("yaxisPtsize")-1, &GDC_yaxis_ptsize},
	{ "xtitlePtsize", sizeof("xtitlePtsize")-1, &GDC_xtitle_ptsize},
	{ "ytitlePtsize", sizeof("ytitlePtsize")-1, &GDC_ytitle_ptsize},
	{ "requestedYmin", sizeof("requestedYmin")-1, &GDC_requested_ymin},
	{ "requestedYmax", sizeof("requestedYmax")-1, &GDC_requested_ymax},
	{ "annotationPtsize", sizeof("annotationPtsize")-1, &GDC_annotation_ptsize},
	{ "requestedYinterval", sizeof("requestedYinterval")-1, &GDC_requested_yinterval},
};

struct gdc_object {
	zend_object zo;

	char **labels;

	GDC_SCATTER_T *scatter;
	int n_scatter;

	float *combo;
	int n_combo;

	float *values;
	int n_values;
	int n_points;

	unsigned long *colors;
	unsigned long *ext_color;
	unsigned long *ext_vol_color;
	unsigned long *explode;

	int counts[4]; /* count of entries for colors & explode, this is needed for cloning */

	unsigned char *missing;
};

PHP_METHOD(GDChart, __construct);
PHP_METHOD(GDChart, out);
PHP_METHOD(GDChart, addValues);
PHP_METHOD(GDChart, addScatter);
PHP_METHOD(GDChart, addCombo);
PHP_METHOD(GDChart, setLabels);
PHP_METHOD(GDChart, setMissing);
PHP_METHOD(GDChart, setExplode);
PHP_METHOD(GDChart, setColors);
PHP_METHOD(GDChart, setExtColors);
PHP_METHOD(GDChart, setExtVolColors);

#define GDC_FREE_LABELS(l)	\
	if (l) {			\
		int i = 0;		\
		while (l[i]) {		\
			efree(l[i++]);	\
		}		\
		efree(l);	\
		l = NULL;	\
	}

#define PHP_GDC_KEY_FREE(k) if (k) { efree(k); k = NULL; }

/* {{{ gdchart_functions[]
 *
 * Every user visible function must have an entry in gdchart_functions[].
 */
function_entry gdchart_functions[] = {
	{NULL, NULL, NULL}	/* Must be the last line in gdchart_functions[] */
};
/* }}} */

/* {{{ gdc_objects_dtor
 */
static void gdc_objects_dtor(void *object, zend_object_handle handle TSRMLS_DC)
{
	struct gdc_object *intern = (struct gdc_object *) object;

	GDC_FREE_LABELS(intern->labels);
	PHP_GDC_KEY_FREE(intern->values)
	PHP_GDC_KEY_FREE(intern->combo)
	PHP_GDC_KEY_FREE(intern->colors)
	PHP_GDC_KEY_FREE(intern->ext_color)
	PHP_GDC_KEY_FREE(intern->ext_vol_color)
	PHP_GDC_KEY_FREE(intern->explode)
	PHP_GDC_KEY_FREE(intern->missing)
	PHP_GDC_KEY_FREE(intern->scatter)

	zend_objects_free_object_storage(object TSRMLS_CC);
}
/* }}} */

static void gdc_objects_clone(void *object, void **object_clone TSRMLS_DC)
{
	struct gdc_object *intern = (struct gdc_object *) object;
	struct gdc_object **intern_clone = (struct gdc_object **) object_clone;
	zval tmp;

	*intern_clone = ecalloc(1, sizeof(struct gdc_object));
	(*intern_clone)->zo.ce = intern->zo.ce;

#if ZEND_MODULE_API_NO >= 20050922
	(*intern_clone)->zo.guards = NULL;
#else
	(*intern_clone)->zo.in_get = 0;
	(*intern_clone)->zo.in_set = 0;
#endif

        ALLOC_HASHTABLE((*intern_clone)->zo.properties);
        zend_hash_init((*intern_clone)->zo.properties, 0, NULL, ZVAL_PTR_DTOR, 0);
	zend_hash_copy((*intern_clone)->zo.properties, intern->zo.properties, (copy_ctor_func_t) zval_add_ref, (void *) &tmp, sizeof(zval *));

	if (intern->scatter) {
		(*intern_clone)->scatter = safe_emalloc(intern->n_scatter,  sizeof(GDC_SCATTER_T), 0);
		memcpy((*intern_clone)->scatter, intern->scatter, intern->n_scatter * sizeof(GDC_SCATTER_T));
		(*intern_clone)->n_scatter = intern->n_scatter;
	}
	if (intern->combo) {
		(*intern_clone)->combo = safe_emalloc(intern->n_combo,  sizeof(float), 0);
		memcpy((*intern_clone)->combo, intern->combo, intern->n_combo * sizeof(float));
		(*intern_clone)->n_combo = intern->n_combo;
	}
	if (intern->values) {
		(*intern_clone)->values = safe_emalloc(intern->n_values,  sizeof(float), 0);
		memcpy((*intern_clone)->values, intern->values, intern->n_values * sizeof(float));
		(*intern_clone)->n_values = intern->n_values;
		(*intern_clone)->n_points = intern->n_points;
	}
	if (intern->labels) {
		int i = 0;
		while (intern->labels[i++]);

		(*intern_clone)->labels = safe_emalloc((i+1), sizeof(char *), 0);
		i = 0;
		while (intern->labels[i]) {
			(*intern_clone)->labels[i] = estrdup(intern->labels[i]);
			i++;
		}
		(*intern_clone)->labels[i] = NULL;
	}
	if (intern->missing) {
		(*intern_clone)->missing = estrdup(intern->missing);
	}
	if (intern->colors) {
		(*intern_clone)->colors = safe_emalloc(intern->counts[0], sizeof(unsigned long), 0);
		memcpy((*intern_clone)->colors, intern->colors, intern->counts[0] * sizeof(unsigned long));
	}
	if (intern->ext_color) {
		(*intern_clone)->ext_color = safe_emalloc(intern->counts[1], sizeof(unsigned long), 0);
		memcpy((*intern_clone)->ext_color, intern->ext_color, intern->counts[1] * sizeof(unsigned long));
	}
	if (intern->ext_vol_color) {
		(*intern_clone)->ext_vol_color = safe_emalloc(intern->counts[2], sizeof(unsigned long), 0);
		memcpy((*intern_clone)->ext_vol_color, intern->ext_vol_color, intern->counts[2] * sizeof(unsigned long));
	}
	if (intern->explode) {
		(*intern_clone)->explode = safe_emalloc(intern->counts[3], sizeof(unsigned long), 0);
		memcpy((*intern_clone)->explode, intern->explode, intern->counts[3] * sizeof(unsigned long));
	}
	memcpy((*intern_clone)->counts, intern->counts, 4 * sizeof(int));
}

/* {{{ gdc_objects_new
 */
PHP_GDC_API zend_object_value gdc_objects_new(zend_class_entry *class_type TSRMLS_DC)
{
	zend_object_value retval;
	struct gdc_object *intern;
	zval tmp;

	intern = ecalloc(1, sizeof(struct gdc_object));
	intern->zo.ce = class_type;

	ALLOC_HASHTABLE(intern->zo.properties);
        zend_hash_init(intern->zo.properties, 0, NULL, ZVAL_PTR_DTOR, 0);
	zend_hash_copy(intern->zo.properties, &class_type->default_properties, (copy_ctor_func_t) zval_add_ref, (void *) &tmp, sizeof(zval *));

#if ZEND_MODULE_API_NO >= 20050922
	intern->zo.guards = NULL;
#else
	intern->zo.in_get = 0;
	intern->zo.in_set = 0;
#endif

	retval.handle = zend_objects_store_put(intern, NULL, (zend_objects_free_object_storage_t) gdc_objects_dtor, gdc_objects_clone TSRMLS_CC);
	retval.handlers = (zend_object_handlers *) &gdc_object_handlers;

	return retval;
}
/* }}} */

zend_object_value gdc_objects_store_clone_obj(zval *zobject TSRMLS_DC)
{
	zend_object_value retval;
	void *new_object;
	zend_object_handle handle = Z_OBJ_HANDLE_P(zobject);
	struct _store_object *obj = &EG(objects_store).object_buckets[handle].bucket.obj;

	if (obj->clone == NULL) {
		php_error(E_ERROR, "Trying to clone an uncloneable object of class %s", Z_OBJCE_P(zobject)->name);
	}		

	obj->clone(obj->object, &new_object TSRMLS_CC);

	retval.handle = zend_objects_store_put(new_object, obj->dtor, obj->free_storage, obj->clone TSRMLS_CC);
	retval.handlers = Z_OBJ_HT_P(zobject);

	return retval;
}

/* {{{ gdchart methods */
static zend_function_entry gdc_functions[] = {
	PHP_ME(GDChart,		__construct, 		NULL,0)
	PHP_ME(GDChart,		out, 			NULL,0)
	PHP_ME(GDChart,		addValues,	 	NULL,0)
	PHP_ME(GDChart,		addCombo,	 	NULL,0)
	PHP_ME(GDChart,		addScatter,	 	NULL,0)
	PHP_ME(GDChart, 	setLabels,		NULL,0)
	PHP_ME(GDChart,		setMissing,	 	NULL,0)
	PHP_ME(GDChart, 	setColors,		NULL,0)
	PHP_ME(GDChart,		setExtColors,	 	NULL,0)
	PHP_ME(GDChart, 	setExtVolColors,	NULL,0)
	PHP_ME(GDChart,		setExplode,	 	NULL,0)
	{NULL, NULL, NULL}
};  /* }}} */

/* {{{ gdchart_module_entry */
zend_module_entry gdchart_module_entry = {
#if ZEND_MODULE_API_NO >= 20010901
	STANDARD_MODULE_HEADER,
#endif
	"gdchart",
	gdchart_functions,
	PHP_MINIT(gdchart),
	PHP_MSHUTDOWN(gdchart),
	NULL,
	NULL,
	PHP_MINFO(gdchart),
#if ZEND_MODULE_API_NO >= 20010901
	"0.2", /* Replace with version number for your extension */
#endif
	STANDARD_MODULE_PROPERTIES
};
/* }}} */

#ifdef COMPILE_DL_GDCHART
ZEND_GET_MODULE(gdchart)
#endif

/* {{{ PHP_MSHUTDOWN_FUNCTION(gdchart)
 */
PHP_MSHUTDOWN_FUNCTION(gdchart)
{
	gdc_class_entry = NULL;
	return SUCCESS;
}
/* }}} */

/* {{{ PHP_MINIT_FUNCTION
 */
PHP_MINIT_FUNCTION(gdchart)
{
	zend_class_entry gdc;

	INIT_CLASS_ENTRY(gdc, "GDChart", gdc_functions);
	gdc.create_object = gdc_objects_new;
	gdc_class_entry = zend_register_internal_class(&gdc);
	memcpy(&gdc_object_handlers, zend_get_std_object_handlers(), sizeof(zend_object_handlers));
	gdc_object_handlers.clone_obj = gdc_objects_store_clone_obj;

#define GDCHART_REG_LONG_PROP(n, v) zend_declare_property_long(gdc_class_entry, n, sizeof(n)-1, v, ZEND_ACC_PUBLIC TSRMLS_CC);
#define GDCHART_REG_BOOL_PROP(n, v) zend_declare_property_bool(gdc_class_entry, n, sizeof(n)-1, v, ZEND_ACC_PUBLIC TSRMLS_CC);
#define GDCHART_REG_DOUBLE_PROP(n, v) zend_declare_property_double(gdc_class_entry, n, sizeof(n)-1, v, ZEND_ACC_PUBLIC TSRMLS_CC);
#define GDCHART_REG_STR_PROP(n, v) zend_declare_property_string(gdc_class_entry, n, sizeof(n)-1, v, ZEND_ACC_PUBLIC TSRMLS_CC);
#define GDCHART_REG_NULL_PROP(n) zend_declare_property_null(gdc_class_entry, n, sizeof(n)-1, ZEND_ACC_PUBLIC TSRMLS_CC);

	GDCHART_REG_LONG_PROP("grid", (long)GDC_TICK_LABELS)
	GDCHART_REG_LONG_PROP("ticks", (long)GDC_TICK_LABELS)
	GDCHART_REG_LONG_PROP("angle", 45)
	GDCHART_REG_LONG_PROP("xaxis", 1)
	GDCHART_REG_LONG_PROP("yaxis", 1)
	GDCHART_REG_LONG_PROP("border", GDC_BORDER_ALL)
	GDCHART_REG_LONG_PROP("yaxis2", 1)
	GDCHART_REG_LONG_PROP("bgColor", 0xffffffL)
	GDCHART_REG_LONG_PROP("barWidth", 75)
	GDCHART_REG_LONG_PROP("capWidth", 25)
	GDCHART_REG_LONG_PROP("volColor", 0xA0A0FF)
	GDCHART_REG_LONG_PROP("pieAngle", 45)
	GDCHART_REG_LONG_PROP("pieDepth", 10)
	GDCHART_REG_LONG_PROP("thumbnail", 0)
	GDCHART_REG_LONG_PROP("hlcStyle", (long)GDC_HLC_CLOSE_CONNECTED)
	GDCHART_REG_LONG_PROP("hardSize", 0)
	GDCHART_REG_LONG_PROP("titleSize", (long)GDC_MEDBOLD)
	GDCHART_REG_LONG_PROP("lineColor", (long)GDC_DFLTCOLOR)
	GDCHART_REG_LONG_PROP("plotColor", 0xC0C0C0)
	GDCHART_REG_LONG_PROP("gridColor", 0xA0A0A0)
	GDCHART_REG_LONG_PROP("labelDist", 5)
	GDCHART_REG_LONG_PROP("labelSize", (long)GDC_SMALL)
	GDCHART_REG_LONG_PROP("edgeColor", (long)GDC_NOCOLOR)
	GDCHART_REG_LONG_PROP("zeroShelf", 1)
	GDCHART_REG_LONG_PROP("stackType", (long)GDC_STACK_DEPTH)
	GDCHART_REG_LONG_PROP("yvalStyle", 1)
	GDCHART_REG_LONG_PROP("hardXorig", 0)
	GDCHART_REG_LONG_PROP("hardYorig", 0)
	GDCHART_REG_LONG_PROP("titleColor", 0x000000)
	GDCHART_REG_LONG_PROP("xtitleSize", (long)GDC_MEDBOLD)
	GDCHART_REG_LONG_PROP("ytitleSize", (long)GDC_MEDBOLD)
	GDCHART_REG_LONG_PROP("perspective", 0)
	GDCHART_REG_LONG_PROP("xtitleColor", 0x000000)
	GDCHART_REG_LONG_PROP("ytitleColor", 0x000000)
	GDCHART_REG_LONG_PROP("xlabelColor", 0x000000)
	GDCHART_REG_LONG_PROP("ylabelColor", 0x000000)
	GDCHART_REG_LONG_PROP("jpegQuality", -1)
	GDCHART_REG_LONG_PROP("ytitle2Color", 0x000000)
	GDCHART_REG_LONG_PROP("ylabel2Color", 0x000000)
	GDCHART_REG_LONG_PROP("extVolColor", 0xA0A0FF)
	GDCHART_REG_LONG_PROP("percentLabels", (long)GDCPIE_PCT_NONE)
	GDCHART_REG_LONG_PROP("xaxisfontSize", (long)GDC_MEDBOLD)
	GDCHART_REG_LONG_PROP("yaxisfontSize", (long)GDC_MEDBOLD)
	GDCHART_REG_LONG_PROP("xlabelSpacing", 5)
	GDCHART_REG_LONG_PROP("ylabelDensity", 80)
	GDCHART_REG_LONG_PROP("interpolations", 0)
	GDCHART_REG_LONG_PROP("transparentBg", 0)
	GDCHART_REG_LONG_PROP("hardGrapheigt", 0)
	GDCHART_REG_LONG_PROP("otherThreshold", -1)
	GDCHART_REG_LONG_PROP("hardGraphwidth", 0)
	GDCHART_REG_LONG_PROP("annotationFontSize", (long)GDC_SMALL)
	GDCHART_REG_LONG_PROP("annoColor", 0)
	GDCHART_REG_LONG_PROP("type", 0)

	GDCHART_REG_STR_PROP("xtitle", "")
	GDCHART_REG_STR_PROP("ytitle", "")
	GDCHART_REG_STR_PROP("title", "")
	GDCHART_REG_STR_PROP("ytitle2", "")
	GDCHART_REG_STR_PROP("bgImage", "")
	GDCHART_REG_STR_PROP("titleFont", "")
	GDCHART_REG_STR_PROP("xaxisFont", "")
	GDCHART_REG_STR_PROP("yaxisFont", "")
	GDCHART_REG_STR_PROP("labelFont", "")
	GDCHART_REG_STR_PROP("ylabelFmt", "")
	GDCHART_REG_STR_PROP("xlabelCtl", "")
	GDCHART_REG_STR_PROP("xtitleFont", "")
	GDCHART_REG_STR_PROP("ytitleFont", "")
	GDCHART_REG_STR_PROP("percentFmt", "%.0f%%")
	GDCHART_REG_STR_PROP("ylabel2Fmt", "")
	GDCHART_REG_STR_PROP("annoNote", "")
	GDCHART_REG_STR_PROP("annotationFont", "")

	GDCHART_REG_BOOL_PROP("labelLine", FALSE)
	
	GDCHART_REG_DOUBLE_PROP("xaxisAngle", 90.0)
	GDCHART_REG_DOUBLE_PROP("depth", 5.0)
	GDCHART_REG_DOUBLE_PROP("titlePtsize", 0.0)
	GDCHART_REG_DOUBLE_PROP("labelPtsize", 0.0)
	GDCHART_REG_DOUBLE_PROP("xaxisPtsize", 0.0)
	GDCHART_REG_DOUBLE_PROP("yaxisPtsize", 0.0)
	GDCHART_REG_DOUBLE_PROP("xtitlePtsize", 0.0)
	GDCHART_REG_DOUBLE_PROP("ytitlePtsize", 0.0)
	GDCHART_REG_DOUBLE_PROP("requestedYmin", 0)
	GDCHART_REG_DOUBLE_PROP("requestedYmax", 0)
	GDCHART_REG_DOUBLE_PROP("annotationPtsize", 0.0)
	GDCHART_REG_DOUBLE_PROP("annoPoint", 0.0)
	GDCHART_REG_DOUBLE_PROP("requestedYinterval", GDC_NOVALUE)

#define REGISTER_GDC_CLASS_CONST_LONG(const_name, value) zend_declare_class_constant_long(gdc_class_entry, const_name, sizeof(const_name)-1, (long)value TSRMLS_CC);

	/* Types of graphs */
	REGISTER_GDC_CLASS_CONST_LONG("PIE_2D", -2);
	REGISTER_GDC_CLASS_CONST_LONG("PIE_3D", -1);
	REGISTER_GDC_CLASS_CONST_LONG("BAR", GDC_BAR);
	REGISTER_GDC_CLASS_CONST_LONG("BAR_3D", GDC_3DBAR);
	REGISTER_GDC_CLASS_CONST_LONG("LINE", GDC_LINE);
	REGISTER_GDC_CLASS_CONST_LONG("AREA", GDC_AREA);
	REGISTER_GDC_CLASS_CONST_LONG("FLOATINGBAR", GDC_FLOATINGBAR);
	REGISTER_GDC_CLASS_CONST_LONG("HILOCLOSE", GDC_HILOCLOSE);
	REGISTER_GDC_CLASS_CONST_LONG("COMBO_LINE_BAR", GDC_COMBO_LINE_BAR);
	REGISTER_GDC_CLASS_CONST_LONG("COMBO_HLC_BAR", GDC_COMBO_HLC_BAR);
	REGISTER_GDC_CLASS_CONST_LONG("COMBO_LINE_AREA", GDC_COMBO_LINE_AREA);
	REGISTER_GDC_CLASS_CONST_LONG("COMBO_LINE_LINE", GDC_COMBO_LINE_LINE);
	REGISTER_GDC_CLASS_CONST_LONG("COMBO_HLC_AREA", GDC_COMBO_HLC_AREA);
	REGISTER_GDC_CLASS_CONST_LONG("HILOCLOSE_3D", GDC_3DHILOCLOSE);
	REGISTER_GDC_CLASS_CONST_LONG("COMBO_LINE_BAR_3D", GDC_3DCOMBO_LINE_BAR);
	REGISTER_GDC_CLASS_CONST_LONG("COMBO_LINE_AREA_3D", GDC_3DCOMBO_LINE_AREA);
	REGISTER_GDC_CLASS_CONST_LONG("COMBO_LINE_LINE_3D", GDC_3DCOMBO_LINE_LINE);
	REGISTER_GDC_CLASS_CONST_LONG("COMBO_HLC_BAR_3D", GDC_3DCOMBO_HLC_BAR);
	REGISTER_GDC_CLASS_CONST_LONG("COMBO_HLC_AREA_3D", GDC_3DCOMBO_HLC_AREA);
	REGISTER_GDC_CLASS_CONST_LONG("FLOATINGBAR_3D", GDC_3DFLOATINGBAR);
	REGISTER_GDC_CLASS_CONST_LONG("AREA_3D", GDC_3DAREA);
	REGISTER_GDC_CLASS_CONST_LONG("LINE_3D", GDC_3DLINE);

	REGISTER_GDC_CLASS_CONST_LONG("FONT_TINY", GDC_TINY);
	REGISTER_GDC_CLASS_CONST_LONG("FONT_SMALL", GDC_SMALL);
	REGISTER_GDC_CLASS_CONST_LONG("FONT_MEDBOLD", GDC_MEDBOLD);
	REGISTER_GDC_CLASS_CONST_LONG("FONT_LARGE", GDC_LARGE);
	REGISTER_GDC_CLASS_CONST_LONG("FONT_GIANT", GDC_GIANT);
	REGISTER_GDC_CLASS_CONST_LONG("LABEL_NONE", GDCPIE_PCT_NONE);
	REGISTER_GDC_CLASS_CONST_LONG("LABEL_ABOVE", GDCPIE_PCT_ABOVE);
	REGISTER_GDC_CLASS_CONST_LONG("LABEL_BELOW", GDCPIE_PCT_BELOW);
	REGISTER_GDC_CLASS_CONST_LONG("LABEL_RIGHT", GDCPIE_PCT_RIGHT);
	REGISTER_GDC_CLASS_CONST_LONG("LABEL_LEFT", GDCPIE_PCT_LEFT);

	REGISTER_GDC_CLASS_CONST_LONG("BORDER_NONE", GDC_BORDER_NONE);
	REGISTER_GDC_CLASS_CONST_LONG("BORDER_ALL", GDC_BORDER_ALL);
	REGISTER_GDC_CLASS_CONST_LONG("BORDER_X", GDC_BORDER_X);
	REGISTER_GDC_CLASS_CONST_LONG("BORDER_Y", GDC_BORDER_Y);
	REGISTER_GDC_CLASS_CONST_LONG("BORDER_Y2", GDC_BORDER_Y2);
	REGISTER_GDC_CLASS_CONST_LONG("BORDER_TOP", GDC_BORDER_TOP);

	REGISTER_GDC_CLASS_CONST_LONG("TICK_LABELS", GDC_TICK_LABELS);
	REGISTER_GDC_CLASS_CONST_LONG("TICK_POINTS", GDC_TICK_POINTS);
	REGISTER_GDC_CLASS_CONST_LONG("TICK_NONE", GDC_TICK_NONE);

	REGISTER_GDC_CLASS_CONST_LONG("STACK_DEPTH", GDC_STACK_DEPTH);
	REGISTER_GDC_CLASS_CONST_LONG("STACK_SUM", GDC_STACK_SUM);
	REGISTER_GDC_CLASS_CONST_LONG("STACK_BESIDE", GDC_STACK_BESIDE);
	REGISTER_GDC_CLASS_CONST_LONG("STACK_LAYER", GDC_STACK_LAYER);

	REGISTER_GDC_CLASS_CONST_LONG("HLC_DIAMOND", GDC_HLC_DIAMOND);
	REGISTER_GDC_CLASS_CONST_LONG("HLC_CLOSE_CONNECTED", GDC_HLC_CLOSE_CONNECTED);
	REGISTER_GDC_CLASS_CONST_LONG("HLC_CONNECTING", GDC_HLC_CONNECTING);
	REGISTER_GDC_CLASS_CONST_LONG("HLC_I_CAP", GDC_HLC_I_CAP);

	REGISTER_GDC_CLASS_CONST_LONG("SCATTER_TRIANGLE_DOWN", GDC_SCATTER_TRIANGLE_DOWN);
	REGISTER_GDC_CLASS_CONST_LONG("SCATTER_TRIANGLE_UP", GDC_SCATTER_TRIANGLE_UP);
	REGISTER_GDC_CLASS_CONST_LONG("SCATTER_CIRCLE", GDC_SCATTER_CIRCLE);

	REGISTER_GDC_CLASS_CONST_LONG("GIF", GDC_GIF);
	REGISTER_GDC_CLASS_CONST_LONG("JPEG", GDC_JPEG);
	REGISTER_GDC_CLASS_CONST_LONG("PNG", GDC_PNG);
	REGISTER_GDC_CLASS_CONST_LONG("WBMP", GDC_WBMP);

	zend_declare_class_constant_double(gdc_class_entry, "NOVALUE", sizeof("NOVALUE")-1, (double)GDC_NOVALUE TSRMLS_CC);

	return SUCCESS;
}
/* }}} */

/* {{{ PHP_MINFO_FUNCTION
 */
PHP_MINFO_FUNCTION(gdchart)
{
	php_info_print_table_start();
	php_info_print_table_header(2, "gdchart support", "enabled");
	php_info_print_table_row(2, "Based on gdchart version", "0.11.4dev");
	php_info_print_table_end();
}
/* }}} */

/* {{{ proto object GDChart::__construct(long type)
 * Create a new GDChart object, specifying the type of graph to generate
 */
PHP_METHOD(GDChart, __construct)
{
	long type;
	zval *obj = getThis();
	struct gdc_object *gdc = (struct gdc_object*)zend_object_store_get_object(obj TSRMLS_CC); 

	if (zend_parse_parameters(ZEND_NUM_ARGS(), "l", &type) == FAILURE) {
		RETURN_FALSE;
	}

	zend_update_property_long(gdc->zo.ce, obj, "type", sizeof("type") - 1, type TSRMLS_CC);
}
/* }}} */

/* {{{ proto bool GDChart::addScatter(double val, double point, long index, long color, long width[, bool clear])
 * Add scatter element set
 */
PHP_METHOD(GDChart, addScatter)
{
	double val, point;
	long ind, color, width=6;
	int i;
	struct gdc_object *gdc = (struct gdc_object*)zend_object_store_get_object(getThis() TSRMLS_CC); 
	zend_bool clear = FALSE;

	if (zend_parse_parameters(ZEND_NUM_ARGS(), "ddll|l", &val, &point, &ind, &color, &width, &clear) == FAILURE) {
		RETURN_FALSE;
	}

	if (clear) {
		PHP_GDC_KEY_FREE(gdc->scatter)
		gdc->n_scatter = 0;
	}

	i = gdc->n_scatter++;
	gdc->scatter = erealloc(gdc->scatter, sizeof(GDC_SCATTER_T) * gdc->n_scatter);
	gdc->scatter[i].val = (float) val;
	gdc->scatter[i].point = (float) point;
	gdc->scatter[i].ind = (GDC_SCATTER_IND_T) ind;
	gdc->scatter[i].color = (unsigned long) color;
	gdc->scatter[i].width = (unsigned short) width;

	RETURN_TRUE;
}
/* }}} */

/* {{{ proto bool GDChart::setLabels(array labels)
 * Add an array of strings to be used as labels for the generated chart.
 */
PHP_METHOD(GDChart, setLabels)
{
	zval *set, **val;
	int cnt, i = 0;
	struct gdc_object *gdc = (struct gdc_object*)zend_object_store_get_object(getThis() TSRMLS_CC); 

	if (zend_parse_parameters(ZEND_NUM_ARGS(), "a", &set) == FAILURE) {
		RETURN_FALSE;
	}

	GDC_FREE_LABELS(gdc->labels);

	/* no labels handler */
	if (!(cnt = zend_hash_num_elements(Z_ARRVAL_P(set)))) {
		RETURN_TRUE;
	}

	gdc->labels = (char **)safe_emalloc((cnt+1), sizeof(char *), 0);
	zend_hash_internal_pointer_reset(Z_ARRVAL_P(set));
	while (zend_hash_get_current_data(Z_ARRVAL_P(set), (void **)&val) == SUCCESS) {
		convert_to_string_ex(val);
		gdc->labels[i++] = estrndup(Z_STRVAL_PP(val), Z_STRLEN_PP(val));
		zend_hash_move_forward(Z_ARRVAL_P(set));
	}
	gdc->labels[i] = NULL;

	RETURN_TRUE;
}
/* }}} */

static void _php_gdc_set_array(INTERNAL_FUNCTION_PARAMETERS, int type)
{
	zval *set, **val;
	int cnt, i = 0;
	unsigned long *vals;
	struct gdc_object *gdc = (struct gdc_object*)zend_object_store_get_object(getThis() TSRMLS_CC);

	if (zend_parse_parameters(ZEND_NUM_ARGS(), "a", &set) == FAILURE) {
		RETURN_FALSE;
	}

	/* no colors handler */
	if (!(cnt = zend_hash_num_elements(Z_ARRVAL_P(set)))) {
		vals = NULL;
	} else {
		vals = safe_emalloc(cnt, sizeof(unsigned long), 0);
		zend_hash_internal_pointer_reset(Z_ARRVAL_P(set));
		while (zend_hash_get_current_data(Z_ARRVAL_P(set), (void **)&val) == SUCCESS) {
			convert_to_long_ex(val);
			vals[i++] = (unsigned long)Z_LVAL_PP(val);
			zend_hash_move_forward(Z_ARRVAL_P(set));
		}
	}

	switch (type) {
		case 1:
			PHP_GDC_KEY_FREE(gdc->colors)
			gdc->colors = vals;
			gdc->counts[0] = cnt;
			break;
		case 2:
			PHP_GDC_KEY_FREE(gdc->ext_color)
			gdc->ext_color = vals;
			gdc->counts[1] = cnt;
			break;
		case 3:
			PHP_GDC_KEY_FREE(gdc->ext_vol_color)
			gdc->ext_vol_color = vals;
			gdc->counts[2] = cnt;
			break;
		case 4:
			PHP_GDC_KEY_FREE(gdc->explode)
			gdc->explode = vals;
			gdc->counts[3] = cnt;
			break;
		default:
			return;
	}

	RETURN_TRUE;
}

/* {{{ proto bool GDChart::setColors(array colors)
 * Add an array of numbers to be used as colors for the entire set of data.
 */
PHP_METHOD(GDChart, setColors)
{
	_php_gdc_set_array(INTERNAL_FUNCTION_PARAM_PASSTHRU, 1);
}
/* }}} */

/* {{{ proto bool GDChart::setExtColors(array colors)
 * Add an array of numbers to be used as colors for the individual data items.
 */
PHP_METHOD(GDChart, setExtColors)
{
	_php_gdc_set_array(INTERNAL_FUNCTION_PARAM_PASSTHRU, 2);	
}
/* }}} */

/* {{{ proto bool GDChart::setExtVolColors(array colors)
 * Add an array of numbers to be used as secondary data individual point plot color.
 */
PHP_METHOD(GDChart, setExtVolColors)
{
	_php_gdc_set_array(INTERNAL_FUNCTION_PARAM_PASSTHRU, 3);
}
/* }}} */

/* {{{ proto bool GDChart::setExplode(array colors)
 * ndicate a list of sections to explode for pie charts
 */
PHP_METHOD(GDChart, setExplode)
{
	_php_gdc_set_array(INTERNAL_FUNCTION_PARAM_PASSTHRU, 4);
}
/* }}} */

/* {{{ proto bool GDChart::setMissing(array missing)
 * Indicate a list of missing sections for pie charts
 */
PHP_METHOD(GDChart, setMissing)
{
	zval *set, **val;
	int cnt, i = 0;
	struct gdc_object *gdc = (struct gdc_object*)zend_object_store_get_object(getThis() TSRMLS_CC); 

	if (zend_parse_parameters(ZEND_NUM_ARGS(), "a", &set) == FAILURE) {
		RETURN_FALSE;
	}

	PHP_GDC_KEY_FREE(gdc->missing)

	if (!(cnt = zend_hash_num_elements(Z_ARRVAL_P(set)))) {
		RETURN_TRUE;
	}

	gdc->missing = safe_emalloc(cnt, sizeof(unsigned char), 0);
	zend_hash_internal_pointer_reset(Z_ARRVAL_P(set));
	while (zend_hash_get_current_data(Z_ARRVAL_P(set), (void **)&val) == SUCCESS) {
		convert_to_boolean_ex(val);
		gdc->missing[i++] = Z_BVAL_PP(val);
		zend_hash_move_forward(Z_ARRVAL_P(set));
	}

	RETURN_TRUE;
}
/* }}} */

/* {{{ proto bool GDChart::addValues(array values[, bool clear])
 * Add an array of floating point values to be used as the data range for the graph.
 * Unless clear parameter is indicated, each call adds a new set of points.
 */
PHP_METHOD(GDChart, addValues)
{
	zval *set, **val;
	struct gdc_object *gdc = (struct gdc_object*)zend_object_store_get_object(getThis() TSRMLS_CC); 
	int i = gdc->n_values;
	zend_bool clear = FALSE;

	if (zend_parse_parameters(ZEND_NUM_ARGS(), "a|b", &set, &clear) == FAILURE) {
		RETURN_FALSE;
	}

	if (clear) {
		gdc->n_values = gdc->n_points = 0;
		PHP_GDC_KEY_FREE(gdc->values);
	}

	if (!gdc->n_points) {
		gdc->n_points = zend_hash_num_elements(Z_ARRVAL_P(set));
	} else if (gdc->n_points != zend_hash_num_elements(Z_ARRVAL_P(set))) {
		php_error_docref(NULL, E_WARNING, "Incorrect number of points, expecting %d, got %d", gdc->n_points, zend_hash_num_elements(Z_ARRVAL_P(set))); 
		RETURN_FALSE;
	}

	gdc->n_values += gdc->n_points;
	gdc->values = erealloc(gdc->values, gdc->n_values * sizeof(float));

	zend_hash_internal_pointer_reset(Z_ARRVAL_P(set));
	while (zend_hash_get_current_data(Z_ARRVAL_P(set), (void **)&val) == SUCCESS) {
		convert_to_double_ex(val);
		gdc->values[i++] = (float)Z_DVAL_PP(val);
		zend_hash_move_forward(Z_ARRVAL_P(set));
	}

	RETURN_TRUE;
}
/* }}} */

/* {{{ proto bool GDChart::addCombo(array values[, bool clear])
 * Add an array of floating point values to be used as the data range for combo graph.
 * Unless clear parameter is indicated, each call adds a new set of points.
 */
PHP_METHOD(GDChart, addCombo)
{
	zval *set, **val;
	struct gdc_object *gdc = (struct gdc_object*)zend_object_store_get_object(getThis() TSRMLS_CC); 
	int cnt, i = gdc->n_combo;
	zend_bool clear = FALSE;

	if (zend_parse_parameters(ZEND_NUM_ARGS(), "a|b", &set, &clear) == FAILURE) {
		RETURN_FALSE;
	}

	if (clear) {
		PHP_GDC_KEY_FREE(gdc->combo);
		gdc->n_combo = 0;
	}

	cnt = zend_hash_num_elements(Z_ARRVAL_P(set));
	if (!cnt) {
		RETURN_FALSE;
	}

	gdc->n_combo += cnt;	
	gdc->combo = erealloc(gdc->combo, gdc->n_combo * sizeof(float));

	zend_hash_internal_pointer_reset(Z_ARRVAL_P(set));
	while (zend_hash_get_current_data(Z_ARRVAL_P(set), (void **)&val) == SUCCESS) {
		convert_to_double_ex(val);
		gdc->combo[i++] = (float)Z_DVAL_PP(val);
		zend_hash_move_forward(Z_ARRVAL_P(set));
	}

	RETURN_TRUE;
}
/* }}} */

/* {{{ proto bool GDChart::out(int width, int height, int image_type[, string filename])
 * Output chart of width x height size as image_type image to screen or specified filename.
 */
PHP_METHOD(GDChart, out)
{
	zval *obj = getThis();
	struct gdc_object *gdc = (struct gdc_object*)zend_object_store_get_object(obj TSRMLS_CC); 
	long width, height, image_type = GDC_PNG;
	zval *type, *tmp;
	char *fname = NULL;
	int fname_len = 0, i = 0, ret = 0;
	FILE *fp = NULL;
	GDC_ANNOTATION_T anno;

	if (zend_parse_parameters(ZEND_NUM_ARGS(), "ll|ls", &width, &height, &image_type, &fname, &fname_len) == FAILURE) {
		RETURN_FALSE;	
	}

	/* determine the type of image to output */
	if (image_type < GDC_GIF || image_type > GDC_WBMP) {
		php_error_docref(NULL, E_WARNING, "Invalid output file type specified"); 
		RETURN_FALSE;
	}
	GDC_image_type = image_type;

	/* determine type of graph to generate */
	type = zend_read_property(gdc->zo.ce, obj, "type", sizeof("type") - 1, 0 TSRMLS_CC);	
	if (Z_TYPE_P(type) != IS_LONG) {
		convert_to_long_ex(&type);
	}
	if (Z_LVAL_P(type) < -2 || Z_LVAL_P(type) > GDC_3DLINE) {
		php_error_docref(NULL, E_WARNING, "Invalid graph type specified"); 
		RETURN_FALSE;
	}

	/* handle all long opts */
	for (i = 0; i < PHP_GDC_LONG_OPT; i++) {
		tmp = zend_read_property(gdc->zo.ce, obj, php_gdc_long_map[i].name, php_gdc_long_map[i].name_len, 0 TSRMLS_CC);
		convert_to_long_ex(&tmp);
		*((long *)php_gdc_long_map[i].value) = Z_LVAL_P(tmp);
	}

	/* handle all string opts */
	for (i = 0; i < PHP_GDC_STR_OPT; i++) {
		tmp = zend_read_property(gdc->zo.ce, obj, php_gdc_string_map[i].name, php_gdc_string_map[i].name_len, 0 TSRMLS_CC);
		convert_to_string_ex(&tmp);
		*((char **)php_gdc_string_map[i].value) = Z_STRLEN_P(tmp) ? (void *) Z_STRVAL_P(tmp) : NULL;
	}

	/* handle all double/float opts */
	for (i = 0; i < PHP_GDC_DOUBLE_OPT; i++) {
		tmp = zend_read_property(gdc->zo.ce, obj, php_gdc_double_map[i].name, php_gdc_double_map[i].name_len, 0 TSRMLS_CC);
		convert_to_double_ex(&tmp);
		*((double *)php_gdc_double_map[i].value) = Z_DVAL_P(tmp);
	}

	tmp = zend_read_property(gdc->zo.ce, obj, "annoNote", sizeof("annoNote") - 1, 0 TSRMLS_CC);
	convert_to_string_ex(&tmp);
	if (Z_STRLEN_P(tmp)) {
		int len = MIN(Z_STRLEN_P(tmp), MAX_NOTE_LEN);
		memcpy(anno.note, Z_STRVAL_P(tmp), len);
		anno.note[len] = '\0';

		tmp = zend_read_property(gdc->zo.ce, obj, "annoPoint", sizeof("annoPoint") - 1, 0 TSRMLS_CC);
		convert_to_double_ex(&tmp);
		anno.point = (float) Z_DVAL_P(tmp);

		tmp = zend_read_property(gdc->zo.ce, obj, "annoColor", sizeof("annoColor") - 1, 0 TSRMLS_CC);
		convert_to_long_ex(&tmp);
		anno.color = (unsigned long) Z_LVAL_P(tmp);
	}

	tmp = zend_read_property(gdc->zo.ce, obj, "labelLine", sizeof("labelLine") - 1, 0 TSRMLS_CC);
	convert_to_boolean_ex(&tmp);
	GDCPIE_label_line = Z_BVAL_P(tmp);

	if (gdc->colors) {
		GDC_SetColor = GDCPIE_Color = gdc->colors;
	} else {
		GDC_SetColor = GDCPIE_Color = gdc_default_colors;
	}

	GDCPIE_missing = gdc->missing;
	GDCPIE_explode = gdc->explode;
	GDC_ExtColor = gdc->ext_color;
	GDC_ExtVolColor = gdc->ext_vol_color;
	GDC_num_scatter_pts = gdc->n_scatter;
	GDC_scatter = gdc->scatter;

	/* handle duplicates */
	GDCPIE_title = GDC_title;
	GDCPIE_BGColor = GDC_BGColor;
	GDCPIE_title_font = GDC_title_font;
	GDCPIE_title_size = GDC_title_size;
	GDCPIE_LineColor = GDC_LineColor;
	GDCPIE_PlotColor = GDC_PlotColor;
	GDCPIE_title_ptsize = GDC_title_ptsize;

	if (fname_len) {
		if (php_check_open_basedir(fname)) {
			RETURN_FALSE;
		}
		fp = VCWD_FOPEN(fname, "wb");
		if (!fp) {
			php_error_docref(NULL, E_WARNING, "Unable to open '%s' for writing", fname); 
			RETURN_FALSE;
		}
	}

	switch(Z_LVAL_P(type)) {
		case -2: /* 2D Pie */
		case -1: /* 3D Pie */
			GDC_out_pie(width,height,fp,Z_LVAL_P(type)+2,gdc->n_points,gdc->labels,gdc->values);
			RETVAL_TRUE;
			break;
		case GDC_HILOCLOSE:
		case GDC_COMBO_HLC_BAR:
		case GDC_COMBO_HLC_AREA:
		case GDC_3DHILOCLOSE:
		case GDC_3DCOMBO_HLC_BAR:
		case GDC_3DCOMBO_HLC_AREA:
			ret = GDC_out_graph(width,height,fp,Z_LVAL_P(type),gdc->n_points,gdc->labels,(int)(gdc->n_values / gdc->n_points / 3),gdc->values,gdc->combo);
			RETVAL_BOOL(ret ? TRUE : FALSE);
			break;
		case GDC_FLOATINGBAR:
		case GDC_3DFLOATINGBAR:
			ret = GDC_out_graph(width,height,fp,Z_LVAL_P(type),gdc->n_points,gdc->labels,(int)(gdc->n_values / gdc->n_points / 2),gdc->values,gdc->combo);
			RETVAL_BOOL(ret ? TRUE : FALSE);
			break;
		default:
			ret = GDC_out_graph(width,height,fp,Z_LVAL_P(type),gdc->n_points,gdc->labels,(int)(gdc->n_values / gdc->n_points),gdc->values,gdc->combo);
			RETVAL_BOOL(ret ? TRUE : FALSE);
			break;
	}

	if (fp) {
		fflush(fp);
		fclose(fp);
	}
}
/* }}} */

/*
 * Local variables:
 * tab-width: 4
 * c-basic-offset: 4
 * End:
 * vim600: noet sw=4 ts=4 fdm=marker
 * vim<600: noet sw=4 ts=4
 */
