/*  xbaseconv.cpp

   Xbase to Xbase converter

   This program converts one DBF file into another DBF file

   Copyright (C) 1998,1999 Piotr Klaban
   email - makler@man.torun.pl

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   V 0.9   15/11/98   - Start working on the program
   V 1.0   31/08/99   - get rid of the message "can not open conversion table"
			message is now visible if verbose flag is used
 */

#include <iostream>
#include <fstream>
#include <sstream>
#include <string>
#include <xbase.h>
#include <strings.h>
#include <ctype.h>
#include <unistd.h>
#include "xbase2pg.hh"

/* set the stack large for dos compiles */
#ifdef DOS
#include <stdio.h>
extern unsigned _stklen = 100000;
#endif

char *subarg = NULL, *convarg = NULL;
int verbose = 0, upper = 0, lower = 0, create = 0, clear = 0, begin = -1,
  stripcont = 0,
  end = -1, do_conv = 0, fieldlower = 0, fieldupper = 0, dumpheader = 0,
  overlay = XB_DONTOVERLAY, memopresent = 0;
xbShort rc;
#ifdef CONVTABLE
char *convtable = CONVTABLE;
#else
char *convtable = "/etc/convtable";
#endif
void Usage (void);
void ConvertStr (char *);
void ReadConvEncodings (char *);
int SetConvEncodings (char *);
xbULong WriteRecs (void);
void ReadSchema (void);
void SubstNames (char *);

char ConvTable[256];

void
Usage (void)
{
	cerr << "xbaseconv (pg2xbase v" << PG2XBASE_VERSION << ")" << endl <<
		"Usage:\txbaseconv [-O] [-U | -L] [-u | -l] [-r] [-C | -D] [-v[v]] [-d[d]]" << endl <<
		"\t\t\t[-b begin] [-e end] [-s oldname=newname[,oldname=newname]]" << endl <<
		"\t\t\t[-V dbfversion] [-G convtablefile] [-g convfrom=convto]" << endl <<
		"\t\t\tinput-dbffile output-dbffile" << endl;
}

xbSchema *MySchema;
class xbDbf *XbaseIn;
class xbDbf *XbaseOut;
AssocArray< string,Table<char,18,0> > ConvStand;

int
main (int argc, char **argv)
{
	char *dbfin = NULL, *dbfout = NULL;
	extern int optind;
	extern char *optarg;
	int i, dbfversion = DEF_DBF_VERSION;

	while ((i = getopt (argc, argv, "b:CdDe:g:G:lLOrs:uUvV:")) != EOF)
	  {
		  switch (i)
		    {
		    case 'b':
			    begin = atoi (optarg);
			    break;
		    case 'C':
			    if (clear)
			      {
				      Usage ();
				      cerr << "Can't use -C and -D at the same time!" << endl;
				      exit (1);
			      }
			    create = 1;
			    break;
		    case 'd':
			    ++dumpheader;
			    if (dumpheader > 3)
				    dumpheader = 3;
			    break;
		    case 'D':
			    if (create)
			      {
				      Usage ();
				      cerr << "Can't use -C and -D at the same time!" << endl;
				      exit (1);
			      }
			    clear = 1;
			    break;
		    case 'e':
			    end = atoi (optarg);
			    break;
		    case 'g':
			    convarg = strdup (optarg);
			    break;
		    case 'G':
			    convtable = strdup (optarg);
			    break;
		    case 'l':
			    if (upper)
			      {
				      Usage ();
				      cerr << "Can't use -u and -l at the same time!" << endl;
				      exit (1);
			      }
			    lower = 1;
			    break;
		    case 'L':
			    if (fieldupper)
			      {
				      Usage ();
				      cerr << "Can't use -f and -F at the same time!" << endl;
				      exit (1);
			      }
			    fieldlower = 1;
			    break;
		    case 'O':
			    overlay = XB_OVERLAY;
			    break;
		    case 'r':
                stripcont = 1;
			    break;
		    case 's':
			    subarg = strdup(optarg);
			    break;
		    case 'u':
			    if (lower)
			      {
				      Usage ();
				      cerr << "Can't use -u and -l at the same time!" << endl;
				      exit (1);
			      }
			    upper = 1;
			    break;
		    case 'U':
			    if (fieldlower)
			      {
				      Usage ();
				      cerr << "Can't use -f and -F at the same time!" << endl;
				      exit (1);
			      }
			    fieldupper = 1;
			    break;
		    case 'v':
			    ++verbose;
			    break;
		    case 'V':
			    dbfversion = atoi (optarg);
			    break;
		    case '?':
			    Usage ();
			    cerr << "Unknown argument: %s" << argv[0] << endl;
			    exit (1);
		    default:
			    break;
		    }
	  }

	argc -= optind;
	argv = &argv[optind];

	if ((argc != 2) && (argc != 1))
	  {
		  Usage ();
		  exit (1);
	  }
	else if (argc == 1)
	  {
		  dumpheader = dumpheader > 0 ? dumpheader : 1;
		  dbfin = argv[0];
	  }
	else
	  {
		  dbfin = argv[0];
		  dbfout = argv[1];
	  }

	if ((begin > end) && (end != -1))
	  {
		  Usage ();
		  cerr << "First record could not be greater than the last one" << endl;
		  exit (1);
	  }

	ReadConvEncodings (convtable);
	if (SetConvEncodings (convarg) < 0)
	  {
		  Usage ();
		  exit (1);
	  }
	else
		do_conv = 1;

	xbXBase x;
	xbDbf DBFtemp1 (&x);
	XbaseIn = &DBFtemp1;

	if ((rc = XbaseIn->OpenDatabase (dbfin)) != XB_NO_ERROR)
	  {
		cout << "ERROR: opening "<< dbfin << ": "; x.DisplayError(rc);
	  }
	else
	  {

#ifdef XB_MEMO_FIELDS
		  memopresent = XbaseIn->MemoFieldsPresent ();
#endif

		  if (dumpheader > 0)
		    {
			    XbaseIn->DumpHeader (dumpheader);
			    return 0;
		    }

		  ReadSchema ();
		  SubstNames (subarg);

		  xbXBase y;
		  xbDbf DBFtemp2 (&y);
		  XbaseOut = &DBFtemp2;

		  // check DB Version
		  XbaseOut->SetVersion (dbfversion);

		  if (create)
		    {
			    rc = XbaseOut->CreateDatabase (dbfout, MySchema, overlay);
		    }
		  else
		    {
			    rc = XbaseOut->OpenDatabase (dbfout);
		    }

		  if (rc != XB_NO_ERROR)
		    {
			cout << "ERROR: creating " << dbfout << ": "; x.DisplayError(rc);
		    }
		  else
		    {
			    if (clear)
				    XbaseOut->Zap (F_SETLKW);

			    WriteRecs ();
			    XbaseOut->CloseDatabase ();		/* Close database and associated indexes */
		    }
	  }

	XbaseIn->CloseDatabase ();	/* Close database and associated indexes */

}


void 
SubstNames (char *subarg)
{
	if (!subarg)
		  return;
	if (verbose > 1)
		cout << "Substituting new field names" << endl;

	int numfields = XbaseIn->FieldCount ();

	AssocArray<string,string> subst;
	subst.SetValues(string(subarg));

	string name, newname;

	for (register int i = 0; i < numfields; ++i)
	{
	    name.assign(MySchema[i].FieldName);
	    if (subst.exist(name)) {
	    	newname = *subst[name];
		strcpy (MySchema[i].FieldName, newname.c_str());
		if (verbose > 1)
		    cout << "Substitute old:" << name << " new:" << newname << endl;
	   }
	}
}

void
ReadSchema ()
{
	int numfields, i;

	numfields = XbaseIn->FieldCount ();
	MySchema = new xbSchema[numfields + 1];
	for (i = 0; i < numfields; ++i)
	  {
		  strcpy (MySchema[i].FieldName, XbaseIn->GetFieldName (i));
		  MySchema[i].Type = XbaseIn->GetFieldType (i);
		  MySchema[i].FieldLen = XbaseIn->GetFieldLen (i);
		  MySchema[i].NoOfDecs = XbaseIn->GetFieldDecimal (i);
	  }
}

xbULong
WriteRecs ()
{
	char *contents = NULL, *fname, ftype;
	int i, numfields, numrecords, flen, recs = 0;
	xbShort t, rc;
	xbLong cbufsize = 0;
#ifdef XB_MEMO_FIELDS
	xbLong mbufsize = 0, mlen = 0;
	char *mcontents = NULL;
#endif

	if (verbose > 0)
		cout << "Begin writing records ..." << endl;
	if (verbose == 2)
		cout << "Each dot is 500 records " << flush;

	numfields = XbaseIn->FieldCount ();
	numrecords = XbaseIn->NoOfRecords ();

	if (begin == -1)
		begin = 1;
	if ((end == -1) || (end > numrecords))
		end = numrecords;

#ifdef XB_MEMO_FIELDS
#ifdef XB_LOCKING_ON
	if (memopresent)
		XbaseIn->LockMemoFile (F_SETLK, F_RDLCK);
#endif
#endif
	for (i = begin; i <= end; ++i)
	  {
		  rc = XbaseIn->GetRecord (i);
		  if (rc == XB_NO_ERROR)
		    {
			    XbaseOut->BlankRecord ();
			    for (t = 0; t < numfields; ++t)
			      {
				      fname = XbaseIn->GetFieldName (t);
				      if (!strlen (fname))
					      continue;
				      ftype = XbaseIn->GetFieldType (t);
				      flen = XbaseIn->GetFieldLen (t);
				      if (cbufsize < flen)
					{
						if (contents)
							delete contents;
						contents = new char[cbufsize = flen + 1];
					}
				      XbaseIn->GetField (t, contents);

				      if (verbose > 4)
					      cout << "Reading " << t << "/" << i << " rec: '" << contents << "'" << endl;

				      if (ftype == XB_MEMO_FLD)
					{
#ifdef XB_MEMO_FIELDS
						mlen = XbaseIn->GetMemoFieldLen (t);
						if (mlen <= 0)
						  {
							  cerr << "Error zero Memo field no. " << t << " of record " << i << endl;
							  continue;
						  }
						if (mbufsize < mlen)
						  {
							  if (mcontents)
								  delete mcontents;
							  mcontents = new char[mbufsize = mlen + 1];
						  }
						if ((rc = XbaseIn->GetMemoField (t, mlen, mcontents, F_SETLKW)) == XB_NO_ERROR)
						  {
							  mcontents[mlen] = '\0';
							  if (do_conv)
								  ConvertStr (mcontents);
							  if (upper)
								  strtoupper (mcontents);
							  if (lower)
								  strtolower (mcontents);
							  if (stripcont)
								  strstrip (mcontents);
						  }
						else
						  {
							  cerr << "Error " << rc << " reading Memo field no. " << t << " of record " << i << endl;
							  continue;
						  }
						XbaseOut->UpdateMemoData (t, mlen, mcontents, F_SETLKW);
#else
						cerr << "Unknown MEMO dbf field no. " << t << ": " << contents << endl;
						continue;
#endif
					}
				      else
					{
				      if (verbose > 3)
					      cout << "Putting " << t << "/" << i << " rec: '" << contents << "'" << endl;
						if (do_conv)
							ConvertStr (contents);
						if (upper)
							strtoupper (contents);
						if (lower)
							strtolower (contents);
						if (stripcont)
							strstrip (contents);

						XbaseOut->PutField (t, contents);
					}
				      if (verbose > 3)
					      cout << "Putting " << t << "/" << i << " rec: '" << contents << "'" << endl;
				      delete contents;
			      }
			    if ((rc = XbaseOut->AppendRecord ()) != XB_NO_ERROR)
				    cout << endl << "Error " << rc << " appending data record." << endl;
			    else if (verbose > 2)
				    cout << "Appending record no. " << i << endl;
			    ++recs;
			    if ((verbose == 2) && ((recs % 500) == 0))
			    {
				cout << "." << flush;
			    }
		    }		/* if */
	  }			/* i */
	  if (verbose == 2)
	  	cout << endl;
#ifdef XB_MEMO_FIELDS
#ifdef XB_LOCKING_ON
	if (memopresent)
		XbaseIn->LockMemoFile (F_SETLK, F_UNLCK);
#endif
#endif
	if (verbose > 0)
		cout << recs << " records has been written (total " << numrecords << ")" << endl;
	return 1;
}

void ConvertStr (char *);
void read_conv_table (char *);

void 
ConvertStr (char *p)
{
	char *q;

	q = p;
	if (!p)
		return;
	if (verbose > 5)
		cout << "Converting string '" << p << "' into the string '";
	while (*p != '\0') {
		  *p = ConvTable[(unsigned int)((unsigned char)(*p))];
		  ++p;
	}
	if (verbose > 5)
		cout << q << "'" << endl;
}

inline void
ConvertStr (string* str)
{
	Point_to_Mem<char> loc(str->length());
	strcpy(loc,str->c_str());
	ConvertStr(loc);
	str->assign(loc);
}

void
ReadConvEncodings (char *file)
{
	int i;
	string str, name;

	ifstream in(file);
	if (in)
	  {
	  	if (verbose > 1)
		  cout << "opening converion table: " << file << endl;
		for (;;)
		{
			if (getline(in, str).eof())
				break;
			if (in.eof()) break;
			if (!in.good())
				in.clear();
			if ((str.empty()) || (str[0] == '#')) continue;
			strtolower(&str);
			istringstream charin(str.c_str());
			charin >> name;
			ConvStand.set(name);
			Table<char,18,0> *tabname = ConvStand[name];
			if (verbose > 3)
				cout << "Read " << name << ": \t" << flush;
			size_t nr = 0;
			for (;nr < 18;) {
				if (!(charin >> i)) {
				    if (charin.eof() || i == 0)
				        break;
				    else {
					cerr << "Conversion table parse error in line: " << str << " " << nr << " " << i << endl;
					exit(2);
				    }
				}
				if (!charin.good()) charin.clear();
				if (verbose > 3)
					cout << i << " " << flush;
				(*tabname)[nr++] = i;
			}
			if (verbose > 3)
				cout << endl;
		}
	  }
	else
	  {
	  	if (verbose > 1)
		  cerr << "WARNING: can not open conversion table " << file << endl;
	  }
}

/*
 * SetConvEncodings("mazovia=iso-latin-2");
 */
int
SetConvEncodings (char *convarg)
{
	register unsigned int i;
	size_t s;

	if (!convarg)
		return 0;

	strtolower(convarg);
	string buffor(convarg);
	if (!(s = buffor.find('=')))
	  {
		  cerr << "Invalid conversion definition: " << buffor << endl;
		  return -1;
	  }

	string from = buffor.substr(0, s);
	string to = buffor.substr(s + 1);

	Table<char,18,0> *fromtab, *totab;
	fromtab = *ConvStand[from];
	totab = *ConvStand[to];
	if (totab == NULL || fromtab == NULL)
	  {
		  cerr << "Unknown standard: " << ((totab == NULL) ? to : from) << endl;
		  return -1;
	  }
#ifdef DEBUG
	if (verbose) {
	    fromtab->dump();
	    totab->dump();
	}
#endif
	for (i = 0; i < 256; ++i)
		  ConvTable[i] = char(i);

	for (i = 0; i < 18; ++i) {
#ifdef DEBUG
		if (verbose)
		cout << "Setting " << i << " ConvTable[" << (int)(unsigned char)((*fromtab)[i])
		<< "] = " << char((*totab)[i]) << endl;
#endif
		  ConvTable[(unsigned int)((unsigned char)(*fromtab)[i])] = char((*totab)[i]);
	}
	return 0;
}
