/*
   xbase2pg.hh

   Xbase to Postgres database converter header file

   This program converts one DBF file into Postgres table

   Copyright (C) 1998,1999 Piotr Klaban
   email - makler@man.torun.pl

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

*/

# define PG2XBASE_VERSION "2.4.1"

# define PG2XBASE_FORMAT_DATE "YYYY-MM-DD"

# define DEF_DBF_VERSION 4

#ifndef DEF_PGPORT_STR
# define DEF_PGPORT_STR "5432"
#endif

using namespace std;

void Usage (void);

template < class T > inline T MIN (const T x, const T y)
{
	return (y > x) ? x : y;
}

template < class T > class Point_to_Mem
{
      public:
	T * p;
	size_t Tsize;

	Point_to_Mem (size_t size)
	{
		p = NULL;
		create(size);
	}
	Point_to_Mem ()
	{
		p = NULL;
		Tsize = 0;
	}
	~Point_to_Mem ()
	{
		delete[]p;
	}
	void renew (size_t newsize)
	{
		T *q;
		if (!p)
		  {
			  create(newsize);
			  return;
		  }
		q = new T[Tsize];
		if (!p)
		  {
			  cerr << "Memory allocation error for size " << Tsize << endl;
			  return;
		  }
		memcpy (q, p, Tsize * sizeof (T));
		delete[]p;
		p = new T[newsize];
		if (!p)
		  {
			  cerr << "Memory allocation error for size " << newsize << endl;
			  delete[]q;
			  return;
		  }
		memcpy (p, q, MIN (Tsize, newsize) * sizeof (T));
		Tsize = newsize;
		delete[]q;
	}
	void create (size_t newsize)
	{
		if (p)
			delete[]p;
		p = new T[newsize];
		if (!p)
		  {
			  cerr << "Memory allocation error for size " << newsize << endl;
			  return;
		  }
		memset (p, 0, newsize);
		Tsize = newsize;
	}
	operator T *()
	{
		return p;
 	}
};

template < class T, size_t S, T def >
class Table
{
	int size;
      public:
	Point_to_Mem<T> tab;
	
	Table() : tab(S) {
		size = int(S);
		for (register int i=0; i < size; i++) {
			tab[i] = def;
		}
	}

	Table& operator= (const Table& right) {
		for (register int i=0; i < size; i++) {
			tab[i] = right[i];
		}
	}

	bool operator== (const Table& right) {
		for (register int i=0; i < size; i++) {
			if (!(tab[i] == right[i]))
				return false;
		}
		return true;
	}
	T& operator[] (size_t pos) {
		if (pos >= size_t(size)) cerr << "Index " << pos << " outside of the size range " << size << endl ;
		return tab[pos];
	}
	operator Table *() {
		return this;
	}
#ifdef DEBUG
	void dump() {
		cout << "Dump Table " << this << " contents" << endl;
		for (register int i=0; i < size; i++) {
			cout << "Nr=" << i << " Val=" << tab[i] << endl;
		}
	}
#endif
};

inline void 
strtoupper (char *str)
{
	while (*str != '\0') {
		  *str = toupper (*str);
		  ++str;
	}
}

inline void 
strtolower (char *str)
{
	while (*str != '\0') {
		  *str = tolower (*str);
		  ++str;
	}
}

inline void 
strtoupper (string* str)
{
	Point_to_Mem<char> loc(str->length());
	strcpy(loc,str->c_str());
	strtoupper(loc);
	str->assign(loc);
}

inline void 
strtolower (string* str)
{
	Point_to_Mem<char> loc(str->length());
	strcpy(loc,str->c_str());
	strtolower(loc);
	str->assign(loc);
}

inline void
strstrip (char *str)
{
    char *p, *q;
    if (!str)
        return;
    long l = (long)strlen(str);

    p = q = str;

    if (p && *p) {
        while (q && isspace(*q) && (q-p < l))
            ++q;
	if (q-p > l)
	    return;
        if (p != q)
            memmove(p, q, l-(q-p)+1);
        l -= (q-p);
        p = q = str+l-1;
        while (q && q >= str && isspace(*q))
            --q;
        if (p != q)
            *(q+1) = '\0';
    }
}

inline void 
strstrip (string* str)
{
	Point_to_Mem<char> loc(str->length());
	strcpy(loc,str->c_str());
	strstrip(loc);
	str->assign(loc);
}

template < class S, class T >
class AssocArray {
	class assoc {
	    public:
		assoc *prev, *next;
		S key;
		T value;
		bool initialized;
		assoc()
		{
			prev = next = NULL;
			initialized = false;
		}
		assoc * add()
		{
			if (!initialized && (prev == NULL)) {
				initialized = true;
				return this;
			}
			this->next = new assoc;
			this->next->prev = this;
			return this->next;
		}
		void del()
		{
			this->next->prev = this->prev;
			this->prev->next = this->next;
			delete this;
		}
		~assoc()
		{
			if (this->next)
				delete this->next;			
		}
	};

	int nelem;
	assoc *first, *last;

	public:
	AssocArray() {
		nelem = 0;
		last = first = new assoc;
	}
	void SetValues(const string& assocstr)
	{
		int p = 0, s = 0;
		if (!(s = assocstr.find('=',p))) {
			cerr << "Wrong substitute argument" << endl;
			Usage();
			exit(0);
		}
		string Key, Value;
		while (1) {
			Key = assocstr.substr(p,s-p);
			if (Key.length() == 0) break;
			p = assocstr.find(',',s+1);
			if (p == -1) p = assocstr.length() + 1;
			Value = assocstr.substr(s+1,p-s-1);
			last = last->add();
			last->key = Key;
			last->value = Value;
			s = assocstr.find('=',++p);
			if (s == -1) break;
		}
	}
	~AssocArray() {
		delete first;
	}
	T* operator[](const S& Key) {
		for (register assoc* a = first;a;a = a->next)
		{
			if (a->key == Key)
				return &(a->value);
		}
		return NULL;
//		last = last->add();
//		return &(last->value);
	}

	bool undef(const S& Key) {
		for (register assoc* a = first;a;a = a->next)
		{
			if (a->key == Key) {
				a.del();
				return true;
			}
		}
		return false;
	}

	bool exist(const S& Key) {
		for (register assoc* a = first;a;a = a->next)
		{
			if (a->key == Key)
				return true;
		}
		return false;
	}

	bool set(const S& Key, const T& Value) {
		for (register assoc* a = first;a;a = a->next)
		{
			if (a->key == Key) {
				a->value = Value;
				return true;
			}
		}
		last = last->add();
		last->key = Key;
		last->value = Value; 
		return false;
	}

	bool set(const S& Key) {
		for (register assoc* a = first;a;a = a->next)
		{
			if (a->key == Key) {
				return true;
			}
		}
		last = last->add();
		last->key = Key;
		return false;
	}

#ifdef DEBUG
	void dump() {
		cout << "First = " << first << endl;
		for (register assoc* a = first;a;a = a->next)
		{
			cout << "Assoc = " << a << " Key = " << a->key << " Value = " << a->value << endl;
		}
	}
#endif
};
