/*

   Xbase to Postgres database converter

   This program converts one DBF file into Postgres table

   Code of that file was based on:
   - dbf2sql-2.2 by Maarten Boekhold (M.Boekhold@et.tudelft.nl)
   * functions: do_inserts, do_substitute, create_table, check_table
   strtoupper, strtolower
   - postgres' and xbase' example files

   Copyright (C) 1998,1999,2004 Piotr Klaban
   email - makler@man.torun.pl

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

 */

using namespace std;

#include <iostream>
#ifdef LINUX
 #ifndef _XOPEN_SOURCE
 # define _XOPEN_SOURCE
 #endif
 #include <unistd.h>
#endif
#include <fstream>
#include <sstream>
#include <string>
#include <strings.h>
#include <xbase.h>
#include <errno.h>
#include <libpq-fe.h>
#include "xbase2pg.hh"

/* set the stack large for dos compiles */
#ifdef DOS
#include <stdio.h>
extern unsigned _stklen = 100000;
#endif

char *subarg;
xbShort rc;
xbXBase x;
int verbose = 0, upper = 0, lower = 0, create = 0, clear = 0, begin = -1,
  stripcont = 0, usefloat = 0, usechar = 0, notnull = 0,
  end = -1, fieldlower = 0, fieldupper = 0, nokeys = 0, paranoid = 0;
#ifdef XB_MEMO_FIELDS
int memopresent = 0;
#endif

xbShort TableExists (const char * table);
xbShort TableNew    (const char * table);
xbShort TableClear  (const char * table);
xbShort TableInsert (const char * table);
void usage (void);
void xbase_to_pg(const char *dbfname, const char *table);

char *
Escape (char *string)
{
    char *escaped, *foo, *bar;
    unsigned long count = 0;

    foo = string;
    while (*foo != '\0')
      {
          if (*foo++ == '\'')
              ++count;
      }

    if (!(escaped = new char[strlen (string) + count + 1]))
      {
          return (char *) -1;
      }

    foo = escaped;
    bar = string;
    while (*bar != '\0')
      {
          if (*bar == '\'')
            {
                *foo++ = '\\';
            }
          *foo++ = *bar++;
      }
    *foo = '\0';

    return escaped;
}

bool
emptystr(char src[], size_t len)
{
   for (size_t i=0; src[i] && i < len; ++i)
     if (src[i] != ' ')
       return 0;
   return 1;
}

void
usage (void)
{
    cerr << "xbase2pg v" << PG2XBASE_VERSION << endl <<
        "usage:\txbase2pg [-h host] [-p port] [-C | -D] [-P] [-v[v[v]]] [-n]" << endl <<
        "\t\t\t[-b begin] [-e end] [-o usechar[,usefloat]] [-s oldname=newname[,oldname=newname]]" << endl <<
        "\t\t\t[-U | -L] [-u | -l] [-r] -d dbase [-t table] {-f dbf-list | dbf-file [dbf-file [...]]}" << endl;
}

PGconn *pgconn;
class xbDbf *XbaseFile;
class xbDate Xdate;

int
main (int argc, char ** argv)
{
    char *dbName = NULL;
    char *dbfname = NULL, *host = NULL, *port = DEF_PGPORT_STR;
    char *deftable = NULL;
    char *filelist = NULL;
    extern int optind;
    extern char *optarg;
    int i;

    pgconn = NULL;

    while ((i = getopt (argc, argv, "b:Cd:De:f:h:lLNno:Pp:rs:t:uUv")) != EOF)
      {
          switch (i)
            {
            case 'b':
                begin = atoi (optarg);
                break;
            case 'C':
                if (clear)
                  {
                      usage ();
                      cerr << "Can't use -C and -D at the same time!" << endl;
                      exit (1);
                  }
                create = 1;
                break;
            case 'd':
                dbName = (char *) strdup (optarg);
                break;
            case 'D':
                if (create)
                  {
                      usage ();
                      cerr << "Can't use -C and -D at the same time!" << endl;
                      exit (1);
                  }
                clear = 1;
                break;
            case 'e':
                end = atoi (optarg);
                break;
            case 'f':
                filelist = (char *) strdup (optarg);
                break;
            case 'h':
                host = (char *) strdup (optarg);
                break;
            case 'l':
                if (upper)
                  {
                      usage ();
                      cerr << "Can't use -u and -l at the same time!" << endl;
                      exit (1);
                  }
                lower = 1;
                break;
            case 'L':
                if (fieldupper)
                  {
                      usage ();
                      cerr << "Can't use -U and -L at the same time!" << endl;
                      exit (1);
                  }
                fieldlower = 1;
                break;
            case 'N':
                notnull = 1;
                break;
            case 'n':
                nokeys = 1;
                break;
            case 'o':
	        if (optarg)
		  {
	              string optlist = string (optarg);
		      if (optlist.find("usechar") == 0 ||
		          optlist.find(",usechar") != string::npos)
		        usechar = 1;
		      if (optlist.find("usefloat") == 0 ||
		          optlist.find(",usefloat") != string::npos)
		        usefloat = 1;
		  }
	        break;
            case 'P':
                paranoid = 1;
		break;
            case 'p':
                port = (char *) strdup (optarg);
                break;
            case 'r':
                stripcont = 1;
                break;
            case 's':
                subarg = (char *) strdup (optarg);
                break;
            case 't':
                deftable = (char *) strdup (optarg);
                break;
            case 'u':
                if (lower)
                  {
                      usage ();
                      cerr << "Can't use -u and -l at the same time!" << endl;
                      exit (1);
                  }
                upper = 1;
                break;
            case 'U':
                if (fieldlower)
                  {
                      usage ();
                      cerr << "Can't use -U and -L at the same time!" << endl;
                      exit (1);
                  }
                fieldupper = 1;
                break;
            case 'v':
                ++verbose;
                break;
            case '?':
                usage ();
                cerr << "Unknown argument: %s" << argv[0] << endl;
                exit (1);
            default:
                break;
            }
      }

    argc -= optind;
    argv = &argv[optind];

    if (dbName == NULL)
      {
          cerr << "Dbase *must* be specified!" << endl;
          usage ();
          exit (1);
      }

    if ((begin > end) && (end != -1))
      {
          usage ();
          cerr << "First record could not be greater than the last one" << endl;
          exit (1);
      }

    if (argc < 1 && filelist == NULL)
      {
          cerr << "No DBF file given" << endl;
          usage ();
          exit (1);
      }

    // make a connection to postgresql server
    pgconn = PQsetdb(host, port, NULL, NULL, dbName);

    if (PQstatus(pgconn) == CONNECTION_BAD)
      {
	  cout << "Connection was unsuccessful..." << endl
		  << "Error message returned: " << PQerrorMessage(pgconn) << endl;
	  return 1;
      }
    else
      {
	  if (verbose > 1)
	    {
		    cout << "Connection to " << dbName << " established OK" << endl;
	    }
      }

    xbDbf DBFtemp (&x);
    XbaseFile = &DBFtemp;    // new xbDbf (&x);

    for (int i = 0; i < argc; ++i)
      {
        string tab;
	dbfname = argv[i];

        if (deftable != NULL)
	  tab = deftable;
	else
	  tab = "";

	if (tab.empty())
	  {
	      size_t s;

	      tab = dbfname;

	      if ((s = tab.rfind('/')) != string::npos)
	          tab = tab.substr(s+1);
	      if ((s = tab.find('.')) != string::npos)
	          tab = tab.substr(0, s);
	      if (tab.empty())
	        {
	          cerr << "Can not build table name based on DBF name: " << dbfname << endl;
		  continue;
		}
	      if (verbose > 2)
	          cout << "built table name from DBF name: " << tab << endl;
	  }

	if (verbose > 1)
	    cout << "would use table name: " << tab << endl;

        xbase_to_pg(dbfname, tab.c_str());
      }


    if (filelist != NULL)
      {
	string str, name, tab;
	ifstream in(filelist);

	if (in)
	  {
	  	if (verbose > 1)
		  cout << "opening DBF filelist: " << filelist << endl;
		for (;;)
		{
			if (deftable)
			    tab = deftable;
			  else
			    tab = "";

			if (getline(in, str).eof())
				break;
			if (in.eof()) break;
			if (!in.good())
				in.clear();
			if ((str.empty()) || (str[0] == '#')) continue;
			istringstream charin(str.c_str());
			charin >> name;
			if (verbose > 1)
			    cout << "read DBF name: " << name << endl;
		        if (!charin.eof())
			  {
			    charin >> tab;
			    if (verbose > 2)
			      cout << "reading table name from file: " << tab << endl;
		          }  

			if (name.empty()) // no DBF name read
			  continue;

			if (tab.empty() && deftable)
			  {
			    tab = deftable;
			  }
			else if (tab.empty())
			  {
			    size_t s;

			    // assign table name from DBF name
			    tab = name;
			    if ((s = tab.rfind('/')) != string::npos)
			      {
			        tab = tab.substr(s+1);
			      }
			    if ((s = tab.find('.')) != string::npos)
			      {
			        tab = tab.substr(0, s);
			      }
			    if (tab.empty())
			      {
			        cerr << "Can not build table name based on DBF name: " << name << endl;
				continue;
			      }

			    if (verbose > 2)
			      cout << "built table name from DBF name: " << tab << endl;
			  }

			if (verbose > 1)
			  cout << "would use table name: " << tab << endl;

        		xbase_to_pg(name.c_str(), tab.c_str());
		}
	  }
      }

    PQfinish(pgconn);
    return 0;
}

void
xbase_to_pg(const char *dbfname, const char *table)
{
    PGresult *res;

    if (!table || !table[0])
      {
        cerr << "Error: no table name given" << endl;
	return;
      }	

    if ((rc = XbaseFile->OpenDatabase (dbfname)) != XB_NO_ERROR)
      {
        cout << "ERROR: opening " << dbfname << ": "; x.DisplayError(rc);
      }
    else
      {
        if (table == NULL)
          {
              // table = XbaseFile->GetDbfName();
              if (table == NULL)
	        {
	          cerr << "Table can not be NULL!" << endl;
	          exit(0);
	        }
          }

	  res = PQexec(pgconn, "BEGIN");
	  if (!res || PQresultStatus(res) != PGRES_COMMAND_OK)
	    {
	        cerr << "Can not BEGIN transaction" << endl;
	        PQclear(res);
	        PQfinish(pgconn);
	        exit(-1);
	    }
	  PQclear(res);

#ifdef XB_MEMO_FIELDS
          memopresent = XbaseFile->MemoFieldsPresent ();
#endif
          rc = 1;
          if (create)
              rc = TableNew (table);
          else if (clear)
              rc = TableClear (table);

          if (rc)
              TableInsert (table);

	  res = PQexec(pgconn, "COMMIT");
	  if (!res || PQresultStatus(res) != PGRES_COMMAND_OK)
	    {
	        cerr << "Can not COMMIT transaction" << endl;
	        PQclear(res);
	        PQfinish(pgconn);
	        exit(-1);
	    }
	  PQclear(res);
      }

    XbaseFile->CloseDatabase ();    /* Close database and associated indexes */
}

xbShort
check_table (const char * table)
{
    PGresult *res;
    char *q = "DECLARE mycursor BINARY CURSOR FOR select relname from pg_class where relkind='r' and relname !~* '^pg'";
    xbLong i = 0, tuples, fnum;
    char const *field;

    if (!table || !table[0])
      {
        cerr << "Error: no table name given" << endl;
	return 0;
      }	

    if (verbose > 0)
        cout << "Checking if table '" << table << "' exists" << endl;

    res = PQexec(pgconn, q);
    if (!res || PQresultStatus(res) != PGRES_COMMAND_OK)
      {
	  cerr << "Error: DECLARE CURSOR failed" << endl;
	  PQclear(res);
	  PQfinish(pgconn);
	  exit (1);
      }
    PQclear(res);

    res = PQexec(pgconn, "FETCH ALL in mycursor");
    if (!res || PQresultStatus(res) != PGRES_TUPLES_OK)
      {
	  if (verbose > 1)
	    {
		    cerr << "Query: " << q << endl;
	    }
	  cerr << "Error: FETCH ALL failed" << endl;
	  PQclear(res);
	  PQfinish(pgconn);
	  exit (1);
      }
	
    tuples = PQntuples(res);
    fnum = PQfnumber(res, "relname");

    for (i = 0; i < tuples; ++i)
      {
          field = PQgetvalue(res, i, fnum);
          if (!strcmp (table, field))
            {
		res = PQexec(pgconn, "CLOSE mycursor");
		PQclear(res);
                return 1;
            }
      }

    res = PQexec(pgconn, "CLOSE mycursor");
    PQclear(res);
    return 0;
}

/* note from dbf2sql-2.2: */
/* patch submitted by Jeffrey Y. Sue <jysue@aloha.net> */
/* Provides functionallity for substituting dBase-fieldnames for others */

char *
do_substitute (char * fname)
{
    char *p = NULL, *newname, *oldname;

    if (!subarg || !fname || !strlen (fname))
      {
          return fname;
      }

    if (verbose > 2)
      {
          cout << "Substituting new field name " << fname << endl;
      }

    static Point_to_Mem < char > locarg (strlen (subarg));
    strcpy (locarg, subarg);

    /* use strstr instead of strtok because of possible empty tokens */
    oldname = locarg;
    while (oldname && strlen (oldname) && (p = strstr (oldname, "=")))
      {
          *p = '\0';    /* mark end of oldname */
          newname = ++p;    /* point past \0 of oldname */
          if (strlen (newname))
            {        /* if not an empty string */
                p = strstr (newname, ",");
                if (p)
                  {
                      *p = '\0';    /* mark end of newname */
                      ++p;    /* point past where the comma was */
                  }
            }
          if (strcmp (fname, oldname) == 0)
            {
                if (verbose > 1)
                  {
                      cout << "Substitute old:" << oldname << " new:" << newname << endl;
                  }
                return strdup (newname);
                break;
            }
          oldname = p;
      }
    return fname;
}                /* do_substitute */

xbShort
TableNew (const char * table)
{
    PGresult *res;
    int i, length;
    string query;

    if (verbose > 1)
      {
          cout << "Building CREATE-clause" << endl;
      }

    if (!table || !table[0])
      {
        cerr << "Error: no table name given" << endl;
	return 0;
      }	

    if (check_table (table))
      {
          query  = "DROP TABLE ";
          query += table;
          if (verbose > 0)
              cout << "Sending delete clause: " << query << endl;
          res = PQexec(pgconn, query.c_str());
          if (!res || PQresultStatus(res) != PGRES_COMMAND_OK)
            {
                cerr << "Error deleting table '" << table << "': " << PQerrorMessage(pgconn) << endl;
	    }
	  PQclear(res);
      }

    query  = "CREATE TABLE ";
    query += table;
    query += " (";
    length = query.length();
    for (i = 0; i < XbaseFile->FieldCount (); ++i)
      {
          char *fname = NULL, *newfname = NULL, ftype, buf[64];
          xbShort flen, fdec;

          fname = strdup (XbaseFile->GetFieldName (i));
          flen = XbaseFile->GetFieldLen (i);
          ftype = XbaseFile->GetFieldType (i);
          fdec = XbaseFile->GetFieldDecimal (i);

          newfname = do_substitute (fname);
          if (newfname != fname)
            {
                delete[]fname;
                fname = newfname;
            }

          if (!strlen (fname))
            {
                delete[]fname;
                continue;
            }
          /* skip field if length of name == 0 */

          if (query.length() != (size_t) length)
              query += ",";

          if (fieldlower)
              strtolower (fname);
          else if (fieldupper)
              strtoupper (fname);

          query += fname;
          delete fname;

          switch (ftype)
            {
            case 'C':
                if (flen > 1)
                  {
	              if (usechar)
                        query += " char(";
		      else
                        query += " varchar(";
                      sprintf(buf,"%d",flen);
                      query += buf;
                      query += ")";
                  }
                else
                  {
                      query += " char";
                  }
                break;
            case 'D':
                query += " date";
                break;
            case 'L':
                query += " bool";
                break;
            case 'F':
	        if (usefloat)
		  query += " float8";
		else 
                  query += " real";
                break;
            case 'N':
	        if (usefloat)
		  query += " float8";
		else
		  {
                      query += " numeric(";
                      sprintf(buf,"%d",flen);
                      query += buf;
                      if (fdec != 0)
                        {
                            query += ",";
                            sprintf(buf,"%d",fdec);
                            query += buf;
                        }
                      query += ")";
                  }
                break;
            case 'M':
#ifdef XB_MEMO_FIELDS
                query += " text";
                break;
#else
                cerr << "Memo field (no." << i << ") is not supported" << endl;
                return (0);
#endif
            default:
                cerr << "Unknown data type (" << ftype << ") in field no. " << i << endl;
                return (0);
            }
      }

    query += ")";

    if (verbose > 1)
      {
          cout << "Sending create-clause:" << endl << query << endl;
      }

    res = PQexec(pgconn, query.c_str());
    if (!res || PQresultStatus(res) != PGRES_COMMAND_OK)
      {
          cerr << "Error creating table '" << table << "': " << PQerrorMessage(pgconn) << endl;
	  PQclear(res);
          return (0);
      }
    PQclear(res);
    return 1;
}

xbShort
TableInsert (const char * table)
{
    PGresult *res;
    int i, h;
    char *foo = NULL;
    u_long key_len = 0;
    u_long val_len = 0;
    xbLong maxflen = 0, numfields, numrecords, countrec = 0, querydiff,
      flen;
    char ftype, *fname;
#ifdef XB_MEMO_FIELDS
    xbLong mbufsize = 0, mlen = 0;
    char *mcontents = NULL;
#endif

    if (verbose > 0)
      {
          cout << "Inserting records" << endl;
      }

    numfields = XbaseFile->FieldCount ();
    numrecords = XbaseFile->NoOfRecords ();

    for (i = 0; i < numfields; ++i)
      {
          fname = XbaseFile->GetFieldName (i);
          flen = XbaseFile->GetFieldLen (i);
          ftype = XbaseFile->GetFieldType (i);

          if (flen > maxflen)
              maxflen = flen;

          // The +1 is for the separating comma */
          key_len += strlen (fname) + 1;
          // take into account the possibility that there are NULL's in the query
          val_len += (flen > 4) ?
              flen + 3 : 7;
      }

    string query;
    Point_to_Mem < char > keys (key_len);
    // multiplicate length with 2 in case of escaping every character
    Point_to_Mem < char > vals (2 * val_len);
    Point_to_Mem < char > contents (2 * maxflen + 1);

    for (i = 0; i < numfields; ++i)
      {
          char *fname = NULL, *newfname = NULL;

          fname = XbaseFile->GetFieldName (i);
          newfname = do_substitute (fname);
          if (!strlen (newfname))
            {
                if (verbose > 3)
                    cout << "Field no. " << i << " has no name ... skipping" << endl;
                continue;
            }
          if (strlen (keys) != 0)    // no comma before the first value

              strcat (keys, ",");

          strcat (keys, newfname);
          if (newfname != fname)
            {
                delete[]newfname;
            }
      }

    if (begin == -1)
        begin = 1;
    if ((end == -1) || (end > numrecords))
        end = numrecords;

#ifdef XB_MEMO_FIELDS
#ifdef XB_LOCKING_ON
    if (memopresent)
        XbaseFile->LockMemoFile (F_SETLK, F_RDLCK);
#endif
#endif
    querydiff = 0;

    for (i = begin; i <= end; ++i)
      {

          rc = XbaseFile->GetRecord (i);
          if (rc == XB_NO_ERROR)
            {
                int currdiff = 0;

                vals[0] = '\0';

                for (h = 0; h < numfields; ++h)
                  {
                      contents[0] = '\0';

                      fname = XbaseFile->GetFieldName (h);
                      if (!strlen (fname))
                          continue;
                      ftype = XbaseFile->GetFieldType (h);
                      XbaseFile->GetField (h, contents);
                      flen = XbaseFile->GetFieldLen (h);

                      switch (ftype)
                    {
                    case 'F':
                    case 'N':
                        if (contents && strlen(contents) > 0) {
                          if (paranoid) {
                            errno = 0;
                            if (!(atof(contents)) && errno) {
                              memset(contents, ' ', flen);
                            }
                          } else { // no paranoid checks
                            // if no digit is found - change all chars to ' '
                            int allspaces = true;
                            for (int i=0; i<flen; ++i) {
                              if (contents[i] >= '0' && contents[i] <= '9') {
                                allspaces = false;
                                break;
                              }
                            }
                            if (allspaces) { // no digits
                              memset(contents, ' ', flen);
                            }
                          }
                        }
                        foo = Escape (contents);
                        break;

                    case 'L':
                        foo = Escape (contents);
                        if (contents[0]=='F') {
			} else if (contents[0]=='?' || contents[0]==' ') { // uninitialized variable
			  foo[0] = '\0';
			}
                        break;
                                                    
                    case 'D':
                        if (contents[0])
                              if (emptystr(contents,strlen(contents)))
                            contents[0] = '\0';
                        if (contents[0])
                          strcpy (contents, (Xdate.FormatDate (PG2XBASE_FORMAT_DATE, contents)));
                        foo = Escape (contents);
                        break;
#ifdef XB_MEMO_FIELDS
                    case 'M':
                        mlen = XbaseFile->GetMemoFieldLen (h);
                        if (mlen <= 0)
                          {
                              cerr << "Error zero Memo field no. " << h << " of record " << i << endl;
                              continue;
                          }
                        if (mbufsize < mlen)
                          {
                              if (mcontents)
                                  delete mcontents;
                              mcontents = new char[mbufsize = mlen + 1];
                          }
                        if ((rc = XbaseFile->GetMemoField (h, mlen, mcontents, F_SETLKW)) == XB_NO_ERROR)
                          {
                              mcontents[mlen] = '\0';
                              /* escape all special chars ('\) */
                              if (upper)
                                  strtoupper (mcontents);
                              if (lower)
                                  strtolower (mcontents);
                              if (stripcont)
                                  strstrip (mcontents);

                              foo = Escape (mcontents);
                          }
                        else
                          {
                              cerr << "Error " << rc << " reading Memo field no. " << h << " of record " << i << endl;
                              continue;
                          }
                        break;
#endif
                    default:
                        if (upper)
                            strtoupper (contents);
                        if (lower)
                            strtolower (contents);
                        if (stripcont)
                            strstrip (contents);
                        foo = Escape (contents);
                        break;
                    }

                      if (flen < (int) strlen (foo))
                    {
                        int diff = strlen (foo) - flen;
                        currdiff += diff;
                        if (querydiff < currdiff)
                          {
                              querydiff += diff;
                          }
                    }
                      if (strlen (vals) != 0)
                          strcat (vals, ",");

                      if (strlen(foo))
                        if (ftype != 'N')
                      {    // we may have to insert a NULL

                        strcat (vals, "'");
                      }
                    else
                      {
                            if (emptystr(foo,strlen(foo)))
                           foo[0] = '\0';
                      }

                      if (*foo == '\0')
			{
			  if (notnull)
                            strcat (vals, "''");
			  else
                            strcat (vals, "NULL");
                	}
		      else
                          strcat (vals, foo);

                      if ((ftype != 'N') && (strlen (foo)))
                    {
                        strcat (vals, "'");
                    }

                      delete[]foo;
                  }    // for (h)

                query  = "INSERT INTO ";
                query += table;
                if (!nokeys) {
                  query += " (";
                  query += keys;
                  query += ")";
                }
                query += " VALUES (";
                query += vals;
                query += ")";

		res = PQexec(pgconn, query.c_str());
		if (!res || PQresultStatus(res) != PGRES_COMMAND_OK)
                  {
                      cerr << "Error sending INSERT in record " << i << ": " << query << endl;
                      cerr << "Error message: " << PQerrorMessage(pgconn) << endl;
		      PQclear(res);
                      break;
                  }
		PQclear(res);
                if ((verbose > 1) && ((i % 100) == 0))
                  {
                      cout.width (4);
                      cout << "Inserting record " << i << endl;
                  }
                if (verbose > 2)
                  {
                      cout.width (4);
                      cout << "Query " << i << ": " << query << endl;
                  }
                ++countrec;
            }        // if (rc)

      }            // for (i)

#ifdef XB_MEMO_FIELDS
    if (mcontents)
        delete mcontents;
#ifdef XB_LOCKING_ON
    if (memopresent)
        XbaseFile->LockMemoFile (F_SETLK, F_UNLCK);
#endif
#endif

    if (verbose > 0)
          cout << "Have inserted " << countrec << " out of total " << numrecords << " records." << endl;

    return 1;
}

xbShort
TableClear (const char * table)
{
    PGresult *res;
    Point_to_Mem<char> query(12 + strlen (table));

    if (check_table (table))
      {
          sprintf (query, "DELETE FROM %s", table);
          if (verbose > 0)
              cout << "Delete all records from the table '" << table << "': " << query << endl;
	  res = PQexec(pgconn, query);
	  if (!res || PQresultStatus(res) != PGRES_COMMAND_OK)
            {
                cerr << "Error deleting from the table '" << table << "': " << PQerrorMessage(pgconn) << endl;
                return (0);
            }
      }
    else
      {
          cerr << "Table '" << table << "' does not exists!" << endl;
          return (0);
      }
    return 1;
}
