/*  pg2xbase.cpp

   Postgres to xbase database converter

   This program creates one DBF file from Postgres table

   Code of that file was based on:
   - dbf2sql-2.2 by Maarten Boekhold (M.Boekhold@et.tudelft.nl)
   * functions: do_inserts, do_substitute, create_table, check_table
   strtoupper, strtolower
   - postgres' psql interface
   * functions: tableDesc
   - postgres' and xbase' example files

   Copyright (C) 1998,1999,2004 Piotr Klaban
   email - makler@man.torun.pl

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 */

using namespace std;

#include <iostream>
#ifdef LINUX
 #ifndef _XOPEN_SOURCE
 # define _XOPEN_SOURCE
 #endif
 #include <unistd.h>
#endif
#include <fstream>
#include <sstream>
#include <string>
#include <strings.h>
#include <xbase.h>
#include <strings.h>
#include <libpq-fe.h>
#include <postgres_fe.h>
#include "xbase2pg.hh"

/* set the stack large for dos compiles */
#ifdef DOS
#include <stdio.h>
extern unsigned _stklen = 100000;

#endif

xbXBase x;
int verbose = 0, upper = 0, lower = 0, create = 0, clear = 0, begin = -1,
  stripcont = 0,
  end = -1, fieldlower = 0, fieldupper = 0, be_quiet = 0, overlay = XB_DONTOVERLAY;
char *subarg = NULL;

int tableDesc (const char * table);
int WriteRecs (const char	 * query);
int rc;
void usage (void);
void pg_to_xbase(const char *table, const char *dbfname);

void
usage (void)
{
	cerr << "pg2xbase v" << PG2XBASE_VERSION << endl <<
		"usage:\tpg2xbase [-h host] [-p port] [-C | -D] [-v[v[v]]]" << endl <<
		"\t\t\t[-b begin] [-e end] [-s oldname=newname[,oldname=newname]]" << endl <<
		"\t\t\t[-U | -L] [-u | -l] [-r] [-V dbfversion] [-q]" << endl <<
		"\t\t\t-d dbase [-t table] {-f dbf-list | dbf-file [dbf-file [...]]}" << endl;
}

class TabFields
{
	Point_to_Mem < xbSchema > Sc;
	xbShort TabSize;
	xbShort LastElem;
	  public:
	  TabFields (int r):Sc (r + 1)
	{
		TabSize = r + 1;
		LastElem = 0;
		for (register int i = 0 ; i < TabSize ; ++i ) {
			memset(&Sc[i],0,sizeof(Sc[i]));
		}
	}
	xbShort AppendField (xbSchema * Field)
	{
		if (LastElem > TabSize)
			  return 0;
		memcpy (&Sc[LastElem], Field, sizeof (Sc[LastElem]));

		LastElem += 1;
		return 1;
	}
	xbSchema *GetSchema ()
	{
		return Sc;
	}
};

class TabFields *MySchema;
PGconn *pgconn;
class xbDbf *XbaseFile;
string selectnames;

/*
 * Change postgres SQL date (MM-DD-CCYY) into DBF 8-char date (CCYYMMDD)
 */
char *
ParseDate (char * timestr)
{
	char pqdate[10];
	char fdate[8];
	int i;

	if (strlen (timestr) != 10) {
		cout << "Date field " << timestr << " is not 10 chars long" << endl;
		return timestr;
	}

	strncpy (pqdate, timestr, 10);
	strncpy (fdate, "00000000", 8);

	// look for YYYY-MM-DD format
	if (pqdate[4] == '-' && pqdate[7] == '-')
	{
		if (verbose > 4)
			cout << "ParseDate: format YYYY-MM-DD recognized" << endl;
		fdate[0] = pqdate[0];
		fdate[1] = pqdate[1];
		fdate[2] = pqdate[2];
		fdate[3] = pqdate[3];
		fdate[4] = pqdate[5];
		fdate[5] = pqdate[6];
		fdate[6] = pqdate[8];
		fdate[7] = pqdate[9];
		
	} else if (pqdate[2] == '-' && pqdate[5] == '-')
	{
		if (verbose > 4)
			cout << "ParseDate: format DD-MM-YYYY recognized" << endl;
		fdate[0] = pqdate[6];
		fdate[1] = pqdate[7];
		fdate[2] = pqdate[8];
		fdate[3] = pqdate[9];
		fdate[4] = pqdate[0];
		fdate[5] = pqdate[1];
		fdate[6] = pqdate[3];
		fdate[7] = pqdate[4];
	} else {
		if (verbose > 4)
			cout << "ParseDate: date format unrecognized" << endl;
		return timestr;
	}

	if (atoi (fdate) == 0) {
		if (verbose > 4)
			cout << "Date field " << timestr << " returned because atoi(" << fdate[0] << fdate[1] <<
				fdate[2] << fdate[3] << fdate[4] << fdate[5] << fdate[6] << fdate[7] << ") is zero" << endl;
		return timestr;
	}

	for (i=0; i<8; ++i) {
		timestr[i] = fdate[i];
	}
	timestr[8] = '\0';

	return timestr;
}

int
WriteRecs (const char * query)
{
	char *contents;
	int nTuples, numfields, i, t, recs = 0, once = -1;
	char ftype, needtype;
	PGresult *res;
	xbLong flen;
	xbSchema *Schema = MySchema->GetSchema();

	if (verbose > 0)
		cout << "Begin writing records ..." << endl;

	res = PQexec(pgconn, query);
	if (!res || PQresultStatus(res) != PGRES_TUPLES_OK)
	  {
	  	  if (verbose > 1)
		      cerr << "Query: " << query << endl;
		  cerr << "Error: Query failed: " << PQresultStatus(res) << PQerrorMessage(pgconn) << endl;
		  PQclear(res);
		  return 0;
	  }

	nTuples = PQntuples(res);
	numfields = PQnfields(res);
	for (i = 0; i < nTuples; ++i)
	  {
		  if ((begin >= 0) && (i < begin))
			  continue;
		  if ((end >= 0) && (i > end))
			  break;
		  XbaseFile->BlankRecord ();
		  for (t = 0; t < numfields; ++t)
		    {
			    ftype = XbaseFile->GetFieldType (t);
			    flen = XbaseFile->GetFieldLen (t);
			    needtype = Schema[t].Type;
			    contents = strdup (PQgetvalue(res, i, t));
                if (stripcont)
                    strstrip (contents);
			    if (upper)
				    strtoupper (contents);
			    if (lower)
				    strtolower (contents);
			    if (ftype == XB_DATE_FLD)
			      {
				      if (!be_quiet && (ftype != needtype) && (once < t)) {
					cerr << "WARNING: dbf field no. " << t << " has different type (" << ftype << ") than Postgres' (" << needtype << ")" << endl;
					once = t;
				      }
				      contents = ParseDate (contents);
				      XbaseFile->PutField (t, contents);
			      }
			    else if (ftype == XB_MEMO_FLD)
			      {
#ifdef XB_MEMO_FIELDS
				      XbaseFile->UpdateMemoData (t, strlen (contents), contents, F_SETLKW);
#else
				      cerr << "Unknown MEMO dbf field no. " << t << ": " << contents << endl;
				      continue;
#endif
			      }
			    else
			      {
				      if (!be_quiet && (ftype != needtype) && (once < t)) {
					cerr << "WARNING: dbf field no. " << t << " has different type (" << ftype << ") than Postgres' (" << needtype << ")" << endl;
					once = t;
				      }
				      if (!be_quiet && (ftype == XB_CHAR_FLD) && (xbShort(strlen(contents)) > flen)) {
					cerr << "WARNING: text field/row " << t << "/" << i << " is only " << 
					    flen << " length (need " << strlen(contents) << "). Truncate." << endl;
				      }
				      XbaseFile->PutField (t, contents);
			      }
			    if (verbose > 2)
				    cout << "Putting " << t << "/" << i << " rec: '" << contents << "'" << endl;
			    delete contents;
		    }
		  if ((rc = XbaseFile->AppendRecord ()) != XB_NO_ERROR)
			  cout << endl << "Error " << rc << " appending data record." << endl;
		  else if (verbose > 2)
			  cout << "Appending record no. " << i << endl;
		  ++recs;
	  }
	if (verbose > 0)
		cout << "Append " << recs << " records (total " << nTuples << ")" << endl;
	PQclear(res);
	return 1;
}

/* note from dbf2sql-2.2: */
/* patch submitted by Jeffrey Y. Sue <jysue@aloha.net> */
/* Provides functionallity for substituting dBase-fieldnames for others */

char *
do_substitute (char * fname)
{
	char *p = NULL, *newname, *oldname;

	if (!subarg)
	  {
		  return fname;
	  }

	if (verbose > 2)
	  {
		  cout << "Substituting new field name " << fname << endl;
	  }

	static Point_to_Mem < char > locarg (strlen (subarg));
	strcpy (locarg, subarg);

	/* use strstr instead of strtok because of possible empty tokens */
	oldname = locarg;
	while (oldname && strlen (oldname) && (p = strstr (oldname, "=")))
	  {
		  *p = '\0';	/* mark end of oldname */
		  newname = ++p;	/* point past \0 of oldname */
		  if (strlen (newname))
		    {		/* if not an empty string */
			    p = strstr (newname, ",");
			    if (p)
			      {
				      *p = '\0';	/* mark end of newname */
				      ++p;	/* point past where the comma was */
			      }
		    }
		  if (strcmp (fname, oldname) == 0)
		    {
			    if (verbose > 1)
			      {
				      cout << "Substitute old:" << oldname << " new:" << newname << endl;
			      }
			    return strdup (newname);
			    break;
		    }
		  oldname = p;
	  }
	return fname;
}				/* do_substitute */

/*
 * Converts Postgres table structure to Schema[] table structure
 */
int
tableDesc (const char * table)
{
	PGresult *res;
	xbSchema Field;
	char descbuf[1024];
	xbShort nTuples;
	char const *rtype;
	char const *rnotnull;
	char const *rhasdef;
	//xbShort i, attlen, typlen, atttypmod;
	int i, attlen, typlen, atttypmod;

	/* Build the query */

	memset(&Field,0,sizeof(Field));

	if (!table)
		return 0;

	descbuf[0] = '\0';
	strcat (descbuf, "SELECT a.attnum, a.attname, t.typname, a.attlen, ");
	strcat (descbuf, "a.atttypmod, a.attnotnull, a.atthasdef, t.typlen ");
	strcat (descbuf, "FROM pg_class c, pg_attribute a, pg_type t ");
	strcat (descbuf, "WHERE c.relname = '");
	strcat (descbuf, table);
	strcat (descbuf, "'");
	strcat (descbuf, "    and a.attnum > 0 ");
	strcat (descbuf, "    and a.attrelid = c.oid ");
	strcat (descbuf, "    and a.atttypid = t.oid ");
	strcat (descbuf, "  ORDER BY attnum ");

	res = PQexec(pgconn, descbuf);
	if (!res || PQresultStatus(res) != PGRES_TUPLES_OK)
	  {
		  if (verbose > 1)
		    {
			    cerr << "Query: " << descbuf << endl;
		    }
		  cerr << "Error: Query failed: " << PQresultStatus(res) << PQerrorMessage(pgconn) << endl;
		  PQclear(res);
		  PQfinish(pgconn);
		  exit (1);
	  }
	 
	if (verbose > 2)
	  {
		  cout << "Sending query: " << descbuf << endl;
	  }
	/* Here the nTuples means "number of table fields" */
	nTuples = PQntuples(res);

	if (nTuples > 0)
	  {
		  char *fname;
		  char ftype;
		  xbShort fsize, fdec;

		  MySchema = new TabFields (nTuples);

		  /* next, print out the instances */
		  for (i = 0; i < nTuples; ++i)
		    {
			    fdec = 0;
			    fsize = -1;
			    fname = strdup (PQgetvalue(res, i, 1));
			    rtype = PQgetvalue(res, i, 2);
			    attlen = atoi (PQgetvalue(res, i, 3));
			    atttypmod = atoi (PQgetvalue(res, i, 4));
			    rnotnull = PQgetvalue(res, i, 5);
			    rhasdef = PQgetvalue(res, i, 6);
			    typlen = atoi (PQgetvalue(res, i, 7));

			    ftype = ' ';
			    if (strcmp (rtype, "bpchar") == 0)
				    ftype = XB_CHAR_FLD;
			    else if (strcmp (rtype, "varchar") == 0)
				    ftype = XB_CHAR_FLD;
			    else if (strncmp (rtype, "char", 4) == 0)
				    ftype = XB_CHAR_FLD;
			    else if (strcmp (rtype, "numeric") == 0)
			      {
				      fsize = ((atttypmod - VARHDRSZ) >> 16) & 0xffff;
				      fdec  = (atttypmod - VARHDRSZ) & 0xffff;
				      ftype = XB_NUMERIC_FLD;
			      }
			    else if (strcmp (rtype, "float4") == 0)
			      {
				      fdec = 6;
				      fsize = 12;
				      ftype = XB_FLOAT_FLD;
			      }
			    else if (strcmp (rtype, "float8") == 0)
			      {
				      fdec = 10;
				      fsize = 19; // no more than 19
				      ftype = XB_FLOAT_FLD;
			      }
			    else if (strncmp (rtype, "int", 3) == 0)
			      {
				      if (typlen > 0)
					  fsize = 5*(typlen/2);
				      if (fsize > 19) // no more than 19 in xbase/dbf..cpp
				          fsize = 19;
				      ftype = XB_NUMERIC_FLD;
			      }
			    else if (strcmp (rtype, "bool") == 0)
				      ftype = XB_LOGICAL_FLD;
				      
			    else if (strcmp (rtype, "date") == 0)
				    ftype = XB_DATE_FLD;
				    
			    else if (strncmp (rtype, "abstime", 7) == 0)
			      {
			      	      fsize = 32;
				      ftype = XB_CHAR_FLD;
			      }
				    
			    else if ((strcmp (rtype, "time") == 0) ||
				     (strcmp (rtype, "datetime") == 0) ||
				     (strcmp (rtype, "timespan") == 0))
			      {
				      cerr << "Can not convert this date/time type " << rtype << " at field " << fname << " (no. " << i << ")" << endl;
				      continue;
			      }
			    else if (rtype[0] == '_')
			      {
				      cerr << "Can not convert array type " << rtype << " at field " << fname << " (no. " << i << ")" << endl;
				      continue;
			      }
			    else if (strcmp (rtype, "text") == 0)
			      {
#ifdef XB_MEMO_FIELDS
				      ftype = XB_MEMO_FLD;
				      fsize = 10;
#else
				      cerr << "Do not know how to include type text at field " << fname << " (no. " << i << ")" << endl;
				      cerr << "Since MEMO field is not supported I would try char(80)" << endl;
				      ftype = XB_CHAR_FLD;
				      fsize = 80;
#endif
			      }
			    else
			      {
				      cerr << "Unknown type " << rtype << " at field " << fname << " (no. " << i << ")" << endl;
				      continue;
			      }

			    if (fsize <= 0)
			      {
				      if (strcmp (rtype, "date") == 0)
					{
						fsize = 8;
					}
				      else if (strcmp (rtype, "bpchar") == 0 ||
					     strcmp (rtype, "varchar") == 0)
					      fsize = atttypmod != -1 ? atttypmod - VARHDRSZ : 0;
				      else
					{
						if (attlen > 0)
							fsize = attlen;
						else
						  {
							  cerr << "Do not know how to include type " << rtype << " at field " << fname << " (no. " << i << endl;
							  continue;
						  }
					}
			      }

			    if (selectnames == "*")
			      selectnames = fname;
			    else
			      {
				selectnames += ", ";
				selectnames += fname;
			      }


			    char *newfname = do_substitute (fname);
			    if (newfname != fname)
			      {
				      delete[]fname;
				      fname = newfname;
			      }
			    if (fieldlower)
				    strtolower (fname);
			    else if (fieldupper)
				    strtoupper (fname);
			    if (strlen (fname) >= 11) { /* fname = 10x[a-z] + NULL */
				    if (verbose > 1)
				      cout << "WARNING: too long field name " << fname;
				    fname[10] = '\0';
				    if (verbose > 1)
				      cout << " is truncated to " << fname << endl;
			    }
			    strcpy (Field.FieldName, fname);
			    Field.Type = ftype;
			    Field.FieldLen = (char) fsize;
			    Field.NoOfDecs = (char) fdec;
			    MySchema->AppendField (&Field);

			    if (verbose > 0)
			      {
				      cout << "New field " << ftype << " (" << fsize;
				      if (fdec > 0)
				        cout << '.' << fdec;
				      cout << ") " << fname << endl;
			      }
		    }
		  PQclear(res);
		  return 1;
	  }
	else
	  {
		  cerr << "Couldn't find table " << table << "!" << endl;
		  PQclear(res);
		  return 0;
	  }
}

int
main (int argc, char ** argv)
{
	char *dbName = NULL, *dbfname, *host = NULL;
	char *port = DEF_PGPORT_STR, *deftable = NULL;
	char *filelist = NULL;
	extern int optind;
	extern char *optarg;
	xbShort dbfversion = DEF_DBF_VERSION; /* 3 or 4 */
	int i;

	while ((i = getopt (argc, argv, "b:d:CDe:f:h:lLOp:rs:t:uUvV:q")) != EOF)
	  {
		  switch (i)
		    {
		    case 'b':
			    begin = atoi (optarg);
			    break;
		    case 'C':
			    if (clear)
			      {
				      usage ();
				      cerr << "Can't use -C and -D at the same time!" << endl;
				      exit (1);
			      }
			    create = 1;
			    break;
		    case 'd':
			    dbName = (char *) strdup (optarg);
			    break;
		    case 'D':
			    if (create)
			      {
				      usage ();
				      cerr << "Can't use -C and -D at the same time!" << endl;
				      exit (1);
			      }
			    clear = 1;
			    break;
		    case 'e':
			    end = atoi (optarg);
			    break;
		    case 'f':
			    filelist = (char *) strdup (optarg);
			    break;
		    case 'h':
			    host = (char *) strdup (optarg);
			    break;
		    case 'l':
			    if (upper)
			      {
				      usage ();
				      cerr << "Can't use -u and -l at the same time!" << endl;
				      exit (1);
			      }
			    lower = 1;
			    break;
		    case 'L':
			    if (fieldupper)
			      {
				      usage ();
				      cerr << "Can't use -U and -L at the same time!" << endl;
				      exit (1);
			      }
			    fieldlower = 1;
			    break;
		    case 'O':
			    overlay = XB_OVERLAY;
			    break;
		    case 'p':
			    port = (char *) strdup (optarg);
			    if (!host)
			      {
				      usage ();
				      cerr << "You must specify also the host name" << endl;
				      exit (1);
			      }
			    break;
		    case 'r':
		            stripcont = 1;
			    break;
		    case 's':
			    subarg = (char *) strdup (optarg);
			    break;
		    case 't':
			    deftable = (char *) strdup (optarg);
			    break;
		    case 'u':
			    if (lower)
			      {
				      usage ();
				      cerr << "Can't use -u and -l at the same time!" << endl;
				      exit (1);
			      }
			    upper = 1;
			    break;
		    case 'U':
			    if (fieldlower)
			      {
				      usage ();
				      cerr << "Can't use -U and -L at the same time!" << endl;
				      exit (1);
			      }
			    fieldupper = 1;
			    break;
		    case 'v':
			    ++verbose;
			    break;
		    case 'V':
			    dbfversion = atoi (optarg);
			    break;
		    case 'q':
			    be_quiet = 1;
			    break;
		    case '?':
			    usage ();
			    cerr << "Unknown argument: " << argv[0] << endl;
			    exit (1);
		    default:
			    break;
		    }
	  }

	argc -= optind;
	argv = &argv[optind];

	if (dbName == NULL)
	  {
		  usage ();
		  cerr << "Dbase *must* be specified!" << endl;
		  exit (1);
	  }

	if (argc < 1 && filelist == NULL)
	  {
		  usage ();
		  exit (1);
	  }

	if ((begin > end) && (end != -1))
	  {
		  usage ();
		  cerr << "First record could not be greater than the last one" << endl;
		  exit (1);
	  }

        // make a connection to postgresql server
	pgconn = PQsetdb(host, port, NULL, NULL, dbName);

	if (PQstatus(pgconn) == CONNECTION_BAD)
	  {
		  cout << "Connection was unsuccessful..." << endl
			  << "Error message returned: " << PQerrorMessage(pgconn) << endl;
		  return 1;
	  }
	else
	  {
		  if (verbose > 1)
		    {
			    cout << "Connection to " << dbName << " established OK" << endl;
		    }
	  }

	xbDbf DBFtemp (&x);
	XbaseFile = &DBFtemp;
	XbaseFile->SetVersion (dbfversion);

    for (int i = 0; i < argc; ++i)
      {
        string tab;
	dbfname = argv[i];

        if (deftable != NULL)
	  tab = deftable;
	else
	  tab = "";

	if (tab.empty())
	  {
	      size_t s;

	      tab = dbfname;

	      if ((s = tab.rfind('/')) != string::npos)
	          tab = tab.substr(s+1);
	      if ((s = tab.find('.')) != string::npos)
	          tab = tab.substr(0, s);
	      if (tab.empty())
	        {
	          cerr << "Can not build table name based on DBF name: " << dbfname << endl;
		  continue;
		}
	      if (verbose > 2)
	          cout << "built table name from DBF name: " << tab << endl;
	  }

	if (verbose > 1)
	    cout << "would use table name: " << tab << endl;

	pg_to_xbase(tab.c_str(), dbfname);
      }

    if (filelist != NULL)
      {
	string str, name, tab;
	ifstream in(filelist);

	if (in)
	  {
	  	if (verbose > 1)
		  cout << "opening DBF filelist: " << filelist << endl;
		for (;;)
		{
			if (deftable)
			    tab = deftable;
			  else
			    tab = "";

			if (getline(in, str).eof())
				break;
			if (in.eof()) break;
			if (!in.good())
				in.clear();
			if ((str.empty()) || (str[0] == '#')) continue;
			istringstream charin(str.c_str());
			charin >> name;
			if (verbose > 1)
			    cout << "read DBF name: " << name << endl;
		        if (!charin.eof())
			  {
			    charin >> tab;
			    if (verbose > 2)
			      cout << "reading table name from file: " << tab << endl;
		          }  

			if (name.empty()) // no DBF name read
			  continue;

			if (tab.empty() && deftable)
			  {
			    tab = deftable;
			  }
			else if (tab.empty())
			  {
			    size_t s;

			    // assign table name from DBF name
			    tab = name;
			    if ((s = tab.rfind('/')) != string::npos)
			      {
			        tab = tab.substr(s+1);
			      }
			    if ((s = tab.find('.')) != string::npos)
			      {
			        tab = tab.substr(0, s);
			      }
			    if (tab.empty())
			      {
			        cerr << "Can not build table name based on DBF name: " << name << endl;
				continue;
			      }

			    if (verbose > 2)
			      cout << "built table name from DBF name: " << tab << endl;
			  }

			if (verbose > 1)
			  cout << "read table name: " << tab << endl;

			pg_to_xbase(tab.c_str(), name.c_str());
		}
	  }
      }


	PQfinish(pgconn);
	return rc;
}

void
pg_to_xbase(const char *table, const char *dbfname)
{
        if (!table || !table[0])
          {
            cerr << "Error: no table name given" << endl;
	    return;
          }	

	selectnames = "*";

	rc = tableDesc (table);
	if (rc)
	  {
	  	  rc = 0;
		  Point_to_Mem < char > query (13 + strlen (table) + selectnames.length());

		  // Select names separately is pointless (commented out on Thu, Jul 11 2002)
		  // sprintf (query, "SELECT %s FROM %s", selectnames.c_str(), table);
		  sprintf (query, "SELECT * FROM %s", table);
		  if (verbose > 1)
			  cout << "Main query is: " << (char *)query << endl;

		  if (create)
		    {
			    rc = XbaseFile->CreateDatabase (dbfname, MySchema->GetSchema (), (xbShort) overlay);
		    }
		  else
		    {
			    rc = XbaseFile->OpenDatabase (dbfname);
		    }

		  if (rc != XB_NO_ERROR)
		    {
			    cout << "ERROR " << rc << ": opening " << dbfname << ": "; x.DisplayError(rc);
		    }
		  else
		    {
			    if (clear)
			      {
			      	    if (verbose > 1)
				      cout << "Erasing DBF data from file " << dbfname << endl;
				    XbaseFile->Zap (F_SETLKW);
			      }

			    WriteRecs (query);
			    XbaseFile->CloseDatabase ();	/* Close database and associated indexes */
		    }

		  //delete MySchema;
	  }
}
