/* -*- pftp-c -*- */
#ifdef HAVE_CONFIG_H
# include "config.h"
#endif
#if HAVE_STDLIB_H
# include <stdlib.h>
#endif
#if HAVE_STDIO_H
# include <stdio.h>
#endif
#if HAVE_UNISTD_H
# include <unistd.h>
#endif
#if HAVE_STRING_H
# include <string.h>
#else
# if HAVE_STRINGS_H
#  include <strings.h>
# endif
#endif
#if HAVE_LIMITS_H
# include <limits.h>
#endif
#if HAVE_DIRENT_H
# include <dirent.h>
#else
# if HAVE_NDIR_H
#  include <ndir.h>
# else
#  if HAVE_SYS_DIR_H
#   include <sys/dir.h>
#  else
#   if HAVE_SYS_NDIR_H
#    include <sys/ndir.h>
#   endif
#  endif
# endif
#endif
#if HAVE_INTTYPES_H
# include <inttypes.h>
#endif
#if HAVE_PWD_H
# include <pwd.h>
#endif
#if HAVE_ERRNO_H
# include <errno.h>
#endif
#include <time.h>
#include <readline/readline.h>

#if !HAVE_D_TYPE
# include <sys/stat.h>
#endif

#ifdef DEBUG
# include <assert.h>
#else
# define assert(x) // x
#endif

#ifdef WITH_DMALLOC
# include <dmalloc.h>
#endif

#include <pftputil.h>
#include <str.h>
#include "ftp_basic_cli.h"
#include "tab-comp.h"
#include "tab_list.h"
#include "nice_list.h"

/* Vars from ftp_basic_cli */
extern const command_t commands[];
extern const size_t no_commands;
extern size_t open_list_len;
extern struct open_list *open_list;
/* End of vars */

static int fix_localpath(const char *pos, size_t pos_len, char **old);
static void restore_localpath(char **old);
static int fix_remotepath(const char *pos, size_t pos_len, pftp_server_t ftp, 
			  char **old);
static void restore_remotepath(pftp_server_t ftp, char **old);
#ifndef WIN32
static void add_users(tab_list_t list, const char *pos, size_t minlen);
#endif
static void add_commands(tab_list_t list, const char *pos, size_t minlen);
static void add_current_sitenames(tab_list_t list, const char *pos, 
				  size_t minlen);
static void add_saved_sitenames(tab_list_t list, const char *pos, 
				size_t minlen);
static void add_localdirs(tab_list_t list, const char *pos, size_t minlen);
static void add_localfiles(tab_list_t list, const char *pos, size_t minlen);
static void add_remotedirs(tab_list_t list, pftp_server_t ftp, const char *pos,
			   size_t minlen);
static void add_remotefiles(tab_list_t list, pftp_server_t ftp, 
			    const char *pos, size_t minlen);
static void matches(const char *pos, size_t pos_len, tab_list_t *list,
		    int *was_a_full_match);
static void beep(void);
static void updateline(char **line, size_t start, size_t len, 
		       const char *match, size_t match_len);
static void show_list(tab_list_t list, int TERM_width);
static void get_word(const char *line, char **word, size_t line_pos, 
		     size_t *word_start, size_t *word_end, int *quoted);
static void str_replace(const char *needle, const char *replace, 
			char **haystack, size_t *haystack_len);
static void remove_dupes(tab_list_t list);
static int alphasort2(const void *a, const void *b);
#ifdef WIN32
static void replace_backslash_with_frontslash(char *path);
#endif

/* tabPressed. The main function, really has very little to do with readline.
   pftp_complete calls this one. */
int tabPressed(char **line, size_t *line_pos, pftp_server_t ftp, 
	       unsigned int mask, int TERM_width)
{
    char *pos = NULL;
    int ret = 0, quoted = 0, fullmatch = 0;
    size_t pos_len = 0, len, word_start = 0, word_end = 0, word_pos;
    tab_list_t list = NULL;
    char *word = NULL;
    size_t add_to_pos_len_at_end = 0;
    
    /* Get the "word" to complete */
    get_word((*line), &word, (*line_pos), &word_start, &word_end, &quoted);
    word_pos = (*line_pos) - word_start;
    
    list = init_list();

#ifdef WIN32
    if (mask & TAB_LOCAL)
	replace_backslash_with_frontslash(word);
#endif
    
    if (mask & TAB_COMMAND) {
	/* Get the word we want to complete */
	pos = word;
	pos_len = word_end - word_start;
	
	add_commands(list, pos, pos_len);
    } else if (mask & TAB_SITENAME) {
	/* Get the word we want to complete */
	pos = word;
	pos_len = word_end - word_start;

	if (!(mask & TAB_LOADSITE)) {
	    add_current_sitenames(list, pos, pos_len);	
	}

	add_saved_sitenames(list, pos, pos_len);	
	remove_dupes(list);
    } else if (mask & (TAB_DIRECTORY | TAB_FILE)) {
	/* Get the path we want to complete */
	char *old = NULL, *path = NULL;
	size_t path_len = word_end - word_start;

	if (!quoted) {
	    size_t old;
	    old = path_len;
	    str_replace("\\ ", " ", &word, &path_len);    
	    add_to_pos_len_at_end = old - path_len;
	}

#ifdef WIN32
	if (path_len >= 3 && word[1] == ':' && word[2] == '/') {
	    path = malloc(path_len + 2);
	    memcpy(path, word, 3);
	    path[3] = '/';
	    memcpy(path + 4, word + 3, path_len - 2);
/*	    path[path_len + 1] = '\0'; */
	} else {
	    path = malloc(path_len + 1);
	    memcpy(path, word, path_len + 1);
/*	    path[path_len] = '\0'; */
	}
#else
	path = malloc(path_len + 1);
	memcpy(path, word, path_len + 1);
/*	path[path_len] = '\0'; */
#endif

	if (!(pos = strrchr(path, '/')))
	    pos = path;
	if (pos > path) {
	    pos[0] = '\0';
	} else {
	    if (path[0] == '/')
		pos++;

	    pos[0] = '\0';
	}
	path_len = pos - path;
	pos = word + path_len;
	if (pos[0] == '/') pos++;
	if ((old = strchr(pos, '/')))
	    pos_len = old - pos;
	else
	    pos_len = strlen(pos);

	/* Read items from FS */
	
	if (mask & TAB_LOCAL) {
	    expand_tilde(&path);
	    path_len = strlen(path);

	    if (!fix_localpath(path, path_len, &old)) {
#ifndef WIN32
		if (pos_len && pos[0] == '~' && path_len == 0) {
		    add_users(list, pos, pos_len);
		} else {
#else
		{
#endif
		    if (mask & TAB_DIRECTORY)
			add_localdirs(list, pos, pos_len);
		    if (mask & TAB_FILE) 
			add_localfiles(list, pos, pos_len);
		}
		
		restore_localpath(&old);
	    }
	} else if (mask & TAB_REMOTE) {
	    if (!fix_remotepath(path, path_len, ftp, &old)) {
		if (mask & TAB_DIRECTORY)
		    add_remotedirs(list, ftp, pos, pos_len);	  
		if (mask & TAB_FILE) 
		    add_remotefiles(list, ftp, pos, pos_len);
		
		restore_remotepath(ftp, &old);   
	    }
	}
	
	free(path);
    }

    /* Look for matches */
    matches(pos, pos_len, &list, &fullmatch);
    len = len_list(list);

    /* To fix diff when removing all "\\ " */
    pos_len += add_to_pos_len_at_end;
    
    switch (len) {
    case 0:
	/* No matches found */
	beep();
	ret = -1;
	break;
    case 1: {
	/* One (1) match found, update line */
	char *match = NULL;
	size_t len = 0;
	get_matchdata(get_first(list), &match, &len);

	/* Fix spaces in unquoted path's */
	if (!quoted && (mask & (TAB_DIRECTORY | TAB_FILE))) {
	    if (fullmatch) {
		if (match[len-1] != ' ') {
		    str_replace(" ", "\\ ", &match, &len);
		} else {
		    match[--len] = '\0';
		    str_replace(" ", "\\ ", &match, &len);
		    match = realloc(match, len + 2);
		    match[len++] = ' ';
		    match[len] = '\0';
		}
	    } else {
		str_replace(" ", "\\ ", &match, &len);
	    }
	}

	updateline(line, word_start + (pos - word), pos_len, match, len);
	(*line_pos) = word_start + (pos - word) + len;
	ret = 1;
	free(match);
    }; break;  
    default:
	/* Multiple matches found, show them */
	show_list(list, TERM_width);
	ret = 0;
    }
    
    free(word);
    free_list(list);
    
    return ret;
}

/* Extract the word to be completed from line and find out if it's quoted */
void get_word(const char *line, char **word, size_t line_pos, 
	      size_t *word_start, size_t *word_end, int *quoted)
{
    char *pos = NULL, *tmp = NULL;
    const char *start = line;
    int open = 0;
    
    for (;;) {
	pos = strchr(start, '\'');
	if (!pos) {
	    pos = strchr(start, '\"');
	} else {
	    tmp = strchr(start, '\"');
	    if (tmp && tmp < pos)
		pos = tmp;
	}
	
	if (!pos)
	    break;
	
	if ((pos - line) > line_pos)
	    break;
	
	open = 1;
	start = pos + 1;
	pos = strchr(start, pos[0]);
	
	if (!pos)
	    break;
	
	open = 0;
	start = pos + 1;
    }
    
    if (open) {
	(*quoted) = 1;
	
	(*word_start) = start - line;
	(*word_end) = strlen(line + (*word_start)) + (*word_start);
    } else {
	size_t c = line_pos;
	(*quoted) = 0;
	
	while (c > 0 && (line[c-1] != ' ' || (c > 1 && line[c-2] == '\\'))) {
	    c--;
	}
	
	(*word_start) = c;
	if ((pos = strchr(line + (*word_start), ' ')) 
	    && ((pos - line) < line_pos)) {
	    (*word_end) = (*word_start) + strlen(line + (*word_start));
	} else {
	    if ((pos = strchr(line + (*word_start), ' ')))
		(*word_end) = pos - line;
	    else
		(*word_end) = (*word_start) + strlen(line + (*word_start));    
	}
    }
    
    (*word) = malloc(((*word_end) - (*word_start)) + 1);
    memcpy((*word), line + (*word_start), (*word_end) - (*word_start));
    (*word)[(*word_end) - (*word_start)] = '\0';
}

#ifndef WIN32
/* adds all usernames longer or the same as minlen to the list.
   Only called if pos[0] == '~' */
void add_users(tab_list_t list, const char *pos, size_t minlen)
{
    struct passwd *data;
    size_t l, tmps = 0;
    char **tmp = NULL;

    setpwent();
    while ((data = getpwent())) {
	if (data->pw_name) {
	    l = strlen(data->pw_name) + 1;	    
	    if (l >= minlen) {
		tmp = realloc(tmp, (tmps + 1) * sizeof(char *));
		tmp[tmps] = malloc(l + 1);
		tmp[tmps][0] = '~';
		memcpy(tmp[tmps] + 1, data->pw_name, l - 1);
		tmp[tmps][l] = '\0';
		tmps++;
	    }
	}
    }
    endpwent();

    qsort(tmp, tmps, sizeof(char *), alphasort2);

    for (l = 0; l < tmps; l++) {
	add_item(list, tmp[l], strlen(tmp[l]));
	free(tmp[l]);
    }

    if (tmp) free(tmp);
}
#endif

/* adds all commands longer or the same as minlen to the list */
void add_commands(tab_list_t list, const char *pos, size_t minlen)
{
    size_t c, l, tmps = 0;
    char **tmp = NULL;
    for (c = 0; c < no_commands; c++) {
	l = strlen(commands[c].name);
	if (l >= minlen) {
	    tmp = realloc(tmp, (tmps + 1) * sizeof(char *));
	    tmp[tmps] = malloc(l + 1);
	    memcpy(tmp[tmps], commands[c].name, l + 1);
	    tmps++;
	}
    }

    qsort(tmp, tmps, sizeof(char *), alphasort2);

    for (l = 0; l < tmps; l++) {
	add_item(list, tmp[l], strlen(tmp[l]));
	free(tmp[l]);
    }

    if (tmp) free(tmp);
}

/* adds all open sitenames available */
void add_current_sitenames(tab_list_t list, const char *pos, size_t minlen)
{
    size_t c, l;

    if (minlen > 0 && (pos[0] == '#' || xisdigit(pos[0])))
	return;

    for (c = 0; c < open_list_len; c++) {
	l  = strlen(open_list[c].name);
	if (l >= minlen)
	    add_item(list, open_list[c].name, l);
    }
}

/* adds all saved sites available */
void add_saved_sitenames(tab_list_t list, const char *pos, size_t minlen)
{
    size_t c, l, len = pftp_getNrOfSettings();
    pftp_settings_t settings;

    if (minlen > 0 && (pos[0] == '#' || xisdigit(pos[0])))
	return;

    for (c = 0; c < len; c++) {
	settings = pftp_getFTPSettings2(c);
	if (settings->name[0] == '\n')
	    continue;
	l = strlen(settings->name);
	if (l >= minlen)
	    add_item(list, settings->name, l);
    }
}

#if !HAVE_SCANDIR
/* If libc don't have SCANDIR, we use this as an sort func for qsort */
int alphasort(const void *a, const void *b)
{
    return strcoll((*(struct dirent **)a)->d_name, 
		   (*(struct dirent **)b)->d_name);
}
#endif

/* Used both by add_localfiles and add_localdirs, add matching directory 
   entities to list (and adds a '/' for dirs and ' ' for files if needed...) */
void add_local(tab_list_t list, size_t minlen, 
	       int (*sel_func)(const struct dirent *),
	       char append)
{
    struct dirent **namelist = NULL;
    int ret = 0;
#if HAVE_SCANDIR
    ret = scandir(".", &namelist, sel_func, alphasort);
#else
    /* Our own scandir */
    DIR *dh = opendir(".");
    
    if (dh) {
	struct dirent *buf;
	ret = 0;
	
	for (;;) {
	    if (!(buf = readdir(dh)))
		break;

	    if (sel_func(buf)) {
		namelist = realloc(namelist, 
				   (ret + 1) * sizeof(struct dirent *));
		namelist[ret] = malloc(buf->d_reclen);
		memcpy(namelist[ret], buf, buf->d_reclen);
		ret++;
	    }
	}
	
	closedir(dh);
	
	qsort(namelist, ret, sizeof(struct dirent *), alphasort);
    } else {
	ret = -1;
    }
    
#endif /* HAVE_SCANDIR */
    
    if (ret == -1) {
	return;
    } else {
	int i;
	size_t l;
	
	if (append) {
	    char *buf = NULL;
	    for (i = 0; i < ret; i++) {
		l = strlen(namelist[i]->d_name);
		if (l >= minlen) {	
		    buf = realloc(buf, l + 1);
		    memcpy(buf, namelist[i]->d_name, l);
		    buf[l] = append;
		    add_item(list, buf, l + 1);
		}
		free(namelist[i]);
	    }
	    if (buf) free(buf);
	} else {
	    for (i = 0; i < ret; i++) {
		l = strlen(namelist[i]->d_name);
		if (l >= minlen) {	
		    add_item(list, namelist[i]->d_name, l);
		}
		free(namelist[i]);
	    }
	}
	
	if (namelist) free(namelist);
    }
}

/* Used when calling add_local to feed to scandir to select what directory
   enteties should be added to list */
int sel_dir(const struct dirent *d)
{
#if HAVE_D_TYPE
    return (d->d_type == DT_DIR && 
	    strcmp(d->d_name, ".") && strcmp(d->d_name, ".."));
#else
    static struct stat buf;
    if (stat(d->d_name, &buf))
	return 0;
    return (S_ISDIR(buf.st_mode) && 
	    strcmp(d->d_name, ".") && strcmp(d->d_name, ".."));
#endif
}

int sel_file(const struct dirent *d)
{
#if HAVE_D_TYPE
    return (d->d_type != DT_DIR);
#else
    static struct stat buf;
    if (stat(d->d_name, &buf))
	return 0;
    return !S_ISDIR(buf.st_mode);
#endif
}

/* Se add_local above */
void add_localdirs(tab_list_t list, const char *pos, size_t minlen)
{  
    add_local(list, minlen, sel_dir, pos[minlen] != '/' ? '/' : 0);
}

void add_localfiles(tab_list_t list, const char *pos, size_t minlen)
{
    add_local(list, minlen, sel_file, pos[minlen] != ' ' ? ' ' : 0);
}

int alphasort2(const void *a, const void *b)
{
    return strcoll((*(const char **)a), (*(const char **)b));
}

/* Used both by add_remotefiles and add_remotedirs, add matching directory 
   entities to list (and adds a '/' for dirs and ' ' for files if needed...) */
void add_remote(tab_list_t list, pftp_server_t ftp, size_t minlen, 
		int (*sel_func)(const pftp_file_t *), char append)
{
    pftp_directory_t dir;
    char **name = NULL, *pos;
    size_t names = 0, len, f;

    if (!pftp_ls(ftp, &dir, 0, NULL)) {
	for (f = 0; f < dir.length; f++) {
	    if (sel_func(dir.files + f)) {
		name = realloc(name, (names + 1) * sizeof(char *)); 
		len = strlen(dir.files[f].name) + 1;
		name[names] = malloc(len);
		memcpy(name[names], dir.files[f].name, len);
		if ((pos = strstr(name[names], " -> "))) {
		    pos[0] = '\0';
		}
		names++;
	    }
	}

	pftp_free_fdt(dir);
	qsort(name, names, sizeof(char *), alphasort2);
    } else {
	return;
    }
    
    if (append) {
	for (f = 0; f < names; f++) {
	    len = strlen(name[f]);
	    if (len >= minlen 
		&& strcmp(name[f], ".") && strcmp(name[f], "..")) {	
		name[f][len] = append;
		add_item(list, name[f], len + 1);
	    }
	    free(name[f]);
	}
    } else {
	for (f = 0; f < names; f++) {
	    len = strlen(name[f]);
	    if (len >= minlen 
		&& strcmp(name[f], ".") && strcmp(name[f], "..")) {	
		add_item(list, name[f], len);
	    }
	    free(name[f]);
	}
    }

    if (name) free(name);
}

int sel_rdir(const pftp_file_t *item)
{
    return (item->type == pft_directory);
}

int sel_rfile(const pftp_file_t *item)
{
    return (item->type != pft_directory);
}

void add_remotedirs(tab_list_t list, pftp_server_t ftp, 
		    const char *pos, size_t minlen)
{
    add_remote(list, ftp, minlen, sel_rdir, pos[minlen] != '/' ? '/' : 0);
}

void add_remotefiles(tab_list_t list, pftp_server_t ftp, 
		     const char *pos, size_t minlen)
{
    add_remote(list, ftp, minlen, sel_rfile, pos[minlen] != ' ' ? ' ' : 0);
}

static void int_matches(const char *pos, size_t pos_len, 
			tab_list_t list, tab_list_t bak, 
			int (*cmpnstrfunc)(const char *, const char *, 
					   size_t len),
			int (*cmpcharfunc)(char, char), int casesensitive,
			int *was_a_fullmatch)
{
    tab_item_t item = get_first(list);
    *was_a_fullmatch = 1;
    
    while (item) {
	if (cmpnstrfunc(pos, get_match(item), pos_len)) {
	    add_item2(bak, item);
	    remove_listitem(list, &item);
	} else {
	    item = get_nextitem(item);
	}
    }
    
    if (len_list(list) > 1) {
	/* Found more than one? Maybe they have something in common atleast */
	size_t new_len = pos_len, new_buf = pos_len + 10;
	char *build_new = malloc(new_buf);
	int same_case = 1, stop;
	tab_item_t item2;

	*was_a_fullmatch = 0;

	memset(build_new, 0, new_buf);
	
	if (casesensitive) {
	    /* If all matches begins with X and pos starts with x, 
	       it's changed. */
	    item = get_first(list);
	    
	    while ((item2 = get_nextitem(item))) {
		if (strncmp(get_match(item), get_match(item2), pos_len)) {
		    same_case = 0;
		    break;
		}
		item = item2;
	    }
	} else {
	    same_case = 0;
	}
	
	item = get_first(list);
	
	if (same_case)
	    memcpy(build_new, get_match(item), pos_len);
	else
	    memcpy(build_new, pos, pos_len);
	
	/* Walk as long as every match still is the same */
	
	for (;;) {      
	    if (!get_match(item)[new_len])
		break;
	    
	    stop = 0;
	    while ((item2 = get_nextitem(item))) {
		if (!get_match(item2)[new_len]) {
		    stop = 1;
		    break;
		}
		
		if (cmpcharfunc(get_match(item)[new_len], 
				get_match(item2)[new_len])) {
		    stop = 1;
		    break;
		}
		
		item = item2;
	    }
	    
	    if (stop)
		break;
	    
	    item = get_first(list);
	    build_new[new_len] = get_match(item)[new_len];
	    new_len++;
	    if (new_len == new_buf) {
		new_buf += 10;
		build_new = realloc(build_new, new_buf);
		memset(build_new + new_len, 0, new_buf - new_len);
	    }		
	}
	
	if (new_len > pos_len || (new_len == pos_len && 
				  strncmp(pos, build_new, pos_len))) {
	    item = get_first(list);
	    while (item) remove_listitem(list, &item);
	    add_item(list, build_new, new_len);
	}
	
	free(build_new);
    }
}

int charcmp(char c1, char c2)
{
    return ((c1 < c2) ? -1 : ((c1 > c2) ? 1 : 0));    
}

int charcasecmp(char c1, char c2)
{
    if (xisupper(c1))
	c1 = tolower(c1);
    if (xisupper(c2))
	c2 = tolower(c2);
    
    return ((c1 < c2) ? -1 : ((c1 > c2) ? 1 : 0));
}

/* Look's for matching completions in list */
void matches(const char *pos, size_t pos_len, tab_list_t *list, 
	     int *was_a_fullmatch)
{
    tab_list_t bak = init_list();
    
    int_matches(pos, pos_len, (*list), bak, strncmp, charcmp, 0, 
		was_a_fullmatch); 
    
    if (len_list((*list)) > 0) {
	/* Found any with same case? Go for it. */
	free_list(bak);
	return;
    }
    
    /* Didn't find any with same case? Check the rest then, without case */
    free_list((*list));
    move_list(list, bak);
    bak = init_list();

    int_matches(pos, pos_len, (*list), bak, strncasecmp, charcasecmp, 1,
		was_a_fullmatch); 
    
    free_list(bak);
}

/* Issued when no matches is found (beeps) */
void beep(void)
{
    rl_ding();
}

/* Update string, replace whatever is between start and start+len with
   item->match */
void updateline(char **word, size_t start, size_t len, const char *match,
	    size_t match_len)
{
    size_t l;

    /* match_len will never be less than len (or it deserves to crash) */
    assert(match_len >= len);

    l = (strlen((*word)) + 1) - len;
    (*word) = realloc((*word), l + match_len);
    memmove((*word) + start + match_len, (*word) + start + len, l - start);
    memcpy((*word) + start, match, match_len);
}

/* Show list of possible completions */
void show_list(tab_list_t list, int TERM_width)
{
    size_t count = len_list(list);
    size_t l = 0, *len = malloc(count * sizeof(size_t));
    size_t columns = 0, *col_width = NULL, lines = 0, y, x;
    tab_item_t item = get_first(list);
    char *space = malloc(TERM_width), **match = malloc(count * sizeof(char *));
    memset(space, ' ', TERM_width);
    
    while (item) {
	match[l] = NULL;
	get_matchdata(item, &match[l], &len[l]);
	l++;
	item = get_nextitem(item);
    }
    
    calc_columns(len, count, TERM_width, 2, &columns, &col_width, &lines);
    item = get_first(list);
    
    puts("");
    
    for (y = 0; y < lines; y++) {
	for (x = 0; x < columns; x++) {
	    if ((l = y+(x*lines)) >= count)
		break;
	    fwrite(match[l], 1, len[l], stdout);
	    if (col_width[x] > len[l])
		fwrite(space, 1, col_width[x] - len[l], stdout);
	    free(match[l]);
	}
	puts("");
    }
    
    free(match);
    free(len);
    free(space);
    free(col_width);
}

/* Get us to the directory path point's at but remember old */
int fix_localpath(const char *path, size_t path_len, char **old)
{
    char *new = NULL;
    size_t size = 100;
    (*old) = NULL;
    
    if (path_len == 0)
	return 0;
    
    for (;;) {
	(*old) = realloc((*old), size);
	if (getcwd((*old), size) == (*old)) 
	    break;
	if (errno != ERANGE) {
	    free((*old));
	    (*old) = NULL;
	    return -1;
	}
	size *= 2;
    }
    
    new = malloc(path_len + 1);
    memcpy(new, path, path_len + 1);
    
    if (new[path_len] == '/' && path_len > 1) 
	new[path_len] = '\0';
    
    if (chdir(new)) {
	free((*old));
	(*old) = NULL;
	free(new);
	return -1;
    }
    
    free(new);
    return 0;
}

/* Fix'es what fix_localpath above did */
void restore_localpath(char **old)
{
    if ((*old)) {
	chdir((*old));
	free((*old));
	(*old) = NULL;
    }
}

int fix_remotepath(const char *path, size_t path_len, pftp_server_t ftp, 
		   char **old)
{
    char *new = NULL;
    (*old) = NULL;
    
    if (path_len == 0)
	return 0;
    
    pftp_curdir(ftp, old, 0);
    if (!(*old))
	return -1;

    new = malloc(path_len + 1);
    memcpy(new, path, path_len + 1);
    
    if (new[path_len] == '/' && path_len > 1) 
	new[path_len] = '\0';

    if (pftp_cd(ftp, new, NULL)) {
	free((*old));
	(*old) = NULL;
	free(new);
	return -1;
    }
    
    free(new);
    return 0;
}

void restore_remotepath(pftp_server_t ftp, char **old)
{
    if ((*old)) {
	pftp_cd(ftp, (*old), NULL);
	free((*old));
	(*old) = NULL;
    }
}

/* Searches string (haystack) for needle and replaces those with replace.
   Replace can contain needle. */
void str_replace(const char *needle, const char *replace, char **haystack,
		 size_t *haystack_len)
{
    char *pos = NULL;
    size_t n_l = strlen(needle), r_l = strlen(replace);
    char *start = (*haystack);
    
    if (n_l > r_l) {
	/* No reallocation needed, but some moving */
	while ((pos = strstr(start, needle))) {
	    memcpy(pos, replace, r_l);
	    memmove(pos + r_l, pos + n_l, 
		    (*haystack_len) - ((pos + n_l) - (*haystack)));
	    (*haystack_len) -= (n_l - r_l);
	    (*haystack)[(*haystack_len)] = '\0';
	    start = pos + r_l;
	}
    } else if (n_l < r_l) {
	/* Reallocation and moving needed */	
	size_t p;
	while ((pos = strstr(start, needle))) {
	    p = pos - (*haystack);
	    (*haystack) = realloc((*haystack), 
				  (*haystack_len) + 1 + (r_l - n_l));
	    pos = (*haystack) + p;
	    memmove(pos + r_l, pos + n_l, 
		    (*haystack_len) - ((pos + n_l) - (*haystack)));
	    memcpy(pos, replace, r_l);
	    (*haystack_len) += (r_l - n_l);
	    (*haystack)[(*haystack_len)] = '\0';
	    start = pos + r_l;
	}
    } else {
	/* No reallocation and no moving */
	while ((pos = strstr(start, needle))) {
	    memcpy(pos, replace, r_l);
	    start = pos + r_l;
	}
    }
}

void remove_dupes(tab_list_t list)
{
    tab_item_t item1, item2;

    item1 = get_first(list);

    while (item1) {
	item2 = get_nextitem(item1);
	while (item2) {
	    if (get_matchlen(item1) == get_matchlen(item2) &&
		memcmp(get_match(item1), get_match(item2), 
		       get_matchlen(item1)) == 0)
		remove_listitem(list, &item2);
	    else
		item2 = get_nextitem(item2);
	}	
	item1 = get_nextitem(item1);
    }
}

#ifdef WIN32
void replace_backslash_with_frontslash(char *path)
{
    char *p;
    p = path;
    while (*p) {
	if (*p == '\\')
	    *p = '/';
	p++;
    }
}
#endif
