/**
 * \file markup.c
 *
 * \brief Markup handling in PennMUSH strings.
 *
 *
 */

#include "config.h"

#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include <limits.h>
#include "copyrite.h"
#include "conf.h"
#include "case.h"
#include "pueblo.h"
#include "parse.h"
#include "externs.h"
#include "ansi.h"
#include "mymalloc.h"
#include "log.h"
#include "game.h"
#include "confmagic.h"

#define ANSI_BEGIN   "\x1B["
#define ANSI_FINISH  "m"

/* COL_* defines */

#define CBIT_HILITE      (1)     /**< ANSI hilite attribute bit */
#define CBIT_INVERT      (2)     /**< ANSI inverse attribute bit */
#define CBIT_FLASH       (4)     /**< ANSI flash attribute bit */
#define CBIT_UNDERSCORE  (8)     /**< ANSI underscore attribute bit */

#define COL_NORMAL      (0)     /**< ANSI normal */
#define COL_HILITE      (1)     /**< ANSI hilite attribute value */
#define COL_UNDERSCORE  (4)     /**< ANSI underscore attribute value */
#define COL_FLASH       (5)     /**< ANSI flag attribute value */
#define COL_INVERT      (7)     /**< ANSI inverse attribute value */

#define COL_BLACK       (30)    /**< ANSI color black */
#define COL_RED         (31)    /**< ANSI color red */
#define COL_GREEN       (32)    /**< ANSI color green */
#define COL_YELLOW      (33)    /**< ANSI color yellow */
#define COL_BLUE        (34)    /**< ANSI color blue */
#define COL_MAGENTA     (35)    /**< ANSI color magenta */
#define COL_CYAN        (36)    /**< ANSI color cyan */
#define COL_WHITE       (37)    /**< ANSI color white */

/* Now the code */

static int write_ansi_close(char *buff, char **bp);
static int write_ansi_letters(const ansi_data *cur, char *buff, char **bp);
static int safe_markup(char const *a_tag, char *buf, char **bp, char type);
static int
 safe_markup_cancel(char const *a_tag, char *buf, char **bp, char type);
static int compare_starts(const void *a, const void *b);
static int escape_marked_str(char **str, char *buff, char **bp);

const char *is_allowed_tag(const char *s, unsigned int len);

static const ansi_data ansi_null = { 0, 0, 0, 0 };

/* ARGSUSED */
FUNCTION(fun_stripansi)
{
  char *cp;
  cp = remove_markup(args[0], NULL);
  safe_str(cp, buff, bp);
}

#ifdef ANSI_DEBUG
/* ARGSUSED */
void inspect_ansi_string(ansi_string *as, dbref who);
FUNCTION(fun_ansiinspect)
{
  char *ptr;
  ansi_string *as;
  char choice;
  int strnum;
  if (nargs < 2 || !args[0] || !*args[0]) {
    choice = 'r';
    strnum = 0;
  } else {
    choice = *args[0];
    strnum = 1;
  }
  switch (choice) {
  case 'i':
    as = parse_ansi_string(args[strnum]);
    inspect_ansi_string(as, executor);
    free_ansi_string(as);
    break;
  case 'r':
    for (ptr = args[strnum]; *ptr; ptr++) {
      if (*ptr == TAG_START)
        *ptr = '<';
      if (*ptr == TAG_END)
        *ptr = '>';
    }
    safe_str(args[strnum], buff, bp);
    break;
  case 'l':
    safe_integer(arglens[strnum], buff, bp);
    break;
  default:
    safe_str("i: inspect r: raw l: length", buff, bp);
    break;
  }
}
#endif

/* ARGSUSED */
FUNCTION(fun_ansi)
{
  ansi_data colors;
  char *save = *bp;
  char *p;
  int i;

  /* Populate the colors struct */
  define_ansi_data(&colors, args[0]);
  if (!(colors.bits || colors.offbits || colors.fore || colors.back)) {
    if (!safe_strl(args[1], arglens[1], buff, bp))
      /* write_ansi_close(buff, bp); */
      return;
  }

  /* Write the colors to buff */
  if (write_ansi_data(&colors, buff, bp)) {
    *bp = save;
    return;
  }

  /* If the contents overrun the buffer, we
   * place an ANSI_ENDALL tag at the end */
  if (safe_str(args[1], buff, bp) || write_ansi_close(buff, bp)) {
    p = buff + BUFFER_LEN - 6;  /* <c/a> */
    for (i = 10; i > 0 && *p != TAG_START; i--, p--) ;
    if (i > 0) {
      /* There's an extant tag, let's just replace that. */
      *bp = p;
      safe_str(ANSI_ENDALL, buff, bp);
    } else {
      *bp = buff + BUFFER_LEN - 6;
      safe_str(ANSI_ENDALL, buff, bp);
    }
  }
}

/* File generated by gperf */
#include "htmltab.c"

/* ARGSUSED */
FUNCTION(fun_html)
{
  if (!Wizard(executor))
    safe_str(T(e_perm), buff, bp);
  else
    safe_tag(args[0], buff, bp);
}

/* ARGSUSED */
FUNCTION(fun_tag)
{
  int i;
  if (!Wizard(executor)
      && !is_allowed_tag(args[0], arglens[0])) {
    safe_str("#-1", buff, bp);
    return;
  }
  safe_chr(TAG_START, buff, bp);
  safe_chr(MARKUP_HTML, buff, bp);
  safe_strl(args[0], arglens[0], buff, bp);
  for (i = 1; i < nargs; i++) {
    if (ok_tag_attribute(executor, args[i])) {
      safe_chr(' ', buff, bp);
      safe_strl(args[i], arglens[i], buff, bp);
    }
  }
  safe_chr(TAG_END, buff, bp);
}

/* ARGSUSED */
FUNCTION(fun_endtag)
{
  if (!Wizard(executor) && !is_allowed_tag(args[0], arglens[0]))
    safe_str("#-1", buff, bp);
  else
    safe_tag_cancel(args[0], buff, bp);
}

/* ARGSUSED */
FUNCTION(fun_tagwrap)
{
  if (!Wizard(executor) && !is_allowed_tag(args[0], arglens[0]))
    safe_str("#-1", buff, bp);
  else {
    if (nargs == 2)
      safe_tag_wrap(args[0], NULL, args[1], buff, bp, executor);
    else
      safe_tag_wrap(args[0], args[1], args[2], buff, bp, executor);
  }
}

/** A version of strlen that ignores ansi and HTML sequences.
 * \param p string to get length of.
 * \return length of string p, not including ansi/html sequences.
 */
int
ansi_strlen(const char *p)
{
  int i = 0;

  if (!p)
    return 0;

  while (*p) {
    if (*p == TAG_START) {
      while ((*p) && (*p != TAG_END))
        p++;
    } else if (*p == ESC_CHAR) {
      while ((*p) && (*p != 'm'))
        p++;
    } else {
      i++;
    }
    p++;
  }
  return i;
}

/** Returns the apparent length of a string, up to numchars visible 
 * characters. The apparent length skips over nonprinting ansi and
 * tags.
 * \param p string.
 * \param numchars maximum size to report.
 * \return apparent length of string.
 */
int
ansi_strnlen(const char *p, size_t numchars)
{
  size_t i = 0;

  if (!p)
    return 0;
  while (*p && numchars > 0) {
    if (*p == ESC_CHAR) {
      while ((*p) && (*p != 'm')) {
        p++;
        i++;
      }
    } else if (*p == TAG_START) {
      while ((*p) && (*p != TAG_END)) {
        p++;
        i++;
      }
    } else
      numchars--;
    i++;
    p++;
  }
  return i;
}

/** Compare two strings, ignoring all ansi and html markup from a string.
 *  Is *NOT* locale safe (a la strcoll)
 * \param a string to compare to
 * \param b Other string
 * \return int - 0 is identical, -1 or 1 for difference.
 */
int
ansi_strcmp(const char *astr, const char *bstr)
{
  const char *a, *b;

  for (a = astr, b = bstr; *a && *b;) {
    a = skip_leading_ansi(a);
    b = skip_leading_ansi(b);
    if (*a != *b)
      return (*a - *b);
    b++;
    a++;
  }
  if (*a)
    a = skip_leading_ansi(a);
  if (*b)
    b = skip_leading_ansi(b);
  return (*a - *b);
}

/** Compare ansi_data for exact equality.
 * \param a ansi_data to compare
 * \param b other ansi_data
 * \return int - 1 is identical, 0 is different
 */
int
ansi_equal(const ansi_data a, const ansi_data b)
{
  return ((a.bits == b.bits) && (a.offbits == b.offbits) &&
          (a.fore == b.fore) && (a.back == b.back));
}

/** Return true if ansi_data contains no ansi values.
 * \param a ansi_data to check
 * \return int 1 on ansi_null, 0 otherwise
 */
int
ansi_isnull(const ansi_data a)
{
  return ((a.bits == 0) && (a.offbits == 0) && (a.fore == 0) && (a.back == 0));
}

/** Strip all ANSI and HTML markup from a string. As a side effect,
 * stores the length of the stripped string in a provided address.
 * NOTE! Length returned is length *including* the terminating NULL,
 * because we usually memcpy the result.
 * \param orig string to strip.
 * \param s_len address to store length of stripped string, if provided.
 * \return pointer to static buffer containing stripped string.
 */
char *
remove_markup(const char *orig, size_t * s_len)
{
  static char buff[BUFFER_LEN];
  char *bp = buff;
  const char *q;
  size_t len = 0;

  if (!orig) {
    if (s_len)
      *s_len = 0;
    return NULL;
  }

  for (q = orig; *q;) {
    switch (*q) {
    case ESC_CHAR:
      /* Skip over ansi */
      while (*q && *q++ != 'm') ;
      break;
    case TAG_START:
      /* Skip over HTML */
      while (*q && *q++ != TAG_END) ;
      break;
    default:
      safe_chr(*q++, buff, &bp);
      len++;
    }
  }
  *bp = '\0';
  if (s_len)
    *s_len = len + 1;
  return buff;
}

static char ansi_chars[50];
static int ansi_codes[255];

#define BUILD_ANSI(letter,ESCcode) \
do { \
  ansi_chars[ESCcode] = letter; \
  ansi_codes[(unsigned char) letter] = ESCcode; \
} while (0)

void
init_ansi_codes(void)
{
  memset(ansi_chars, 0, sizeof(ansi_chars));
  memset(ansi_codes, 0, sizeof(ansi_codes));
/* 
  BUILD_ANSI('n', COL_NORMAL);
  BUILD_ANSI('f', COL_FLASH);
  BUILD_ANSI('h', COL_HILITE);
  BUILD_ANSI('i', COL_INVERT);
  BUILD_ANSI('u', COL_UNDERSCORE);
*/
  BUILD_ANSI('x', COL_BLACK);
  BUILD_ANSI('X', COL_BLACK + 10);
  BUILD_ANSI('r', COL_RED);
  BUILD_ANSI('R', COL_RED + 10);
  BUILD_ANSI('g', COL_GREEN);
  BUILD_ANSI('G', COL_GREEN + 10);
  BUILD_ANSI('y', COL_YELLOW);
  BUILD_ANSI('Y', COL_YELLOW + 10);
  BUILD_ANSI('b', COL_BLUE);
  BUILD_ANSI('B', COL_BLUE + 10);
  BUILD_ANSI('m', COL_MAGENTA);
  BUILD_ANSI('M', COL_MAGENTA + 10);
  BUILD_ANSI('c', COL_CYAN);
  BUILD_ANSI('C', COL_CYAN + 10);
  BUILD_ANSI('w', COL_WHITE);
  BUILD_ANSI('W', COL_WHITE + 10);
}

#undef BUILD_ANSI

int
write_ansi_data(ansi_data *cur, char *buff, char **bp)
{
  int retval = 0;
  retval += safe_chr(TAG_START, buff, bp);
  retval += safe_chr(MARKUP_COLOR, buff, bp);
  retval += write_ansi_letters(cur, buff, bp);
  retval += safe_chr(TAG_END, buff, bp);
  return retval;
}

static int
write_ansi_close(char *buff, char **bp)
{
  int retval = 0;
  retval += safe_chr(TAG_START, buff, bp);
  retval += safe_chr(MARKUP_COLOR, buff, bp);
  retval += safe_chr('/', buff, bp);
  retval += safe_chr(TAG_END, buff, bp);
  return retval;
}

static int
write_ansi_letters(const ansi_data *cur, char *buff, char **bp)
{
  int retval = 0;
  char *save;
  save = *bp;
  if (cur->fore == 'n') {
    retval += safe_chr(cur->fore, buff, bp);
  } else {
#define CBIT_SET(x,y) (x->bits & y)
    if (CBIT_SET(cur, CBIT_FLASH))
      retval += safe_chr('f', buff, bp);
    if (CBIT_SET(cur, CBIT_HILITE))
      retval += safe_chr('h', buff, bp);
    if (CBIT_SET(cur, CBIT_INVERT))
      retval += safe_chr('i', buff, bp);
    if (CBIT_SET(cur, CBIT_UNDERSCORE))
      retval += safe_chr('u', buff, bp);
#undef CBIT_SET
#define CBIT_SET(x,y) (x->offbits & y)
    if (CBIT_SET(cur, CBIT_FLASH))
      retval += safe_chr('F', buff, bp);
    if (CBIT_SET(cur, CBIT_HILITE))
      retval += safe_chr('H', buff, bp);
    if (CBIT_SET(cur, CBIT_INVERT))
      retval += safe_chr('I', buff, bp);
    if (CBIT_SET(cur, CBIT_UNDERSCORE))
      retval += safe_chr('U', buff, bp);
#undef CBIT_SET

    if (cur->fore)
      retval += safe_chr(cur->fore, buff, bp);
    if (cur->back)
      retval += safe_chr(cur->back, buff, bp);
  }

  if (retval)
    *bp = save;
  return retval;
}


void
nest_ansi_data(ansi_data *old, ansi_data *cur)
{
  if (cur->fore != 'n') {
    cur->bits |= old->bits;
    cur->bits &= ~cur->offbits;
    if (!cur->fore)
      cur->fore = old->fore;
    if (!cur->back)
      cur->back = old->back;
  } else {
    cur->bits = 0;
    cur->offbits = 0;
    cur->back = 0;
  }
}

/* We need EDGE_UP to return 1 if x has bit set and y doesn't. */
#define EDGE_UP(x,y,z) ((x.bits & z) != (y->bits & z))
int
write_raw_ansi_data(ansi_data *old, ansi_data *cur, char *buff, char **bp)
{
  int f = 0;
  ansi_data past = *old;
  ansi_data pres = *cur;
  /* This shouldn't happen */
  if (pres.fore == 'n') {
    if (past.bits || (past.fore != 'n') || past.back) {
      return safe_str(ANSI_RAW_NORMAL, buff, bp);
    }
  }
  if (pres.fore == 'd')
    pres.fore = 0;
  if (pres.back == 'D')
    pres.back = 0;

  /* Do we *unset* anything in cur? */
  if ((past.bits & ~(pres.bits)) ||
      (past.fore && !pres.fore) || (past.back && !pres.back)) {
    safe_str(ANSI_RAW_NORMAL, buff, bp);
    past = ansi_null;
  }

  if (past.fore == pres.fore && past.back == pres.back &&
      past.bits == pres.bits)
    return 0;

  if (!(pres.fore || pres.back || pres.bits)) {
    if (past.fore != 'n')
      return safe_str(ANSI_RAW_NORMAL, buff, bp);
    return 0;
  }

  safe_str(ANSI_BEGIN, buff, bp);
  if (EDGE_UP(past, cur, CBIT_HILITE)) {
    if (f++)
      safe_chr(';', buff, bp);
    safe_integer(COL_HILITE, buff, bp);
  }
  if (EDGE_UP(past, cur, CBIT_INVERT)) {
    if (f++)
      safe_chr(';', buff, bp);
    safe_integer(COL_INVERT, buff, bp);
  }
  if (EDGE_UP(past, cur, CBIT_FLASH)) {
    if (f++)
      safe_chr(';', buff, bp);
    safe_integer(COL_FLASH, buff, bp);
  }
  if (EDGE_UP(past, cur, CBIT_UNDERSCORE)) {
    if (f++)
      safe_chr(';', buff, bp);
    safe_integer(COL_UNDERSCORE, buff, bp);
  }

  if (pres.fore && pres.fore != past.fore) {
    if (f++)
      safe_chr(';', buff, bp);
    safe_integer(ansi_codes[(unsigned char) pres.fore], buff, bp);
  }
  if (pres.back && pres.back != past.back) {
    if (f++)
      safe_chr(';', buff, bp);
    safe_integer(ansi_codes[(unsigned char) pres.back], buff, bp);
  }

  return safe_str(ANSI_FINISH, buff, bp);
}

#undef EDGE_UP

void
define_ansi_data(ansi_data *store, const char *str)
{
  *store = ansi_null;

  for (; str && *str && (*str != TAG_END); str++) {
    switch (*str) {
    case 'n':                  /* normal */
      /* This is explicitly normal, it'll never be colored */
      store->bits = 0;
      store->offbits = ~0;
      store->fore = 'n';
      store->back = 0;
      break;
    case 'f':                  /* flash */
      store->bits |= CBIT_FLASH;
      store->offbits &= ~CBIT_FLASH;
      break;
    case 'h':                  /* hilite */
      store->bits |= CBIT_HILITE;
      store->offbits &= ~CBIT_HILITE;
      break;
    case 'i':                  /* inverse */
      store->bits |= CBIT_INVERT;
      store->offbits &= ~CBIT_INVERT;
      break;
    case 'u':                  /* underscore */
      store->bits |= CBIT_UNDERSCORE;
      store->offbits &= ~CBIT_UNDERSCORE;
      break;
    case 'F':                  /* flash */
      store->offbits |= CBIT_FLASH;
      break;
    case 'H':                  /* hilite */
      store->offbits |= CBIT_HILITE;
      break;
    case 'I':                  /* inverse */
      store->offbits |= CBIT_INVERT;
      break;
    case 'U':                  /* underscore */
      store->offbits |= CBIT_UNDERSCORE;
      break;
    case 'b':                  /* blue fg */
    case 'c':                  /* cyan fg */
    case 'g':                  /* green fg */
    case 'm':                  /* magenta fg */
    case 'r':                  /* red fg */
    case 'w':                  /* white fg */
    case 'x':                  /* black fg */
    case 'y':                  /* yellow fg */
    case 'd':                  /* default fg */
      store->fore = *str;
      break;
    case 'B':                  /* blue bg */
    case 'C':                  /* cyan bg */
    case 'G':                  /* green bg */
    case 'M':                  /* magenta bg */
    case 'R':                  /* red bg */
    case 'W':                  /* white bg */
    case 'X':                  /* black bg */
    case 'Y':                  /* yellow bg */
    case 'D':                  /* default fg */
      store->back = *str;
      break;
    }
  }
  store->bits &= ~(store->offbits);
}

int
read_raw_ansi_data(ansi_data *store, const char *codes)
{
  int curnum;
  if (!codes || !store)
    return 0;
  store->bits = 0;
  store->offbits = 0;
  store->fore = 0;
  store->back = 0;

  /* 'codes' can point at either the ESC_CHAR,
   * the '[', or the following byte. */

  /* Skip to the first ansi number */
  while (*codes && !isdigit((unsigned char) *codes) && *codes != 'm')
    codes++;

  while (*codes && (*codes != 'm')) {
    curnum = atoi(codes);
    if (curnum < 10) {
      switch (curnum) {
      case COL_HILITE:
        store->bits |= CBIT_HILITE;
        store->offbits &= ~CBIT_HILITE;
        break;
      case COL_UNDERSCORE:
        store->bits |= CBIT_UNDERSCORE;
        store->offbits &= ~CBIT_UNDERSCORE;
        break;
      case COL_FLASH:
        store->bits |= CBIT_FLASH;
        store->offbits &= ~CBIT_FLASH;
        break;
      case COL_INVERT:
        store->bits |= CBIT_INVERT;
        store->offbits &= ~CBIT_INVERT;
        break;
      case COL_NORMAL:
        store->bits = 0;
        store->offbits = ~0;
        store->fore = 'n';
        store->back = 0;
        break;
      }
    } else if (curnum < 40) {
      store->fore = ansi_chars[curnum];
    } else if (curnum < 50) {
      store->back = ansi_chars[curnum];
    }
    /* Skip current and find the next ansi number */
    while (*codes && isdigit((unsigned char) *codes))
      codes++;
    while (*codes && !isdigit((unsigned char) *codes) && (*codes != 'm'))
      codes++;
  }
  return 1;
}

/** Return a string pointer past any ansi/html markup at the start.
 * \param p a string.
 * \return pointer to string after any initial ansi/html markup.
 */

char *
skip_leading_ansi(const char *p)
{
  if (!p)
    return NULL;
  while (*p == ESC_CHAR || *p == TAG_START) {
    if (*p == ESC_CHAR) {
      while (*p && *p != 'm')
        p++;
    } else {                    /* TAG_START */
      while (*p && *p != TAG_END)
        p++;
    }
    if (*p)
      p++;
  }
  return (char *) p;

}

static char *
parse_tagname(const char *ptr)
{
  static char tagname[BUFFER_LEN];
  char *tag = tagname;
  if (!ptr || !*ptr)
    return NULL;
  while (*ptr && !isspace((unsigned char) *ptr) && *ptr != TAG_END) {
    *(tag++) = *(ptr++);
  }
  *tag = '\0';
  return tagname;
}

static void
free_markup_info(markup_information *info)
{
  if (info) {
    if (info->start_code) {
      mush_free(info->start_code, "markup_code");
      info->start_code = NULL;
    }
    if (info->stop_code) {
      mush_free(info->stop_code, "markup_code");
      info->stop_code = NULL;
    }
  }
}

/** Convert a string into an ansi_string.
 * This takes a string that may contain ansi/html markup codes and
 * converts it to an ansi_string structure that separately stores
 * the plain string and the markup codes for each character.
 * \param source string to parse.
 * \return pointer to an ansi_string structure representing the src string.
 */
ansi_string *
parse_ansi_string(const char *source)
{
  return real_parse_ansi_string(source);
}

/* This does the actual work of parse_ansi_string. */
ansi_string *
real_parse_ansi_string(const char *source)
{
  ansi_string *data = NULL;
  char src[BUFFER_LEN], *sptr;
  char tagbuff[BUFFER_LEN];
  char *ptr, *txt;
  ansi_data *col;
  char *tmp;
  char type;

  int pos = 0;
  int num = 0;

  int priority = 0;
  markup_information *info;

  ansi_data ansistack[BUFFER_LEN];
  int stacktop = 0;

  ansi_data tmpansi;
  int oldcodes = 0;

  ansistack[0] = ansi_null;

  if (!source)
    return NULL;

  info = NULL;

  sptr = src;
  safe_str(source, src, &sptr);
  *sptr = '\0';

  data = mush_malloc(sizeof(ansi_string), "ansi_string");
  if (!data)
    return NULL;

  /* Set it to zero */
  memset(data, 0, sizeof(ansi_string));

  txt = data->text;
  col = data->ansi;

  for (ptr = src; *ptr;) {
    switch (*ptr) {
    case TAG_START:
      /* In modern Penn, this is both Pueblo/HTML and color defining code */

      for (tmp = ptr; *tmp && *tmp != TAG_END; tmp++) ;
      if (*tmp)
        *tmp = '\0';
      else
        tmp--;
      ptr++;
      /* Now ptr is at TAG_START and tmp is at TAG_END (nulled) */

      type = *(ptr++);
      switch (type) {
      case MARKUP_COLOR:
        if (!*ptr)
          break;
        if (oldcodes == 1) {
          oldcodes = 0;
          stacktop--;
        }
        /* Start or end tag? */
        if (*ptr != '/') {
          define_ansi_data(&tmpansi, ptr);
          nest_ansi_data(&(ansistack[stacktop]), &tmpansi);
          stacktop++;
          ansistack[stacktop] = tmpansi;
        } else {
          if (*(ptr + 1) == 'a') {
            stacktop = 0;       /* Endall tag */
          } else {
            if (stacktop > 0)
              stacktop--;
          }
        }
        break;
      case MARKUP_HTML:
        if (*ptr && *ptr != '/') {
          /* We're at the start tag. */
          info = &(data->markup[data->nmarkups++]);
          info->start_code = mush_strdup(ptr, "markup_code");
          snprintf(tagbuff, BUFFER_LEN, "/%s", parse_tagname(ptr));
          info->stop_code = mush_strdup(tagbuff, "markup_code");
          info->type = MARKUP_HTML;
          info->start = pos;
          info->end = -1;
          info->priority = priority++;
        } else if (*ptr) {
          /* Closing tag */
          for (num = data->nmarkups - 1; num >= 0; num--) {
            if (data->markup[num].end < 0 && data->markup[num].stop_code &&
                strcasecmp(data->markup[num].stop_code, ptr) == 0) {
              break;
            }
          }
          if (num >= 0) {
            data->markup[num].end = pos;
          } else {
            /* This is greviously wrong, we can't find the begin tag?
             * Consider this a standalone tag with no close tag. */
            info = &(data->markup[data->nmarkups++]);
            /* Start code is where we are */
            info->stop_code = mush_strdup(ptr, "markup_code");
            info->type = MARKUP_HTML;
            info->priority = priority++;
            info->start = -1;
            info->end = pos;
          }
        }
        break;
      default:
        /* This is a broken string. Are we near or at buffer_len? */
        if (ptr - source < BUFFER_LEN - 4) {
          /* If we're not, this is broken in more ways than I can think */
          goto broken_string;
        }
        break;
      }
      ptr = tmp;
      ptr++;
      break;
    case ESC_CHAR:
      /* ESC_CHAR tags shouldn't be used anymore, so hopefully
       * we won't get here. 
       * To parse these, we assume they can't have the new tag-style
       * ANSI codes in them, as this should always be true when loading
       * from attributes. Assuming that, this code is separate from the
       * "actual" tags, creating a single temporary holding space on the
       * top of the ansi-stack and playing with the colors there.
       */
      for (tmp = ptr; *tmp && *tmp != 'm'; tmp++) ;

      /* Store the "background" colors */
      tmpansi = ansistack[stacktop];
      if (oldcodes == 0) {
        oldcodes = 1;
        stacktop++;
        ansistack[stacktop] = tmpansi;
        ansistack[stacktop].offbits = 0;
      }

      read_raw_ansi_data(&tmpansi, ptr);
      ansistack[stacktop].bits |= tmpansi.bits;
      ansistack[stacktop].bits &= ~(tmpansi.offbits);   /* ANSI_RAW_NORMAL */
      if (tmpansi.fore)
        ansistack[stacktop].fore = tmpansi.fore;
      if (tmpansi.back)
        ansistack[stacktop].back = tmpansi.back;

      ptr = tmp;
      if (*tmp)
        ptr++;
      break;
    default:
      col[pos] = ansistack[stacktop];
      txt[pos++] = *(ptr++);
    }
  }

  txt[pos] = '\0';
  data->len = pos;

  /* For everything left on the stack:
   * If it's an HTML code, assume it's a standalone, and leave
   *   its stop point where it is. */
  for (num = 0; num < data->nmarkups; num++) {
    info = &(data->markup[num]);
    switch (info->type) {
    case MARKUP_HTML:
      /* If it's HTML, we treat it as standalone (<IMG>, <BR>, etc)
       * This is ugly - it's not a "start" but a "stop" */
      if (info->end < 0) {
        mush_free(info->stop_code, "markup_code");
        info->stop_code = info->start_code;
        info->start_code = NULL;
        info->end = info->start;
        info->start = -1;
      }
      break;
    }
  }
  return data;
broken_string:
  /* This stinks. We treat this as if it's not pueblo-safe */
  if (data == NULL)
    return NULL;
  strncpy(data->text, source, BUFFER_LEN);
  data->len = strlen(data->text);
  for (num = data->nmarkups - 1; num >= 0; num--) {
    free_markup_info(&(data->markup[num]));
  }
  data->nmarkups = 0;
  return data;
}

/** Reverse an ansi string, preserving its ansification.
 * This function destructively modifies the ansi_string passed.
 * \param as pointer to an ansi string.
 */
void
flip_ansi_string(ansi_string *as)
{
  int i, j;
  markup_information *info;
  char tmptext;
  ansi_data tmpansi;
  int mid;
  int len = as->len;

  /* Reverse the text */
  mid = len / 2;                /* Midpoint */
  for (i = len - 1, j = 0; i >= mid; j++, i--) {
    tmptext = as->text[i];
    as->text[i] = as->text[j];
    as->text[j] = tmptext;

    tmpansi = as->ansi[i];
    as->ansi[i] = as->ansi[j];
    as->ansi[j] = tmpansi;
  }

  /* Now reverse the html-markup. */
  for (i = as->nmarkups - 1; i >= 0; i--) {
    int start, end;
    info = &(as->markup[i]);
    if (info->start == -1) {
      /* Standalones */
      info->end = len - info->end;
    } else {
      end = len - info->start;
      start = len - info->end;
      info->start = start;
      info->end = end;
    }
  }
}

/** Free an ansi_string.
 * \param as pointer to ansi_string to free.
 */
void
free_ansi_string(ansi_string *as)
{
  int i;

  if (!as)
    return;

  for (i = as->nmarkups - 1; i >= 0; i--) {
    free_markup_info(&(as->markup[i]));
  }
  mush_free(as, "ansi_string");
}


static int
compare_starts(const void *a, const void *b)
{
  markup_information *ai, *bi;

  ai = (markup_information *) a;
  bi = (markup_information *) b;

  return ai->start - bi->start;
}



void
optimize_ansi_string(ansi_string *as)
{
  int i, j;
  int target = -1;
  int len = 0;

  if (!as)
    return;

  /* If we've only got 1 or 0, or we've already optimized, do nothing. */
  if (as->nmarkups > 1 && as->optimized == 0) {
    /* Sort the markup codes by their start position */
    qsort(as->markup, as->nmarkups, sizeof(markup_information), compare_starts);

    for (i = 0; i < as->nmarkups; i++) {

      /* If end is negative, it's removed or broken. Either way... */
      if (as->markup[i].end < 0)
        continue;

      for (j = i + 1; j < as->nmarkups; j++) {

        /* Already removed? */
        if (as->markup[j].end < 0 && as->markup[j].start < 0)
          continue;

        /* End if we can't stretch markup[i] any farther */
        if (as->markup[i].end < as->markup[j].start)
          break;

        /* If there's an identical code within our bounds, merge it. */
        if (as->markup[i].start_code && as->markup[j].start_code &&
            (strcmp(as->markup[j].start_code, as->markup[i].start_code) == 0)
          ) {
          if (as->markup[j].end > as->markup[i].end)
            as->markup[i].end = as->markup[j].end;
          as->markup[j].start = -1;
          as->markup[j].end = -1;
        }                       /* end if_indentical */
      }                         /* end inner loop */
    }                           /* end outer loop */
  }
  /* end if_optimized */
  j = 0;

  /* Get rid of all removed markups
   * "target" is non-negative when we've pegged a destination
   * "len" begins counting when we have a target set and we hit a
   *   block of non-removed markup 
   * If len is non-zero and we hit a removed markup, shift the block left.
   * The end of the removed string is our new target (it's removable anyway)
   */

  for (i = 0; i < as->nmarkups; i++) {
    /* Valid tag? */
    if (as->markup[i].end >= 0 && (as->markup[i].start < as->markup[i].end)) {
      if (target != -1)
        len++;
      j++;
    } else {
      free_markup_info(&(as->markup[i]));
      if (len > 0 && target != -1)
        memmove(&(as->markup[target]), &(as->markup[i - len]),
                len * sizeof(markup_information));
      if (len > 0 || target == -1) {
        target = j;
        len = 0;
      }
    }
  }
  if (len > 0)
    memmove(&(as->markup[target]), &(as->markup[i - len]),
            len * sizeof(markup_information));

  as->nmarkups = j;
  as->optimized = 1;
}

/* Copy the start code for a particular markup_info */
static int
copy_start_code(markup_information *info, char *buff, char **bp)
{
  int retval = 0;
  char *save;
  save = *bp;
  if (info && info->start_code) {
    retval += safe_chr(TAG_START, buff, bp);
    retval += safe_chr(info->type, buff, bp);
    retval += safe_str(info->start_code, buff, bp);
    retval += safe_chr(TAG_END, buff, bp);
  }
  if (retval)
    *bp = save;
  return retval;
}

/* Copy the stop code for a particular markup_info */
static int
copy_stop_code(markup_information *info, char *buff, char **bp)
{
  int retval = 0;
  char *save;
  save = *bp;
  if (info && info->stop_code) {
    retval += safe_chr(TAG_START, buff, bp);
    retval += safe_chr(MARKUP_HTML, buff, bp);
    retval += safe_str(info->stop_code, buff, bp);
    retval += safe_chr(TAG_END, buff, bp);
  }
  if (retval)
    *bp = save;
  return retval;
}

#ifdef ANSI_DEBUG
void
inspect_ansi_string(ansi_string *as, dbref who)
{
  markup_information *info;
  int count = 0;
  int j;
  notify_format(who, "Inspecting ansi string");
  notify_format(who, "  Text: %s", as->text);
  notify_format(who, "  Nmarkups: %d", as->nmarkups);
  for (j = 0; j < as->nmarkups; j++) {
    info = &(as->markup[j]);
    if (info->type == MARKUP_HTML) {
      notify_format(who,
                    "    %d (%s): (start: %d end: %d) start_code: %s stop_code: %s",
                    count++, (info->type == MARKUP_HTML ? "html" : "ansi"),
                    info->start, info->end, info->start_code, info->stop_code);
    }
  }
  notify_format(who, "Inspecting ansi string complete");
}
#endif

/** Delete a portion of an ansi string.
 * \param as ansi_string to delete from
 * \param start start point to remove
 * \param size length of string to remove
 * \retval 0 success
 * \reval 1 failure.
 */

int
ansi_string_delete(ansi_string *as, int start, int count)
{
  int i;
  int end;
  markup_information *dm;

  /* Nothing to delete */
  if (start >= as->len || count <= 0)
    return 0;

  /* We can't delete from a negative index */
  if (start < 0) {
    /* start is negative: this *decreases* count. */
    count += start;
    start = 0;
  }

  /* We can't delete past the end of our string */
  if ((start + count) > as->len)
    count = as->len - start;

  /* Nothing to delete */
  if (count <= 0)
    return 0;

  end = start + count;

  dm = as->markup;

  /* Remove or shrink the Pueblo markup on dst */
  for (i = 0; i < as->nmarkups; i++) {
    if (dm[i].start >= start && dm[i].end <= end) {
      dm[i].start = -1;
      dm[i].end = -1;
    }
    if (dm[i].start >= start) {
      dm[i].start -= count;
      if (dm[i].start < start)
        dm[i].start = start;
    }
    if (dm[i].end > start) {
      dm[i].end -= count;
      if (dm[i].end < start)
        dm[i].end = start;
    }
  }

  if (as->nmarkups > 0)
    as->optimized = 0;

  /* Shift text over */
  memmove(as->text + start, as->text + end, as->len - end);
  memmove(as->ansi + start, as->ansi + end,
          (as->len - end) * sizeof(ansi_data));
  as->len -= count;
  as->text[as->len] = '\0';
  as->ansi[as->len] = ansi_null;
  return 0;
}

#define copyto(x,y) \
do { \
  x.type = y.type; \
  x.priority = y.priority; \
  if (y.start_code) x.start_code = mush_strdup(y.start_code,"markup_code"); \
  else (x.start_code = NULL); \
  if (y.stop_code) x.stop_code = mush_strdup(y.stop_code,"markup_code"); \
  else (x.stop_code = NULL); \
} while (0)

/** Insert an ansi string into another ansi_string
 * with markups kept as straight as possible.
 * \param dst ansi_string to insert into.
 * \param loc Location to insert into, 0-indexed
 * \param src ansi_string to insert
 * \retval 0 success
 * \retval 1 failure.
 */

int
ansi_string_insert(ansi_string *dst, int loc, ansi_string *src)
{
  int i, j;
  int len;
  int retval = 0;
  int src_len = src->len;

  markup_information *dm, *sm;

  ansi_data backansi;


  /* If src->len == 0, we might have only markup. Stand-alones. Ew! */
  if (src->len <= 0 && src->nmarkups <= 0)
    return 0;
  if (dst->len >= BUFFER_LEN)
    return 1;

  if (src_len >= BUFFER_LEN)
    src_len = BUFFER_LEN - 1;
  if (src_len < 0)
    src_len = 0;

  if (loc > dst->len)
    loc = dst->len;
  if (loc < 0)
    loc = 0;

  if (dst->nmarkups > 0 || src->nmarkups > 0)
    dst->optimized = 0;

  dm = dst->markup;
  sm = src->markup;

  /* shift or widen the Pueblo markup on dst */
  for (i = 0; i < dst->nmarkups; i++) {
    if (loc <= dm[i].start)
      dm[i].start += src_len;
    if (loc < dm[i].end)
      dm[i].end += src_len;
  }

  /* Copy markup onto the end of dst */
  for (j = 0; j < src->nmarkups; j++) {

    /* It's possible, but not at all easy, to get this much Pueblo markup */
    if (i >= BUFFER_LEN)
      break;

    /* Is it a valid tag? */
    if (sm[j].end >= 0 && sm[j].start < sm[j].end &&
        ((sm[j].start < 0 && sm[j].end <= src_len)
         || (sm[j].start < src_len && sm[j].start >= 0))) {

      copyto(dm[i], sm[j]);

      /* If our start is non-negative, start+loc is its position in dm */
      if (sm[j].start >= 0)
        dm[i].start = sm[j].start + loc;
      else
        dm[i].start = -1;

      /* Make sure the end position is within the bounds of its own string */
      if (sm[j].end <= src_len)
        dm[i].end = sm[j].end + loc;
      else
        dm[i].end = src_len + loc;

      i++;
    }
  }

  for (; i >= BUFFER_LEN; i--)
    free_markup_info(&(dm[i]));
  dst->nmarkups = i;

  dst->len += src_len;
  if (dst->len >= BUFFER_LEN) {
    retval = 1;
    dst->len = BUFFER_LEN - 1;
  }

  len = dst->len - src->len - loc;

  /* Determine what old ansi might stretch across the new text.
   * This sets backansi to any ansi values (bits, colors) that
   * are continuous across an entire length of text. */
  backansi = ansi_null;
  if (0 < loc && loc < dst->len) {
    backansi.offbits = dst->ansi[loc - 1].offbits & dst->ansi[loc].offbits;
    backansi.bits = dst->ansi[loc - 1].bits & dst->ansi[loc].bits;
    if (dst->ansi[loc - 1].fore == dst->ansi[loc].fore)
      backansi.fore = dst->ansi[loc].fore;
    if (dst->ansi[loc - 1].back == dst->ansi[loc].back)
      backansi.back = dst->ansi[loc].back;
  }

  /* Shift text over */
  if (0 < len) {
    /* The length beyond our insertion that can *actually* be moved. */
    if (loc + src_len + len >= BUFFER_LEN)
      len = BUFFER_LEN - loc - src_len - len - 1;
    memmove(dst->text + loc + src_len, dst->text + loc, len);
    memmove(dst->ansi + loc + src_len, dst->ansi + loc,
            len * sizeof(ansi_data));
  }


  /* Copy text from src */
  if (loc + src_len >= BUFFER_LEN)
    src_len = BUFFER_LEN - 1 - loc;
  if (src_len > 0) {
    memcpy(dst->text + loc, src->text, src_len);
    for (i = 0; i < src_len; i++) {
      j = loc + i;
      dst->ansi[j].back = src->ansi[i].back ? src->ansi[i].back : backansi.back;
      dst->ansi[j].fore = src->ansi[i].fore ? src->ansi[i].fore : backansi.fore;
      dst->ansi[j].offbits = src->ansi[i].offbits | backansi.offbits;
      dst->ansi[j].bits =
        ~(dst->ansi[j].offbits) & (src->ansi[i].bits | backansi.bits);
    }
  }
  dst->text[dst->len] = '\0';
  dst->ansi[dst->len] = ansi_null;
  return retval;
}


/** Replace a portion of an ansi string with
 *  another ansi string, keeping markups as
 *  straight as possible.
 * \param dst ansi_string to insert into.
 * \param loc Location to  insert into, 0-indexed
 * \param len Length of string inside dst to replace
 * \param src ansi_string to insert
 * \retval 0 success
 * \retval 1 failure.
 */
int
ansi_string_replace(ansi_string *dst, int loc, int len, ansi_string *src)
{
  int i, j;
  int end, d_end;
  int diff;
  int retval = 0;
  ansi_data backansi;
  markup_information *dm, *sm;

  if (loc < 0)
    loc = 0;

  /* Is it really an insert? */
  if (loc >= dst->len || len <= 0)
    return ansi_string_insert(dst, loc, src);

  /* Is it really a delete? */
  if (src->len <= 0)
    return ansi_string_delete(dst, loc, len);

  /* We can't delete past the end of our string. */
  if ((len + loc) > dst->len)
    len = dst->len - loc;

  end = loc + len;              /* End of the removed section */
  d_end = loc + src->len;       /* End of the new string within the dst string */
  diff = src->len - len;        /* Total change in length */

  if (diff > 0 && dst->len >= BUFFER_LEN)
    return 1;

  dst->optimized = 0;

  dm = dst->markup;
  sm = src->markup;

  /* Modify, remove, stretch, and mangle Pueblo markup */
  for (i = 0; i < dst->nmarkups; i++) {

    /* If it doesn't cross into the replaced part, leave as is */
    if (dm[i].end <= loc)
      continue;

    /* Debatable: If it surrounds the replaced part exactly, try
     * to keep it, stretching it to wrap around the replacement. */
    if (loc <= dm[i].start && dm[i].end <= end) {
      /* If the locations match, stretch it, otherwise overwrite it. */
      if (dm[i].start == loc && dm[i].end == end) {
        dm[i].end = loc + src->len;
      } else {
        dm[i].start = -1;
        dm[i].end = -1;
      }
      continue;
    }

    /* Shift the beginning if necessary */
    if (loc < dm[i].start) {
      /* If we're beyond the markup, just shift start. */
      if (end < dm[i].start) {
        dm[i].start += diff;
      } else {
        /* Otherwise it's inside; push it to the right of the new markup. */
        dm[i].start = loc + src->len;
      }
      if (dm[i].start > BUFFER_LEN) {
        dm[i].start = -1;
        dm[i].end = -1;
      }
    }

    /* If this markup ends before the new one, squish it. */
    if (dm[i].end < end) {
      dm[i].end = (dm[i].start >= 0) ? loc : -1;        /* Or erase stand-alones */
    } else {
      dm[i].end += diff;        /* Otherwise, shift it. */
      if (dm[i].end > BUFFER_LEN)
        dm[i].end = BUFFER_LEN;
    }

  }

  /* Copy markup. Code taken from ansi_string_insert. */
  for (j = 0; j < src->nmarkups; j++) {

    /* It's possible, but not at all easy, to get this much pueblo markup */
    if (i >= BUFFER_LEN)
      break;

    /* Is it a valid tag? */
    if (sm[j].end >= 0 && sm[j].start < sm[j].end &&
        ((sm[j].start < 0 && sm[j].end <= src->len)
         || (sm[j].start < src->len && sm[j].start >= 0))) {

      copyto(dm[i], sm[j]);

      /* If our start is non-negative, start+loc is its position in dm */
      if (sm[j].start >= 0)
        dm[i].start = sm[j].start + loc;
      else
        dm[i].start = -1;

      /* Make sure the end position is within the bounds of its own string */
      if (sm[j].end <= src->len)
        dm[i].end = sm[j].end + loc;
      else
        dm[i].end = src->len + loc;

      i++;
    }
  }

  for (; i >= BUFFER_LEN; i--)
    free_markup_info(&(dm[i]));
  dst->nmarkups = i;

  /* length of original string after replace bits */
  len = dst->len - end;

  dst->len += diff;
  if (dst->len >= BUFFER_LEN) {
    retval = 1;
    dst->len = BUFFER_LEN - 1;
  }

  /* Determine what old ansi might stretch across the new text.
   * This sets backansi to any ansi values (bits, colors) that
   * are continuous across an entire length of text.
   */
  backansi = dst->ansi[loc];
  for (i = loc; i < end && !ansi_isnull(backansi); i++) {
    backansi.offbits &= dst->ansi[i].offbits;
    backansi.bits &= dst->ansi[i].bits;
    if (backansi.fore != dst->ansi[i].fore)
      backansi.fore = 0;
    if (backansi.back != dst->ansi[i].back)
      backansi.back = 0;
  }

  /* Shift text over */
  if (diff != 0) {
    if (d_end + len >= BUFFER_LEN)
      len = BUFFER_LEN - (1 + d_end);
    if (len > 0) {
      memmove(dst->text + d_end, dst->text + end, len);
      memmove(dst->ansi + d_end, dst->ansi + end, len * sizeof(ansi_data));
    }
  }

  /* Copy text from src */
  len = src->len;
  if (loc + len >= BUFFER_LEN)
    len = BUFFER_LEN - loc - 1;
  if (len > 0) {
    memcpy(dst->text + loc, src->text, len);
    for (i = 0; i < len; i++) {
      j = loc + i;
      dst->ansi[j].back = src->ansi[i].back ? src->ansi[i].back : backansi.back;
      dst->ansi[j].fore = src->ansi[i].fore ? src->ansi[i].fore : backansi.fore;
      dst->ansi[j].offbits = src->ansi[i].offbits | backansi.offbits;
      dst->ansi[j].bits =
        ~(dst->ansi[j].offbits) & (src->ansi[i].bits | backansi.bits);
    }
  }
  dst->text[dst->len] = '\0';
  dst->ansi[dst->len] = ansi_null;
  return retval;
}


/** Scrambles an ansi_string, returning a pointer to the new string.
 * \param as ansi_string to scramble.
 */
ansi_string *
scramble_ansi_string(ansi_string *as)
{
  int i, j, k;
  int pos[BUFFER_LEN];
  markup_information *dm;
  ansi_string *tmp = NULL;

  if (!as)
    return NULL;

  optimize_ansi_string(as);

  tmp = mush_malloc(sizeof(ansi_string), "ansi_string");
  if (!tmp)
    return NULL;

  memset(tmp, 0, sizeof(ansi_string));

  /* Scramble the text */
  tmp->len = as->len;

  for (i = 0; i < as->len; i++)
    pos[i] = i;

  for (i = 0; i < as->len; i++) {
    j = get_random32(0, as->len - 1);
    k = pos[i];
    pos[i] = pos[j];
    pos[j] = k;
  }

  /* The old scramble did new[i] = old[pos[i]],
   * but handling markup tags is easier if we do it this way. */
  /* Scramble the text and ansi... */
  for (i = 0; i < as->len; i++) {
    tmp->text[pos[i]] = as->text[i];
    tmp->ansi[pos[i]] = as->ansi[i];
  }

  /* Scramble the Pueblo markup */
  dm = tmp->markup;
  if (as->nmarkups > 0)
    tmp->optimized = 0;

  /* Copy the standalones (tags with -1 for start) */
  for (i = 0; i < as->nmarkups && as->markup[i].start == -1; i++) {
    copyto(dm[i], as->markup[i]);
    dm[i].end = pos[i];
  }

  /* Copy the rest */
  j = i;
  for (; i < as->nmarkups; i++) {
    for (k = as->markup[i].start; k < as->markup[i].end; k++) {
      copyto(dm[j], as->markup[i]);
      dm[j].start = pos[k];
      dm[j].end = dm[j].start + 1;
      j++;
      if (j >= BUFFER_LEN) {
        optimize_ansi_string(tmp);
        tmp->optimized = 0;
        j = tmp->nmarkups;
        if (j >= BUFFER_LEN)
          return tmp;
      }
    }
  }
  tmp->nmarkups = j;
  optimize_ansi_string(tmp);
  return tmp;
}

#undef copyto

/** Safely append an ansi_string into a buffer as a real string,
 * \param as pointer to ansi_string to append.
 * \param start position in as to start copying from.
 * \param len length in characters to copy from as.
 * \param buff buffer to insert into.
 * \param bp pointer to pointer to insertion point of buff.
 * \retval 0 success.
 * \retval 1 failure.
 */
int
safe_ansi_string(ansi_string *as, int start, int len, char *buff, char **bp)
{
  int i, j;
  int cur;
  markup_information *info;
  int nextstart, nextend, next;
  int end = start + len;
  int retval = 0;
  ansi_data curansi;

  if (!as)
    return 0;

  optimize_ansi_string(as);

  if (len <= 0)
    return 0;
  if (as->len > BUFFER_LEN)
    as->len = BUFFER_LEN;
  if (start >= as->len)
    return 0;
  if (end > as->len)
    end = as->len;

  /* Standalones (Stop codes with -1 for start) */
  for (j = 0; j < as->nmarkups; j++) {
    info = &(as->markup[j]);
    if (info->start != -1)
      break;                    /* No more standalone tags to copy */
    if (info->stop_code && info->end == start)
      retval += safe_str(info->stop_code, buff, bp);
  }

  /* Now, start codes of everything that impacts us. */
  for (; j < as->nmarkups; j++) {
    info = &(as->markup[j]);
    if (info->start > start)
      break;                    /* No more tags to copy. */
    if (info->end > start)
      retval += copy_start_code(info, buff, bp);
  }

  /* Find the next changes--new tags, or a prior tag ending. */
  i = start;
  nextend = BUFFER_LEN + 1;

  /* If there is another start, it's our next one; we have a sorted list. */
  if (j < as->nmarkups)
    nextstart = as->markup[j].start;
  else
    nextstart = BUFFER_LEN + 1;

  /* To find the next END is harder, since it isn't sorted... */
  /* Scan forward. Stop once we find a tag with a start beyond
   * this one's end. Anything beyond that can't be the nextend,
   * so we'll backtrack from there. */
  if (as->nmarkups > 0) {
    cur = j;
    if (cur >= as->nmarkups)
      cur--;
    info = &(as->markup[cur]);
    while (cur < as->nmarkups && as->markup[cur].start < info->end)
      cur++;
    cur--;
    if (info->end > as->markup[cur].start) {
      for (; cur >= 0; cur--) {
        if (as->markup[cur].end > i && as->markup[cur].end < nextend)
          nextend = as->markup[cur].end;
      }
    }
  }

  next = (nextend < nextstart) ? nextend : nextstart;

  if (end < next)
    next = end;

  curansi = as->ansi[start];
  if (!ansi_isnull(curansi))
    write_ansi_data(&curansi, buff, bp);
  /* If there's any text/ansi between start and next, print it */
  for (i = start; i < next && i < as->len; i++) {
    if (as->text[i]) {
      if (!ansi_equal(curansi, as->ansi[i])) {
        if (!ansi_isnull(curansi))
          write_ansi_close(buff, bp);
        curansi = as->ansi[i];
        if (!ansi_isnull(curansi))
          write_ansi_data(&curansi, buff, bp);
      }
      safe_chr(as->text[i], buff, bp);
    }
  }

  cur = j;                      /* Our current markup */
  i = next;                     /* Our current position */

  /* Basically the same thing as above, in loop form. */
  while (i < end) {

    if (i >= nextend) {
      nextend = BUFFER_LEN + 2;
      j = cur;
      /* Find the last markup that could possibly have a relevant end code */
      while (j < as->nmarkups && as->markup[j].start < as->markup[cur].end)
        j++;
      j--;
      /* We MUST have markup if we're here, so no nmarkup > 0 check. */
      /* Print relevant stop codes, and find our nextend */
      for (; j >= 0; j--) {
        info = &(as->markup[j]);
        if (info->end >= i) {
          if (info->end == i)
            retval += copy_stop_code(info, buff, bp);
          else if (info->end < nextend)
            nextend = info->end;
        }
      }
    }

    if (i >= nextstart) {
      /* Print out all the relevant start codes */
      for (; cur < as->nmarkups && as->markup[cur].start == i; cur++)
        retval += copy_start_code(&(as->markup[cur]), buff, bp);
      if (cur < as->nmarkups)
        nextstart = as->markup[cur].start;
      else
        nextstart = BUFFER_LEN + 2;
    }

    next = (nextend < nextstart) ? nextend : nextstart;
    if (end < next)
      next = end;


    for (; i < next && i < as->len; i++) {
      if (as->text[i]) {
        if (!ansi_equal(curansi, as->ansi[i])) {
          if (!ansi_isnull(curansi))
            write_ansi_close(buff, bp);
          curansi = as->ansi[i];
          if (!ansi_isnull(curansi))
            write_ansi_data(&curansi, buff, bp);
        }
        safe_chr(as->text[i], buff, bp);
      }
    }
  }

  /* Now, find all things that end for us */
  if (as->nmarkups > 0) {
    j = cur;
    while (cur < as->nmarkups && as->markup[j].end > as->markup[cur].start)
      cur++;
    cur--;
    for (; cur >= 0; cur--) {
      info = &(as->markup[cur]);
      if (info->start < i && info->end >= i)
        retval += copy_stop_code(info, buff, bp);
    }
  }

  if (!retval && !ansi_isnull(curansi))
    retval += write_ansi_close(buff, bp);
  return retval;
}

/* Following functions are used for
 * decompose_str()
 */

extern char escaped_chars[UCHAR_MAX + 1];

static int
escape_marked_str(char **str, char *buff, char **bp)
{
  unsigned char *in;
  int retval = 0;
  int dospace = 1;
  int spaces = 0;
  int i;

  if (!str || !*str || !**str)
    return 0;
  in = (unsigned char *) *str;
  for (; *in && *in != ESC_CHAR && *in != TAG_START; in++) {
    if (*in == ' ') {
      spaces++;
    } else {
      if (spaces) {
        if (spaces >= 5) {
          retval += safe_str("[space(", buff, bp);
          retval += safe_number(spaces, buff, bp);
          retval += safe_str(")]", buff, bp);
        } else {
          if (dospace) {
            spaces--;
            retval += safe_str("%b", buff, bp);
          }
          while (spaces) {
            retval += safe_chr(' ', buff, bp);
            if (--spaces) {
              --spaces;
              retval += safe_str("%b", buff, bp);
            }
          }
        }
      }
      spaces = 0;
      dospace = 0;
      switch (*in) {
      case '\n':
        retval += safe_str("%r", buff, bp);
        break;
      case '\t':
        retval += safe_str("%t", buff, bp);
        break;
      case BEEP_CHAR:
        for (i = 1; *(in + 1) == BEEP_CHAR && i < 5; in++, i++) ;
        retval += safe_format(buff, bp, "[beep(%d)]", i);
        break;
      default:
        if (escaped_chars[*in])
          retval += safe_chr('\\', buff, bp);
        retval += safe_chr(*in, buff, bp);
        break;
      }
    }
  }
  if (spaces) {
    if (spaces >= 5) {
      retval += safe_str("[space(", buff, bp);
      retval += safe_number(spaces, buff, bp);
      retval += safe_str(")]", buff, bp);
    } else {
      spaces--;                 /* This is for the final %b space */
      if (spaces && dospace) {
        spaces--;
        retval += safe_str("%b", buff, bp);
      }
      while (spaces) {
        safe_chr(' ', buff, bp);
        if (--spaces) {
          --spaces;
          retval += safe_str("%b", buff, bp);
        }
      }
      retval += safe_str("%b", buff, bp);
    }
  }
  *str = (char *) in;
  return retval;
}

/* Does the work of decompose_str, which is found in look.c.
 * Even handles ANSI and Pueblo, which is why it's so ugly.
 * Code based off of real_parse_ansi_string, not safe_ansi_string.
 */
int
real_decompose_str(char *orig, char *buff, char **bp)
{
  int i;
  char *str = orig;
  char *tmp;
  char *pstr;
  char type;

  ansi_data ansistack[BUFFER_LEN];
  ansi_data oldansi;
  ansi_data tmpansi;
  int ansitop = 0;
  int ansiheight = 0;
  int howmanyopen = 0;
  int oldcodes = 0;

  char *pueblostack[BUFFER_LEN];
  char tagbuff[BUFFER_LEN];
  int pueblotop = -1;

  int retval = 0;

  ansistack[0] = ansi_null;

  if (!str || !*str)
    return 0;

  retval += escape_marked_str(&str, buff, bp);

  while (str && *str && *str != '\0') {
    oldansi = ansistack[ansitop];
    ansiheight = ansitop;
    while (*str == TAG_START || *str == ESC_CHAR) {
      switch (*str) {
      case TAG_START:
        for (tmp = str; *tmp && *tmp != TAG_END; tmp++) ;
        if (*tmp) {
          *tmp = '\0';
        } else {
          tmp--;
        }
        str++;
        type = *(str++);
        switch (type) {
        case MARKUP_COLOR:
          if (!*str)
            break;
          if (oldcodes == 1) {
            ansitop--;
            oldcodes = 0;
          }
          /* Start or end tag? */
          if (*str != '/') {
            define_ansi_data(&tmpansi, str);
            nest_ansi_data(&(ansistack[ansitop]), &tmpansi);
            ansitop++;
            ansistack[ansitop] = tmpansi;
          } else {
            if (*(str + 1) == 'a') {
              ansitop = 0;
            } else {
              if (ansitop > 0) {
                ansitop--;
              }
            }
          }
          break;
        case MARKUP_HTML:
          if (!*str)
            break;
          if (*str != '/') {
            pueblotop++;
            snprintf(tagbuff, BUFFER_LEN, "%s", parse_tagname(str));
            pueblostack[pueblotop] = mush_strdup(tagbuff, "markup_code");

            retval += safe_str("[tag(", buff, bp);
            retval += safe_str(tagbuff, buff, bp);
            str += strlen(tagbuff);
            if (str && *str) {
              while (str && str != tmp) {
                str++;
                pstr = strchr(str, '=');
                if (pstr) {
                  *pstr = '\0';
                  retval += safe_chr(',', buff, bp);
                  retval += safe_str(str, buff, bp);
                  retval += safe_chr('=', buff, bp);
                  str = pstr + 1;
                  pstr = strchr(str, '\"');

                  retval += safe_chr('\"', buff, bp);
                  if (str == pstr) {
                    str++;
                    pstr = strchr(str, '\"');
                  } else {
                    pstr = strchr(str, ' ');
                  }

                  if (!pstr)
                    pstr = tmp;

                  *pstr = '\0';
                  retval += safe_str(str, buff, bp);
                  retval += safe_chr('\"', buff, bp);
                  str = pstr;
                } else {
                  safe_str(str, buff, bp);
                  break;
                }
              }
            }
            retval += safe_str(")]", buff, bp);

          } else {
            if (pueblotop > -1) {
              i = (*(str + 1) == 'a') ? 0 : pueblotop;
              for (i--; pueblotop > i; pueblotop--) {
                retval += safe_str("[endtag(", buff, bp);
                retval += safe_str(pueblostack[pueblotop], buff, bp);
                retval += safe_str(")]", buff, bp);
                mush_free(pueblostack[pueblotop], "markup_code");
              }
            }
          }
          break;
        }
        tmp++;
        str = tmp;
        break;
      case ESC_CHAR:
        /* It SHOULD be impossible to get here... */
        for (tmp = str; *tmp && *tmp != 'm'; tmp++) ;

        /* Store the "background" colors */
        tmpansi = ansistack[ansitop];
        if (oldcodes == 0) {
          oldcodes = 1;
          ansitop++;
          ansistack[ansitop] = tmpansi;
          ansistack[ansitop].offbits = 0;
        }

        read_raw_ansi_data(&tmpansi, str);
        ansistack[ansitop].bits |= tmpansi.bits;
        ansistack[ansitop].bits &= ~(tmpansi.offbits);  /* ANSI_RAW_NORMAL */
        if (tmpansi.fore)
          ansistack[ansitop].fore = tmpansi.fore;
        if (tmpansi.back)
          ansistack[ansitop].back = tmpansi.back;

        str = tmp;
        if (*tmp)
          str++;
        break;
      }
    }

    /* Handle ANSI/Text */
    tmpansi = ansistack[ansitop];
    if (ansitop > 0 || ansiheight > 0) {
      /* Close existing tags as necessary to cleanly open the next. */
      /*  oldansi = ansistack[ansiheight]; */
      if (!ansi_equal(oldansi, tmpansi)) {
        while (ansiheight > 0) {
          if (howmanyopen > 0) {
            howmanyopen--;
            retval += safe_str(")]", buff, bp);
          }
          ansiheight--;
        }
      }
      if (!ansi_isnull(tmpansi) && !ansi_equal(oldansi, tmpansi)) {
        retval += safe_str("[ansi(", buff, bp);
        retval += write_ansi_letters(&tmpansi, buff, bp);
        retval += safe_chr(',', buff, bp);
        howmanyopen++;
      }
    }
    retval += escape_marked_str(&str, buff, bp);
  }

  for (; howmanyopen > 0; howmanyopen--)
    retval += safe_str(")]", buff, bp);
  for (; pueblotop > -1; pueblotop--) {
    retval += safe_str("[endtag(", buff, bp);
    retval += safe_str(pueblostack[pueblotop], buff, bp);
    retval += safe_str(")]", buff, bp);
  }

  return retval;
}

/** Our version of pcre_copy_substring, with ansi-safeness.
 * \param as the ansi_string whose .text value was matched against.
 * \param ovector the offset vectors
 * \param stringcount the number of subpatterns
 * \param stringnumber the number of the desired subpattern
 * \param buff buffer to copy the subpattern to
 * \param bp pointer to the end of buffer
 * \return size of subpattern, or -1 if unknown pattern
 */
int
ansi_pcre_copy_substring(ansi_string *as, int *ovector,
                         int stringcount, int stringnumber,
                         int nonempty, char *buff, char **bp)
{
  int yield;
  if (stringnumber < 0 || stringnumber >= stringcount)
    return -1;
  stringnumber *= 2;
  yield = ovector[stringnumber + 1] - ovector[stringnumber];
  if (!nonempty || yield) {
    safe_ansi_string(as, ovector[stringnumber], yield, buff, bp);
    **bp = '\0';
  }
  return yield;
}


/** Our version of pcre_copy_named_substring, with ansi-safeness.
 * \param code the pcre compiled code
 * \param as the ansi_string whose .text value was matched against.
 * \param ovector the offset vectors
 * \param stringcount the number of subpatterns
 * \param stringname the name of the desired subpattern
 * \param buff buffer to copy the subpattern to
 * \param bp pointer to the end of buffer
 * \return size of subpattern, or -1 if unknown pattern
 */
int
ansi_pcre_copy_named_substring(const pcre * code, ansi_string *as,
                               int *ovector, int stringcount,
                               const char *stringname, int ne,
                               char *buff, char **bp)
{
  int n = pcre_get_stringnumber(code, stringname);
  if (n <= 0)
    return -1;
  return ansi_pcre_copy_substring(as, ovector, stringcount, n, ne, buff, bp);
}

/** Safely add a tag into a buffer.
 * If we support pueblo, this function adds the tag start token,
 * the tag, and the tag end token. If not, it does nothing.
 * If we can't fit the tag in, we don't put any of it in.
 * \param a_tag the html tag to add.
 * \param buf the buffer to append to.
 * \param bp pointer to address in buf to insert.
 * \retval 0, successfully added.
 * \retval 1, tag wouldn't fit in buffer.
 */
static int
safe_markup(char const *a_tag, char *buf, char **bp, char type)
{
  int result = 0;
  char *save = buf;
  result = safe_chr(TAG_START, buf, bp);
  result = safe_chr(type, buf, bp);
  result = safe_str(a_tag, buf, bp);
  result = safe_chr(TAG_END, buf, bp);
  /* If it didn't all fit, rewind. */
  if (result)
    *bp = save;
  return result;
}

int
safe_tag(char const *a_tag, char *buff, char **bp)
{
  if (SUPPORT_PUEBLO)
    return safe_markup(a_tag, buff, bp, MARKUP_HTML);
  return 0;
}

/** Safely add a closing tag into a buffer.
 * If we support pueblo, this function adds the tag start token,
 * a slash, the tag, and the tag end token. If not, it does nothing.
 * If we can't fit the tag in, we don't put any of it in.
 * \param a_tag the html tag to add.
 * \param buf the buffer to append to.
 * \param bp pointer to address in buf to insert.
 * \retval 0, successfully added.
 * \retval 1, tag wouldn't fit in buffer.
 */
static int
safe_markup_cancel(char const *a_tag, char *buf, char **bp, char type)
{
  int result = 0;
  char *save = buf;
  result = safe_chr(TAG_START, buf, bp);
  result = safe_chr(type, buf, bp);
  result = safe_chr('/', buf, bp);
  result = safe_str(a_tag, buf, bp);
  result = safe_chr(TAG_END, buf, bp);
  /* If it didn't all fit, rewind. */
  if (result)
    *bp = save;
  return result;
}

int
safe_tag_cancel(char const *a_tag, char *buf, char **bp)
{
  if (SUPPORT_PUEBLO)
    return safe_markup_cancel(a_tag, buf, bp, MARKUP_HTML);
  return 0;
}

/** Safely add a tag, some text, and a matching closing tag into a buffer.
 * If we can't fit the stuff, we don't put any of it in.
 * \param a_tag the html tag to add.
 * \param params tag parameters.
 * \param data the text to wrap the tag around.
 * \param buf the buffer to append to.
 * \param bp pointer to address in buf to insert.
 * \param player the player involved in all this, or NOTHING if internal.
 * \retval 0, successfully added.
 * \retval 1, tagged text wouldn't fit in buffer.
 */
int
safe_tag_wrap(char const *a_tag, char const *params,
              char const *data, char *buf, char **bp, dbref player)
{
  int result = 0;
  char *save = buf;
  if (SUPPORT_PUEBLO) {
    result = safe_chr(TAG_START, buf, bp);
    result = safe_chr(MARKUP_HTML, buf, bp);
    result = safe_str(a_tag, buf, bp);
    if (params && *params && ok_tag_attribute(player, params)) {
      result = safe_chr(' ', buf, bp);
      result = safe_str(params, buf, bp);
    }
    result = safe_chr(TAG_END, buf, bp);
  }
  result = safe_str(data, buf, bp);
  if (SUPPORT_PUEBLO) {
    result = safe_tag_cancel(a_tag, buf, bp);
  }
  /* If it didn't all fit, rewind. */
  if (result)
    *bp = save;
  return result;
}
