/*
 * Copyright Staffan Gimåker 2006-2009.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */


#include <cctype>
#include <cassert>
#include <fstream>
#include <iostream>
#include <string>
#include <vector>
#include <list>
#include <map>
#include <set>
#include <stdexcept>
#include <boost/tuple/tuple.hpp>
#include <boost/tuple/tuple_comparison.hpp>
#include <boost/function.hpp>
#include <boost/bind.hpp>
#include <boost/progress.hpp>
#include <boost/lexical_cast.hpp>
#include <Eigen/Core>
#include <Eigen/Geometry>

#include "../../Types.hh"
#include "../../GeometryToolbox.hh"


using namespace peekabot;



class ObjToPbmfConverter
{
    struct Material
    {
        Material()
            : m_ambient_reflectance(0.2, 0.2, 0.2),
              m_diffuse_reflectance(0.8, 0.8, 0.8),
              m_specular_reflectance(1.0, 1.0, 1.0),
              m_alpha(1.0),
              m_shininess(0.0)
        {
        }

        RGBColor m_ambient_reflectance;
        RGBColor m_diffuse_reflectance;
        RGBColor m_specular_reflectance;
        float m_alpha;
        float m_shininess;
        std::string m_texture;
    };

    typedef boost::tuple<int, int, int, const Material *> IndexTuple;
    typedef std::map<IndexTuple, uint32_t> IndexTranslationMap;

    IndexTranslationMap m_index_translation_map;

    typedef std::map<std::string, Material> MaterialMap;

    MaterialMap m_materials;

    int m_vertex_count;
    int m_tri_count;

    int m_src_vertex_count;
    int m_src_normal_count;
    int m_src_uv_coord_count;
    
    std::vector<float> m_src_vertices;
    std::vector<float> m_src_normals;
    std::vector<float> m_src_colors;
    std::vector<float> m_src_uv_coords;
    std::vector<IndexTuple> m_src_indices;

    std::vector<float> m_vertices;
    std::vector<float> m_normals;
    std::vector<float> m_colors;
    std::vector<float> m_uv_coords;
    std::vector<uint32_t>  m_indices;

    typedef std::vector<std::string> ArgList;

    const Material m_default_material;
    const Material *m_active_material;

    // vertex (v) -> normal
    typedef std::map<int, Eigen::Vector3f> AutoNormalsMap;

    AutoNormalsMap m_auto_normals;
    bool m_override_normals;
    bool m_no_colors;

    typedef std::set<std::string> UndefinedMaterials;
    UndefinedMaterials m_undefined_materials;

    typedef std::list<std::string> Log;
    Log m_log;

    std::string m_texture;

    Eigen::Vector3f m_offset;
    
private:
    void split_at(std::string &str, std::string &head, char delimiter)
    {
        std::string::iterator delim = find(str.begin(), str.end(), delimiter);
        head = std::string(str.begin(), delim);
        str.replace(str.begin(), delim+1, "");
    }

    void collapse_spaces(std::string &str)
    {
        // Replace all "space characters" with plain blankspaces
        for( std::string::iterator it = str.begin(); it != str.end(); it++ )
            if( isspace(*it) )
                *it = ' ';

        if( str.size() < 2 )
            return;

        // Collapse sequences of spaces to a single space
        std::string::iterator prev_it = str.begin();
        for( std::string::iterator it = prev_it+1; it != str.end(); it++ )
        {
            if( isspace(*prev_it) && isspace(*it) )
            {
                str.erase(it);
                it = prev_it;
            }
            else
                prev_it = it;
        }
    }


    Eigen::Vector3f &get_auto_normal(int i)
    {
        AutoNormalsMap::iterator it = m_auto_normals.find(i);

        if( it == m_auto_normals.end() )
            it = m_auto_normals.insert(
                std::make_pair(i, Eigen::Vector3f(0,0,0))).first;

        return it->second;
    }

    
    void exec_command(const std::string &command, 
                      ArgList &args)
    {
        if( command == "v" )
            add_vertex(args);
        else if( command == "vn" )
            add_vertex_normal(args);
        else if( command == "f" )
            add_face(args);
        else if( command == "vt" )
            add_tex_coord(args);
        else if( command == "mtllib" )
        {
            for( ArgList::iterator it = args.begin();
                 it != args.end(); it++ )
                parse_mtl(*it);
        }
        else if( command == "usemtl" )
            use_mtl(args);
        else
        {
            // TODO
        }
    }
    
    
    void add_vertex(const ArgList &args)
    {
        if( args.size() != 3 && args.size() != 4 )
            throw std::runtime_error(
                "Malformed vertex definition");
             
        int i = 0;
        for( ArgList::const_iterator it = args.begin();
             it != args.end() && i < 3; it++, i++ )
        {
                float val = atof((*it).c_str());
                m_src_vertices.push_back(val);
        }
        
        m_src_vertex_count++;
    }
    

    void add_vertex_normal(const ArgList &args)
    {
        if( args.size() != 3 )
            throw std::runtime_error(
                "Malformed vertex normal definition");
        
        for( ArgList::const_iterator it = args.begin();
             it != args.end(); it++ )
        {
            float val = atof((*it).c_str());
                m_src_normals.push_back(val);
        }

        m_src_normal_count++;
    }
    
    
    void add_face(ArgList &args)
    {
        // Overview:
        // 0. vertices < 3 => error
        // 1. triangle? add it
        // 2. !triangle? triangulate it:
        //    1. [0,2] \in V is a triangle
        //    2. V\1 is a polygon of less order than V, recurse!

        
        if( args.size() == 3 )
        {
            add_triangle(args);
        }
        else if( args.size() > 3 )
        {
            ArgList::iterator v0, v1, v2, v3;
            v3 = args.begin();
            v0 = v3++; // hax
            v1 = v3++;
            v2 = v3++;
            
            add_triangle(ArgList(v0, v3));
            args.erase(v1);
                add_face(args);
        }
        else
            m_log.push_back(
                "Warning: Face ignored - faces must consist of 3+ vertices.");
            /*throw std::runtime_error(
                "Faces must consist of 3+ vertices");*/
    }
    
    
    // pre: args.size() == 3)
    void add_triangle(const ArgList &args)
    {
        std::vector<int> v;
        std::vector<int> vn;
        std::vector<int> vt;

        // Convert strings to indices...
        for( ArgList::const_iterator it = args.begin();
             it != args.end(); it++ )
        {
            std::string data = *it, v_str, vt_str, vn_str;
            
            split_at(data, v_str, '/');
            split_at(data, vt_str, '/');
            split_at(data, vn_str, '/');

            int i = atoi(v_str.c_str())-1;
            int j = atoi(vn_str.c_str())-1;
            int k = atoi(vt_str.c_str())-1;
            
            // TEMP: force auto normals
            //j = -1;

            if( i < 0 || j < -1 || k < -1 )
            {
                throw std::runtime_error(
                    "Missing face data (invalid) or relative face data references (unsupported) encountered");
            }

            v.push_back(i);
            vn.push_back(j);
            vt.push_back(k);
        }


        // Calculate face normal, used to auto generate missing vertex normals
        Eigen::Vector3f p1(m_src_vertices[3*v[0]+0], 
                    m_src_vertices[3*v[0]+1], 
                    m_src_vertices[3*v[0]+2]);
        Eigen::Vector3f p2(m_src_vertices[3*v[1]+0], 
                    m_src_vertices[3*v[1]+1], 
                    m_src_vertices[3*v[1]+2]);
        Eigen::Vector3f p3(m_src_vertices[3*v[2]+0], 
                    m_src_vertices[3*v[2]+1], 
                    m_src_vertices[3*v[2]+2]);
        Eigen::Vector3f face_normal = (p1-p3).cross(p2-p3);
        face_normal.normalize();


        for( int i = 0; i < 3; i++ )
        {
            get_auto_normal(v[i]) += face_normal;

            IndexTuple idx(v[i], vn[i], vt[i], m_active_material);
            m_src_indices.push_back(idx);
        }

        m_tri_count++;
    }

        
    void add_tex_coord(const ArgList &args)
    {
        if( args.size() != 2 && args.size() != 3 )
        {
            throw std::runtime_error(
                "Malformed UV coordinate");
        }

        ArgList::const_iterator it = args.begin();

        float u = atof(args[0].c_str());
        float v = atof(args[1].c_str());

        m_src_uv_coords.push_back(u);
        m_src_uv_coords.push_back(1-v);

        m_src_uv_coord_count++;
    }


    void use_mtl(const ArgList &args)
    {
        if( args.size() != 1 )
            throw std::runtime_error(
                "Invalid usemtl command, number of arguments must be one");

        MaterialMap::const_iterator it = m_materials.find(args[0]);

        if( it == m_materials.end() )
        {
            if( m_undefined_materials.insert(args[0]).second )
                m_log.push_back(
                    "Warning: Material '" + args[0] + "' has not been defined");
            
            m_active_material = &m_default_material;
        }
        else
            m_active_material = &(it->second);
    }


    void parse_mtl(const std::string &filename)
    {
        std::ifstream ifs(filename.c_str());
        std::string buf, cmd;
        ArgList args;

        if( !ifs.is_open() )
        {
            m_log.push_back(
                "Warning: Material file '" + filename + "' not found");

            return;
        }

        while( !ifs.eof() )
        {
            getline(ifs, buf, '\n');

            // Get rid of extraneous space/tabs/etc.
            collapse_spaces(buf);
            // Extract command
            split_at(buf, cmd, ' ');

            while( !buf.empty() )
            {
                std::string arg;
                split_at(buf, arg, ' ');
                args.push_back(arg);
            }

            exec_mtl_command(cmd, args);
            args.clear();
        }

        ifs.close();
    }


    void exec_mtl_command(const std::string &command, ArgList &args)
    {
        static MaterialMap::iterator current_it = m_materials.end();
        
        if( command == "#" || command.size() == 0 )
            return;
        if( command == "newmtl" )
        {
            if( args.size() != 1 )
            {
                std::string error = "Invalid material specification, ";
                error += "newmtl takes 1 argument, ";
                error += args.size();
                error += " given";

                throw std::runtime_error(error);
            }

            current_it = m_materials.insert(
                std::make_pair(args[0], Material())).first;
        }
        else
        {
            if( current_it == m_materials.end() )
                throw std::runtime_error(
                    "Invalid command '" + command +
                    "', must be preceded by a newmtl command");

            if( command == "Ka" )
                set_mtl_color(current_it->second.m_ambient_reflectance, args);
            else if( command == "Kd" )
                set_mtl_color(current_it->second.m_diffuse_reflectance, args);
            else if( command == "Ks" )
                set_mtl_color(current_it->second.m_specular_reflectance, args);
            else if( command == "d" || command == "Tr" )
            {
                // TODO
            }
            else if( command == "Ns" )
            {
                // TODO
            }
            else if( command == "map_Ka" )
            {
                // TODO
            }
            else if( command == "illum" )
            {
                // TODO, if illum = 1 we should set specular to (0,0,0), otherwise ignore it
            }
            /*else
                throw std::runtime_error(
                "Unsupported mtl command (" + command + ") encountered");*/
        }
    }


    void set_mtl_color(RGBColor &color, const ArgList &args)
    {
        if( args.size() != 3 )
            throw std::runtime_error(
                "Invalid color specification, erroneous number of arguments");

        // Clamp color values to [0, 1]
        color.r = std::min(1.0, std::max(0.0, atof( args[0].c_str() )));
        color.g = std::min(1.0, std::max(0.0, atof( args[1].c_str() )));
        color.b = std::min(1.0, std::max(0.0, atof( args[2].c_str() )));
    }


    void convert_source_data()
    {
        boost::progress_display pd( 3*m_tri_count );

        for( int i = 0; i < 3*m_tri_count; i++ )
        {
            IndexTuple &triple = m_src_indices[i];

            int i = triple.get<0>();
            int j = triple.get<1>();
            int k = triple.get<2>();
            const Material *mtl = triple.get<3>();

            assert( mtl != 0 );

            if( i >= m_src_vertex_count ||
                j >= m_src_normal_count ||
                k >= m_src_uv_coord_count )
            {
                throw std::runtime_error(
                    "Corrupted or missing data - unavailable data referenced");
            }

            IndexTranslationMap::iterator it = 
                m_index_translation_map.find(triple);

            if( it == m_index_translation_map.end() )
            {
                // Vertex not added yet.. add it
                it = m_index_translation_map.insert(
                    std::make_pair(triple, m_vertex_count)).first;
                m_vertex_count++;

                m_vertices.push_back(m_src_vertices[3*i+0] + m_offset(0));
                m_vertices.push_back(m_src_vertices[3*i+1] + m_offset(1));
                m_vertices.push_back(m_src_vertices[3*i+2] + m_offset(2));

                if( j == -1 || m_override_normals )
                {
                    // Missing normal...
                    Eigen::Vector3f &n = get_auto_normal(i);
                    n.normalize();
                    m_normals.push_back(n(0));
                    m_normals.push_back(n(1));
                    m_normals.push_back(n(2));
                }
                else
                {
                    m_normals.push_back(m_src_normals[3*j+0]);
                    m_normals.push_back(m_src_normals[3*j+1]);
                    m_normals.push_back(m_src_normals[3*j+2]);
                }

                m_colors.push_back(mtl->m_diffuse_reflectance.r);
                m_colors.push_back(mtl->m_diffuse_reflectance.g);
                m_colors.push_back(mtl->m_diffuse_reflectance.b);

                if( k >= 0 )
                {
                    m_uv_coords.push_back(m_src_uv_coords[2*k+0]);
                    m_uv_coords.push_back(m_src_uv_coords[2*k+1]);
                }
            }

            uint32_t idx = it->second;
            
            m_indices.push_back(idx);

            pd += 1;
        }
    }


    void calculate_bounding_sphere(Eigen::Vector3f &pos, float &r)
    {
        int n = m_src_vertex_count;
        std::vector<Eigen::Vector3f> tmp;

        for( int i = 0; i < n; i++ )
        {
            Eigen::Vector3f p(m_src_vertices[3*i+0],
                       m_src_vertices[3*i+1],
                       m_src_vertices[3*i+2]);
            tmp.push_back(p);
        }

        geom::calc_optimal_bounding_sphere(
            tmp.begin(), tmp.end(), pos, r,
            1, 1, 1);

        pos += m_offset;
    }


    void write_pbfm_model(std::ofstream &ofs)
    {
        //SerializationInterface buf(*ofs.rdbuf());

        // Write header
        char uid[] = "pbmf";
        ofs.write((char *)uid, 4);

        uint8_t zero = 0;
        uint8_t one = 1;
        ofs.write((char *)&zero, 1); // Byte order
        ofs.write((char *)&zero, 1); // Version

        ofs.write((char *)&m_vertex_count, 4);
        ofs.write((char *)&m_tri_count, 4);

        // Has uv coords?
        if( m_uv_coords.empty() )
            ofs.write((char *)&zero, 1);
        else
            ofs.write((char *)&one, 1);
        
        // Has colors?
        if( m_no_colors )
            ofs.write((char *)&zero, 1);
        else
            ofs.write((char *)&one, 1);

        float r; 
        Eigen::Vector3f pos;
        calculate_bounding_sphere(pos, r);

        ofs.write((char *)&r, 4); // bsphere info
        ofs.write((char *)&pos(0), 4);
        ofs.write((char *)&pos(1), 4);
        ofs.write((char *)&pos(2), 4);

        float dummy;
        ofs.write((char *)&zero, 1); // Disable culling?
        ofs.write((char *)&zero, 1); // Always draw back to front?
        
        ofs.write((char *)&dummy, 4);
        ofs.write((char *)&dummy, 4);
        ofs.write((char *)&dummy, 4);
        //
        ofs.write((char *)&dummy, 4);
        ofs.write((char *)&dummy, 4);
        ofs.write((char *)&dummy, 4);
        //
        ofs.write((char *)&dummy, 4);
        ofs.write((char *)&dummy, 4);
        ofs.write((char *)&dummy, 4);
        //
        ofs.write((char *)&dummy, 4);
        ofs.write((char *)&dummy, 4);
        ofs.write((char *)&dummy, 4);

        ofs.write((char *)&dummy, 4);
        
        // strlen...
        //buf << m_texture;
        uint32_t _strlen = m_texture.length();
        ofs.write((char *)&_strlen, 4);
        ofs.write(m_texture.c_str(), _strlen);

        ofs.write((char *)&m_vertices[0], 4*3*m_vertex_count);
        ofs.write((char *)&m_normals[0], 4*3*m_vertex_count);

        if( !m_uv_coords.empty() )
            ofs.write((char *)&m_uv_coords[0], 4*2*m_vertex_count);

        if( !m_no_colors )
            ofs.write((char *)&m_colors[0], 4*3*m_vertex_count);

        // TODO: uv-coords, colors
        ofs.write((char *)&m_indices[0], 4*3*m_tri_count);
    }



public:
    
    ObjToPbmfConverter()
        : m_default_material()
    {
    }

    void convert(const std::string &input_filename,
                 const std::string &output_filename,
                 bool override_normals,
                 bool no_colors,
                 const std::string &texture_name,
                 const Eigen::Vector3f &offset)
        throw(std::runtime_error)
    {
        std::ifstream ifs(input_filename.c_str());
        std::ofstream ofs(output_filename.c_str(), std::ios::binary);

        if( !ifs.is_open() )
            throw std::runtime_error(
                "File '" + input_filename + "' could not be opened for reading");

        if( !ofs.is_open() )
            throw std::runtime_error(
                "File '" + output_filename + "' could not be opened for writing");


        m_override_normals = override_normals;
        m_no_colors = no_colors;
        m_texture = texture_name;
        m_offset = offset;

        m_tri_count = 0;
        m_vertex_count = 0;
        m_src_vertex_count = 0;
        m_src_normal_count = 0;
        m_src_uv_coord_count = 0;

        m_indices.clear();
        m_normals.clear();
        m_uv_coords.clear();
        m_colors.clear();

        m_src_vertices.clear();
        m_src_normals.clear();
        m_src_uv_coords.clear();

        m_materials.clear();
        m_active_material = &m_default_material;
        m_undefined_materials.clear();

        m_log.clear();
        

        std::string buf, cmd;
        ArgList args;

        
        std::cout << "Parsing input file..." << std::endl;

        ifs.seekg(0, std::ios::end);
        boost::progress_display pd( ifs.tellg() );
        ifs.seekg(0, std::ios::beg);

        while( !ifs.eof() )
        {
            std::streamoff prev_pos = ifs.tellg();

            getline(ifs, buf, '\n');

            // Get rid of extraneous space/tabs/etc.
            collapse_spaces(buf);



            split_at(buf, cmd, ' ');

            args.clear();
            while( !buf.empty() )
            {
                std::string arg;
                split_at(buf, arg, ' ');
                args.push_back(arg);
            }

            exec_command(cmd, args);

            if( !ifs.eof() )
                pd += (ifs.tellg() - prev_pos);
        }

        ifs.close();

        std::cout << "\nConverting input..." << std::endl;
        
        // Convert the source data into something usable...
        convert_source_data();

        std::cout << "\nWriting pbmf model..." << std::endl;

        write_pbfm_model(ofs);        
        ofs.close();

        std::cout << std::endl;
        for( Log::iterator it = m_log.begin(); it != m_log.end(); ++it )
        {
            std::cout << *it << std::endl;
        }

        std::cout << "\nSuccessfully converted model with " << m_vertex_count 
                  << " vertices, " << m_tri_count << " triangles and " 
                  << m_materials.size() << " materials" << std::endl;

        m_log.clear();
    }
};


int main(int argc, char **argv)
{
    typedef std::vector<std::string> ArgList;
    ArgList args;


    for( int i = 1; i < argc; i++ )
        args.push_back(argv[i]);


    bool override_normals = false;
    bool no_colors = false;
    std::string texture_name = "";
    float x_offset = 0, y_offset = 0, z_offset = 0;
    
    // Check options...

    for( ArgList::iterator it = args.begin(); it != args.end(); )
    {
        if( *it == "--override-normals" )
        {
            override_normals = true;
            it = args.erase(it);
        }
        else if( *it == "--texture" )
        {
            it = args.erase(it);
            if( it != args.end() )
            {
                texture_name = *it;
                it = args.erase(it);
            }
        }
        else if( *it == "--no-colors" )
        {
            no_colors = true;
            it = args.erase(it);
        }
        else if( *it == "--x-offset" )
        {
            it = args.erase(it);

            if( it != args.end() )
            {
                x_offset = boost::lexical_cast<float>(*it);
                it = args.erase(it);
            }
        }
        else if( *it == "--y-offset" )
        {
            it = args.erase(it);

            if( it != args.end() )
            {
                y_offset = boost::lexical_cast<float>(*it);
                it = args.erase(it);
            }
        }
        else if( *it == "--z-offset" )
        {
            it = args.erase(it);

            if( it != args.end() )
            {
                z_offset = boost::lexical_cast<float>(*it);
                it = args.erase(it);
            }
        }
        else
        {
            ++it;
        }
    }


    if( args.size() < 2 )
    {
        std::cout << "Usage: " << argv[0] 
                  << " [--override-normals] [--texture <filename>]"
                  << " [--no-colors] [--x-offset <float>] [--y-offset <float>]"
                  << " [--z-offset <float>] <input.obj> <output.pbmf>" 
                  << std::endl;
        return 0;
    }

    try
    {
        ObjToPbmfConverter converter;
        converter.convert(
            args[0], args[1], override_normals, 
            no_colors, texture_name, 
            Eigen::Vector3f(x_offset, y_offset, z_offset));
    }
    catch(std::exception &e)
    {
        std::cout << "Error: " << e.what() << std::endl;

        return -1;
    }

    return 0;
}
