/*
 * Copyright Staffan Gimåker 2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "TriMeshEntity.hh"
#include "../Mesh.hh"
#include "../PrepareRenderContext.hh"
#include "../../GeometryToolbox.hh"

#include <GL/glew.h>
#include <Eigen/Geometry>


using namespace peekabot;
using namespace peekabot::renderer;


TriMeshEntity::TriMeshEntity()
    : m_double_sided(true),
      m_ff_mesh(new TriMesh),
      m_bf_mesh(new TriMesh),
      m_max_index(-1)
{
    // Cull front faces on the back-face mesh
    m_bf_mesh->get_state().set(new statelets::CullFrontFaces(true));
}


TriMeshEntity::TriMeshEntity(const TriMeshEntity &other)
    : m_double_sided(other.m_double_sided),
      m_ff_mesh(other.m_ff_mesh->clone()),
      m_bf_mesh(other.m_bf_mesh->clone()),
      m_max_index(other.m_max_index)
{
}


TriMeshEntity::~TriMeshEntity()
{
}


TriMeshEntity *TriMeshEntity::clone() const
{
    return new TriMeshEntity(*this);
}


void TriMeshEntity::get_renderables(PrepareRenderContext &context) const
{
    if( m_max_index >= (boost::int64_t)vertex_count() )
    {
        // One or more index is out-of-bounds => it's not safe to render
        // the current indices
        return;
    }

    if( m_ff_mesh->has_indices() )
    {
        context.prepare(m_ff_mesh.get());

        if( m_double_sided )
            context.prepare(m_bf_mesh.get());
    }
}


void TriMeshEntity::set_indices(const Indices &indices)
{
    boost::shared_ptr<IndexBuffer> buf(new IndexBuffer(true));
    buf->set_usage_hint(IndexBuffer::DYNAMIC_DRAW);
    buf->set_data(&indices[0], sizeof(Indices::value_type)*indices.size());
    m_ff_mesh->set_indices(buf);
    m_bf_mesh->set_indices(buf);
    m_ff_mesh->set_element_count(indices.size()/3);
    m_bf_mesh->set_element_count(indices.size()/3);

    // Update m_max_index
    m_max_index = -1;
    for( std::size_t i = 0; i < indices.size(); ++i )
    {
        if( indices[i] > m_max_index )
            m_max_index = indices[i];
    }

    if( !indices.empty() && m_ff_mesh->has_vertices() )
    {
        assert( m_ff_mesh->has_normals() );
        assert( m_bf_mesh->has_normals() );

        VertexBuffer::ReadWriteMapping<Eigen::Vector3f> ff_normals(
            *m_ff_mesh->get_normals());
        VertexBuffer::ReadWriteMapping<Eigen::Vector3f> bf_normals(
            *m_bf_mesh->get_normals());
        VertexBuffer::ReadOnlyMapping<Eigen::Vector3f> vertices(
            *m_ff_mesh->get_vertices());;

        update_normals(
            &indices[0], indices.size(),
            vertices.get(), vertex_count(),
            ff_normals.get(), bf_normals.get());
    }
}


void TriMeshEntity::add_indices(const Indices &new_indices)
{
    if( new_indices.empty() )
        return;

    if( !m_ff_mesh->has_indices() )
    {
        set_indices(new_indices);
        return;
    }

    // Update the index buffer
    boost::shared_ptr<IndexBuffer> ib = m_ff_mesh->get_indices();
    std::size_t old_size = ib->size();
    std::size_t n = sizeof(Indices::value_type)*new_indices.size();
    ib->grow_back(n);
    ib->set_sub_data(&new_indices[0], n, old_size);
    m_ff_mesh->set_element_count(tri_count());
    m_bf_mesh->set_element_count(tri_count());

    // Update normals
    if( m_ff_mesh->has_vertices() )
    {
        VertexBuffer::ReadWriteMapping<Eigen::Vector3f> ff_normals(
            *m_ff_mesh->get_normals());
        VertexBuffer::ReadWriteMapping<Eigen::Vector3f> bf_normals(
            *m_bf_mesh->get_normals());
        VertexBuffer::ReadOnlyMapping<Eigen::Vector3f> vertices(
            *m_ff_mesh->get_vertices());

        update_normals(
            &new_indices[0], new_indices.size(),
            vertices.get(), vertex_count(),
            ff_normals.get(), bf_normals.get());
    }
}


void TriMeshEntity::get_indices(Indices &indices) const
{
    indices.clear();

    if( m_ff_mesh->has_indices() )
    {
        IndexBuffer::ReadOnlyMapping<boost::uint32_t> inds(
            *m_ff_mesh->get_indices());

        for( std::size_t i = 0; i < 3*tri_count(); ++i )
            indices.push_back(inds[i]);
    }
}


void TriMeshEntity::toggle_double_sided(bool double_sided)
{
    // TODO: Setting single-sided drawing should somehow disable generating
    // normals for the back-face mesh
    m_double_sided = double_sided;
}


void TriMeshEntity::set_vertices_helper(
    const Vertices &vertices)
{
    boost::shared_ptr<VertexBuffer> vb(new VertexBuffer);
    vb->set_usage_hint(VertexBuffer::DYNAMIC_DRAW);
    vb->set_data(&vertices[0], 3*sizeof(float)*vertices.size());
    m_ff_mesh->set_vertices(vb, 0, 0);
    m_bf_mesh->set_vertices(vb, 0, 0);

    // NOTE: Given that we map ff_nb, bf_nb and the vertex buffer
    // simultaneously in the current implementation must put the normal buffers
    // in main memory
    // TODO: change it so we can have the normals in video
    // memory too if possible!
    boost::shared_ptr<VertexBuffer> ff_nb(new VertexBuffer(true));
    boost::shared_ptr<VertexBuffer> bf_nb(new VertexBuffer(true));
    ff_nb->set_usage_hint(VertexBuffer::DYNAMIC_DRAW);
    bf_nb->set_usage_hint(VertexBuffer::DYNAMIC_DRAW);
    ff_nb->resize(3*sizeof(float)*vertices.size());
    bf_nb->resize(3*sizeof(float)*vertices.size());
    m_ff_mesh->set_normals(ff_nb, 0, 0);
    m_bf_mesh->set_normals(bf_nb, 0, 0);

    VertexBuffer::ReadWriteMapping<Eigen::Vector3f> ff_normals(*ff_nb);
    VertexBuffer::ReadWriteMapping<Eigen::Vector3f> bf_normals(*bf_nb);
    // Zero the normals
    for( std::size_t i = 0; i < vertices.size(); ++i )
        ff_normals[i] = bf_normals[i] = Eigen::Vector3f(0, 0, 0);

    // Calculate normals
    if( m_ff_mesh->has_indices() && !vertices.empty() )
    {
        boost::shared_ptr<IndexBuffer> ib = m_ff_mesh->get_indices();
        IndexBuffer::ReadOnlyMapping<boost::uint32_t> indices(*ib);

        update_normals(
            indices.get(), 3*tri_count(),
            &vertices[0], vertices.size(),
            ff_normals.get(), bf_normals.get());
    }

    // Calculate bounding sphere
    Eigen::Vector3f midpoint;
    float r;
    geom::calc_approx_bounding_sphere(
        vertices, midpoint, r,
        1, 1, 1);

    set_bounding_sphere(BoundingSphere(midpoint, r));
}


void TriMeshEntity::add_vertices_helper(
    const Vertices &new_vertices, const Vertices &all_vertices)
{
    // Update the vertex buffer
    boost::shared_ptr<VertexBuffer> vb = m_ff_mesh->get_vertices();
    std::size_t old_size = vb->size();
    std::size_t n = 3*sizeof(float)*new_vertices.size();
    vb->grow_back(n);
    vb->set_sub_data(&new_vertices[0], n, old_size);

    // Update the normal buffers
    boost::shared_ptr<VertexBuffer> ff_nb = m_ff_mesh->get_normals();
    boost::shared_ptr<VertexBuffer> bf_nb = m_bf_mesh->get_normals();
    ff_nb->grow_back(n);
    bf_nb->grow_back(n);

    VertexBuffer::WriteOnlyMapping<Eigen::Vector3f> ff_normals(*ff_nb);
    VertexBuffer::WriteOnlyMapping<Eigen::Vector3f> bf_normals(*bf_nb);
    // Zero the new normals
    for( std::size_t i = vertex_count()-new_vertices.size();
         i < vertex_count(); ++i )
        ff_normals[i] = bf_normals[i] = Eigen::Vector3f(0, 0, 0);

    // Update the bounding sphere
    Eigen::Vector3f midp = get_bounding_sphere_lc().get_pos();
    float r = get_bounding_sphere_lc().get_radius();
    if( geom::extend_bounding_sphere(new_vertices, midp, r, 1.5) )
    {
        VertexBuffer::ReadOnlyMapping<Eigen::Vector3f> verts(*vb);
        geom::calc_approx_bounding_sphere(
            verts.get(), verts.get()+vertex_count(),
            midp, r,
            1, 1, 1);
    }

    set_bounding_sphere(BoundingSphere(midp, r));
}


void TriMeshEntity::set_vertices(
    const Vertices &vertices)
{
    // Clear color pointers
    m_ff_mesh->set_colors(boost::shared_ptr<VertexBuffer>());
    m_bf_mesh->set_colors(boost::shared_ptr<VertexBuffer>());

    set_vertices_helper(vertices);
}


void TriMeshEntity::add_vertices(
    const Vertices &new_vertices,
    const Vertices &all_vertices)
{
    assert( !m_ff_mesh->get_colors() );
    assert( !m_bf_mesh->get_colors() );

    if( new_vertices.empty() )
        return;

    if( !m_ff_mesh->has_vertices() )
    {
        assert( all_vertices.size() == new_vertices.size() );
        set_vertices(new_vertices);
        return;
    }

    add_vertices_helper(new_vertices, all_vertices);
}


void TriMeshEntity::get_vertices(Vertices &vertices) const
{
    vertices.clear();

    if( m_ff_mesh->has_vertices() )
    {
        VertexBuffer::ReadOnlyMapping<Eigen::Vector3f> verts(
            *m_ff_mesh->get_vertices());

        for( std::size_t i = 0; i < vertex_count(); ++i )
            vertices.push_back(verts[i]);
    }
}


void TriMeshEntity::set_colored_vertices(
    const Vertices &vertices,
    const VertexColors &colors)
{
    assert( 3*vertices.size() == colors.size() );

    // Set color pointers
    boost::shared_ptr<VertexBuffer> cb(new VertexBuffer);
    cb->set_usage_hint(VertexBuffer::DYNAMIC_DRAW);
    cb->set_data(&colors[0], colors.size());
    m_ff_mesh->set_colors(cb);
    m_bf_mesh->set_colors(cb);

    set_vertices_helper(vertices);
}


void TriMeshEntity::add_colored_vertices(
    const Vertices &new_vertices,
    const Vertices &all_vertices,
    const VertexColors &new_colors,
    const VertexColors &all_colors)
{
    assert( 3*new_vertices.size() == new_colors.size() );
    assert( 3*all_vertices.size() == all_colors.size() );

    if( new_vertices.empty() )
        return;

    if( !m_ff_mesh->has_vertices() )
    {
        assert( new_vertices.size() == all_vertices.size() );
        assert( new_colors.size() == all_colors.size() );
        set_colored_vertices(new_vertices, new_colors);
        return;
    }

    assert( m_ff_mesh->get_colors() );
    assert( m_bf_mesh->get_colors() );

    // Update the color buffer
    boost::shared_ptr<VertexBuffer> cb = m_ff_mesh->get_colors();
    std::size_t old_size = cb->size();
    std::size_t n = new_colors.size();
    cb->grow_back(n);
    cb->set_sub_data(&new_colors[0], n, old_size);

    add_vertices_helper(new_vertices, all_vertices);
}


std::size_t TriMeshEntity::vertex_count() const
{
    if( m_ff_mesh->has_vertices() )
        return m_ff_mesh->get_vertices()->size()/3/sizeof(float);
    else
        return 0;
}


std::size_t TriMeshEntity::tri_count() const
{
    if( m_ff_mesh->has_indices() )
        return m_ff_mesh->get_indices()->size()/3/sizeof(boost::uint32_t);
    else
        return 0;
}


void TriMeshEntity::update_normals(
    const boost::uint32_t *indices, std::size_t n_indices,
    const Eigen::Vector3f *vertices, std::size_t n_vertices,
    Eigen::Vector3f *ff_normals,
    Eigen::Vector3f *bf_normals)
{
    if( n_vertices == 0 )
        return;

    bool incremental = n_vertices > 3*n_indices;

    for( std::size_t i = 0; i < n_indices; i += 3 )
    {
        Indices::value_type i1 = indices[i+0];
        Indices::value_type i2 = indices[i+1];
        Indices::value_type i3 = indices[i+2];

        if( i1 >= n_vertices || i2 >= n_vertices || i3 >= n_vertices )
            continue;

        const Eigen::Vector3f &p1 = vertices[i1];
        const Eigen::Vector3f &p2 = vertices[i2];
        const Eigen::Vector3f &p3 = vertices[i3];

        Eigen::Vector3f face_normal = (p1-p3).cross(p2-p3);
        face_normal.normalize();

        // NOTE: We don't need to normalize the normals we have GL_NORMALIZE
        // enabled to do this for us (needed for scaling).
        ff_normals[i1] += face_normal;
        ff_normals[i2] += face_normal;
        ff_normals[i3] += face_normal;

        if( incremental && bf_normals )
        {
            bf_normals[i1] -= face_normal;
            bf_normals[i2] -= face_normal;
            bf_normals[i3] -= face_normal;
        }
    }

    if( !incremental && bf_normals )
    {
        for( std::size_t i = 0; i < n_vertices; ++i )
        {
            bf_normals[i] = -ff_normals[i];
        }
    }
}
