/*
 * Copyright Staffan Gimåker 2009-2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <vector>
#include <boost/shared_ptr.hpp>
#include <boost/cstdint.hpp>

#include "TransformationGuides.hh"
#include "../SceneManager.hh"
#include "../PrepareRenderContext.hh"
#include "../Mesh.hh"
#include "../ObjectEntity.hh"
#include "Camera.hh"
#include "../statelets/Transformation.hh"


using namespace peekabot;
using namespace peekabot::renderer;


namespace
{
    const Axis AXIS_MASK[] = { X_AXIS, Y_AXIS, Z_AXIS };

    const RGBColor AXIS_COLOR[3] = {
        RGBColor(0.7, 0.2, 0.2),
        RGBColor(0.2, 0.7, 0.2),
        RGBColor(0.2, 0.2, 0.7)
    };
}


TransformationGuides::TransformationGuides(
    const SceneManager &scene_manager)
    : m_scene_manager(scene_manager),
      m_mode(NO_TRANSFORMATION_GUIDES),
      m_coord_sys(WORLD_COORDINATES),
      m_pivot(AVERAGE_CENTER),
      m_axes(Axis(X_AXIS | Y_AXIS | Z_AXIS))
{
}


void TransformationGuides::get_renderables(PrepareRenderContext &context) const
{
    if( m_mode == MOVE_TRANSFORMATION_GUIDES )
        generate_move_guides(context);
    else if( m_mode == ROTATE_TRANSFORMATION_GUIDES )
        generate_rotate_guides(context);
}


void TransformationGuides::set_mode(TransformationGuidesMode mode)
{
    m_mode = mode;
}


void TransformationGuides::set_coord_sys(CoordinateSystem coord_sys)
{
    m_coord_sys = coord_sys;
}


void TransformationGuides::set_pivot(RotationCenterpoint pivot)
{
    m_pivot = pivot;
}


void TransformationGuides::set_axes(Axis axes)
{
    m_axes = axes;
}


void TransformationGuides::generate_move_guides(
    PrepareRenderContext &context) const
{
    typedef SceneManager::SelectedSet SS;
    const SS &selected = m_scene_manager.get_selected();

    if( selected.empty() )
        return;

    std::vector<float> vertices;
    std::vector<boost::uint8_t> colors;
    const Camera *cam = context.get_camera();


    for( SS::const_iterator it = selected.begin(); it != selected.end(); ++it )
    {
        const Eigen::Vector3f pos = (*it)->get_transformation().translation();

        for( int i = 0; i < 3; ++i )
        {
            if( m_axes & AXIS_MASK[i] )
            {
                Eigen::Vector3f v1, v2;

                switch( m_coord_sys )
                {
                    case LOCAL_COORDINATES:
                        v1 = pos - (*it)->get_transformation().linear().col(i)*1000;
                        v2 = pos + (*it)->get_transformation().linear().col(i)*1000;
                        break;

                    case PARENT_COORDINATES:
                        v1 = pos - (*it)->get_parent_transformation().linear().col(i)*1000;
                        v2 = pos + (*it)->get_parent_transformation().linear().col(i)*1000;
                        break;

                    case WORLD_COORDINATES:
                        v1 = pos - Eigen::Matrix3f::Identity().col(i)*1000;
                        v2 = pos + Eigen::Matrix3f::Identity().col(i)*1000;
                        break;

                    case CAMERA_COORDINATES:
                        v1 = pos - cam->get_viewer_mtow().linear().col(i)*1000;
                        v2 = pos + cam->get_viewer_mtow().linear().col(i)*1000;
                        break;

                    default:
                        assert( false );
                }

                vertices.push_back(v1(0));
                vertices.push_back(v1(1));
                vertices.push_back(v1(2));
                vertices.push_back(v2(0));
                vertices.push_back(v2(1));
                vertices.push_back(v2(2));

                colors.push_back(255*AXIS_COLOR[i].r);
                colors.push_back(255*AXIS_COLOR[i].g);
                colors.push_back(255*AXIS_COLOR[i].b);
                colors.push_back(255*AXIS_COLOR[i].r);
                colors.push_back(255*AXIS_COLOR[i].g);
                colors.push_back(255*AXIS_COLOR[i].b);
            }
        }
    }

    size_t n = vertices.size()/6;
    boost::shared_ptr<VertexBuffer> vb(new VertexBuffer(true));
    boost::shared_ptr<VertexBuffer> cb(new VertexBuffer(true));
    vb->set_usage_hint(VertexBuffer::STREAM_DRAW);
    cb->set_usage_hint(VertexBuffer::STREAM_DRAW);
    vb->set_data(&vertices[0], 6*n*sizeof(float));
    cb->set_data(&colors[0], 6*n);

    LineMesh *mesh = new LineMesh(
        vb, cb,
        boost::shared_ptr<VertexBuffer>(),
        boost::shared_ptr<IndexBuffer>(),
        0, n);
    mesh->get_state().set(new statelets::Transformation());

    context.prepare(mesh, true);
}


void TransformationGuides::generate_rotate_guides(
    PrepareRenderContext &context) const
{
    typedef SceneManager::SelectedSet SS;
    const SS &selected = m_scene_manager.get_selected();

    if( selected.empty() )
        return;

    std::vector<float> solid_vertices, dashed_vertices, solid_pt_vertices;
    std::vector<boost::uint8_t> solid_colors;
    const Camera *cam = context.get_camera();
    Eigen::Vector3f avg_pos(0,0,0);

    if( m_pivot == AVERAGE_CENTER )
    {
        // Calculate the average position of the selected objects
        // Could be done more efficiently using a trigger... then we wouldn't
        // have to recalculate it every frame
        for( SS::const_iterator it = selected.begin();
             it != selected.end(); ++it )
            avg_pos += (*it)->get_transformation().translation();
        avg_pos /= selected.size();
    }

    for( SS::const_iterator it = selected.begin(); it != selected.end(); ++it )
    {
        Eigen::Vector3f pivpos;

        if( m_pivot == LOCAL_CENTER )
            pivpos = (*it)->get_transformation().translation();
        else if( m_pivot == PARENT_CENTER )
            pivpos = (*it)->get_parent_transformation().translation();
        else if( m_pivot == WORLD_CENTER )
            pivpos = Eigen::Vector3f(0,0,0);
        else // AVERAGE_CENTER
            pivpos = avg_pos;

        const Eigen::Vector3f pos = (*it)->get_transformation().translation();

        for( int i = 0; i < 3; ++i )
        {
            if( m_axes & AXIS_MASK[i] )
            {
                Eigen::Vector3f v1, v2, axis;

                if( m_coord_sys == LOCAL_COORDINATES )
                    axis = (*it)->get_transformation().linear().col(i);
                else if( m_coord_sys == PARENT_COORDINATES )
                    axis = (*it)->get_parent_transformation().linear().col(i);
                else if( m_coord_sys == WORLD_COORDINATES )
                    axis = Eigen::Matrix3f::Identity().col(i);
                else if( m_coord_sys == CAMERA_COORDINATES )
                    axis = cam->get_viewer_mtow().linear().col(i);
                else
                    assert( false );

                v1 = pivpos - axis*1000;
                v2 = pivpos + axis*1000;

                if( m_coord_sys != CAMERA_COORDINATES || i != 0 )
                {
                    solid_vertices.push_back(v1(0));
                    solid_vertices.push_back(v1(1));
                    solid_vertices.push_back(v1(2));
                    solid_vertices.push_back(v2(0));
                    solid_vertices.push_back(v2(1));
                    solid_vertices.push_back(v2(2));

                    solid_colors.push_back(255*AXIS_COLOR[i].r);
                    solid_colors.push_back(255*AXIS_COLOR[i].g);
                    solid_colors.push_back(255*AXIS_COLOR[i].b);
                    solid_colors.push_back(255*AXIS_COLOR[i].r);
                    solid_colors.push_back(255*AXIS_COLOR[i].g);
                    solid_colors.push_back(255*AXIS_COLOR[i].b);
                }

                if( m_pivot != LOCAL_CENTER )
                {
                    Eigen::Vector3f tmp;
                    if( m_coord_sys != CAMERA_COORDINATES || i != 0 )
                    {
                        Eigen::Vector3f proj = axis * (
                            axis.dot(pivpos-pos) / axis.norm());
                        tmp = pivpos - proj;
                    }
                    else
                        tmp = pivpos;

                    dashed_vertices.push_back(tmp(0));
                    dashed_vertices.push_back(tmp(1));
                    dashed_vertices.push_back(tmp(2));
                    dashed_vertices.push_back(pos(0));
                    dashed_vertices.push_back(pos(1));
                    dashed_vertices.push_back(pos(2));

                    solid_pt_vertices.push_back(tmp(0));
                    solid_pt_vertices.push_back(tmp(1));
                    solid_pt_vertices.push_back(tmp(2));
                }
            }
        }
    }

    if( !solid_vertices.empty() )
    {
        size_t n_solid = solid_vertices.size()/6;
        boost::shared_ptr<VertexBuffer> solid_vb(new VertexBuffer(true));
        boost::shared_ptr<VertexBuffer> solid_cb(new VertexBuffer(true));

        solid_vb->set_usage_hint(VertexBuffer::STREAM_DRAW);
        solid_cb->set_usage_hint(VertexBuffer::STREAM_DRAW);

        solid_vb->set_data(&solid_vertices[0], 6*n_solid*sizeof(float));
        solid_cb->set_data(&solid_colors[0], 6*n_solid);

        LineMesh *solid = new LineMesh(
            solid_vb, solid_cb,
            boost::shared_ptr<VertexBuffer>(),
            boost::shared_ptr<IndexBuffer>(),
            0, n_solid);

        solid->get_state().set(new statelets::Transformation());
        context.prepare(solid, true);
    }


    if( !solid_pt_vertices.empty() )
    {
        size_t n_solid_pt = solid_pt_vertices.size()/3;
        boost::shared_ptr<VertexBuffer> solid_pt_vb(new VertexBuffer(true));

        solid_pt_vb->set_usage_hint(VertexBuffer::STREAM_DRAW);

        solid_pt_vb->set_data(
            &solid_pt_vertices[0], 3*n_solid_pt*sizeof(float));

        PointMesh *solid_pt = new PointMesh(
            solid_pt_vb,
            boost::shared_ptr<VertexBuffer>(),
            n_solid_pt);

        solid_pt->get_state().set(new statelets::PointParams(4, true));
        solid_pt->get_state().set(
            new statelets::Color(RGBColor(0.2, 0.2, 0.2)));
        solid_pt->get_state().set(new statelets::Transformation());
        context.prepare(solid_pt, true);
    }


    if( !dashed_vertices.empty() )
    {
        size_t n_dashed = dashed_vertices.size()/6;
        boost::shared_ptr<VertexBuffer> dashed_vb(new VertexBuffer(true));
        dashed_vb->set_usage_hint(VertexBuffer::STREAM_DRAW);
        dashed_vb->set_data(&dashed_vertices[0], 6*n_dashed*sizeof(float));

        LineMesh *dashed = new LineMesh(
            dashed_vb, boost::shared_ptr<VertexBuffer>(),
            boost::shared_ptr<VertexBuffer>(),
            boost::shared_ptr<IndexBuffer>(),
            0, n_dashed);

        dashed->get_state().set(new statelets::Transformation());
        dashed->get_state().set(new statelets::LineParams(1, true, 6, 43690));
        dashed->get_state().set(
            new statelets::Color(RGBColor(0.2, 0.2, 0.2)));
        context.prepare(dashed, true);
    }
}
