/*
 * Copyright Staffan Gimåker 2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef PEEKABOT_RENDERER_OG3D_NODE_HH_INCLUDED
#define PEEKABOT_RENDERER_OG3D_NODE_HH_INCLUDED


#include <cstddef>
#include <Eigen/Core>
#include <boost/cstdint.hpp>


namespace peekabot
{
    namespace renderer
    {
        class OccupancyGrid3D;
        class Og3dTraversalContext;
        class Og3dRenderContext;

        /**
         *
         * Invariants:
         *   A leaf node is never unknown, i.e. it has an occupancy value set.
         */
        class Og3dNode
        {
        public:
            Og3dNode();

            ~Og3dNode();

            inline bool is_leaf() const throw()
            {
                return m_children == 0;
            }

            /**
             * \brief Get the i:th child of a non-leaf node.
             *
             * \pre <tt>is_leaf() == false</tt>
             * \pre \f$ 0 \leq i < 8 \f$
             *
             * \return A pointer to the i:th child node, or null if the i:th
             * child does not exist.
             */
            inline const Og3dNode *get_child(int i) const
            {
                return m_children[i];
            }

            /**
             * \copydoc get_child(int) const
             */
            inline Og3dNode *get_child(int i)
            {
                return m_children[i];
            }

            /**
             * \brief If the node is a leaf, the method returns the node's
             * occupancy value. If the node is a not a leaf, it returns the
             * maximum occupancy value contained in its subtree.
             */
            inline boost::uint8_t get_belief() const
            {
                return m_belief;
            }

            /**
             * \brief Set the belief of the cell located at the given
             * coordinate \c x. This method should not be used directly,
             * instead use OccupancyGrid3D::set_belief().
             */
            void set_belief(
                Og3dTraversalContext &tc,
                const Eigen::Vector3f &x,
                boost::uint8_t belief,
                const Eigen::Vector3f &center, float node_size);

            /**
             * \brief Get the i:th child's center offset relative the parent
             * node.
             *
             * I.e., <tt>node_center + node->octant_offset(0, node_size,
             * hf)<tt> returns the center position of the first child of \c
             * node.
             *
             * \pre \f$ 0 \leq i < 8 \f$
             *
             * \param i The octant for which to get the offset
             * from its parent node.
             * \param node_size The node size of \c this.
             * \param hf The OG-map height factor, i.e. how tall the cell
             * is in relation to a side of its base area.
             *
             * \return A vector describing the offset of node i from its parent.
             *
             * \sa \c m_children (for octant indexing).
             */
            inline Eigen::Vector3f octant_offset(
                int i, float node_size, float hf) const throw()
            {
                return Eigen::Vector3f(
                    node_size/4*(2*(i&1)-1),
                    node_size/4*((i&2)-1),
                    hf*node_size/4*(((i&4)>>1)-1) );
            }

            /**
             * \brief Check if a coordinate is enclosed in the
             * octree node.
             *
             * \return \c true if the node encloses the given coordinate,
             *         \c false otherwise.
             */
            bool encloses(
                const Eigen::Vector3f &x,
                const Eigen::Vector3f &center, float node_size,
                float hf) const throw();

            /**
             * \brief Check if a coordinate is enclosed the node's
             *        <em>i</em>:th quadrant.
             *
             * \pre The coordinate is enclosed in the node, i.e.
             * <tt>encloses(x) == true</tt>.
             *
             * \param i The quadrant to check against, \f$0 \leq i < 8\f$.
             * \return \c true if the coordinate is enclosed by the
             *         node's <em>i</em>:th octant, \c false
             *         otherwise.
             *
             * \remarks Note that this doesn't check if the coordinate
             * is enclosed by a child node, or even demands that any
             * child nodes exist.
             *
             * \sa encloses(),
             * m_children (for quadrant indexing).
             */
            inline bool enclosed_in_octant(
                const Eigen::Vector3f &x, int i,
                const Eigen::Vector3f &center) const throw()
            {
                return ( ( i&1 ? x(0) >= center(0) : x(0) < center(0) ) &&
                         ( i&2 ? x(1) >= center(1) : x(1) < center(1) ) &&
                         ( i&4 ? x(2) >= center(2) : x(2) < center(2) ) );
            }

            /**
             *
             * \param plane_mask Bit-mask of frustum planes that the node is
             * possible outside or intersecting. Only the 8 least significant
             * bits should ever be set.
             */
            void render(
                Og3dTraversalContext &tc, Og3dRenderContext &rc,
                const Eigen::Vector3f &center, float node_size,
                int plane_mask);

        private:
            void render_no_cull(
                Og3dTraversalContext &tc, Og3dRenderContext &rc,
                const Eigen::Vector3f &center, float node_size,
                std::size_t *n_drawn = 0);

            static void update_subtree_beliefs(Og3dTraversalContext &tc);

            /**
             * \brief Optimize the storage by merging neighbouring nodes
             * if possible.
             *
             * \pre \a node is a leaf node.
             */
            static void optimize(Og3dTraversalContext &tc, Og3dNode *node);

        public:
            /**
             * \brief Pointers to the child nodes of the node.
             *
             * \invariant m_children == 0 \f$\Leftrightarrow\f$ the
             * node is a leaf.
             */
            Og3dNode **m_children;

            /**
             * \brief For leaf nodes this stores the node's occupancy
             * value. For non-leaf nodes this store's the maximum occupancy
             * value of its subtree.
             *
             * It's initialized to zero upon construction.
             */
            boost::uint8_t m_belief;

            boost::int8_t m_counter;
        };
    }
}


#endif // PEEKABOT_RENDERER_OG3D_NODE_HH_INCLUDED
