/*
 * Copyright Staffan Gimåker 2007-2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "../Mesh.hh"
#include "MeshEntities.hh"
#include "../PrepareRenderContext.hh"
#include "../../GeometryToolbox.hh"

#include <GL/glew.h>
#include <boost/scoped_array.hpp>


using namespace peekabot;
using namespace peekabot::renderer;


MeshEntity::MeshEntity(boost::shared_ptr<Mesh> mesh) throw()
    : m_mesh(mesh)
{
}

MeshEntity::~MeshEntity()
{
}

void MeshEntity::get_renderables(PrepareRenderContext &context) const
{
    if( m_mesh->has_vertices() )
        context.prepare(m_mesh.get());
}


void MeshEntity::set_vertices(const Vertices &vertices) throw()
{
    boost::shared_ptr<VertexBuffer> buf;

    if( !vertices.empty() )
    {
        buf.reset(new VertexBuffer);

        buf->set_data(
            &vertices[0],
            3*sizeof(float)*vertices.size());

        if( !m_mesh->has_indices() )
            m_mesh->set_element_count(vertices.size()/m_mesh->vertices_per_element());
    }

    m_mesh->set_vertices(buf);
    m_mesh->set_colors(boost::shared_ptr<VertexBuffer>());

    calculate_bvs(vertices);
}

void MeshEntity::add_vertices(
    const Vertices &new_vertices,
    const Vertices &all_vertices) throw()
{
    if( new_vertices.empty() )
        return;

    assert( !m_mesh->has_colors() );

    if( !m_mesh->has_vertices() )
    {
        set_vertices(new_vertices);
        return;
    }

    // Update the vertex buffer
    boost::shared_ptr<VertexBuffer> vb = m_mesh->get_vertices();
    std::size_t old_size = vb->size();
    std::size_t n = 3*sizeof(float)*new_vertices.size();
    vb->grow_back(n);
    vb->set_sub_data(&new_vertices[0], n, old_size);

    std::size_t n_vertices = vb->size()/3/sizeof(float);

    // Update the element count
    m_mesh->set_element_count(n_vertices/m_mesh->vertices_per_element());

    // Update the bounding sphere
    //
    // For entities with few vertices, recalculate the BV from scratch. For
    // entities with many vertices, try to extend (this is "best effort", it's
    // not "optimal") the existing bounding sphere to save computations.
    if( n_vertices < 100 )
    {
        calculate_bvs(all_vertices);
    }
    else
    {
        extend_bvs(new_vertices);
    }
}

void MeshEntity::get_vertices(Vertices &vertices) const throw()
{
    vertices.clear();

    if( !m_mesh->has_vertices() )
        return;

    const boost::shared_ptr<VertexBuffer> verts = m_mesh->get_vertices();

    VertexBuffer::ReadOnlyMapping<Eigen::Vector3f> p(*verts);

    std::size_t N = verts->size()/sizeof(float)/3;
    for( std::size_t i = 0; i < N; ++i )
        vertices.push_back(p[i]);
}


void MeshEntity::set_colored_vertices(
    const Vertices &vertices,
    const VertexColors &colors)
{
    assert( 3*vertices.size() == colors.size() );

    boost::shared_ptr<VertexBuffer> vb;
    boost::shared_ptr<VertexBuffer> cb;

    if( !vertices.empty() )
    {
        vb.reset(new VertexBuffer);
        vb->set_data(&vertices[0], 3*sizeof(float)*vertices.size());

        cb.reset(new VertexBuffer);
        cb->set_data(&colors[0], colors.size());

        if( !m_mesh->has_indices() )
            m_mesh->set_element_count(vertices.size()/m_mesh->vertices_per_element());
    }

    m_mesh->set_vertices(vb);
    m_mesh->set_colors(cb);

    calculate_bvs(vertices);
}


void MeshEntity::add_colored_vertices(
    const Vertices &new_vertices,
    const Vertices &all_vertices,
    const VertexColors &new_colors,
    const VertexColors &all_colors)
{
    assert( 3*all_vertices.size() == all_colors.size() );
    assert( 3*new_vertices.size() == new_colors.size() );

    if( new_vertices.empty() )
        return;

    assert( m_mesh->has_colors() );

    if( !m_mesh->has_vertices() )
    {
        set_colored_vertices(new_vertices, new_colors);
        return;
    }

    // Update the vertex buffer
    boost::shared_ptr<VertexBuffer> vb = m_mesh->get_vertices();
    assert( vb );
    {
        std::size_t old_size = vb->size();
        std::size_t n = 3*sizeof(float)*new_vertices.size();
        vb->grow_back(n);
        vb->set_sub_data(&new_vertices[0], n, old_size);
    }

    // Update the element count
    std::size_t n_vertices = vb->size()/3/sizeof(float);
    m_mesh->set_element_count(n_vertices/m_mesh->vertices_per_element());

    // Update the color buffer
    boost::shared_ptr<VertexBuffer> cb = m_mesh->get_colors();
    assert( cb );
    {
        std::size_t old_size = cb->size();
        std::size_t n = new_colors.size();
        cb->grow_back(n);
        cb->set_sub_data(&new_colors[0], n, old_size);
    }

    // Update the bounding sphere
    //
    // For entities with few vertices, recalculate the BV from scratch. For
    // entities with many vertices, try to extend (this is "best effort", it's
    // not "optimal") the existing bounding sphere to save computations.
    if( n_vertices < 100 )
    {
        calculate_bvs(all_vertices);
    }
    else
    {
        extend_bvs(new_vertices);
    }
}


void MeshEntity::calculate_bvs(const Vertices &vertices) throw()
{
    // TODO: this is necessarily optimal for indexed primitives, since there
    // might be vertices that aren't referenced by the indices

    //
    // Calculate the midpoint of the vertices
    //
    Eigen::Vector3f midpoint;
    float r;

    geom::calc_approx_bounding_sphere(
        vertices, midpoint, r,
        1, 1, 1);

    set_bounding_sphere(BoundingSphere(midpoint, r));
}


void MeshEntity::extend_bvs(const Vertices &new_vertices) throw()
{
    Eigen::Vector3f midp = get_bounding_sphere_lc().get_pos();
    float r = get_bounding_sphere_lc().get_radius();

    if( geom::extend_bounding_sphere(new_vertices, midp, r, 1.5) )
    {
        VertexBuffer::ReadOnlyMapping<Eigen::Vector3f> verts(
            *m_mesh->get_vertices());

        std::size_t n_vertices = m_mesh->get_vertices()->size()/3/sizeof(float);

        geom::calc_approx_bounding_sphere(
            verts.get(), verts.get()+n_vertices,
            midp, r,
            1, 1, 1);
    }

    set_bounding_sphere(BoundingSphere(midp, r));
}


boost::shared_ptr<Mesh> MeshEntity::get_mesh() throw()
{
    return m_mesh;
}

const boost::shared_ptr<Mesh> 
MeshEntity::get_mesh() const throw()
{
    return m_mesh;
}


//


PointMeshEntity::PointMeshEntity() throw()
    : MeshEntity(boost::shared_ptr<Mesh>(new PointMesh))
{
}

PointMeshEntity::PointMeshEntity(const PointMeshEntity &x) throw()
    : MeshEntity(boost::shared_ptr<Mesh>(x.get_mesh()->clone()))
{
}

PointMeshEntity::~PointMeshEntity()
{
}

PointMeshEntity *PointMeshEntity::clone() const throw()
{
    return new PointMeshEntity(*this);
}

//

LineMeshEntity::LineMeshEntity() throw()
    : MeshEntity(boost::shared_ptr<Mesh>(new LineMesh))
{
}

LineMeshEntity::LineMeshEntity(const LineMeshEntity &x) throw()
    : MeshEntity(boost::shared_ptr<Mesh>(x.get_mesh()->clone()))
{
}

LineMeshEntity::~LineMeshEntity()
{
}

LineMeshEntity *LineMeshEntity::clone() const throw()
{
    return new LineMeshEntity(*this);
}


void LineMeshEntity::set_line_style(
    float line_width, 
    uint16_t stipple_pattern, 
    int stipple_factor)
{
    m_mesh->get_state().set(
        new statelets::LineParams(
            line_width, stipple_pattern != 0xFFFF,
            stipple_factor, stipple_pattern));
}
