/*
 * Copyright Staffan Gimåker 2006-2009.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef PEEKABOT_RENDERER_CUBE_HH_INCLUDED
#define PEEKABOT_RENDERER_CUBE_HH_INCLUDED


#include <GL/glew.h>
#include <Eigen/Core>

#include "../CullableEntity.hh"
#include "../Renderable.hh"
#include "../PrepareRenderContext.hh"


namespace peekabot
{
    namespace renderer
    {
        class Cube : public CullableEntity, public Renderable
        {
            float m_width;
            float m_height;
            float m_depth;

        public:
            Cube(float width = 1.0f, float height = 1.0f, float depth = 1.0f)
                : CullableEntity(
                    BoundingSphere(
                        Eigen::Vector3f(0,0,0),
                        sqrt(3)*std::max(width, std::max(height, depth)))),
                  m_width(width),
                  m_height(height),
                  m_depth(depth) {}

            virtual Cube *clone() const throw()
            {
                return new Cube(m_width, m_height, m_depth);
            }

            virtual void get_renderables(PrepareRenderContext &context) const
            {
                context.prepare(this);
            }

            virtual void render(RenderContext &context) const
            {
                float x = m_width/2;
                float y = m_depth/2;
                float z = m_height/2;

                glBegin(GL_QUADS);


                glNormal3f(0, 1, 0);
                glVertex3f( x, y,-z);			// Top Right Of The Quad (Top)
                glVertex3f(-x, y,-z);			// Top Left Of The Quad (Top)
                glVertex3f(-x, y, z);			// Bottom Left Of The Quad (Top)
                glVertex3f( x, y, z);			// Bottom Right Of The Quad (Top)


                glNormal3f(0, -1, 0);
                glVertex3f( x,-y, z);			// Top Right Of The Quad (Bottom)
                glVertex3f(-x,-y, z);			// Top Left Of The Quad (Bottom)
                glVertex3f(-x,-y,-z);			// Bottom Left Of The Quad (Bottom)
                glVertex3f( x,-y,-z);			// Bottom Right Of The Quad (Bottom)


                glNormal3f(0, 0, 1);
                glVertex3f( x, y, z);			// Top Right Of The Quad (Front)
                glVertex3f(-x, y, z);			// Top Left Of The Quad (Front)
                glVertex3f(-x,-y, z);			// Bottom Left Of The Quad (Front)
                glVertex3f( x,-y, z);			// Bottom Right Of The Quad (Front)


                glNormal3f(0, 0, -1);
                glVertex3f( x,-y,-z);			// Bottom Left Of The Quad (Back)
                glVertex3f(-x,-y,-z);			// Bottom Right Of The Quad (Back)
                glVertex3f(-x, y,-z);			// Top Right Of The Quad (Back)
                glVertex3f( x, y,-z);			// Top Left Of The Quad (Back)


                glNormal3f(-1, 0, 0);
                glVertex3f(-x, y, z);			// Top Right Of The Quad (Left)
                glVertex3f(-x, y,-z);			// Top Left Of The Quad (Left)
                glVertex3f(-x,-y,-z);			// Bottom Left Of The Quad (Left)
                glVertex3f(-x,-y, z);			// Bottom Right Of The Quad (Left)

                glNormal3f(1, 0, 0);
                glVertex3f( x, y,-z);			// Top Right Of The Quad (Right)
                glVertex3f( x, y, z);			// Top Left Of The Quad (Right)
                glVertex3f( x,-y, z);			// Bottom Left Of The Quad (Right)
                glVertex3f( x,-y,-z);			// Bottom Right Of The Quad (Right)

                glEnd();
            }

            float get_width() const throw() { return m_width; }

            float get_height() const throw() { return m_height; }

            float get_depth() const throw() { return m_depth; }

            void set_width(float x) throw()
            {
                m_width = x;
                set_bounding_sphere(
                    BoundingSphere(
                        Eigen::Vector3f(0,0,0),
                        sqrt(3)*std::max(m_width, std::max(m_height, m_depth))));
            }

            void set_height(float x) throw()
            {
                m_height = x;
                set_bounding_sphere(
                    BoundingSphere(
                        Eigen::Vector3f(0,0,0),
                        sqrt(3)*std::max(m_width, std::max(m_height, m_depth))));
            }

            void set_depth(float x) throw()
            {
                m_depth = x;
                set_bounding_sphere(
                    BoundingSphere(
                        Eigen::Vector3f(0,0,0),
                        sqrt(3)*std::max(m_width, std::max(m_height, m_depth))));
            }
        };
    }
}


#endif // PEEKABOT_RENDERER_CUBE_HH_INCLUDED
