/*
 * Copyright Staffan Gimåker 2008-2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef PEEKABOT_RENDERER_TESSELLATOR_HH_INCLUDED
#define PEEKABOT_RENDERER_TESSELLATOR_HH_INCLUDED


#include <GL/glew.h>
#include <stdexcept>
#include <vector>
#include <boost/function.hpp>
#include <Eigen/Core>
#include <boost/utility.hpp>

#ifdef _WIN32
#  include <windows.h>
#else
#  define APIENTRY
#endif


namespace peekabot
{
    namespace renderer
    {
        class Tessellator : public boost::noncopyable
        {
            typedef void (APIENTRY *GluTessCallbackType)();

            struct Vertex
            {
                Eigen::Vector3f vertex;
                std::size_t index;
            };

        public:
            typedef boost::function<void (
                std::size_t idx1, std::size_t idx2, std::size_t idx3,
                const Eigen::Vector3f &v1,
                const Eigen::Vector3f &v2,
                const Eigen::Vector3f &v3)> EmitFaceCallback;

            /**
             * \brief Called when a new vertex needs to be created.
             *
             * This callback allows the caller to combine other data associated
             * with the vertex as well, such as color or texture coordinates.
             *
             * Any user data should be combined as a linear combination of the
             * user data in idx[0]-idx[3], with the corresponding weights (which
             * sum to 1). The new vertex will be assigned the index returned by
             * the callback. The user is response for making sure the returned
             * index is greater or equal to the number of vertices input using
             * the vertex() method.
             *
             * However, if you don't plan on uses the indices it's safe to
             * return an arbitrary value.
             */
            typedef boost::function<std::size_t (
                const Eigen::Vector3f &v,
                std::size_t idx1, std::size_t idx2,
                std::size_t idx3, std::size_t idx4,
                const float w[4])> CombineCallback;

            Tessellator(
                const EmitFaceCallback &emit_callback,
                const CombineCallback &combine_callback);

            ~Tessellator();

            void begin_polygon();

            void end_polygon();

            void begin_contour();

            void vertex(float x, float y, float z);

            Eigen::Vector3f last_vertex() const
            {
                if( m_verts.empty() )
                    throw std::logic_error("No vertices added");
                return m_verts.back()->vertex;
            }

            std::size_t vertices_input() const throw()
            {
                return m_verts.size() + m_q.size();
            }

        private:
            void end_contour();

            void triangulate_fan(const std::vector<Vertex *> &v);

            void triangulate_strip(const std::vector<Vertex *> &v);

            void triangulate_triangles(const std::vector<Vertex *> &v);

            inline static bool is_ccw(Vertex *v1, Vertex *v2, Vertex *v3) throw()
            {
                return (v2->vertex(1)-v1->vertex(1)) * (v3->vertex(0)-v1->vertex(0)) <
                    (v3->vertex(1)-v1->vertex(1)) * (v2->vertex(0)-v1->vertex(0));
                // else, CW or colinear
            }


        private:
            GLUtesselator *m_tess;
            std::vector<Vertex *> m_verts;
            std::size_t m_idx;

            std::vector<Vertex *> m_q;
            GLenum m_type;

            EmitFaceCallback m_emit_callback;
            CombineCallback m_combine_callback;

            std::size_t m_contour_count;

        private:
            static void tess_callback(Vertex *vertex_data, void *polygon_data);

            static void combine_callback(
                GLdouble coords[3],
                Vertex *vertex_data[4],
                GLfloat weight[4],
                void **outdata,
                void *polygon_data);

            static void begin_callback(GLenum type, void *polygon_data);

            static void end_callback(void *polygon_data);

            static void error_callback(GLenum _errno);
        };
    }
}


#endif // PEEKABOT_RENDERER_TRI_TESSELLATOR_HH_INCLUDED
