/*
 * Copyright Staffan Gimåker 2007-2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "RenderTraverser.hh"
#include "RenderContext.hh"
#include "Entity.hh"
#include "Renderable.hh"
#include "entities/Camera.hh"
#include "statelets/Texture2D.hh"
#include "Mesh.hh"


using namespace peekabot;
using namespace peekabot::renderer;

//
// ------------------------------
//


RenderTraverser::Batch::~Batch()
{
    for( size_t i = 0; i < m_items.size(); ++i )
    {
        if( m_items[i].m_override_state )
            delete m_items[i].m_override_state;
    }

    m_items.clear();
}


void RenderTraverser::Batch::render(RenderContext &context, bool front_to_back)
{
    if( front_to_back )
    {
        for( Items::iterator it = m_items.begin(); it != m_items.end(); ++it )
        {
            if( it->m_override_state )
                it->m_override_state->apply(it->m_template_state);
            else
                it->m_renderable->get_state().apply(it->m_template_state);
            it->m_renderable->render(context);
        }
    }
    else
    {
        for( Items::reverse_iterator it = m_items.rbegin(); it != m_items.rend(); ++it )
        {
            if( it->m_override_state )
                it->m_override_state->apply(it->m_template_state);
            else
                it->m_renderable->get_state().apply(it->m_template_state);
            it->m_renderable->render(context);
        }
    }
}


//
// ------------------------------
//

RenderTraverser::RenderTraverser(
    const Camera &camera,
    const bool *visible_layers)
    : PrepareRenderContext(&camera),
      m_visible_layers(visible_layers),
      m_entity_tmp(0),
      m_context(&camera)
{
}


RenderTraverser::~RenderTraverser()
{
    for( std::set<const Renderable *>::const_iterator it = m_delete_set.begin();
         it != m_delete_set.end(); ++it )
    {
        delete *it;
    }
    m_delete_set.clear();
}


void RenderTraverser::traverse(const Entity *entity)
{
    const ObjectEntity *object =
        dynamic_cast<const ObjectEntity *>(entity);

    if( object )
    {
        if(  m_visible_layers[object->get_layer()] )
        {
            m_entity_tmp = object;
            entity->get_renderables(*this);
            m_entity_tmp = 0;

            if( object->is_selected() )
                make_tripod(object);
        }
    }
    else
    {
        entity->get_renderables(*this);
    }
}


void RenderTraverser::prepare(const Renderable *r, bool assume_ownership)
{
    handle_renderable(r, m_entity_tmp);

    if( assume_ownership )
        m_delete_set.insert(r);
}


void RenderTraverser::render_opaque() throw()
{
    render_batch(m_opaque, true);
}


void RenderTraverser::render_translucent() throw()
{
    render_batch(m_translucent, false);
}


void RenderTraverser::render_always_on_top() throw()
{
    // Draw always-on-top renderables
    State::push_baseline();
    State baseline = State::get_baseline_state();
    State::OverrideSet ignore_set;

    baseline.set(new statelets::DepthTest(statelets::DepthTest::ALWAYS));

    ignore_set.insert(statelets::DepthTest::get_type());
    State::set_baseline(baseline, ignore_set);

    render_batch(m_opaque_on_top, true);
    render_batch(m_translucent_on_top, false);

    State::pop_baseline();
}


void RenderTraverser::handle_renderable(
    const Renderable *renderable,
    const ObjectEntity *object)
{
    //
    // Is the renderable translucent? We sort opaque and translucent renderables
    // into different batches, so we need to find out.
    // It is translucent when the alpha used for rendering is less than ~1,
    // which it is when
    // 1) Its alpha statelet reports an alpha less than 1.
    // 2) Its parent node's template state reports an alpha less than 1 and
    //    the renderable statelet is unset.
    //
    float alpha = 1.0f;
    if( renderable->get_state().is_set<statelets::Alpha>() )
        alpha = renderable->get_state().get<statelets::Alpha>()->get();
    else if( object && object->get_template_state().is_set<statelets::Alpha>() )
        alpha = object->get_template_state().get<statelets::Alpha>()->get();

    if( alpha > 0.98f ) // cheat a little :)
    {
        /*if( renderable->is_always_on_top() )
            m_opaque_on_top.push_back(
                std::make_pair(renderable, &object->get_template_state()));
                else*/

        if( object )
        {
            if( object->is_selected() || object->has_selected_ancestor() )
            {
                State s = renderable->get_state();
                s.set(new statelets::StencilTest(true));
                m_opaque.add(renderable, &object->get_template_state(), s);
            }
            else
                m_opaque.add(renderable, &object->get_template_state());
        }
        else
        {
            m_opaque.add(renderable, 0);
        }
    }
    else
    {
        /*if( renderable->is_always_on_top() )
            m_translucent_on_top.push_back(
                std::make_pair(renderable, &object->get_template_state()));
                else*/

        if( object )
        {
            if( object->is_selected() || object->has_selected_ancestor() )
            {
                State s = renderable->get_state();
                s.set(new statelets::StencilTest(true));
                m_translucent.add(renderable, &object->get_template_state(), s);
            }
            else
                m_translucent.add(renderable, &object->get_template_state());
        }
        else
        {
            m_translucent.add(renderable, 0);
        }
    }


    if( object )
    {
        if( object->is_selected() )
            m_selected.add(renderable, &object->get_template_state());
        else if( object->has_selected_ancestor() )
            m_ancestor_selected.add(renderable, &object->get_template_state());
    }

    /*if( m_opaque.size() > 100 )
    {
        render_batch(m_opaque);
        m_opaque.clear();
    }*/
}



void RenderTraverser::render_batch(Batch &batch, bool front_to_back) throw()
{
    batch.render(m_context, front_to_back);
}



void RenderTraverser::render_outlines(
    const RGBColor &selected_color,
    const RGBColor &ancestor_selected_color) throw()
{
    State::push_baseline();

    State baseline = State::get_baseline_state();
    State::OverrideSet ignore_set;

    baseline.set(new statelets::ColorMaterial(true));
    baseline.set(new statelets::Lighting(false));
    baseline.set(new statelets::StencilTest(true));
    baseline.set(new statelets::StencilFunc(
                     GL_NOTEQUAL, 1, static_cast<GLuint>(-1),
                     GL_KEEP, GL_KEEP, GL_REPLACE));
    baseline.set(new statelets::PointParams(4, true));
    baseline.set(new statelets::LineParams(4, false, 0, 0));
    baseline.set(new statelets::PolygonMode(
                     statelets::PolygonMode::FRONT,
                     statelets::PolygonMode::LINE));


    ignore_set.insert(statelets::Name::get_type());
    ignore_set.insert(statelets::Lighting::get_type());
    ignore_set.insert(statelets::LightModel::get_type());
    ignore_set.insert(statelets::Material::get_type());
    ignore_set.insert(statelets::Color::get_type());
    ignore_set.insert(statelets::ColorMaterial::get_type());
    ignore_set.insert(statelets::StencilTest::get_type());
    ignore_set.insert(statelets::StencilFunc::get_type());
    ignore_set.insert(statelets::LineParams::get_type());
    ignore_set.insert(statelets::PointParams::get_type());
    ignore_set.insert(statelets::ColorPointer::get_type());
    ignore_set.insert(statelets::NormalPointer::get_type());
    ignore_set.insert(statelets::TexCoordPointer::get_type());
    ignore_set.insert(statelets::Alpha::get_type());
    ignore_set.insert(statelets::BlendMode::get_type());
    ignore_set.insert(statelets::Texture2D::get_type());


    baseline.get<statelets::Color>()->set(selected_color);
    State::set_baseline(baseline, ignore_set);
    render_batch(m_selected, true);

    baseline.get<statelets::Color>()->set(ancestor_selected_color);
    State::set_baseline(baseline, ignore_set);
    render_batch(m_ancestor_selected, true);

    State::pop_baseline();
}


void RenderTraverser::make_tripod(const ObjectEntity *object)
{
    const Camera *cam = m_context.get_camera();
    const Eigen::Transform3f &mtow = object->get_transformation();
    const Eigen::Vector3f wp(mtow.translation());

    // v = The tripods position relative the camera used to render it
    Eigen::Vector3f v = wp-cam->get_viewer_mtow().translation();
    // u = Camera viewing vector
    Eigen::Vector3f u = cam->get_transformation().linear().col(0);

    // Overwrite v with the projection of v on u.
    // |v| is now the distance to the tripod along the view vector
    v = u.dot(v) * u;

    // We can calculate how big the tripod should be by:
    // 1) calculating how big (in meters) it should be to be rendered as 64
    //    pixels if it was situated at depth x=near
    // 2) use similarity and the result from 1 to calculate how big it has to
    //    be given its depth of x=|v|
    float k = (
        64/cam->get_pixels_per_meter() * // 1
        v.norm()/cam->get_near_plane()); // 2!

    float kx = k/object->get_x_scale();
    float ky = k/object->get_y_scale();
    float kz = k/object->get_z_scale();

    // X axis
    m_tripod_verts.push_back(wp(0));
    m_tripod_verts.push_back(wp(1));
    m_tripod_verts.push_back(wp(2));

    m_tripod_verts.push_back(wp(0) + mtow(0,0)*kx);
    m_tripod_verts.push_back(wp(1) + mtow(1,0)*kx);
    m_tripod_verts.push_back(wp(2) + mtow(2,0)*kx);

    m_tripod_verts.push_back(wp(0) + mtow(0,0)*kx);
    m_tripod_verts.push_back(wp(1) + mtow(1,0)*kx);
    m_tripod_verts.push_back(wp(2) + mtow(2,0)*kx);

    m_tripod_verts.push_back(wp(0) + mtow(0,0)*kx*0.9 + mtow(0,1)*ky*0.1);
    m_tripod_verts.push_back(wp(1) + mtow(1,0)*kx*0.9 + mtow(1,1)*ky*0.1);
    m_tripod_verts.push_back(wp(2) + mtow(2,0)*kx*0.9 + mtow(2,1)*ky*0.1);

    m_tripod_verts.push_back(wp(0) + mtow(0,0)*kx);
    m_tripod_verts.push_back(wp(1) + mtow(1,0)*kx);
    m_tripod_verts.push_back(wp(2) + mtow(2,0)*kx);

    m_tripod_verts.push_back(wp(0) + mtow(0,0)*kx*0.9 - mtow(0,1)*ky*0.1);
    m_tripod_verts.push_back(wp(1) + mtow(1,0)*kx*0.9 - mtow(1,1)*ky*0.1);
    m_tripod_verts.push_back(wp(2) + mtow(2,0)*kx*0.9 - mtow(2,1)*ky*0.1);

    for( int i = 0; i < 6; ++i )
    {
        m_tripod_colors.push_back(255);
        m_tripod_colors.push_back(0);
        m_tripod_colors.push_back(0);
    }

    // Y axis
    m_tripod_verts.push_back(wp(0));
    m_tripod_verts.push_back(wp(1));
    m_tripod_verts.push_back(wp(2));

    m_tripod_verts.push_back(wp(0) + mtow(0,1)*ky);
    m_tripod_verts.push_back(wp(1) + mtow(1,1)*ky);
    m_tripod_verts.push_back(wp(2) + mtow(2,1)*ky);

    m_tripod_verts.push_back(wp(0) + mtow(0,1)*ky);
    m_tripod_verts.push_back(wp(1) + mtow(1,1)*ky);
    m_tripod_verts.push_back(wp(2) + mtow(2,1)*ky);

    m_tripod_verts.push_back(wp(0) + mtow(0,1)*ky*0.9 + mtow(0,0)*kx*0.1);
    m_tripod_verts.push_back(wp(1) + mtow(1,1)*ky*0.9 + mtow(1,0)*kx*0.1);
    m_tripod_verts.push_back(wp(2) + mtow(2,1)*ky*0.9 + mtow(2,0)*kx*0.1);

    m_tripod_verts.push_back(wp(0) + mtow(0,1)*ky);
    m_tripod_verts.push_back(wp(1) + mtow(1,1)*ky);
    m_tripod_verts.push_back(wp(2) + mtow(2,1)*ky);

    m_tripod_verts.push_back(wp(0) + mtow(0,1)*ky*0.9 - mtow(0,0)*kx*0.1);
    m_tripod_verts.push_back(wp(1) + mtow(1,1)*ky*0.9 - mtow(1,0)*kx*0.1);
    m_tripod_verts.push_back(wp(2) + mtow(2,1)*ky*0.9 - mtow(2,0)*kx*0.1);

    for( int i = 0; i < 6; ++i )
    {
        m_tripod_colors.push_back(0);
        m_tripod_colors.push_back(255);
        m_tripod_colors.push_back(0);
    }

    // Z axis
    m_tripod_verts.push_back(wp(0));
    m_tripod_verts.push_back(wp(1));
    m_tripod_verts.push_back(wp(2));

    m_tripod_verts.push_back(wp(0) + mtow(0,2)*kz);
    m_tripod_verts.push_back(wp(1) + mtow(1,2)*kz);
    m_tripod_verts.push_back(wp(2) + mtow(2,2)*kz);

    m_tripod_verts.push_back(wp(0) + mtow(0,2)*kz);
    m_tripod_verts.push_back(wp(1) + mtow(1,2)*kz);
    m_tripod_verts.push_back(wp(2) + mtow(2,2)*kz);

    m_tripod_verts.push_back(wp(0) + mtow(0,2)*kz*0.9 + mtow(0,0)*kx*0.1);
    m_tripod_verts.push_back(wp(1) + mtow(1,2)*kz*0.9 + mtow(1,0)*kx*0.1);
    m_tripod_verts.push_back(wp(2) + mtow(2,2)*kz*0.9 + mtow(2,0)*kx*0.1);

    m_tripod_verts.push_back(wp(0) + mtow(0,2)*kz);
    m_tripod_verts.push_back(wp(1) + mtow(1,2)*kz);
    m_tripod_verts.push_back(wp(2) + mtow(2,2)*kz);

    m_tripod_verts.push_back(wp(0) + mtow(0,2)*kz*0.9 - mtow(0,0)*kx*0.1);
    m_tripod_verts.push_back(wp(1) + mtow(1,2)*kz*0.9 - mtow(1,0)*kx*0.1);
    m_tripod_verts.push_back(wp(2) + mtow(2,2)*kz*0.9 - mtow(2,0)*kx*0.1);

    for( int i = 0; i < 6; ++i )
    {
        m_tripod_colors.push_back(0);
        m_tripod_colors.push_back(0);
        m_tripod_colors.push_back(255);
    }
}


void RenderTraverser::render_tripods() throw()
{
    boost::shared_ptr<VertexBuffer> vb =
        boost::shared_ptr<VertexBuffer>(new VertexBuffer);
    vb->set_usage_hint(VertexBuffer::STREAM_DRAW);
    boost::shared_ptr<VertexBuffer> cb =
        boost::shared_ptr<VertexBuffer>(new VertexBuffer);
    cb->set_usage_hint(VertexBuffer::STREAM_DRAW);

    std::size_t n = m_tripod_verts.size() / 6;

    vb->set_data(&m_tripod_verts[0], 3*2*n*sizeof(float));
    m_tripod_verts.clear();

    cb->set_data(&m_tripod_colors[0], 3*2*n*sizeof(float));
    m_tripod_colors.clear();

    LineMesh m(
        vb, cb,
        boost::shared_ptr<VertexBuffer>(),
        boost::shared_ptr<IndexBuffer>(),
        0, n);

    m.get_state().set(new statelets::DepthTest(statelets::DepthTest::ALWAYS));
    m.get_state().set(new statelets::LineParams(2, false, 0, 0));

    m.get_state().apply();
    m.render(m_context);
}
