/*
 * Copyright Staffan Gimåker 2006-2009.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */


#ifndef PEEKABOT_RENDERER_OCTREE_HH_INCLUDED
#define PEEKABOT_RENDERER_OCTREE_HH_INCLUDED


#include <Eigen/Core>
#include <boost/unordered_set.hpp>
#include <boost/unordered_map.hpp>


namespace peekabot
{
    namespace renderer
    {
        class CullableEntity;
        class SceneTraverser;
        class Camera;
        class Frustum;

        /**
         * \brief A loose octree implementation.
         *
         * The octree divides the populated space, in its entirety, into an 
         * hierarchy of axis-aligned bounding boxes (cubes), residing in world space
         * coordinates. Each cube is represented by an \c OctreeNode.
         *
         * It is implemented to be fast, but not extravagantly gluttonous for memory. 
         * In practice, it means that the octree is \e not fully expanded. Rather,
         * it is subdivided and modified as needed, using only as much memory as
         * it actually needs.
         *
         * There is presently only one parameter that control subdivision depth. 
         * The \e minimum size of a node is set upon constructing the octree. 
         * It is then guaranteed that the tree  will never produce nodes smaller
         * than set by subdividing itself.
         *
         * \e Size and \e half-size always refers to the bounding cubes' side
         * length (or half side length), never the shortest distance to the cube
         * from it's center point or anything such.
         *
         * \remarks The octree is initially centered around the world origin, but is
         * dynamically extended as needed and best suited.
         * \remarks The <em>loose octree</em> was invented and dubbed by Thatcher 
         * Ulrich, refer to his website for tidbits of good information.
         */
        class Octree
        {
            struct OctreeNode;

        public:
            Octree(float initial_half_size, float min_node_half_size);

            ~Octree();

            void add(const CullableEntity *e);

            void remove(const CullableEntity *e);

            bool contains(const CullableEntity *e) const throw();

            void update(const CullableEntity *e);

            void traverse(
                const Camera &camera,
                SceneTraverser &traverser) throw();

        private:
            OctreeNode *grow_to_accomodate(const CullableEntity *e);

            void trickle_down(const CullableEntity *e);

            static bool is_leaf(const OctreeNode *node) throw();

            static bool is_root(const OctreeNode *node) throw();

            static bool is_superfluous(const OctreeNode *node) throw();

            static void remove_superfluous(OctreeNode *node);

            static bool encloses(
                OctreeNode *node, const CullableEntity *e) throw();

            static bool enclosed_in_octant(
                OctreeNode *node, int octant, const CullableEntity *e) throw();

            inline static Eigen::Vector3f octant_offset(OctreeNode *node, int octant) throw()
            {
                return octant_offset(node->m_half_size, octant);
            }

            static Eigen::Vector3f octant_offset(float half_size, int octant) throw();

            void traverse(
                OctreeNode *node,
                const Frustum &camera,
                SceneTraverser &traverser, 
                const int order[],
                int plane_mask);

            static void traverse_no_cull(
                OctreeNode *node,
                SceneTraverser &traverser,
                const int order[]);

            static void object_level_cull(
                const CullableEntity *e, 
                const Frustum &view_frustum,
                SceneTraverser &traverser,
                int plane_mask);

        private:
            struct OctreeNode
            {
                OctreeNode(
                    OctreeNode *parent,
                    float half_size,
                    const Eigen::Vector3f &pos);

                ~OctreeNode();

                OctreeNode *m_parent;

                const float m_half_size;
                const Eigen::Vector3f m_pos;

                OctreeNode *m_children[8];

                typedef boost::unordered_set<const CullableEntity *> ContainedSet;
                ContainedSet m_contained;
            };

            const float m_min_node_half_size;

            OctreeNode *m_root;

            typedef boost::unordered_map<const CullableEntity *, OctreeNode *> EntityNodeMap;
            EntityNodeMap m_node_map;
        };
    }
}


#endif // PEEKABOT_RENDERER_OCTREE_HH_INCLUDED
