/*
 * Copyright Staffan Gimåker 2006-2009.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <iostream>
#include <vector>

#include <cmath>
#include <cassert>
#include <stdexcept>

#include "Octree.hh"
#include "CullableEntity.hh"
#include "Frustum.hh"
#include "entities/Camera.hh"
#include "SceneTraverser.hh"


using namespace peekabot;
using namespace peekabot::renderer;



//
// ------------------------ OctreeNode implementation ------------------------
//


Octree::OctreeNode::OctreeNode(
    OctreeNode *parent,
    float half_size,
    const Eigen::Vector3f &pos)
    : m_parent(parent),
      m_half_size(half_size),
      m_pos(pos)
{
    for( size_t i = 0; i < 8; ++i )
        m_children[i] = 0;
}


Octree::OctreeNode::~OctreeNode()
{
    for( size_t i = 0; i < 8; ++i )
    {
        if( m_children[i] )
        {
            m_children[i]->m_parent = 0;
            delete m_children[i];
            m_children[i] = 0;
        }
    }
}


//
// -------------------------- Octree implementation --------------------------
//


Octree::Octree(float initial_half_size, float min_node_half_size)
    : m_min_node_half_size(min_node_half_size),
      m_root(new OctreeNode(0, initial_half_size, Eigen::Vector3f(0,0,0)))
{
}


Octree::~Octree()
{
    delete m_root;
    m_root = 0;
}


void Octree::add(const CullableEntity *e)
{
    if( contains(e) )
        throw std::runtime_error(
            "The octree already contains the given entity");

    if( encloses(m_root, e) )
    {
        m_node_map[e] = m_root;
        m_root->m_contained.insert(e);
    }
    else
    {
        m_node_map[e] = grow_to_accomodate(e);
        m_node_map[e]->m_contained.insert(e);
    }
}


void Octree::remove(const CullableEntity *e)
{
    if( !contains(e) )
        throw std::runtime_error(
            "The octree doesn't contain the given entity");

    OctreeNode *node = m_node_map[e];
    node->m_contained.erase(e);
    m_node_map.erase(e);

    remove_superfluous(node);
}


bool Octree::contains(const CullableEntity *e) const throw()
{
    return m_node_map.find(e) != m_node_map.end();
}


void Octree::update(const CullableEntity *e)
{
    EntityNodeMap::iterator node_map_it = m_node_map.find(e);
    //assert( node_map_it != m_node_map.end() );

    OctreeNode *prev_node = node_map_it->second;

    // If it's still enclosed in the same node (tightly or not) there is
    // nothing more to do here... if it's not, let it "trickle" up the tree to
    // a fit where it fits
    if( !encloses(prev_node, e) )
    {
        // find ancestor of prev_node that encloses e
        OctreeNode *node = prev_node->m_parent;
        while( node != 0 && !encloses(node, e) )
            node = node->m_parent;

        // invariant: node encloses e, and null if it doesn't
        // node = null => the e is outside the octree (omg!)

        prev_node->m_contained.erase(e);

        if( node != 0 )
        {
            node_map_it->second = node;
            node->m_contained.insert(e);
        }
        else
        {
            node_map_it->second = grow_to_accomodate(e);
            node_map_it->second->m_contained.insert(e);
        }

        if( is_superfluous(prev_node) )
            remove_superfluous(prev_node);
    }
}


Octree::OctreeNode *Octree::grow_to_accomodate(const CullableEntity *e)
{
    // e is outside the octree!
    // deal with special case... by extending the tree outwards,
    // that is, by create a new big node and use the current
    // root node as a child octant in the new node.
    do
    {
        assert( !encloses(m_root, e) );

        OctreeNode *old_root = m_root;

        Eigen::Vector3f configuration[] =
        {
            Eigen::Vector3f(-1, -1, -1),
            Eigen::Vector3f( 1, -1,  1),
            Eigen::Vector3f(-1,  1, -1),
            Eigen::Vector3f( 1,  1,  1),
            Eigen::Vector3f(-1, -1, -1),
            Eigen::Vector3f( 1, -1,  1),
            Eigen::Vector3f(-1,  1, -1),
            Eigen::Vector3f( 1,  1,  1)
        };

        float min_dist = 0;
        int min_conf = -1;

        for( int i = 0; i < 8; i++ )
        {
            Eigen::Vector3f confpos = old_root->m_pos + (
                old_root->m_half_size/2)*configuration[i];

            float d = (confpos - e->get_bounding_sphere_wc().get_pos()).norm();

            if( min_conf == -1 || d < min_dist )
            {
                min_conf = i;
                min_dist = d;
            }
        }

        // Now we just have to figure out in which directions
        // to extend the octree.
        // This can easily be done by find the configuration p
        // which minimizes the distance dist(e,p). There
        // are 8 possible configurations.
        //
        // The different possible configurations are the 
        // permutations of <x,y,z>, x,y,z \in {-1,1}. I.e.
        // all the ways the new cube can be offset from the
        // old root node.
        //
        // Having found the best configuration, we implicitly
        // also have the index of the new root node to put the 
        // old one in.
        // index = (x/2+0.5)+2*(y/2+0.5)+4*(z/2+0.5)

        assert( min_conf != -1 );

        m_root = new OctreeNode(
            0, 
            old_root->m_half_size*2, 
            old_root->m_pos + old_root->m_half_size/2*configuration[min_conf]);

        int index = -1;
        for( int i = 0; i < 8; ++i )
        {
            if( (octant_offset(m_root, i) + m_root->m_pos).isApprox(
                    old_root->m_pos, m_root->m_half_size * 1e-6) )
            {
                index = i;
                break;
            }
        }

        assert( index != -1 );

        m_root->m_children[index] = old_root;
        old_root->m_parent = m_root;

        remove_superfluous(old_root);
    }
    while( !encloses(m_root, e) );

    return m_root;
}


void Octree::traverse(
    const Camera &camera,
    SceneTraverser &traverser) throw()
{
    Eigen::Vector3f view_dir(
        camera.get_transformation()(0,0),
        camera.get_transformation()(1,0),
        camera.get_transformation()(2,0));
    std::vector<std::pair<float, int> > tmp(8);

    for( int i = 0; i < 8; i++ )
    {
        float z_dist = view_dir.dot(view_dir + octant_offset(1, i));
        tmp[i] = std::make_pair(z_dist, i);
    }

    std::sort(tmp.begin(), tmp.end());

    int order[8];

    for( int i = 0; i < 8; i++ )
    {
        order[i] = tmp[i].second;
    }

    const boost::shared_ptr<Frustum> view_frustum(camera.get_frustum());
    traverse(m_root, *view_frustum.get(), traverser, order, 0x3F);
}


void Octree::traverse(
    OctreeNode *node,
    const Frustum &view_frustum, 
    SceneTraverser &traverser, 
    const int order[],
    int plane_mask)
{
    // node_bs = the "full-sized" node bsphere
    BoundingSphere node_bs(node->m_pos, node->m_half_size*sqrtf(3));

    // Cull the node?
    IntersectionTestResult culltest = node_bs.contained_in(
        view_frustum.get_bounding_sphere());

    // Refine node cull test?
    if( culltest != DISJOINT )
        culltest = view_frustum.is_in_frustum(node_bs, plane_mask);


    if( culltest == DISJOINT )
    {
        // Cull all child nodes!
        return;
    }
    else if( culltest == INSIDE )
    {
        traverse_no_cull(node, traverser, order);
    }
    else // culltest == INTERSECT
    {
        // Partial intersection! Do the following:
        // 0. Trickle "dirty" objects down the tree
        // 1. Do object level tests for contained, non-dirty objects
        // 2. Traverse child nodes

        for( OctreeNode::ContainedSet::const_iterator it = node->m_contained.begin();
             it != node->m_contained.end(); )
        {
            OctreeNode::ContainedSet::const_iterator curr_it = it;
            ++it;

            if( (*curr_it)->get_bounding_sphere_wc().get_radius() <= node->m_half_size/2 &&
                node->m_half_size/2 >= m_min_node_half_size )
            {
                // Trickle it down the tree!
                trickle_down(*curr_it);
            }
            else
            {
                object_level_cull(*curr_it, view_frustum, traverser, plane_mask);
            }
        }


        for( size_t i = 0; i < 8; i++ )
        {
            OctreeNode *child = node->m_children[order[i]];
            if( child )
                traverse(child, view_frustum, traverser, order, plane_mask);
        }
    }
}


void Octree::trickle_down(const CullableEntity *e)
{
    EntityNodeMap::iterator node_map_it = m_node_map.find(e);
    //assert( node_map_it != m_node_map.end() );

    OctreeNode *prev_node = node_map_it->second;
    OctreeNode *node = 0;

    //assert( encloses(prev_node, e) );

    for( size_t i = 0; i < 8; i++ )
    {
        if( enclosed_in_octant(prev_node, i, e) )
        {
            if( prev_node->m_children[i] == 0 )
            {
                // invariant: m_half_size/2 > m_min_node_half_size
                prev_node->m_children[i] = new OctreeNode(
                    prev_node, prev_node->m_half_size/2,
                    prev_node->m_pos + octant_offset(prev_node, i));
            }

            node = prev_node->m_children[i];
            break;
        }
    }

    if( !node )
    {
        TheMessageHub::instance().publish(
            ERROR_MESSAGE,
            "Pending assertion failure! Object not enclosed in a child node, "
            "even though it should meet the requirements to do so. "
            "bounding sphere: pos=(%f, %f, %f), r=%f", 
            e->get_bounding_sphere_wc().get_pos()(0),
            e->get_bounding_sphere_wc().get_pos()(1),
            e->get_bounding_sphere_wc().get_pos()(2),
            e->get_bounding_sphere_wc().get_radius());

        assert(false); // Should never happen
    }

    //assert( encloses(node, e) );

    prev_node->m_contained.erase(e);
    node_map_it->second = node;
    node->m_contained.insert(e);
}


void Octree::traverse_no_cull(
    OctreeNode *node,
    SceneTraverser &traverser,
    const int order[])
{
    for( OctreeNode::ContainedSet::const_iterator it = node->m_contained.begin();
         it != node->m_contained.end(); ++it )
    {
        traverser.traverse(*it);
    }

    for( size_t i = 0; i < 8; i++ )
    {
        OctreeNode *child = node->m_children[order[i]];
        if( child )
            traverse_no_cull(child, traverser, order);
    }
}


void Octree::object_level_cull(
    const CullableEntity *e, 
    const Frustum &view_frustum,
    SceneTraverser &traverser,
    int plane_mask)
{
    IntersectionTestResult culltest = e->get_bounding_sphere_wc().contained_in(
        view_frustum.get_bounding_sphere());

    if( culltest != DISJOINT )
        culltest = view_frustum.is_in_frustum(
            e->get_bounding_sphere_wc(), plane_mask);

    if( culltest != DISJOINT )
        traverser.traverse(e);
}



bool Octree::encloses(OctreeNode *node, const CullableEntity *e) throw()
{
    if( e->get_bounding_sphere_wc().get_radius() > node->m_half_size )
        return false;

    const float x = e->get_bounding_sphere_wc().get_pos()(0);
    const float y = e->get_bounding_sphere_wc().get_pos()(1);
    const float z = e->get_bounding_sphere_wc().get_pos()(2);

    const float qs = node->m_half_size/2;

    return ( x >= node->m_pos(0) - qs && x < node->m_pos(0) + qs &&
             y >= node->m_pos(1) - qs && y < node->m_pos(1) + qs &&
             z >= node->m_pos(2) - qs && z < node->m_pos(2) + qs );
}


bool Octree::enclosed_in_octant(
    OctreeNode *node, int octant, const CullableEntity *e) throw()
{
    if( e->get_bounding_sphere_wc().get_radius() > node->m_half_size/2 )
        return false;

    const float x = e->get_bounding_sphere_wc().get_pos()(0);
    const float y = e->get_bounding_sphere_wc().get_pos()(1);
    const float z = e->get_bounding_sphere_wc().get_pos()(2);

    return ( ( octant&1 ? x >= node->m_pos(0) : x < node->m_pos(0) ) && 
             ( octant&2 ? y >= node->m_pos(1) : y < node->m_pos(1) ) && 
             ( octant&4 ? z >= node->m_pos(2) : z < node->m_pos(2) ) );
}


Eigen::Vector3f Octree::octant_offset(float half_size, int octant) throw()
{
    float k = half_size/4;

    return Eigen::Vector3f(
        k*(2*(octant&1)-1),
        k*((octant&2)-1),
        k*(((octant&4)>>1)-1) );
}


bool Octree::is_leaf(const OctreeNode *node) throw()
{
    for( size_t i = 0; i < 8; ++i )
        if( node->m_children[i] )
            return false;
    return true;
}


bool Octree::is_root(const OctreeNode *node) throw()
{
    return node->m_parent == 0;
}


bool Octree::is_superfluous(const OctreeNode *node) throw()
{
    return node->m_contained.empty() && is_leaf(node) && !is_root(node);
}


void Octree::remove_superfluous(OctreeNode *node)
{
    if( is_superfluous(node) )
    {
        OctreeNode *tmp1 = node;
        OctreeNode *tmp2 = node->m_parent;

        while( is_superfluous(tmp2) )
        {
            tmp1 = tmp2;
            tmp2 = tmp1->m_parent;
        }

        for( size_t i = 0; i < 8; ++i )
            if( tmp2->m_children[i] == tmp1 )
                tmp2->m_children[i] = 0;

        delete tmp1;
    }
}
