/*
 * Copyright Staffan Gimåker 2007-2008, 2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef PEEKABOT_RENDERER_MESH_HH_INCLUDED
#define PEEKABOT_RENDERER_MESH_HH_INCLUDED


#include <stdexcept>
#include <boost/shared_ptr.hpp>

#include "../Types.hh"
#include "VertexBuffer.hh"
#include "IndexBuffer.hh"
#include "Renderable.hh"


namespace peekabot
{
namespace renderer
{
    class MeshVisitor;



    /**
     * \brief Mesh superclass for indexed and unindexed geometry.
     */
    class Mesh : public Renderable
    {
    public:
        Mesh();

        Mesh(boost::shared_ptr<VertexBuffer> vertices,
             boost::shared_ptr<VertexBuffer> colors,
             boost::shared_ptr<VertexBuffer> tex_coords,
             boost::shared_ptr<IndexBuffer> indices,
             uint32_t index_offset,
             uint32_t element_count);

        Mesh(const Mesh &mesh);

        virtual ~Mesh() {}

        virtual Mesh *clone() const = 0;

        bool has_vertices() const throw();

        boost::shared_ptr<VertexBuffer> &get_vertices() throw(std::runtime_error);

        const boost::shared_ptr<VertexBuffer> &get_vertices() const throw(std::runtime_error);

        bool has_colors() const throw();

        boost::shared_ptr<VertexBuffer> &get_colors() throw(std::runtime_error);

        const boost::shared_ptr<VertexBuffer> &get_colors() const throw(std::runtime_error);

        bool has_tex_coords() const throw();

        boost::shared_ptr<VertexBuffer> &get_tex_coords() throw(std::runtime_error);

        const boost::shared_ptr<VertexBuffer> &get_tex_coords() const throw(std::runtime_error);

        void set_vertices(
            const boost::shared_ptr<VertexBuffer> &buffer,
            std::size_t stride = 0, std::size_t offset = 0);

        void set_colors(
            const boost::shared_ptr<VertexBuffer> &buffer,
            std::size_t stride = 0, std::size_t offset = 0);

        void set_tex_coords(boost::shared_ptr<VertexBuffer> tex_coords) throw();

        inline uint32_t get_element_count() const throw()
        {
            return m_element_count;
        }

        void set_element_count(uint32_t element_count) throw();
       
        inline boost::shared_ptr<IndexBuffer> &get_indices() throw()
        {
            return m_indices;
        }

        void set_indices(boost::shared_ptr<IndexBuffer> indices)
        {
            m_indices = indices;
        }

        inline const boost::shared_ptr<IndexBuffer> &get_indices() const throw()
        {
            return m_indices;
        }

        inline bool has_indices() const throw()
        {
            return m_indices;
        }

        inline uint32_t get_index_offset() const throw()
        {
            return m_index_offset;
        }

        void set_index_offset(uint32_t offset) throw();

        virtual uint32_t vertices_per_element() const throw() = 0;


        virtual void accept(MeshVisitor &visitor) throw() = 0;

    private:
        boost::shared_ptr<IndexBuffer> m_indices;
        uint32_t m_index_offset;
        uint32_t m_element_count;
    };


    class TriMesh : public Mesh
    {
    public:
        TriMesh();

        TriMesh(boost::shared_ptr<VertexBuffer> vertices,
                boost::shared_ptr<VertexBuffer> normals,
                boost::shared_ptr<VertexBuffer> colors,
                boost::shared_ptr<VertexBuffer> tex_coords,
                boost::shared_ptr<IndexBuffer> indices,
                uint32_t index_offset, 
                uint32_t element_count);

        TriMesh(const TriMesh &mesh);

        virtual TriMesh *clone() const throw();

        bool has_normals() const throw();

        boost::shared_ptr<VertexBuffer> &get_normals() throw(std::runtime_error);

        const boost::shared_ptr<VertexBuffer> &get_normals() const throw(std::runtime_error);

        virtual uint32_t vertices_per_element() const throw();

        virtual void render(RenderContext &context) const;

        void set_normals(
            const boost::shared_ptr<VertexBuffer> &buffer,
            std::size_t stride = 0, std::size_t offset = 0);

        virtual void accept(MeshVisitor &visitor) throw();
    };


    class LineMesh : public Mesh
    {
    public:
        LineMesh();

        LineMesh(boost::shared_ptr<VertexBuffer> vertices,
                 boost::shared_ptr<VertexBuffer> colors,
                 boost::shared_ptr<VertexBuffer> tex_coords,
                 boost::shared_ptr<IndexBuffer> indices,
                 uint32_t index_offset, 
                 uint32_t element_count);

        LineMesh(const LineMesh &mesh);

        virtual LineMesh *clone() const throw();

        virtual uint32_t vertices_per_element() const throw();

        virtual void render(RenderContext &context) const;

        virtual void accept(MeshVisitor &visitor) throw();
    };


    class LineStripMesh : public Mesh
    {
    public:
        LineStripMesh();

        LineStripMesh(
            boost::shared_ptr<VertexBuffer> vertices,
            boost::shared_ptr<VertexBuffer> colors,
            boost::uint32_t element_count);

        LineStripMesh(const LineStripMesh &mesh);

        virtual LineStripMesh *clone() const throw();

        virtual boost::uint32_t vertices_per_element() const throw();

        virtual void render(RenderContext &context) const;

        virtual void accept(MeshVisitor &visitor) throw();
    };


    class PointMesh : public Mesh
    {
    public:
        PointMesh();

        PointMesh(boost::shared_ptr<VertexBuffer> vertices,
                  boost::shared_ptr<VertexBuffer> colors,
                  uint32_t element_count);

        PointMesh(const PointMesh &mesh);

        virtual PointMesh *clone() const throw();

        virtual uint32_t vertices_per_element() const throw();

        virtual void render(RenderContext &context) const;

        virtual void accept(MeshVisitor &visitor) throw();
    };

}
}


#endif // PEEKABOT_RENDERER_MESH_HH_INCLUDED
