/*
 * Copyright Staffan Gimåker 2006-2009.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef PEEKABOT_RENDERER_BOUNDING_VOLUMES_HH_INCLUDED
#define PEEKABOT_RENDERER_BOUNDING_VOLUMES_HH_INCLUDED


#include <Eigen/Core>


namespace peekabot
{
namespace renderer
{
    class BoundingSphere;
    class BoundingCone;
    class Plane;
    class AABB;
    class OBB;

    /**
     * \brief Trinary result for intersection tests.
     */
    enum IntersectionTestResult
    {
        /**
         * \brief The volume is fully contained in the volume it was
         * tested against. I.e., it is a subset of the other volume.
         *
         * When testing against a plane, this enumerator signifies that the
         * object lies on plane's negative normal side, without intersecting
         * the plane.
         */
        INSIDE = 2,
        /**
         * \brief The volume is partially contained in the volume it was
         * tested against. I.e., the volumes intersect.
         */
        INTERSECT = 1,
        /**
         * \brief The two tested volumes are disjoint.
         */
        DISJOINT = 0
    };


    /**
     * \brief Polymorphic bounding volume class.
     *
     * Provides (abstract) interfaces for testing a bounding volume (BV)
     * against a set of common bounding volumes.
     */
    class BoundingVolume
    {
    public:
        virtual ~BoundingVolume() {}

        virtual IntersectionTestResult intersect(const BoundingSphere &bv) const throw() = 0;
        virtual IntersectionTestResult intersect(const Plane &bv) const throw() = 0;
    };

    /*struct BVTests
    {
        static IntersectionTestResult intersect(const Sphere &contains, 
                                           const Vector4 &containee);
        static IntersectionTestResult intersect(const Sphere &contains, 
                                           const Vector4 &containee);
    };*/


    class BoundingSphere : public BoundingVolume
    {
    public:
        BoundingSphere() throw() {}

        BoundingSphere(const Eigen::Vector3f &pos, float r) throw()
            : m_pos(pos), m_r(r) {}

        inline float get_radius() const throw() { return m_r; }

        inline const Eigen::Vector3f &get_pos() const throw() { return m_pos; }

        inline void set_radius(float r) throw() { m_r = r; }

        inline void set_pos(const Eigen::Vector3f &pos) throw() { m_pos = pos; }

        IntersectionTestResult contained_in(const BoundingCone &container) const throw();
        IntersectionTestResult contained_in(const BoundingSphere &container) const throw();
        IntersectionTestResult contained_in(const Plane &container) const throw();

        virtual IntersectionTestResult intersect(const BoundingSphere &bv) const throw();
        virtual IntersectionTestResult intersect(const Plane &bv) const throw();

    private:
        Eigen::Vector3f m_pos;
        float m_r;
    };


    class BoundingCone : public BoundingVolume
    {
    public:
        BoundingCone() throw();

        BoundingCone(
            const Eigen::Vector3f &vertex,
            const Eigen::Vector3f &axis,
            float angle) throw();

        inline const Eigen::Vector3f &get_vertex() const throw()
        {
            return m_vertex;
        }

        inline void set_vertex(const Eigen::Vector3f &vertex) throw()
        {
            m_vertex = vertex;
        }

        inline const Eigen::Vector3f &get_axis() const throw()
        {
            return m_axis;
        }

        inline void set_axis(const Eigen::Vector3f &axis) throw()
        {
            m_axis = axis;
            m_axis.normalize();
        }

        inline void set_angle(float angle) throw()
        {
            m_angle = angle;
            m_sin_angle = sinf(angle);
            m_cos_angle = cosf(angle);
        }

        inline float get_angle() const throw()
        {
            return m_angle;
        }

        inline float get_sin_angle() const throw()
        {
            return m_sin_angle;
        }

        inline float get_cos_angle() const throw()
        {
            return m_sin_angle;
        }

        virtual IntersectionTestResult intersect(const BoundingSphere &bv) const throw();
        virtual IntersectionTestResult intersect(const Plane &bv) const throw();

    private:
        Eigen::Vector3f m_vertex;
        Eigen::Vector3f m_axis; // of unit length
        float m_angle; // in rad
        float m_sin_angle;
        float m_cos_angle;
    };


    class Plane : public BoundingVolume
    {
    public:
        Plane() throw() {}

        Plane(const Eigen::Vector3f &normal, float d) throw()
            : m_normal(normal), m_d(d) {}

        inline const Eigen::Vector3f &get_normal() const throw()
        {
            return m_normal;
        }

        inline void set_normal(const Eigen::Vector3f &normal) throw()
        {
            m_normal = normal;
        }

        inline void set_normal(float x, float y, float z) throw()
        {
            m_normal(0) = x; m_normal(1) = y; m_normal(2) = z;
        }

        inline float get_d() const throw()
        {
            return m_d;
        }

        inline void set_d(float d) throw()
        {
            m_d = d;
        }

        inline void normalize() throw()
        {
            m_normal.normalize();
        }

        virtual IntersectionTestResult intersect(const BoundingSphere &bv) const throw();
        virtual IntersectionTestResult intersect(const Plane &bv) const throw();

    private:
        Eigen::Vector3f m_normal;
        float m_d;
    };


    class Point : public BoundingVolume
    {
        Eigen::Vector3f m_pos;
    public:
    };

    class AABB : public BoundingVolume
    {
    };

    class OBB : public BoundingVolume
    {
    };


}
}


#endif // PEEKABOT_RENDERER_BOUNDING_VOLUMES_HH_INCLUDED
