/*
 * Copyright Staffan Gimåker 2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "SceneViewFrame.hh"
#include "FrameLayoutController.hh"
#include "MainWindowController.hh"
#include "ToolController.hh"
#include "Gui.hh"
#include "../Server.hh"
#include "RendererRelay.hh"
#include "../CameraObject.hh"
#include "../SceneTree.hh"
#include "../FsToolbox.hh"
#include "../Config.hh"
#include "../ObjectTypes.hh"
#include "Tool.hh"

#include <sstream>
#include <cstdio>
#include <cmath>
#include <boost/bind.hpp>
#include <boost/filesystem/operations.hpp>
#include <boost/foreach.hpp>


using namespace peekabot;
using namespace peekabot::gui;


SceneViewFrame::SceneViewFrame(FrameLayoutController &layout)
    : Frame(layout),
      m_gui(layout.get_gui()),
      m_gl_area(m_gui.get_gl_config(), true),
      m_cameras_model(Gtk::ListStore::create(m_camera_cols)),
      m_cameras(m_cameras_model),
      m_cam_id(0xFFFFFFFF),
      m_layer_table(2, 5, true)
{
    for( std::size_t i = 0; i < NUMBER_OF_LAYERS; ++i )
        m_layers[i] = (i == 0);

    setup_gl_area();
    set_main_widget(&m_gl_area);
    m_gl_area.show();

    toggle_toolbar_controls(false);

    setup_layer_ctrls();
    setup_screenshot_ctrls();
    setup_cam_ctrls();

    {
        m_tool_ortho.set_icon_name("peekabot-ortho");
        m_tool_ortho.set_label("Orthographic");
        prepend_toolbar_item(m_tool_ortho);
        m_tool_ortho.signal_toggled().connect(
            sigc::mem_fun(*this, &SceneViewFrame::on_ortho_toggled));
        m_tool_ortho.set_tooltip_text("Toggle orthographic mode");
    }

    {
        m_cameras_model->set_sort_column(0, Gtk::SORT_ASCENDING);
        m_cameras.pack_start(m_camera_cols.m_name);
        m_cameras.signal_changed().connect(
            sigc::mem_fun(*this, &SceneViewFrame::on_camera_changed));

        m_tool_cameras.add(m_cameras);
        m_tool_cameras.set_size_request(128, -1);
        prepend_toolbar_item(m_tool_cameras);
        m_tool_cameras.set_tooltip_text("The active viewport camera");

        const Gui::Cameras &cams = m_gui.get_cameras();
        for( Gui::Cameras::const_iterator it = cams.begin();
             it != cams.end(); ++it )
        {
            on_object_attached(it->first, it->second, CAMERA_OBJECT);
        }

        if( !cams.empty() )
            m_cameras.set_active(0);

        m_gui.object_attached_signal().connect(
            boost::bind(&SceneViewFrame::on_object_attached, this, _1, _3, _4));

        m_gui.object_detached_signal().connect(
            boost::bind(&SceneViewFrame::on_object_detached, this, _1));

        m_gui.object_renamed_signal().connect(
            boost::bind(&SceneViewFrame::on_object_renamed, this, _1, _2));
    }

    // Connect slots
    m_gui.selection_changed_signal().connect(
        boost::bind(&SceneViewFrame::on_selection_changed, this));
}


SceneViewFrame::~SceneViewFrame()
{
    if( m_view_mode_set_conn.connected() )
        m_view_mode_set_conn.disconnect();

    m_gui.object_attached_signal().disconnect(
        boost::bind(&SceneViewFrame::on_object_attached, this, _1, _3, _4));

    m_gui.object_detached_signal().disconnect(
        boost::bind(&SceneViewFrame::on_object_detached, this, _1));

    m_gui.object_renamed_signal().disconnect(
        boost::bind(&SceneViewFrame::on_object_renamed, this, _1, _2));

    m_gui.selection_changed_signal().disconnect(
        boost::bind(&SceneViewFrame::on_selection_changed, this));
}


void SceneViewFrame::setup_gl_area()
{
    // Drawing and context handling:
    m_gl_area.signal_expose_event().connect(
        sigc::mem_fun(*this, &SceneViewFrame::on_gl_area_expose_event));

    m_gl_area.set_can_focus(true);

    // Mouse motion etc.:
    m_gl_area.add_events(
        Gdk::POINTER_MOTION_MASK |
        Gdk::BUTTON_MOTION_MASK |
        Gdk::BUTTON_PRESS_MASK |
        Gdk::BUTTON_RELEASE_MASK |
        Gdk::KEY_RELEASE_MASK |
        Gdk::ENTER_NOTIFY_MASK |
        Gdk::SCROLL_MASK);

    m_gl_area.signal_motion_notify_event().connect(
        sigc::mem_fun(*this, &SceneViewFrame::on_gl_area_mouse_motion));
    m_gl_area.signal_scroll_event().connect(
        sigc::mem_fun(*this, &SceneViewFrame::on_gl_area_scroll));
    m_gl_area.signal_button_release_event().connect(
        sigc::mem_fun(*this, &SceneViewFrame::on_gl_area_button_release));
    m_gl_area.signal_button_press_event().connect(
        sigc::mem_fun(*this, &SceneViewFrame::on_gl_area_button_press));
    m_gl_area.signal_key_release_event().connect(
        sigc::mem_fun(*this, &SceneViewFrame::on_gl_area_key_release));
    m_gl_area.signal_enter_notify_event().connect(
        sigc::mem_fun(*this, &SceneViewFrame::on_gl_area_enter_notify));
}


void SceneViewFrame::setup_cam_ctrls()
{
    Gtk::MenuItem *item_reset_cam = manage(
        new Gtk::MenuItem("Reset camera"));
    m_menu_cam_ctrls.append(*item_reset_cam);
    item_reset_cam->signal_activate().connect(
        sigc::mem_fun(*this, &SceneViewFrame::on_reset_camera));
    item_reset_cam->set_tooltip_text(
        "Reset the camera to its original pose");

    m_center_on_selection_item.set_label("Center on selection");
    m_menu_cam_ctrls.append(m_center_on_selection_item);
    m_center_on_selection_item.set_sensitive(!m_gui.get_selection().empty());
    m_center_on_selection_item.signal_activate().connect(
        sigc::mem_fun(*this, &SceneViewFrame::on_center_on_selection));
    m_center_on_selection_item.set_tooltip_text(
        "Move the camera so that its pivot point is centered on "
        "the current selection");

    Gtk::SeparatorMenuItem *sep = manage(new Gtk::SeparatorMenuItem());
    m_menu_cam_ctrls.append(*sep);

    Gtk::MenuItem *item_pos_z = manage(new Gtk::MenuItem("Top view"));
    item_pos_z->signal_activate().connect(
        sigc::bind(
            sigc::mem_fun(*this, &SceneViewFrame::on_align_camera), 2, -1));
    m_menu_cam_ctrls.append(*item_pos_z);

    Gtk::MenuItem *item_neg_z = manage(new Gtk::MenuItem("Bottom view"));
    item_neg_z->signal_activate().connect(
        sigc::bind(
            sigc::mem_fun(*this, &SceneViewFrame::on_align_camera), 2, 1));
    m_menu_cam_ctrls.append(*item_neg_z);

    Gtk::MenuItem *item_pos_x = manage(new Gtk::MenuItem("Align to +X"));
    item_pos_x->signal_activate().connect(
        sigc::bind(
            sigc::mem_fun(*this, &SceneViewFrame::on_align_camera), 0, 1));
    m_menu_cam_ctrls.append(*item_pos_x);

    Gtk::MenuItem *item_neg_x = manage(new Gtk::MenuItem("Align to -X"));
    item_neg_x->signal_activate().connect(
        sigc::bind(
            sigc::mem_fun(*this, &SceneViewFrame::on_align_camera), 0, -1));
    m_menu_cam_ctrls.append(*item_neg_x);

    Gtk::MenuItem *item_pos_y = manage(new Gtk::MenuItem("Align to +Y"));
    item_pos_y->signal_activate().connect(
        sigc::bind(
            sigc::mem_fun(*this, &SceneViewFrame::on_align_camera), 1, 1));
    m_menu_cam_ctrls.append(*item_pos_y);

    Gtk::MenuItem *item_neg_y = manage(new Gtk::MenuItem("Align to -Y"));
    item_neg_y->signal_activate().connect(
        sigc::bind(
            sigc::mem_fun(*this, &SceneViewFrame::on_align_camera), 1, -1));
    m_menu_cam_ctrls.append(*item_neg_y);

    m_menu_cam_ctrls.show_all();

    m_tool_cam_ctrls.set_label("Top-down view");
    m_tool_cam_ctrls.set_icon_name("peekabot-top-down");
    m_tool_cam_ctrls.signal_clicked().connect(
        sigc::bind(
            sigc::mem_fun(*this, &SceneViewFrame::on_top_down),
            Eigen::Vector3f(0,0,-1)));

    m_tool_cam_ctrls.set_menu(m_menu_cam_ctrls);
    prepend_toolbar_item(m_tool_cam_ctrls);
}


void SceneViewFrame::setup_screenshot_ctrls()
{
    Gtk::Image *img = manage(new Gtk::Image());
    img->set_from_icon_name("peekabot-screenshot", Gtk::ICON_SIZE_MENU);

    Gtk::ImageMenuItem *item_screenshot = manage(
        new Gtk::ImageMenuItem(*img, "Save screenshot..."));
    m_menu_screenshot_ctrls.add(*item_screenshot);
    item_screenshot->set_tooltip_text(
        "Save a screenshot at a user-specified location and resolution");
    item_screenshot->signal_activate().connect(
        boost::bind(&SceneViewFrame::on_screenshot, this));

    m_tool_screenshot_ctrls.set_tooltip_text(
        "Save a screenshot of the current view, in the "
        "configured screenshot directory");
    m_tool_screenshot_ctrls.set_label("Quick-shot");
    m_tool_screenshot_ctrls.set_menu(m_menu_screenshot_ctrls);
    m_tool_screenshot_ctrls.set_icon_name("peekabot-screenshot");
    m_tool_screenshot_ctrls.signal_clicked().connect(
        boost::bind(&SceneViewFrame::on_quickshot, this));
    m_menu_screenshot_ctrls.show_all();

    prepend_toolbar_item(m_tool_screenshot_ctrls);
}


void SceneViewFrame::setup_layer_ctrls()
{
    m_layer_table.set_border_width(1);

    for( int row = 0; row < 2; ++row )
    {
        for( int col = 0; col < 5; ++col )
        {
            int idx = 5*row + col;
            m_layer_btn[idx].set_size_request(17, 17);
            m_layer_btn[idx].set_can_focus(false);
            m_layer_btn[idx].set_relief(Gtk::RELIEF_HALF);
            m_layer_btn[idx].signal_toggled().connect(
                sigc::bind(
                    sigc::mem_fun(*this, &SceneViewFrame::on_layer_toggled),
                    idx) );

            std::ostringstream ss;
            ss << "Layer " << idx + 1;
            m_layer_btn[idx].set_tooltip_text(ss.str());

            m_layer_table.attach(
                m_layer_btn[idx], col, col+1, row, row+1,
                Gtk::FILL, Gtk::FILL);
        }
    }

    m_tool_layers.add(m_layer_table);
    prepend_toolbar_item(m_tool_layers);

    for( std::size_t i = 0; i < NUMBER_OF_LAYERS; ++i )
        m_layer_btn[i].set_active(m_layers[i]);
}


bool SceneViewFrame::on_gl_area_expose_event(GdkEventExpose *event)
{
    if( m_cameras.get_active() )
    {
        Glib::RefPtr<Gdk::GL::Window> gl_window = m_gl_area.get_gl_window();
        assert( gl_window );

        if( !gl_window->gl_begin(m_gui.get_gl_context()) )
        {
            // TODO: handle/report error
            assert( false );
            return true;
        }

        assert( m_gui.get_renderer() );

        try
        {
            m_gui.get_renderer()->render(
                m_gl_area.get_width(), m_gl_area.get_height(),
                m_cam_id, m_layers);
        }
        catch(...)
        {
            // TODO: handle/report error
        }

        if( gl_window->is_double_buffered() )
            gl_window->swap_buffers();
        else
            glFlush();

        gl_window->gl_end();

        // HACK! This should be a no-op but for some reason inserting this code
        // makes the program not mysteriously crash on my Intel card with
        // Fedora 14.
        gl_window->make_current(m_gui.get_dummy_gl_context());
    }
    else
    {
        Cairo::RefPtr<Cairo::Context> cr =
            m_gl_area.get_window()->create_cairo_context();

        const int w = m_gl_area.get_width();
        const int h = m_gl_area.get_height();

        cr->set_source_rgb(0, 0, 0);
        cr->paint();

        cr->set_source_rgb(1, 1, 1);
        cr->set_font_size(20);
        Cairo::TextExtents te;
        const std::string str = "No cameras available";
        cr->get_text_extents(str, te);
        cr->move_to((w-te.width)/2, (h+te.height)/2);
        cr->show_text(str);
    }

    return true;
}


bool SceneViewFrame::on_gl_area_mouse_motion(GdkEventMotion *event)
{
    gdouble dx = event->x - m_pointer_x;
    gdouble dy = event->y - m_pointer_y;

    m_gui.get_tool_controller().get_active_tool().mouse_drag(
        this, event, dx, dy);

    m_pointer_x = event->x;
    m_pointer_y = event->y;

    return true;
}


bool SceneViewFrame::on_gl_area_scroll(GdkEventScroll *event)
{
    m_gui.get_tool_controller().get_active_tool().scroll(this, event);
    return true;
}


bool SceneViewFrame::on_gl_area_button_press(GdkEventButton *event)
{
    m_button_press_x[event->button-1] = event->x;
    m_button_press_y[event->button-1] = event->y;

    return true;
}


bool SceneViewFrame::on_gl_area_button_release(GdkEventButton *event)
{
    double dx = event->x - m_button_press_x[event->button-1];
    double dy = event->y - m_button_press_y[event->button-1];

    if( dx*dx + dy*dy <= 1 )
        m_gui.get_tool_controller().get_active_tool().button_click(this, event);

    m_gui.get_tool_controller().get_active_tool().button_release(this, event);

    return true;
}


void SceneViewFrame::toggle_ortho(ServerData &sd, ObjectID cam_id, bool ortho)
{
    CameraObject *cam = dynamic_cast<CameraObject *>(
        sd.m_scene->get_object(cam_id));

    if( cam )
    {
        cam->set_orthographic(ortho);
    }
}


void SceneViewFrame::set_camera_transform(
    ServerData &sd, ObjectID cam_id,
    Eigen::Transform3f trans)
{
    CameraObject *cam = dynamic_cast<CameraObject *>(
        sd.m_scene->get_object(cam_id));

    if( cam )
    {
        cam->set_transformation(trans);
    }
}


void SceneViewFrame::on_ortho_toggled()
{
    m_gui.server_post(
        boost::bind(&SceneViewFrame::toggle_ortho, this,
                    _1, m_cam_id, m_tool_ortho.get_active()));
}


void SceneViewFrame::on_reset_camera()
{
    Eigen::Transform3f tmp;
    tmp.setIdentity();

    m_gui.server_post(
        boost::bind(&SceneViewFrame::set_camera_transform, this,
                    _1, m_cam_id, tmp));
}


void SceneViewFrame::on_object_attached(
    ObjectID id, const std::string &name, ObjectType type)
{
    if( type != CAMERA_OBJECT )
        return;

    bool was_empty = m_cameras_model->children().size() == 0;

    Gtk::TreeModel::Row row = *(m_cameras_model->append());
    row[m_camera_cols.m_id] = id;
    row[m_camera_cols.m_name] = name;

    if( was_empty )
    {
        m_cam_id = id;
        m_cameras.set_active(0);
        toggle_toolbar_controls(true);
    }
}


void SceneViewFrame::on_object_detached(ObjectID id)
{
    Gtk::TreeModel::Children children = m_cameras_model->children();
    for( Gtk::TreeModel::iterator it = children.begin(); it != children.end(); ++it )
    {
        Gtk::TreeModel::Row row = *it;
        if( row[m_camera_cols.m_id] == id )
        {
            m_cameras_model->erase(it);
            break;
        }
    }

    if( m_cameras_model->children().empty() )
    {
        toggle_toolbar_controls(false);
    }
}


void SceneViewFrame::on_object_renamed(ObjectID id, const std::string &name)
{
    Gtk::TreeModel::Children children = m_cameras_model->children();
    for( Gtk::TreeModel::iterator it = children.begin(); it != children.end(); ++it )
    {
        Gtk::TreeModel::Row row = *it;
        if( row[m_camera_cols.m_id] == id )
        {
            row[m_camera_cols.m_name] = name;
            break;
        }
    }
}


void SceneViewFrame::on_camera_changed()
{
    if( m_view_mode_set_conn.connected() )
        m_view_mode_set_conn.disconnect();

    Gtk::TreeModel::iterator it = m_cameras.get_active();
    if( it )
    {
        Gtk::TreeModel::Row row = *it;
        if( row )
        {
            m_cam_id = row[m_camera_cols.m_id];

            m_gl_area.queue_draw();

            m_gui.server_post(
                boost::bind(&SceneViewFrame::connect_on_camera_view_mode_set,
                            this, _1, m_cam_id));
        }
    }
    else if( !m_cameras_model->children().empty() )
    {
        m_cameras.set_active(0);
    }
}


void SceneViewFrame::connect_on_camera_view_mode_set(
    ServerData &sd, ObjectID cam_id)
{
    CameraObject *cam = dynamic_cast<CameraObject *>(
        sd.m_scene->get_object(cam_id));

    if( cam )
    {
        if( m_view_mode_set_conn.connected() )
            m_view_mode_set_conn.disconnect();

        m_view_mode_set_conn = cam->ortho_set_signal().connect(
            boost::bind(&SceneViewFrame::on_camera_view_mode_set, this, cam));

        on_camera_view_mode_set(cam);
    }
}


void SceneViewFrame::on_camera_view_mode_set(CameraObject *cam)
{
    m_gui.post(
        boost::bind(&Gtk::ToggleToolButton::set_active,
                    &m_tool_ortho, cam->is_orthographic()));
}


void SceneViewFrame::on_screenshot()
{
    m_gui.get_main_window_controller().show_screenshot_dialog(
        m_cam_id, m_layers);
}


void SceneViewFrame::on_quickshot()
{
    int w = m_gl_area.get_width();
    int h = m_gl_area.get_height();

    if( w <= 0 || h <= 0 )
        return;

    m_gui.save_screenshot(
        w, h, m_cam_id, m_layers,
        generate_snapshot_filename("", ".png"));
}


std::string SceneViewFrame::generate_snapshot_filename(
    const std::string &prefix,
    const std::string &postfix) const
{
    boost::filesystem::path snapshot_dir(
        m_gui.get_config().get_option<boost::filesystem::path>(
            "resources.snapshot_path",
            fs::get_resource_path() / "snapshots"));

    static std::map<std::string, unsigned int> counter_map;
    char buf[256];
    boost::filesystem::path snapshot_path;
    do
    {
        sprintf(buf, "%s%04u%s",
                prefix.c_str(),
                ++counter_map[prefix],
                postfix.c_str());
        snapshot_path = snapshot_dir / buf;
    }
    while( boost::filesystem::exists(snapshot_path) );

    return snapshot_path.string();
}


void SceneViewFrame::redraw_scene_view()
{
    m_gl_area.queue_draw();
}


void SceneViewFrame::on_layer_toggled(int layer_no)
{
    m_layers[layer_no] = m_layer_btn[layer_no].get_active();
    redraw_scene_view();
}


void SceneViewFrame::on_center_on_selection()
{
    m_gui.server_post(
        boost::bind(&SceneViewFrame::center_on_selection_handler, this,
                    _1, m_cam_id, m_gui.get_selection()));
}


void SceneViewFrame::center_on_selection_handler(
    ServerData &sd, ObjectID cam_id,
    std::set<ObjectID> selection)
{
    CameraObject *cam = dynamic_cast<CameraObject *>(
        sd.m_scene->get_object(cam_id));

    if( cam )
    {
        Eigen::Vector3f pos(0,0,0);
        std::size_t n = 0;
        BOOST_FOREACH( ObjectID id, selection )
        {
            const SceneObject *obj = sd.m_scene->get_object(id);
            if( obj )
            {
                pos += obj->get_mtow().translation();
                ++n;
            }
        }

        if( n > 0 )
        {
            cam->set_world_position(pos);
        }
    }
}


void SceneViewFrame::on_selection_changed()
{
    m_center_on_selection_item.set_sensitive(!m_gui.get_selection().empty());
}


ObjectID SceneViewFrame::get_camera() const
{
    return m_cam_id;
}


ObjectID SceneViewFrame::pick(double x, double y)
{
    int w = m_gl_area.get_width();
    int h = m_gl_area.get_height();

    if( w <= 1 || h <= 1 )
        return 0;

    if( !m_gui.get_renderer() )
        return 0;

    Glib::RefPtr<Gdk::GL::Window> gl_window = m_gl_area.get_gl_window();
    if( !gl_window->gl_begin(m_gui.get_gl_context()) )
        return 0;

    ObjectID id = 0;
    try
    {
        id = m_gui.get_renderer()->pick(
            w, h, m_cam_id, m_layers,
            x, m_gl_area.get_height()-y-1,
            1.0);
    }
    catch(...)
    {
    }

    gl_window->gl_end();

    return id;
}


bool SceneViewFrame::on_gl_area_key_release(GdkEventKey *event)
{
    m_gui.get_tool_controller().get_active_tool().key_release(this, event);
    return true;
}


bool SceneViewFrame::on_gl_area_enter_notify(GdkEventCrossing *event)
{
    m_gl_area.grab_focus();
    return true;
}


void SceneViewFrame::toggle_toolbar_controls(bool sensitive)
{
    m_tool_ortho.set_sensitive(sensitive);
    m_tool_cam_ctrls.set_sensitive(sensitive);
    m_tool_screenshot_ctrls.set_sensitive(sensitive);
    m_tool_layers.set_sensitive(sensitive);
}


void SceneViewFrame::on_top_down(Eigen::Vector3f v)
{
    m_gui.server_post(
        boost::bind(
            &SceneViewFrame::ss_top_down, this, _1, m_cam_id, v));
}


void SceneViewFrame::ss_top_down(
    ServerData &sd, ObjectID cam_id, Eigen::Vector3f v)
{
    SceneObject *cam = sd.m_scene->get_object(cam_id);

    if( cam )
    {
        cam->set_orientation(v, WORLD_COORDINATES);
    }
}


void SceneViewFrame::on_align_camera(std::size_t dim, int sign)
{
    m_gui.server_post(
        boost::bind(
            &SceneViewFrame::ss_align_camera, this, _1, m_cam_id, dim, sign));
}


void SceneViewFrame::ss_align_camera(
    ServerData &sd, ObjectID cam_id, std::size_t dim, int sign)
{
    assert( sign == 1 || sign == -1 );
    assert( dim < 3 );

    SceneObject *cam = sd.m_scene->get_object(cam_id);

    if( cam )
    {
        Eigen::Transform3f m = cam->get_mtow();
        Eigen::Vector3f tmp(0,0,0);
        tmp(dim) = sign;

        m.linear().col(0) = tmp;
        m.linear().col(2) =
            (dim != 2) ? Eigen::Vector3f(0,0,1) : Eigen::Vector3f(1,0,0);
        m.linear().col(1) = m.linear().col(2).cross(tmp);

        cam->set_transformation(m, WORLD_COORDINATES);
    }
}
