/*
 * Copyright Staffan Gimåker 2006-2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "RendererRelay.hh"
#include "Gui.hh"
#include "WorkQueue.hh"
#include "../ObjectVisitor.hh"
#include "../Server.hh"
#include "../SceneTree.hh"

#include "../renderer/RenderEngine.hh"
#include "../renderer/ObjectEntity.hh"
#include "../renderer/entities/Camera.hh"
#include "../renderer/entities/Cube.hh"
#include "../renderer/entities/Polygon.hh"
#include "../renderer/entities/MeshBased.hh"
#include "../renderer/PrimitiveFactory.hh"
#include "../renderer/VertexBased.hh"
#include "../renderer/LineBased.hh"
#include "../renderer/entities/Polyline.hh"
#include "../renderer/entities/MeshEntities.hh"
#include "../renderer/entities/TriMeshEntity.hh"
#include "../renderer/entities/Label.hh"
#include "../renderer/entities/OccupancyGrid2D.hh"
#include "../renderer/entities/OccupancyGrid3D.hh"
#include "../renderer/PbmfLoader.hh"
#include "../renderer/entities/TransformationGuides.hh"
#include "../renderer/entities/Dummy.hh"

#include "../SceneObject.hh"
#include "../SphereObject.hh"
#include "../CylinderObject.hh"
#include "../CubeObject.hh"
#include "../CameraObject.hh"
#include "../GroupObject.hh"
#include "../CircleObject.hh"
#include "../ScalableObject.hh"
#include "../Hinge.hh"
#include "../Slider.hh"
#include "../ModelObject.hh"
#include "../GridObject.hh"
#include "../PointCloud.hh"
#include "../LineCloud.hh"
#include "../OccupancyGrid2D.hh"
#include "../OccupancyGrid3D.hh"
#include "../Label.hh"
#include "../TriMesh.hh"
#include "../Polyline.hh"
#include "../PolygonObject.hh"
#include "../LineBased.hh"

#include <set>
#include <map>
#include <vector>
#include <cassert>
#include <cmath>
#include <boost/bind.hpp>
#include <boost/foreach.hpp>


using namespace peekabot;
using namespace peekabot::gui;


class RendererRelay::Impl :
    private ObjectVisitor,
    public boost::noncopyable
{
public:
    Impl(Gui &gui)
        : m_gui(gui),
          m_renderer(0),
          m_guides(0)
    {
        m_work_queue.set_pre_processing_callback(
            boost::bind(&RendererRelay::Impl::pre_process_posts, this));

        m_work_queue.set_post_processing_callback(
            boost::bind(&RendererRelay::Impl::post_process_posts, this));
    }

    ~Impl()
    {
        // TODO: Clean up! We're leaking memory as it is now.

        if( m_fps_timeout_conn.connected() )
            m_fps_timeout_conn.disconnect();

        if( m_guides )
        {
            delete m_guides;
            m_guides = 0;
        }

        if( m_renderer )
        {
            delete m_renderer;
            m_renderer = 0;
        }
    }

    void render(
        unsigned int w, unsigned int h, ObjectID camera_id,
        const bool *visible_layers)
    {
        assert( m_renderer );

        renderer::Camera *cam = get_object_entity<renderer::Camera>(camera_id);

        if( !cam )
            throw std::runtime_error("No such camera");

        m_renderer->render(w, h, *cam, visible_layers);
    }

    ObjectID pick(
        unsigned int w, unsigned int h, ObjectID camera_id,
        const bool *visible_layers,
        int x, int y, float sensitivity)
    {
        assert( m_renderer );

        renderer::Camera *cam = get_object_entity<renderer::Camera>(camera_id);

        if( !cam )
            throw std::runtime_error("No such camera");

        return m_renderer->pick(
            w, h, *cam, visible_layers, x, y, sensitivity);
    }

    void init()
    {
        assert( !m_renderer );
        m_renderer = new renderer::RenderEngine();

        m_gui.gl_begin();

        m_renderer->init();
        m_renderer->init_context();

        m_guides = new renderer::TransformationGuides(
            m_renderer->get_scene_manager());
        m_renderer->get_scene_manager().add_entity(m_guides);

        m_gui.gl_end();

        m_gui.server_post(
            boost::bind(&RendererRelay::Impl::populate_scene, this, _1));

        m_work_queue.start();

        m_fps_t = boost::posix_time::microsec_clock::local_time();
        m_fps_counter = 0;
        m_fps_timeout_conn = Glib::signal_timeout().connect(
            sigc::mem_fun(*this, &RendererRelay::Impl::fps_timeout), 1000);
    }

    void set_tg_mode(renderer::TransformationGuidesMode mode)
    {
        assert( m_guides );
        post(
            boost::bind(
                &renderer::TransformationGuides::set_mode, m_guides, mode));
    }

    void set_tg_coord_sys(CoordinateSystem coord_sys)
    {
        assert( m_guides );
        post(
            boost::bind(
                &renderer::TransformationGuides::set_coord_sys,
                m_guides, coord_sys));
    }

    void set_tg_pivot(RotationCenterpoint pivot)
    {
        assert( m_guides );
        post(
            boost::bind(
                &renderer::TransformationGuides::set_pivot,
                m_guides, pivot));
    }

    void set_tg_axes(Axis axes)
    {
        assert( m_guides );
        post(
            boost::bind(
                &renderer::TransformationGuides::set_axes,
                m_guides, axes));
    }

    void set_max_fps(float fps)
    {
        m_work_queue.set_processing_freq(fps);
    }

    void set_background_color(const RGBColor &color)
    {
        // Note: Setting the background color does not require having a valid
        // OpenGL context so this is safe
        m_renderer->set_background_color(color);
        m_gui.redraw_scene_views();
    }

    void set_selected_color(const RGBColor &color)
    {
        // Note: Setting the selected color does not require having a valid
        // OpenGL context so this is safe
        m_renderer->set_selected_color(color);
        m_gui.redraw_scene_views();
    }

    void set_ancestor_selected_color(const RGBColor &color)
    {
        // Note: Setting the ancestor selected color does not require having a
        // valid OpenGL context so this is safe
        m_renderer->set_ancestor_selected_color(color);
        m_gui.redraw_scene_views();
    }

    FpsUpdatedSignal &fps_updated_signal() const
    {
        return m_fps_updated_signal;
    }

private:
    inline void post(const boost::function<void ()> &handler)
    {
        m_work_queue.post(handler);
    }

    void pre_process_posts()
    {
        assert( m_renderer );
        m_gui.gl_begin();
    }

    void post_process_posts()
    {
        m_gui.gl_end();
        m_gui.redraw_scene_views();
        ++m_fps_counter;
    }

    /// \name Methods executed in the server thread
    /// @{

    virtual void visit(CameraObject *obj)
    {
        post(
            boost::bind(
                &RendererRelay::Impl::add_generic<renderer::Camera>,
                this, obj->get_object_id()));

        connect_scene_object_slots(obj);

        record_connection(
            obj, obj->ortho_set_signal().connect(
                boost::bind(&RendererRelay::Impl::on_ortho_set, this, obj)));

        record_connection(
            obj, obj->fov_set_signal().connect(
                boost::bind(&RendererRelay::Impl::on_fov_set, this, obj)));

        record_connection(
            obj, obj->zoom_distance_set_signal().connect(
                boost::bind(
                    &RendererRelay::Impl::on_zoom_distance_set, this, obj)));

        record_connection(
            obj, obj->near_plane_set_signal().connect(
                boost::bind(
                    &RendererRelay::Impl::on_near_plane_set, this, obj)));

        record_connection(
            obj, obj->far_plane_set_signal().connect(
                boost::bind(
                    &RendererRelay::Impl::on_far_plane_set, this, obj)));

        on_ortho_set(obj);
        on_fov_set(obj);
        on_zoom_distance_set(obj);
        on_near_plane_set(obj);
        on_far_plane_set(obj);
    }

    void on_ortho_set(const CameraObject *obj)
    {
        post(
            boost::bind(
                &renderer::Camera::set_orthographic,
                boost::bind(
                    &RendererRelay::Impl::get_object_entity<renderer::Camera>,
                    this, obj->get_object_id()),
                obj->is_orthographic()));
    }

    void on_fov_set(const CameraObject *obj)
    {
        post(
            boost::bind(
                &renderer::Camera::set_fov,
                boost::bind(
                    &RendererRelay::Impl::get_object_entity<renderer::Camera>,
                    this, obj->get_object_id()),
                obj->get_fov()));
    }

    void on_zoom_distance_set(const CameraObject *obj)
    {
        post(
            boost::bind(
                &renderer::Camera::set_zoom_distance,
                boost::bind(
                    &RendererRelay::Impl::get_object_entity<renderer::Camera>,
                    this, obj->get_object_id()),
                obj->get_zoom_distance()));
    }

    void on_near_plane_set(const CameraObject *obj)
    {
        post(
            boost::bind(
                &renderer::Camera::set_near_plane,
                boost::bind(
                    &RendererRelay::Impl::get_object_entity<renderer::Camera>,
                    this, obj->get_object_id()),
                obj->get_near_plane()));
    }

    void on_far_plane_set(const CameraObject *obj)
    {
        post(
            boost::bind(
                &renderer::Camera::set_far_plane,
                boost::bind(
                    &RendererRelay::Impl::get_object_entity<renderer::Camera>,
                    this, obj->get_object_id()),
                obj->get_far_plane()));
    }

    virtual void visit(ModelObject *obj)
    {
        post(
            boost::bind(
                &RendererRelay::Impl::add_model,
                this, obj->get_object_id(), obj->get_model_path()));

        connect_scene_object_slots(obj);
    }

    // Executed in the GUI thread
    void add_model(ObjectID id, Path path)
    {
        renderer::PbmfLoader loader;
        add_object_entity(id, loader.load(path));
    }

    virtual void visit(SphereObject *obj)
    {
        post(
            boost::bind(
                &RendererRelay::Impl::add_sphere,
                this, obj->get_object_id()));

        connect_scene_object_slots(obj);
    }

    // Executed in the GUI thread
    void add_sphere(ObjectID id)
    {
        add_object_entity(
            id, renderer::ThePrimitiveFactory::instance().get_icosphere());
    }

    virtual void visit(GroupObject *obj)
    {
        post(
            boost::bind(
                &RendererRelay::Impl::add_generic<renderer::Dummy>,
                this, obj->get_object_id()));

        connect_scene_object_slots(obj);
    }

    virtual void visit(CubeObject *obj)
    {
        post(
            boost::bind(
                &RendererRelay::Impl::add_generic<renderer::Cube>,
                this, obj->get_object_id()));

        connect_scene_object_slots(obj);
    }

    virtual void visit(PolygonObject *obj)
    {
        post(
            boost::bind(
                &RendererRelay::Impl::add_generic<renderer::Polygon>,
                this, obj->get_object_id()));

        connect_scene_object_slots(obj);
    }

    virtual void visit(CylinderObject *obj)
    {
        post(
            boost::bind(
                &RendererRelay::Impl::add_cylinder,
                this, obj->get_object_id()));

        connect_scene_object_slots(obj);
    }

    // Executed in the GUI thread
    void add_cylinder(ObjectID id)
    {
        add_object_entity(
            id, renderer::ThePrimitiveFactory::instance().get_cylinder());
    }

    virtual void visit(CircleObject *obj)
    {
        post(
            boost::bind(
                &RendererRelay::Impl::add_circle,
                this, obj->get_object_id()));

        connect_scene_object_slots(obj);
    }

    // Executed in the GUI thread
    void add_circle(ObjectID id)
    {
        add_object_entity(
            id, renderer::ThePrimitiveFactory::instance().get_circle());
    }

    virtual void visit(Polyline *obj)
    {
        post(
            boost::bind(
                &RendererRelay::Impl::add_generic<renderer::Polyline>,
                this, obj->get_object_id()));

        connect_scene_object_slots(obj);
    }

    virtual void visit(Hinge *obj)
    {
        // Joints act like groups, i.e. have no visual representation
        post(
            boost::bind(
                &RendererRelay::Impl::add_generic<renderer::Dummy>,
                this, obj->get_object_id()));

        connect_scene_object_slots(obj);
    }

    virtual void visit(Slider *obj)
    {
        // Joints act like groups, i.e. have no visual representation
        post(
            boost::bind(
                &RendererRelay::Impl::add_generic<renderer::Dummy>,
                this, obj->get_object_id()));

        connect_scene_object_slots(obj);
    }

    virtual void visit(GridObject *obj)
    {
        post(
            boost::bind(
                &RendererRelay::Impl::add_generic<renderer::LineMeshEntity>,
                this, obj->get_object_id()));

        connect_scene_object_slots(obj);

        record_connection(
            obj, obj->segment_count_set_signal().connect(
                boost::bind(
                    &RendererRelay::Impl::set_grid_vertices, this, obj)));

        record_connection(
            obj, obj->grid_type_set_signal().connect(
                boost::bind(
                    &RendererRelay::Impl::set_grid_vertices, this, obj)));

        record_connection(
            obj, obj->central_angle_set_signal().connect(
                boost::bind(
                    &RendererRelay::Impl::set_grid_vertices, this, obj)));

        set_grid_vertices(obj);
    }

    void set_grid_vertices(const GridObject *obj)
    {
        std::vector<Eigen::Vector3f> verts;
        boost::uint32_t segments = obj->get_segment_count();

        if( obj->get_grid_type() == REGULAR_GRID )
        {
            float foo = segments/2.0f;

            for( unsigned int i = 0; i < segments+1; ++i )
            {
                verts.push_back(Eigen::Vector3f(-foo+i, -foo, 0));
                verts.push_back(Eigen::Vector3f(-foo+i,  foo, 0));
            }
            for( unsigned int i = 0; i < segments+1; ++i )
            {
                verts.push_back(Eigen::Vector3f(-foo, -foo+i, 0));
                verts.push_back(Eigen::Vector3f( foo, -foo+i, 0));
            }
        }
        else if( obj->get_grid_type() == RADIAL_GRID )
        {
            float theta = obj->get_central_angle();
            float offset = -theta/2;

            for( unsigned int i = 0; i < segments; ++i )
            {
                unsigned int r = i+1;
                unsigned int circle_segments = static_cast<unsigned int>(
                    ceilf((64+10*i) * theta/M_PI/2));

                for( unsigned int j = 0; j < circle_segments; ++j )
                {
                    verts.push_back(
                        Eigen::Vector3f(r*cosf(theta*j/circle_segments+offset),
                                        r*sinf(theta*j/circle_segments+offset), 0));
                    verts.push_back(
                        Eigen::Vector3f(r*cosf(theta*(j+1)/circle_segments+offset),
                                        r*sinf(theta*(j+1)/circle_segments+offset), 0));
                }
            }
        }
        else if( obj->get_grid_type() == ANGULAR_GRID )
        {
            float theta = obj->get_central_angle();
            float offset = -theta/2;

            unsigned int lines;
            if( fabsf(theta-2*M_PI) < 0.000001 )
                lines = segments;
            else
                lines = segments+1;

            for( unsigned int i = 0; i < lines; ++i )
            {
                verts.push_back(Eigen::Vector3f(0,0,0));
                verts.push_back(
                    Eigen::Vector3f(cosf(theta*i/segments+offset),
                                    sinf(theta*i/segments+offset), 0));
            }
        }
        else
            assert( false );

        post(
            boost::bind(
                &renderer::VertexBased::set_vertices,
                boost::bind(
                    &RendererRelay::Impl::get_object_entity<renderer::VertexBased>,
                    this, obj->get_object_id()),
                verts));
    }

    virtual void visit(PointCloud *obj)
    {
        post(
            boost::bind(
                &RendererRelay::Impl::add_generic<renderer::PointMeshEntity>,
                this, obj->get_object_id()));

        connect_scene_object_slots(obj);
    }

    virtual void visit(LineCloud *obj)
    {
        post(
            boost::bind(
                &RendererRelay::Impl::add_generic<renderer::LineMeshEntity>,
                this, obj->get_object_id()));

        connect_scene_object_slots(obj);
    }

    virtual void visit(OccupancyGrid2D *obj)
    {
        post(
            boost::bind(
                &RendererRelay::Impl::add_og2d,
                this, obj->get_object_id(), obj->get_cell_size()));

        connect_scene_object_slots(obj);

        record_connection(
            obj, obj->cells_set_signal().connect(
                boost::bind(
                    &RendererRelay::Impl::on_og2d_cells_set,
                    this, obj, _1)));

        record_connection(
            obj, obj->unoccupied_color_set_signal().connect(
                boost::bind(
                    &RendererRelay::Impl::on_og2d_unoccopied_color_set,
                    this, obj)));

        record_connection(
            obj, obj->occupied_color_set_signal().connect(
                boost::bind(
                    &RendererRelay::Impl::on_og2d_occopied_color_set,
                    this, obj)));

        on_og2d_unoccopied_color_set(obj);
        on_og2d_occopied_color_set(obj);
    }

    // Executed in the GUI thread
    void add_og2d(ObjectID id, float cell_size)
    {
        add_object_entity(
            id, new renderer::OccupancyGrid2D(cell_size, 1024*256));
    }

    void on_og2d_cells_set(
        const OccupancyGrid2D *obj,
        const std::vector<std::pair<Eigen::Vector2f, float> > &cells)
    {
        post(
            boost::bind(
                &renderer::OccupancyGrid2D::set_cells,
                boost::bind(
                    &RendererRelay::Impl::get_object_entity<renderer::OccupancyGrid2D>,
                    this, obj->get_object_id()),
                cells));
    }

    void on_og2d_unoccopied_color_set(const OccupancyGrid2D *obj)
    {
        post(
            boost::bind(
                &renderer::OccupancyGrid2D::set_unoccupied_color,
                boost::bind(
                    &RendererRelay::Impl::get_object_entity<renderer::OccupancyGrid2D>,
                    this, obj->get_object_id()),
                obj->get_unoccupied_color()));
    }

    void on_og2d_occopied_color_set(const OccupancyGrid2D *obj)
    {
        post(
            boost::bind(
                &renderer::OccupancyGrid2D::set_occupied_color,
                boost::bind(
                    &RendererRelay::Impl::get_object_entity<renderer::OccupancyGrid2D>,
                    this, obj->get_object_id()),
                obj->get_occupied_color()));
    }

    virtual void visit(OccupancyGrid3D *obj)
    {
        post(
            boost::bind(
                &RendererRelay::Impl::add_og3d,
                this, obj->get_object_id(),
                obj->get_cell_xy_size(), obj->get_cell_z_size()));

        connect_scene_object_slots(obj);

        record_connection(
            obj, obj->cells_set_signal().connect(
                boost::bind(
                    &RendererRelay::Impl::on_og3d_cells_set,
                    this, obj, _1)));

        record_connection(
            obj, obj->color_mapping_set_signal().connect(
                boost::bind(
                    &RendererRelay::Impl::on_og3d_color_mapping_set,
                    this, obj)));

        record_connection(
            obj, obj->color_mapping_z_min_set_signal().connect(
                boost::bind(
                    &RendererRelay::Impl::on_og3d_color_mapping_set,
                    this, obj)));

        record_connection(
            obj, obj->color_mapping_z_max_set_signal().connect(
                boost::bind(
                    &RendererRelay::Impl::on_og3d_color_mapping_set,
                    this, obj)));

        on_og3d_color_mapping_set(obj);
    }

    // Executed in the GUI thread
    void add_og3d(ObjectID id, float cell_xy_size, float cell_z_size)
    {
        add_object_entity(
            id, new renderer::OccupancyGrid3D(cell_xy_size, cell_z_size));
    }

    void on_og3d_cells_set(
        const OccupancyGrid3D *obj,
        const std::vector<std::pair<Eigen::Vector3f, float> > &cells)
    {
        post(
            boost::bind(
                &renderer::OccupancyGrid3D::set_cells,
                boost::bind(
                    &RendererRelay::Impl::get_object_entity<renderer::OccupancyGrid3D>,
                    this, obj->get_object_id()),
                cells));
    }

    void on_og3d_color_mapping_set(const OccupancyGrid3D *obj)
    {
        post(
            boost::bind(
                &renderer::OccupancyGrid3D::set_color_mapping,
                boost::bind(
                    &RendererRelay::Impl::get_object_entity<renderer::OccupancyGrid3D>,
                    this, obj->get_object_id()),
                obj->is_color_mapping_enabled(),
                obj->get_color_mapping_z_min(),
                obj->get_color_mapping_z_max()));
    }

    virtual void visit(Label *obj)
    {
        post(
            boost::bind(
                &RendererRelay::Impl::add_label,
                this, obj->get_object_id()));

        connect_scene_object_slots(obj);

        record_connection(
            obj, obj->text_set_signal().connect(
                boost::bind(
                    &RendererRelay::Impl::on_label_text_set,
                    this, obj)));

        record_connection(
            obj, obj->alignment_set_signal().connect(
                boost::bind(
                    &RendererRelay::Impl::on_label_alignment_set,
                    this, obj)));

        on_label_text_set(obj);
        on_label_alignment_set(obj);
    }

    // Executed in the GUI thread
    void add_label(ObjectID id)
    {
        add_object_entity(id, new renderer::Label(m_gui.get_config()));
    }

    void on_label_text_set(const Label *obj)
    {
        post(
            boost::bind(
                &renderer::Label::set_text,
                boost::bind(
                    &RendererRelay::Impl::get_object_entity<renderer::Label>,
                    this, obj->get_object_id()),
                obj->get_text()));
    }

    void on_label_alignment_set(const Label *obj)
    {
        post(
            boost::bind(
                &renderer::Label::set_alignment,
                boost::bind(
                    &RendererRelay::Impl::get_object_entity<renderer::Label>,
                    this, obj->get_object_id()),
                obj->get_alignment()));
    }

    virtual void visit(TriMesh *obj)
    {
        post(
            boost::bind(
                &RendererRelay::Impl::add_generic<renderer::TriMeshEntity>,
                this, obj->get_object_id()));

        connect_scene_object_slots(obj);

        record_connection(
            obj, obj->indices_set_signal().connect(
                boost::bind(
                    &RendererRelay::Impl::on_tri_mesh_indices_set,
                    this, obj)));

        record_connection(
            obj, obj->indices_added_signal().connect(
                boost::bind(
                    &RendererRelay::Impl::on_tri_mesh_indices_added,
                    this, obj, _1)));

        if( !obj->get_indices().empty() )
            on_tri_mesh_indices_set(obj);
    }

    void on_tri_mesh_indices_set(const TriMesh *obj)
    {
        post(
            boost::bind(
                &renderer::TriMeshEntity::set_indices,
                boost::bind(
                    &RendererRelay::Impl::get_object_entity<renderer::TriMeshEntity>,
                    this, obj->get_object_id()),
                obj->get_indices()));
    }

    void on_tri_mesh_indices_added(
        const TriMesh *obj,
        const std::vector<boost::uint32_t> &indices)
    {
        post(
            boost::bind(
                &renderer::TriMeshEntity::add_indices,
                boost::bind(
                    &RendererRelay::Impl::get_object_entity<renderer::TriMeshEntity>,
                    this, obj->get_object_id()),
                indices));
    }

    void record_connection(
        const SceneObject *obj,
        const boost::signals2::connection &connection)
    {
        m_connection_map[obj].push_back(connection);
    }

    void populate_scene(ServerData &sd)
    {
        on_object_attached(sd.m_scene->get_root());
    }

    void connect_scene_object_slots(const SceneObject *obj)
    {
        record_connection(
            obj, obj->child_attached_signal().connect(
                boost::bind(
                    &RendererRelay::Impl::on_object_attached, this, _1)));

        record_connection(
            obj, obj->detached_signal().connect(
                boost::bind(
                    &RendererRelay::Impl::on_object_detached, this, obj)));

        record_connection(
            obj, obj->mtow_changed_signal().connect(
                boost::bind(
                    &RendererRelay::Impl::on_mtow_changed,
                    this, obj)));

        record_connection(
            obj, obj->accum_opacity_changed_signal().connect(
                boost::bind(
                    &RendererRelay::Impl::on_accum_opacity_changed,
                    this, obj)));

        record_connection(
            obj, obj->color_set_signal().connect(
                boost::bind(
                    &RendererRelay::Impl::on_color_set, this, obj)));

        record_connection(
            obj, obj->layer_set_signal().connect(
                boost::bind(
                    &RendererRelay::Impl::on_layer_set, this, obj)));

        record_connection(
            obj, obj->visibility_changed_signal().connect(
                boost::bind(
                    &RendererRelay::Impl::on_visibility_changed,
                    this, obj)));

        record_connection(
            obj, obj->selected_set_signal().connect(
                boost::bind(
                    &RendererRelay::Impl::on_selected_set, this, obj)));

        record_connection(
            obj, obj->has_selected_ancestor_changed_signal().connect(
                boost::bind(
                    &RendererRelay::Impl::on_has_selected_ancestor_changed,
                    this, obj)));

        on_mtow_changed(obj);
        on_accum_opacity_changed(obj);
        on_color_set(obj);
        on_layer_set(obj);
        on_visibility_changed(obj);
        on_selected_set(obj);
        on_has_selected_ancestor_changed(obj);

        // Set the renderer entity's "name", used for picking, equal to its
        // corresponding object id.
        post(
            boost::bind(
                &renderer::ObjectEntity::set_name,
                boost::bind(
                    &RendererRelay::Impl::get_object_entity<renderer::ObjectEntity>,
                    this, obj->get_object_id()),
                obj->get_object_id()));

        // Set up parent pointers for all non-root objects
        if( obj->get_parent() )
        {
            post(
                boost::bind(
                    &renderer::ObjectEntity::set_parent,
                    boost::bind(
                        &RendererRelay::Impl::get_object_entity<renderer::ObjectEntity>,
                        this, obj->get_object_id()),
                    boost::bind(
                        &RendererRelay::Impl::get_object_entity<renderer::ObjectEntity>,
                        this, obj->get_parent()->get_object_id())));
        }

        const ScalableObject *scalable =
            dynamic_cast<const ScalableObject *>(obj);
        if( scalable )
        {
            record_connection(
                obj, scalable->scale_set_signal().connect(
                    boost::bind(
                        &RendererRelay::Impl::on_mtow_changed,
                        this, obj)));
        }

        const VertexObject *vb = dynamic_cast<const VertexObject *>(obj);
        if( vb )
        {
            record_connection(
                obj, vb->vertices_set_signal().connect(
                    boost::bind(
                        &RendererRelay::Impl::on_vertices_set,
                        this, obj)));

            record_connection(
                obj, vb->vertices_added_signal().connect(
                    boost::bind(
                        &RendererRelay::Impl::on_vertices_added,
                        this, obj, _1, _2)));

            on_vertices_set(obj);
        }

        const LineBased *lb = dynamic_cast<const LineBased *>(obj);
        if( lb )
        {
            record_connection(
                obj, lb->line_width_set_signal().connect(
                    boost::bind(
                        &RendererRelay::Impl::on_line_prop_set,
                        this, obj)));

            record_connection(
                obj, lb->line_style_set_signal().connect(
                    boost::bind(
                        &RendererRelay::Impl::on_line_prop_set,
                        this, obj)));

            record_connection(
                obj, lb->stipple_factor_set_signal().connect(
                    boost::bind(
                        &RendererRelay::Impl::on_line_prop_set,
                        this, obj)));

            on_line_prop_set(obj);
        }
    }

    void on_mtow_changed(const SceneObject *obj)
    {
        const ScalableObject *scalable(
            dynamic_cast<const ScalableObject *>(obj));

        Eigen::Transform3f m;

        if( scalable )
        {
            m = obj->get_mtow();
            m.matrix().col(0) *= scalable->get_x_scale();
            m.matrix().col(1) *= scalable->get_y_scale();
            m.matrix().col(2) *= scalable->get_z_scale();
        }
        else
        {
            m = obj->get_mtow();
        }

        post(
            boost::bind(
                &renderer::ObjectEntity::set_transformation,
                boost::bind(
                    &RendererRelay::Impl::get_object_entity<renderer::ObjectEntity>,
                    this, obj->get_object_id()), m));
    }

    void on_accum_opacity_changed(const SceneObject *obj)
    {
        post(
            boost::bind(
                &renderer::ObjectEntity::set_opacity,
                boost::bind(
                    &RendererRelay::Impl::get_object_entity<renderer::ObjectEntity>,
                    this, obj->get_object_id()),
                obj->get_accumulated_opacity()));
    }

    void on_color_set(const SceneObject *obj)
    {
        post(
            boost::bind(
                &renderer::ObjectEntity::set_color,
                boost::bind(
                    &RendererRelay::Impl::get_object_entity<renderer::ObjectEntity>,
                    this, obj->get_object_id()),
                obj->get_color()));
    }

    void on_layer_set(const SceneObject *obj)
    {
        post(
            boost::bind(
                &renderer::ObjectEntity::set_layer,
                boost::bind(
                    &RendererRelay::Impl::get_object_entity<renderer::ObjectEntity>,
                    this, obj->get_object_id()),
                obj->get_layer()));
    }

    void on_visibility_changed(const SceneObject *obj)
    {
        post(
            boost::bind(
                &renderer::ObjectEntity::set_visibility,
                boost::bind(
                    &RendererRelay::Impl::get_object_entity<renderer::ObjectEntity>,
                    this, obj->get_object_id()),
                obj->is_visible()));
    }

    void on_selected_set(const SceneObject *obj)
    {
        post(
            boost::bind(
                &renderer::ObjectEntity::set_selected,
                boost::bind(
                    &RendererRelay::Impl::get_object_entity<renderer::ObjectEntity>,
                    this, obj->get_object_id()),
                obj->is_selected()));
    }

    void on_has_selected_ancestor_changed(const SceneObject *obj)
    {
        post(
            boost::bind(
                &renderer::ObjectEntity::set_has_selected_ancestor,
                boost::bind(
                    &RendererRelay::Impl::get_object_entity<renderer::ObjectEntity>,
                    this, obj->get_object_id()),
                obj->has_selected_ancestor()));
    }

    void on_object_attached(const SceneObject *obj)
    {
        const_cast<SceneObject *>(obj)->accept(this);

        for( SceneObject::ConstChildIterator it = obj->begin();
             it != obj->end(); ++it )
        {
            on_object_attached(*it);
        }
    }

    void on_object_detached(const SceneObject *obj)
    {
        // Disconnect all slots for the object
        ConnectionMap::iterator conn_it = m_connection_map.find(obj);

        if( conn_it != m_connection_map.end() )
        {
            Connections &connections = conn_it->second;

            for( Connections::iterator it = connections.begin();
                 it != connections.end(); ++it )
            {
                it->disconnect();
            }

            m_connection_map.erase(conn_it);
        }

        // Remove the object entity associated with the object, if there is one
        post(
            boost::bind(
                &RendererRelay::Impl::remove_entity,
                this, obj->get_object_id()));

        // Disconnect slots and remove entities for child objects
        for( SceneObject::ConstChildIterator it = obj->begin();
             it != obj->end(); ++it )
        {
            on_object_detached(*it);
        }
    }

    void on_vertices_set(const SceneObject *obj)
    {
        const VertexObject *vobj = dynamic_cast<const VertexObject *>(obj);

        if( vobj->has_vertex_colors() )
        {
            post(
                boost::bind(
                    &renderer::VertexBased::set_colored_vertices,
                    boost::bind(
                        &RendererRelay::Impl::get_object_entity<
                            renderer::VertexBased>,
                        this, obj->get_object_id()),
                    vobj->get_vertices(), vobj->get_vertex_colors()));
        }
        else
        {
            post(
                boost::bind(
                    &renderer::VertexBased::set_vertices,
                    boost::bind(
                        &RendererRelay::Impl::get_object_entity<
                            renderer::VertexBased>,
                        this, obj->get_object_id()),
                    vobj->get_vertices()));
        }
    }

    void on_vertices_added(
        const SceneObject *obj,
        const VertexObject::Vertices &new_vertices,
        const VertexObject::VertexColors *new_colors)
    {
        const VertexObject *vobj = dynamic_cast<const VertexObject *>(obj);

        if( vobj->has_vertex_colors() )
        {
            post(
                boost::bind(
                    &renderer::VertexBased::add_colored_vertices,
                    boost::bind(
                        &RendererRelay::Impl::get_object_entity<
                            renderer::VertexBased>,
                        this, obj->get_object_id()),
                    new_vertices, vobj->get_vertices(),
                    *new_colors, vobj->get_vertex_colors()));
        }
        else
        {
            post(
                boost::bind(
                    &renderer::VertexBased::add_vertices,
                    boost::bind(
                        &RendererRelay::Impl::get_object_entity<
                            renderer::VertexBased>,
                        this, obj->get_object_id()),
                    new_vertices, vobj->get_vertices()));
        }
    }

    void on_line_prop_set(const SceneObject *obj)
    {
        const LineBased *lb = dynamic_cast<const LineBased *>(obj);

        post(
            boost::bind(
                &renderer::LineBased::set_line_style,
                boost::bind(
                    &RendererRelay::Impl::get_object_entity<renderer::LineBased>,
                    this, obj->get_object_id()),
                lb->get_line_width(),
                (boost::uint16_t)lb->get_line_style(),
                lb->get_stipple_factor()));
    }

    /// @}
    /// \name Methods executed in the GUI thread
    /// @{

    template<typename T> T *get_object_entity(ObjectID id)
    {
        ObjectEntityMap::iterator it = m_object_entities.find(id);
        if( it == m_object_entities.end() )
            return 0;
        else
            return dynamic_cast<T *>(it->second);
    }

    void add_object_entity(ObjectID id, renderer::ObjectEntity *oe)
    {
        m_object_entities.insert(std::make_pair(id, oe));
        m_renderer->get_scene_manager().add_entity(oe);
    }

    template<typename T> void add_generic(ObjectID id)
    {
        add_object_entity(id, new T);
    }

    void remove_entity(ObjectID id)
    {
        ObjectEntityMap::iterator it = m_object_entities.find(id);

        if( it != m_object_entities.end() )
        {
            renderer::ObjectEntity *oe = it->second;
            m_renderer->get_scene_manager().remove_entity(oe);
            delete oe;
            m_object_entities.erase(it);
        }
    }

    bool fps_timeout()
    {
        boost::posix_time::ptime now =
            boost::posix_time::microsec_clock::local_time();

        m_fps_updated_signal(
            1000.0f * m_fps_counter / (now-m_fps_t).total_milliseconds());

        m_fps_t = now;
        m_fps_counter = 0;

        return true;
    }

    /// @}

private:
    Gui &m_gui;

    WorkQueue m_work_queue;

    renderer::RenderEngine *m_renderer;

    typedef std::map<ObjectID, renderer::ObjectEntity *> ObjectEntityMap;

    // Accessed only in the GUI thread!
    ObjectEntityMap m_object_entities;

    typedef std::vector<boost::signals2::connection> Connections;
    typedef std::map<const SceneObject *, Connections> ConnectionMap;

    // Accessed only in the server thread!
    ConnectionMap m_connection_map;

    renderer::TransformationGuides *m_guides;

    int m_fps_counter;
    boost::posix_time::ptime m_fps_t;
    sigc::connection m_fps_timeout_conn;

    mutable FpsUpdatedSignal m_fps_updated_signal;
};

//
// ---------------------------------------------------------------
//

RendererRelay::RendererRelay(Gui &gui)
    : m_impl(0)
{
    m_impl = new Impl(gui);
}


RendererRelay::~RendererRelay()
{
    if( m_impl )
    {
        delete m_impl;
        m_impl = 0;
    }
}


void RendererRelay::render(
    unsigned int w, unsigned int h, ObjectID camera_id,
    const bool *visible_layers)
{
    m_impl->render(w, h, camera_id, visible_layers);
}


ObjectID RendererRelay::pick(
    unsigned int w, unsigned int h, ObjectID camera_id,
    const bool *visible_layers,
    int x, int y, float sensitivity)
{
    return m_impl->pick(w, h, camera_id, visible_layers, x, y, sensitivity);
}


void RendererRelay::init()
{
    m_impl->init();
}

void RendererRelay::set_tg_mode(renderer::TransformationGuidesMode mode)
{
    m_impl->set_tg_mode(mode);
}


void RendererRelay::set_tg_coord_sys(CoordinateSystem coord_sys)
{
    m_impl->set_tg_coord_sys(coord_sys);
}


void RendererRelay::set_tg_pivot(RotationCenterpoint pivot)
{
    m_impl->set_tg_pivot(pivot);
}


void RendererRelay::set_tg_axes(Axis axes)
{
    m_impl->set_tg_axes(axes);
}


void RendererRelay::set_max_fps(float fps)
{
    m_impl->set_max_fps(fps);
}


void RendererRelay::set_background_color(const RGBColor &color)
{
    m_impl->set_background_color(color);
}


void RendererRelay::set_selected_color(const RGBColor &color)
{
    m_impl->set_selected_color(color);
}


void RendererRelay::set_ancestor_selected_color(const RGBColor &color)
{
    m_impl->set_ancestor_selected_color(color);
}


RendererRelay::FpsUpdatedSignal &RendererRelay::fps_updated_signal() const
{
    return m_impl->fps_updated_signal();
}
