/*
 * Copyright Staffan Gimåker 2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "NavigateTool.hh"
#include "Gui.hh"
#include "ToolController.hh"
#include "SceneViewFrame.hh"
#include "../Server.hh"
#include "../CameraObject.hh"
#include "../SceneTree.hh"


using namespace peekabot;
using namespace peekabot::gui;


NavigateTool::NavigateTool(Gui &gui)
    : m_gui(gui)
{
}


void NavigateTool::button_click(
    SceneViewFrame *sf, const GdkEventButton *event)
{
    if( event->button == 1 )
    {
        // Left button: Pick
        ObjectID id = sf->pick(event->x, event->y);

        if( id != 0 )
        {
            if( event->state & GDK_SHIFT_MASK )
            {
                m_gui.set_selected(id, true);
            }
            else if( event->state & GDK_CONTROL_MASK )
            {
                m_gui.toggle_select(id);
            }
            else
            {
                Gui::Selection sel;
                sel.insert(id);
                m_gui.set_selection(sel);
            }
        }
    }
    else if( event->button == 3 )
    {
        // Right button: Clear selection
        m_gui.clear_selection();
    }
}


void NavigateTool::mouse_drag(
    SceneViewFrame *sf, const GdkEventMotion *event,
    double dx, double dy)
{
    float rate_mod = 1;
    if( event->state & GDK_CONTROL_MASK )
    {
        rate_mod /= 5;
    }

    if( event->state & GDK_BUTTON1_MASK )
    {
        m_gui.server_post(
            boost::bind(&NavigateTool::camera_pan, this,
                        _1, sf->get_camera(), dx, dy, rate_mod));
    }
    else if( event->state & GDK_BUTTON3_MASK )
    {
        m_gui.server_post(
            boost::bind(&NavigateTool::camera_rotate, this,
                        _1, sf->get_camera(), dx, dy, rate_mod));
    }
    else if( event->state & GDK_BUTTON2_MASK )
    {
        m_gui.server_post(
            boost::bind(&NavigateTool::camera_zoom, this,
                        _1, sf->get_camera(), dy, rate_mod, true));
    }
}


void NavigateTool::scroll(
    SceneViewFrame *sf, const GdkEventScroll *event)
{
    float rate_mod = 1;
    if( event->state & GDK_CONTROL_MASK )
    {
        rate_mod /= 5;
    }

    if( event->direction == GDK_SCROLL_UP )
    {
        m_gui.server_post(
            boost::bind(&NavigateTool::camera_zoom, this,
                        _1, sf->get_camera(), -10, rate_mod, true));
    }
    else if( event->direction == GDK_SCROLL_DOWN )
    {
        m_gui.server_post(
            boost::bind(&NavigateTool::camera_zoom, this,
                        _1, sf->get_camera(), 10, rate_mod, true));
    }
}


void NavigateTool::camera_pan(
    ServerData &sd, ObjectID cam_id, float dx, float dy, float rate_mod)
{
    float rate = 1e-3 * rate_mod;

    CameraObject *cam = dynamic_cast<CameraObject *>(
        sd.m_scene->get_object(cam_id));

    if( cam )
    {
        Eigen::Transform3f m = cam->get_transformation();

        m.translation() -= (
            rate * dx * cam->get_fov() * -m.linear().col(1) -
            rate * dy * cam->get_fov() * m.linear().col(2));

        cam->set_transformation(m);
    }
}


void NavigateTool::camera_rotate(
    ServerData &sd, ObjectID cam_id, float dx, float dy, float rate_mod)
{
    float rate = 0.2e-3 * rate_mod;

    CameraObject *cam = dynamic_cast<CameraObject *>(
        sd.m_scene->get_object(cam_id));

    if( cam )
    {
        Eigen::Transform3f m = cam->get_mtow();
        Eigen::Vector3f trans = m.translation();
        m.translation() = Eigen::Vector3f(0,0,0);

        m = (Eigen::AngleAxisf(
                 rate * -dx * cam->get_fov(), Eigen::Vector3f::UnitZ()) *
             Eigen::AngleAxisf(
                 rate * dy * cam->get_fov(), cam->get_mtow().linear().col(1)) *
             m);
        m.translation() = trans;

        cam->set_transformation(m, WORLD_COORDINATES);
    }
}


void NavigateTool::camera_zoom(
    ServerData &sd, ObjectID cam_id, float dy, float rate_mod, bool rel)
{
    float rate = 1e-2 * rate_mod;

    CameraObject *cam = dynamic_cast<CameraObject *>(
        sd.m_scene->get_object(cam_id));

    if( cam )
    {
        if( rel )
        {
            float delta = cam->get_zoom_distance() * rate * dy;
            if( delta > 2 )
                delta = 2;
            else if( delta < -2 )
                delta = -2;

            cam->set_zoom_distance(
                std::max(0.01f, cam->get_zoom_distance() + delta));
        }
        else
        {
            cam->set_zoom_distance(std::max(0.01f, dy));
        }
    }
}


void NavigateTool::key_release(
    SceneViewFrame *sf, const GdkEventKey *event)
{
    if( event->keyval == GDK_g || event->keyval == GDK_m )
    {
        m_gui.get_tool_controller().activate_move_tool();
    }
    else if( event->keyval == GDK_r )
    {
        m_gui.get_tool_controller().activate_rotate_tool();
    }
}
