/*
 * Copyright Staffan Gimåker 2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "LayerPropMediator.hh"
#include "../SceneObject.hh"
#include "../SceneTree.hh"
#include "../props/LayerPropAdapter.hh"
#include "../Server.hh"

#include <stack>
#include <cassert>
#include <boost/bind.hpp>
#include <boost/lexical_cast.hpp>


using namespace peekabot;
using namespace peekabot::gui;


namespace
{
    void propagate_layer(
        ServerData &sd, ObjectID object_id, boost::uint8_t layer)
    {
        SceneObject *obj = sd.m_scene->get_object(object_id);
        if( obj )
        {
            std::stack<SceneObject *> s;
            for( SceneObject::ChildIterator it = obj->begin();
                 it != obj->end(); ++it )
                s.push(*it);

            while( !s.empty() )
            {
                SceneObject *p = s.top();
                s.pop();
                p->set_layer(layer);
                for( SceneObject::ChildIterator it = p->begin();
                     it != p->end(); ++it )
                {
                    s.push(*it);
                }
            }
        }
    }
}


LayerPropMediator::LayerPropMediator(
    PropInspectorController &pi, SceneObject *obj,
    LayerPropAdapter *prop, PropKey prop_key)
    : PropMediator(pi),
      m_obj(obj),
      m_prop(prop),
      m_object_id(obj->get_object_id()),
      m_prop_key(prop_key),
      m_n_updates_queued(0),
      m_hbox(0), m_combo_box(0),
      m_propagate_btn(0), m_propagate_img(0),
      m_lbl(0)
{
    // Create and init widget
    post(
        boost::bind(&LayerPropMediator::create_widget, this,
                    m_obj->get_prop_name(m_prop_key),
                    any_cast<boost::uint8_t>(m_prop->get(m_obj))));
    // Connect slots
    prop->signal(m_obj).connect(
        boost::bind(&LayerPropMediator::on_prop_set, this));
}


// Must be executed in the server thread
void LayerPropMediator::destroy()
{
    // Post to destroy the GUI widget
    post(
        boost::bind(&LayerPropMediator::destroy_widget, this));
    // Disconnect slots
    m_prop->signal(m_obj).disconnect(
        boost::bind(&LayerPropMediator::on_prop_set, this));
}


// Executed in the GUI thread
void LayerPropMediator::create_widget(std::string prop_name, boost::uint8_t val)
{
    assert( !m_hbox && !m_combo_box && !m_propagate_btn && !m_lbl );

    m_combo_box = new Gtk::ComboBoxText();
    for( unsigned int i = 1; i <= NUMBER_OF_LAYERS; ++i )
        m_combo_box->append_text(boost::lexical_cast<std::string>(i));
    m_combo_box->set_active(val);
    m_widget_set_conn = m_combo_box->signal_changed().connect(
        sigc::mem_fun(*this, &LayerPropMediator::on_widget_set));

    m_propagate_img = new Gtk::Image(Gtk::Stock::OK, Gtk::ICON_SIZE_BUTTON);
    m_propagate_btn = new Gtk::Button();
    m_propagate_btn->set_image(*m_propagate_img);
    m_propagate_btn->set_tooltip_text(
        "Propagate the layer setting to all descendants");
    m_propagate_btn->signal_clicked().connect(
        sigc::mem_fun(*this, &LayerPropMediator::on_propagate_clicked));

    m_hbox = new Gtk::HBox();
    m_hbox->pack_start(*m_combo_box, true, true);
    m_hbox->pack_start(*m_propagate_btn, false, false);

    m_lbl = new Gtk::Label(prop_name + ":", 1.0, 0.5);

    add_prop_widgets(m_lbl, m_hbox);
}


// Executed in the GUI thread
void LayerPropMediator::destroy_widget()
{
    assert( m_hbox && m_combo_box && m_propagate_btn && m_lbl );

    erase_prop_widgets(m_lbl, m_hbox);
    delete m_hbox;
    m_hbox = 0;
    delete m_combo_box;
    m_combo_box = 0;
    delete m_propagate_btn;
    m_propagate_btn = 0;
    delete m_propagate_img;
    m_propagate_img = 0;
    delete m_lbl;
    m_lbl = 0;

    delete this;
}


// Executed in the server thread
void LayerPropMediator::on_prop_set()
{
    boost::uint8_t val = any_cast<boost::uint8_t>(m_prop->get(m_obj));
    post(
        boost::bind(
            &LayerPropMediator::update_widget, this, val));
}


// Executed in the GUI thread
void LayerPropMediator::update_widget(boost::uint8_t val)
{
    // If there are more than one widget update queued up, skip all but the
    // last one
    m_n_updates_queued = std::max(m_n_updates_queued-1, 0);
    if( m_n_updates_queued > 0 )
        return;

    m_widget_set_conn.block();
    m_combo_box->set_active(val);
    m_widget_set_conn.unblock();
}


// Executed in the GUI thread
void LayerPropMediator::on_widget_set()
{
    ++m_n_updates_queued;
    set_prop(
        m_object_id, m_prop_key,
        boost::uint8_t(m_combo_box->get_active_row_number()));
}


void LayerPropMediator::on_propagate_clicked()
{
    server_post(
        boost::bind(&propagate_layer, _1, m_object_id,
                    m_combo_box->get_active_row_number()));
}
