/*
 * Copyright Staffan Gimåker 2010.
 *
 * ---
 *
 * Distributed under the Boost Software License, Version 1.0.
 * (See accompanying file LICENSE_1_0.txt or copy at
 * http://www.boost.org/LICENSE_1_0.txt)
 */

#include "TriMeshProxy.hh"
#include "../PeekabotClient.hh"
#include "../../ObjectTypes.hh"

#include "../../actions/AddObject.hh"
#include "../../actions/Assign.hh"
#include "../../actions/SetIndices.hh"


using namespace peekabot;
using namespace peekabot::client;


//
// ------------------ IndexSet implementation --------------------
//


struct IndexSet::Impl
{
    inline void add_triangle(
        boost::uint32_t v1,
        boost::uint32_t v2,
        boost::uint32_t v3)
    {
        m_inds.push_back(v1);
        m_inds.push_back(v2);
        m_inds.push_back(v3);
    }

    inline void clear()
    {
        m_inds.clear();
    }

    inline bool empty() const
    {
        return m_inds.empty();
    }

    std::vector<boost::uint32_t> m_inds;
};


IndexSet::IndexSet()
    : m_impl(new IndexSet::Impl)
{
}

IndexSet::IndexSet(const IndexSet &other)
    : m_impl(new Impl(*other.m_impl))
{
}

IndexSet::~IndexSet()
{
}

IndexSet &IndexSet::operator=(const IndexSet &other)
{
    m_impl.reset(new Impl(*other.m_impl));
    return *this;
}

void IndexSet::add_triangle(
    boost::uint32_t v1,
    boost::uint32_t v2,
    boost::uint32_t v3)
{
    m_impl->add_triangle(v1, v2, v3);
}

void IndexSet::add_quad(
    boost::uint32_t v1,
    boost::uint32_t v2,
    boost::uint32_t v3,
    boost::uint32_t v4)
{
    m_impl->add_triangle(v1, v2, v4);
    m_impl->add_triangle(v4, v2, v3);
}

void IndexSet::add_convex_polygon(
    const std::vector<boost::uint32_t> &inds)
{
    if( inds.size() < 3 )
        throw std::runtime_error(
            "Invalid polygon - must contain at least 3 vertices");

    // Triangulate the polygon
    for( std::size_t i = 1; i < inds.size()-1; ++i )
        add_triangle(0, i, i+1);
}

void IndexSet::clear()
{
    m_impl->clear();
}

bool IndexSet::empty() const
{
    return m_impl->empty();
}


//
// ------------------ TriMeshProxyBase implementation -------------------
//


TriMeshProxyBase::TriMeshProxyBase()
{
}


TriMeshProxyBase::TriMeshProxyBase(TriMeshProxyBase &p)
    : ObjectProxyBase(p),
      ScalableProxyBase(p),
      VertexBasedProxyBase(p)
{
}


DelayedDispatch TriMeshProxyBase::set_indices(const IndexSet &indices)
{
    return DelayedDispatch(
        get_client_impl(),
        new SetIndices(get_object_id(), indices.m_impl->m_inds));
}


DelayedDispatch TriMeshProxyBase::add_indices(const IndexSet &indices)
{
    return DelayedDispatch(
        get_client_impl(),
        new SetIndices(get_object_id(), indices.m_impl->m_inds, true));
}


//
// ------------------ TriMeshProxy implementation -------------------
//


TriMeshProxy::TriMeshProxy()
{
}


TriMeshProxy::TriMeshProxy(TriMeshProxyBase &p)
    : ObjectProxyBase(p),
      TriMeshProxyBase(p)
{
}


TriMeshProxy &TriMeshProxy::operator=(const TriMeshProxy &p)
{
    return *this = (TriMeshProxyBase &)p;
}


TriMeshProxy &TriMeshProxy::operator=(const TriMeshProxyBase &p)
{
    unchecked_assign(unchecked_get_client_impl(p), get_pseudonym(p));
    return *this;
}


DelayedDispatch TriMeshProxy::assign(const ObjectProxyBase &p)
{
    unchecked_assign(get_client_impl(p), allocate_pseudonym());

    return DelayedDispatch(
        get_client_impl(),
        new Assign(PathIdentifier(get_object_id(p)),
                   get_object_id(), TRI_MESH_OBJECT));
}


DelayedDispatch TriMeshProxy::assign(
    PeekabotClient &client, const std::string &path)
{
    unchecked_assign(get_client_impl(client), allocate_pseudonym());

    return DelayedDispatch(
        get_client_impl(),
        new Assign(PathIdentifier(path), get_object_id(), TRI_MESH_OBJECT));
}


DelayedDispatch TriMeshProxy::assign(
    const ObjectProxyBase &parent,
    const std::string &rel_path)
{
    unchecked_assign(get_client_impl(parent), allocate_pseudonym());

    return DelayedDispatch(
        get_client_impl(), new Assign(
            PathIdentifier(get_object_id(parent), rel_path),
            get_object_id(), TRI_MESH_OBJECT));
}


DelayedDispatch TriMeshProxy::add(
    PeekabotClient &client,
    const std::string &path,
    NameConflictPolicy conflict_policy)
{
    unchecked_assign(get_client_impl(client), allocate_pseudonym());

    return DelayedDispatch(
        get_client_impl(),
        new AddObject(PathIdentifier(path), conflict_policy,
                      get_object_id(), TRI_MESH_OBJECT));
}


DelayedDispatch TriMeshProxy::add(
    const ObjectProxyBase &parent,
    const std::string &name,
    NameConflictPolicy conflict_policy)
{
    unchecked_assign(get_client_impl(parent), allocate_pseudonym());

    return DelayedDispatch(
        get_client_impl(),
        new AddObject(PathIdentifier(get_object_id(parent), name),
                      conflict_policy, get_object_id(), TRI_MESH_OBJECT));
}
